const { Inventory } = require('engine/managers/Inventory');
const { Audio } = require('engine/managers/Audio');

//-----------------------------------------------------------------------------
// Game_Map
//
// The game object class for a map. It contains scrolling and passage
// determination functions.


Objects.Map = class Map {
  constructor() {
    this.initialize(...arguments);
  }

  get zoom() {
    return this._zoom;
  }

  initialize() {
    this._interpreter = new Objects.Interpreter();
    this._mapId = 0;
    this._realMapId = 0;
    this._tilesetId = 0;
    this._events = [];
    this._farmObjectsXy = {};
    this._villagers = [];
    this._creatures = [];
    this._commonEvents = [];
    this._displayX = 0;
    this._displayY = 0;
    this._nameDisplay = true;
    this._scrollDirection = 2;
    this._scrollRest = 0;
    this._scrollSpeed = 4;
    this._scrollTargetX = false;
    this._scrollTargetY = false;
    this._parallaxName = '';
    this._parallaxZero = false;
    this._parallaxLoopX = false;
    this._parallaxLoopY = false;
    this._parallaxSx = 0;
    this._parallaxSy = 0;
    this._parallaxX = 0;
    this._parallaxY = 0;
    this._buttonUpdateDelay = Utils.getFrameCount(10);
    this._grassUpdateDelay = Utils.getFrameCount(10);
    this._fullyLoaded = false;
    this._mouseDelay = 0;
    this._grassEventsPendingUpdate = [];
    this._eventsStarting = 0;
    this._allowRain = true;
    this._collisionBoxesXy = {};
    this._soundSources = [];
    this._soundDelay = 0;
    this._villagerUpdateDelay = 0;
    // this._creatureUpdateDelay = 0;

    const zoom = Utils.getResolutionZoomLevel(Graphics._resolution, Graphics._zoomType);
    this._zoom = new PIXI.Point(zoom, zoom);
  }

  restoreData(data) {
    const propertiesToSkip = ['_interpreter', '_zoom', '_commonEvents'];
    Utils.assignProperties(this, data, propertiesToSkip);

    if (data._interpreter) {
      Utils.assignProperties(this._interpreter, data._interpreter);
    }

    if (data._commonEvents) {
      this._commonEvents = [];

      data._commonEvents.forEach((commonEvent) => {
        const newEvent = new Objects.CommonEvent(commonEvent._commonEventId);

        // Utils.assignProperties(newEvent, commonEvent);
        this._commonEvents.push(newEvent);
      });
    }
  }

  getDesiredZoomType() {
    if (!Managers.Map.isZoomOutEnabled()) {
      return 'default';
    }

    return 'out';
  }

  setup(mapId) {
    if (!$dataMap) {
      throw new Error('The map data is not available');
    }

    this._fullyLoaded = false;
    this.setMapId(mapId);
    this._soundSources = [];
    this._soundDelay = 0;
    this._villagerUpdateDelay = 0;
    this._creatureUpdateDelay = 0;
    $gameTemp.clearCustomTiles();
    this.loadCollisionBoxes();

    const zoom = Utils.getResolutionZoomLevel(Graphics._resolution, this.getDesiredZoomType());
    this._zoom = new PIXI.Point(zoom, zoom);

    Managers.Map.updateMapLights();

    if ($dataMap.tilesetId <= 4) {
      $dataMap.tilesetId = Managers.Time.month;
    } else if ($dataMap.tilesetId >= 9 && $dataMap.tilesetId <= 16) {
      $dataMap.tilesetId = Math.floor(($dataMap.tilesetId - 1) / 4) * 4 + Managers.Time.month;
    }

    if ($dataMap.meta.norain) {
      this._allowRain = false;
    } else {
      this._allowRain = true;

    }
    this._tilesetId = $dataMap.tilesetId;

    if (!Managers.Map.isHomeMap(mapId)) {
      $gameTemp.setCurrentTvChannel(false);
    }

    this._displayX = 0;
    this._displayY = 0;

    $gameTemp.clearReservedAsyncCommonEvents();
    Managers.Mailbox.updateMailboxSwitch();

    this.setupEvents();
    this.setupScroll();
    this.setupParallax();
    this.setupCustomMapEvents();
    this._needsRefresh = false;

    Managers.Lighting.clear();
    Managers.Time.resetFrameCount();
    $gameTemp.requestFarmObjectsRefresh();
    this._fullyLoaded = true;
  }

  isMapFullyLoaded() {
    return this._fullyLoaded;
  }

  isRainAllowed() {
    return this._allowRain !== false;
  }

  getTileCollisionBox(boxType) {
    // 6..9 is the same as 0..3, except on winter
    if (boxType >= 6 && boxType <= 9) {
      if (Managers.Map.mapMonth($gameMap._mapId) == Seasons.WINTER) {
        return false;
      }

      boxType -= 6;
    }

    const list = {
      // WATER CORNERS:

      // diagonally blocked (bottom/left side is blocked, top/right side is free)
      0: [
        { x: 0, y: 0.25, width: 0.25, height: 0.75},
        { x: 0.25, y: 0.5, width: 0.25, height: 0.5},
      ],
      // diagonally blocked (bottom/right side is blocked, top/left side is free)
      1: [
        { x: 0.75, y: 0.25, width: 0.25, height: 0.75},
        { x: 0.5, y: 0.5, width: 0.25, height: 0.5},
      ],
      // diagonally blocked (top/left side is blocked, bottom/right side is free)
      2: [
        { x: 0, y: 0, width: 0.75, height: 0.25 },
        { x: 0, y: 0.25, width: 0.5, height: 0.25 },
        { x: 0, y: 0.5, width: 0.25, height: 0.25 },
      ],
      // diagonally blocked (top/right side is blocked, bottom/left side is free)
      3: [
        { x: 0.75, y: 0.5, width: 0.25, height: 0.25 },
        { x: 0.5, y: 0.25, width: 0.5, height: 0.25 },
        { x: 0.25, y: 0, width: 0.25, height: 0.25 },
      ],

      // MOUNTAIN CORNERS:

      // diagonally blocked (bottom/left side is blocked, top/right side is free)
      4: [
        { x: 0, y: 0, width: 0.25, height: 1},
        { x: 0.25, y: 0.25, width: 0.25, height: 0.75},
        { x: 0.5, y: 0.5, width: 0.25, height: 0.5},
        { x: 0.75, y: 0.75, width: 0.25, height: 0.25},
      ],
      // diagonally blocked (bottom/right side is blocked, top/left side is free)
      5: [
        { x: 0.75, y: 0, width: 0.25, height: 1 },
        { x: 0.5, y: 0.25, width: 0.25, height: 0.75},
        { x: 0.25, y: 0.5, width: 0.25, height: 0.5 },
        { x: 0, y: 0.75, width: 0.25, height: 0.25 },
      ],

      // WALL CORNERS:

      // block a small corner on the top left
      10: [
        { x: 0, y: 0, width: 0.25, height: 0.5 },
      ],

      // block a small corner on the top right
      11: [
        { x: 0.75, y: 0, width: 0.25, height: 0.5 },
      ],
    };

    return list[boxType];
  }

  loadCollisionBoxes() {
    const { width, height } = $dataMap;
    this._collisionBoxesXy = {};

    for (let x = 0; x < width; x++) {
      for (let y = 0; y < height; y++) {
        const regionId = $dataMap.data[(5 * height + y) * width + x] || 0;

        if (regionId >= Region.COLLISION_BOX_1 && regionId <= Region.COLLISION_BOX_12) {
          if (!this._collisionBoxesXy[x]) {
            this._collisionBoxesXy[x] = {};
          }

          this._collisionBoxesXy[x][y] = this.getTileCollisionBox(regionId - Region.COLLISION_BOX_1);
          continue;
        }
      }
    }
  }

  collisionBoxesXy(x, y) {
    if (!this._collisionBoxesXy[x]) return false;
    return this._collisionBoxesXy[x][y] || false;
  }

  fixPixelPositions() {
    const events = this.allEvents();
    const len = events.length;
    for (let i = 0; i < len; i++) {
      const event = events[i];
      if (!event) continue;
      if (event._erased) continue;

      event.fixPixelPositions();
    }


    $gamePlayer.fixPixelPositions();

    //#ToDo: Update followers


  }

  resetMap(destroyObjects) {
    $gameSystem.resetMap(this._mapId);

    if (destroyObjects) {
      Managers.FarmObjects._farmObjects[this._mapId] = [];
    }
  }

  isEventRunning() {
    return this._interpreter.isRunning() || this.isAnyEventStarting();
  }

  decreaseStartingFlag() {
    if (this._eventsStarting > 0) {
      this._eventsStarting--;
    }
  }

  increaseStartingFlag() {
    this._eventsStarting++;
  }

  tileWidth() {
    return Constants.TILE_SIZE;
  }

  tileHeight() {
    return Constants.TILE_SIZE;
  }

  mapId() {
    return this._mapId;
  }

  realMapId() {
    return this._realMapId;
  }

  setMapId(mapId) {
    this._realMapId = Managers.Map.getRealMapId(mapId);
    this._mapId = mapId;
  }

  tilesetId() {
    return this._tilesetId;
  }

  displayX() {
    return this._displayX;
  }

  displayY() {
    return this._displayY;
  }

  parallaxName() {
    return this._parallaxName;
  }

  requestRefresh() /*mapId*/{
    this._needsRefresh = true;
  }

  isNameDisplayEnabled() {
    return this._nameDisplay;
  }

  disableNameDisplay() {
    this._nameDisplay = false;
  }

  enableNameDisplay() {
    this._nameDisplay = true;
  }

  setupEvents() {
    this._farmObjectsXy = {};
    this._villagers = [];
    this._creatures = [];
    this._events = [];
    this._eventsStarting = 0;
    this._grassEventsPendingUpdate = [];

    for (let i = 0; i < $dataMap.events.length; i++) {
      const eventData = $dataMap.events[i];
      if (eventData) {
        const notes = eventData.note || '';
        if (notes.includes('<dynamic>')) {
          this._events[i] = new Objects.MapEvent(this._mapId, i);
        } else {
          this._events[i] = new Objects.StaticEvent(this._mapId, i);
        }
      }
    }
    this._commonEvents = this.parallelCommonEvents().map(commonEvent => new Objects.CommonEvent(commonEvent.id));

    const firstOfTheDay = !$gameSystem.hasMapGeneratedDailyEvents($gameMap._mapId);

    Managers.FarmObjects.setupMapEvents();
    Managers.Creatures.setupMapEvents();
    Managers.Villagers.refreshMapVillagers();
    Managers.Content.setupMapEvents(firstOfTheDay);

    const customEvents = $gameSystem.getCustomEvents(this._mapId);
    for (const eventId in customEvents) {
      if (customEvents[eventId] === undefined) continue;
      const newEventId = this.getIndexForNewEvent();

      customEvents[eventId].eventId = newEventId;
      this._events[newEventId] = new Objects.CustomEvent(this._mapId, newEventId, customEvents[eventId]);
    }

    this.refreshTileEvents();

    Managers.Map.afterSetupEvents(this._mapId);
  }

  addEventAt(eventData, x, y, temporary, classType, reservedId = undefined) {
    eventData.x = x;
    eventData.y = y;
    return this.addEvent(eventData, temporary, classType, reservedId);
  }

  makeShinyEvent(x, y) {
    if (Switches.isInsideFestival) return;
  }

  iterateFarmObjectEvents(callbackFn) {
    for (const x in this._farmObjectsXy) {
      if (!this._farmObjectsXy.hasOwnProperty(x)) {
        continue;
      }

      for (const y in this._farmObjectsXy[x]) {
        if (!this._farmObjectsXy[x].hasOwnProperty(y)) {
          continue;
        }

        const farmObjects = this._farmObjectsXy[x][y];
        const len = farmObjects.length;
        for (let i = 0; i < len; i++) {
          callbackFn(farmObjects[i]);
        }
      }
    }
  }

  farmObjectEvents(filterFn) {
    const list = [];

    for (const x in this._farmObjectsXy) {
      if (this._farmObjectsXy.hasOwnProperty(x)) {
        for (const y in this._farmObjectsXy[x]) {
          if (this._farmObjectsXy[x].hasOwnProperty(y)) {
            const farmObjects = this._farmObjectsXy[x][y];
            const len = farmObjects.length;
            for (let i = 0; i < len; i++) {
              if (!filterFn || filterFn(farmObjects[i])) {
                list.push(farmObjects[i]);
              }
            }
          }
        }
      }
    }

    return list;
  }

  villagerEvents() {
    return this._villagers.filter(event => !!event);
  }

  creatureEvents() {
    return this._creatures.filter(event => !!event);
  }

  villagerEvent(actorId) {
    return this._villagers[actorId];
  }

  creatureEvent(creatureId) {
    return this._creatures.filter(event => !!event && !!event._creatureData && event._creatureData.id == creatureId && !event._erased).shift();
  }

  villagerEventByName(villagerName) {
    const actorId = $gameActors.findActorId(villagerName);
    if (actorId === 0) return undefined;

    return this.villagerEvent(actorId);
  }

  eraseVillager(actorId, onlyRegulars) {
    if (this._villagers[actorId]) {
      if (!!this._villagers.isCutscene && !!onlyRegulars) {
        return;
      }
      this._villagers[actorId].erase();
    }
    this._villagers[actorId] = undefined;
  }

  removeVillagerFromList(actorId) {
    this._villagers[actorId] = undefined;
  }

  events() {
    return this._events.filter(event => !!event);
  }

  event(eventId) {
    return this._events[eventId];
  }

  eraseEvent(eventId) {
    this._events[eventId].erase();
  }

  parallelCommonEvents() {
    return $dataCommonEvents.filter((commonEvent) => {
      if (!commonEvent) return false;
      if (commonEvent.trigger !== 2 && !commonEvent.name.includes('_async')) return false;

      if (commonEvent.name.indexOf('_map_') > 0) {
        if (!commonEvent.name.includes(`_map_${this._mapId}`)) return false;
      }

      return true;
    });
  }

  isCommonEventMarkedAsParallel(commonEventId) {
    for (let commonEvent of this._commonEvents) {
      if (!commonEvent) continue;

      if (commonEvent._commonEventId == commonEventId) {
        return true;
      }
    }

    return false;
  }

  markCommonEventAsParallel(commonEventId) {
    if (this.isCommonEventMarkedAsParallel(commonEventId)) {
      return;
    }

    this._commonEvents.push(new Objects.CommonEvent(commonEventId));
  }

  setupScroll() {
    this._scrollDirection = 2;
    this._scrollRest = 0;
    this._scrollSpeed = 4;
    this._scrollTargetX = false;
    this._scrollTargetY = false;
  }

  setupParallax() {
    this._parallaxName = $dataMap.parallaxName || '';
    this._parallaxZero = Managers.Images.isZeroParallax(this._parallaxName);
    this._parallaxLoopX = $dataMap.parallaxLoopX;
    this._parallaxLoopY = $dataMap.parallaxLoopY;
    this._parallaxSx = $dataMap.parallaxSx;
    this._parallaxSy = $dataMap.parallaxSy;
    this._parallaxX = 0;
    this._parallaxY = 0;
  }

  setDisplayPos(x, y) {
    if (this.isLoopHorizontal()) {
      this._displayX = x.mod(this.width());
      this._parallaxX = x;
    } else {
      const endX = this.width() - this.screenTileX();
      this._displayX = endX < 0 ? endX / 2 : x.clamp(0, endX);
      this._parallaxX = this._displayX;
    }
    if (this.isLoopVertical()) {
      this._displayY = y.mod(this.height());
      this._parallaxY = y;
    } else {
      const endY = this.height() - this.screenTileY();
      this._displayY = endY < 0 ? endY / 2 : y.clamp(0, endY);
      this._parallaxY = this._displayY;
    }

    this.fixScrollBounds();
  }

  parallaxOx() {
    if (this._parallaxZero) {
      return this._parallaxX * this.tileWidth();
    } else if (this._parallaxLoopX) {
      return this._parallaxX * this.tileWidth() / 2;
    } else {
      return 0;
    }
  }

  parallaxOy() {
    if (this._parallaxZero) {
      return this._parallaxY * this.tileHeight();
    } else if (this._parallaxLoopY) {
      return this._parallaxY * this.tileHeight() / 2;
    } else {
      return 0;
    }
  }

  tileset() {
    return $dataTilesets[this._tilesetId];
  }

  tilesetFlags() {
    const tileset = this.tileset();
    if (tileset) {
      return tileset.flags;
    } else {
      return [];
    }
  }

  displayName() {
    return $dataMap.displayName;
  }

  width() {
    return $dataMap.width;
  }

  height() {
    return $dataMap.height;
  }

  data() {
    return $dataMap.data;
  }

  isLoopHorizontal() {
    return $dataMap.scrollType === 2 || $dataMap.scrollType === 3;
  }

  isLoopVertical() {
    return $dataMap.scrollType === 1 || $dataMap.scrollType === 3;
  }

  screenTileX() {
    return (Graphics.width / this.tileWidth()) / this.zoom.x;
  }

  screenTileY() {
    return (Graphics.height / this.tileHeight()) / this.zoom.y;
  }

  adjustX(x) {
    // if (this.isLoopHorizontal() && x < this._displayX - (this.width() - this.screenTileX()) / 2) {
    //   return x - this._displayX + $dataMap.width;
    // } else {
    return x - this._displayX;
    // }
  }

  adjustY(y) {
    // if (this.isLoopVertical() && y < this._displayY - (this.height() - this.screenTileY()) / 2) {
    //   return y - this._displayY + $dataMap.height;
    // } else {
    return y - this._displayY;
    // }
  }

  roundX(x) {
    return this.isLoopHorizontal() ? x.mod(this.width()) : x;
  }

  roundY(y) {
    return this.isLoopVertical() ? y.mod(this.height()) : y;
  }

  xWithDirection(x, d) {
    if (DirectionHelper.goesLeft(d)) {
      return x - 1;
    } else if (DirectionHelper.goesRight(d)) {
      return x + 1;
    } else {
      return x;
    }
  }

  yWithDirection(y, d) {
    if (DirectionHelper.goesDown(d)) {
      return y + 1;
    } else if (DirectionHelper.goesUp(d)) {
      return y - 1;
    } else {
      return y;
    }
  }

  roundXWithDirection(x, d) {
    return this.roundX(x + (d === 6 ? 1 : d === 4 ? -1 : 0));
  }

  roundYWithDirection(y, d) {
    return this.roundY(y + (d === 2 ? 1 : d === 8 ? -1 : 0));
  }

  deltaX(x1, x2) {
    let result = x1 - x2;
    if (this.isLoopHorizontal() && Math.abs(result) > this.width() / 2) {
      if (result < 0) {
        result += this.width();
      } else {
        result -= this.width();
      }
    }
    return result;
  }

  deltaY(y1, y2) {
    let result = y1 - y2;
    if (this.isLoopVertical() && Math.abs(result) > this.height() / 2) {
      if (result < 0) {
        result += this.height();
      } else {
        result -= this.height();
      }
    }
    return result;
  }

  distanceBetweenRects(rect1, rect2, defaultX, defaultY) {
    if (defaultX === undefined) defaultX = 0;
    if (defaultY === undefined) defaultY = 0;

    const left1 = rect1.left;
    const left2 = rect2.left;
    const right1 = rect1.right;
    const right2 = rect2.right;
    const top1 = rect1.top;
    const top2 = rect2.top;
    const bottom1 = rect1.bottom;
    const bottom2 = rect2.bottom;

    let xDistance = 999;
    let yDistance = 999;

    if ((left1 >= left2 && left1 <= right2) || (left2 >= left1 && left2 <= right1)) {
      if (right2 - left2 <= 1) {
        xDistance = 0;
      } else if (right1 - left1 <= 1) {
        xDistance = 0;
      } else {
        xDistance = Math.min(Math.abs(left1 - left2), Math.abs(right1 - right2), Math.abs(left1 - right2), Math.abs(right1 - left2));
      }
    } else if (right1 < left2) {
      xDistance = left2 - right1;
    } else if (right2 < left1) {
      xDistance = left1 - right2;
    } else {
      xDistance = defaultX;
    }

    if ((top1 >= top2 && top1 <= bottom2) || (top2 >= top1 && top2 <= bottom1)) {
      if (top2 - bottom2 <= 1) {
        yDistance = 0;
      } else if (top1 - bottom1 <= 1) {
        yDistance = 0;
      } else {
        yDistance = Math.min(Math.abs(top1 - top2), Math.abs(bottom1 - bottom2), Math.abs(bottom1 - top2), Math.abs(top1 - bottom2));
      }
    } else if (bottom1 < top2) {
      yDistance = top2 - bottom1;
    } else if (bottom2 < top1) {
      yDistance = top1 - bottom2;
    } else {
      yDistance = defaultY;
    }

    xDistance = Math.fix(xDistance);
    yDistance = Math.fix(yDistance);

    return Math.abs(xDistance) + Math.abs(yDistance);
  }

  distanceBetweenPoints(point1, point2) {
    const left1 = point1.x;
    const left2 = point2.x;
    const top1 = point1.y;
    const top2 = point2.y;

    const xDistance = Math.abs(left1 - left2);
    const yDistance = Math.abs(top1 - top2);

    return Math.fix(xDistance + yDistance);
  }

  distanceBetweenRectAndCenter(rect, center) {
    let xDistance;
    let yDistance;

    if (rect.left <= center.x && rect.right >= center.x) {
      xDistance = 0;
    } else {
      xDistance = Math.min(Math.abs(center.x - rect.left), Math.abs(center.x - rect.right));
    }

    if (rect.top <= center.y && rect.bottom >= center.y) {
      yDistance = 0;
    } else {
      yDistance = Math.min(Math.abs(center.y - rect.top), Math.abs(center.y - rect.bottom));
    }

    xDistance = Math.abs(Math.fix(xDistance));
    yDistance = Math.abs(Math.fix(yDistance));

    const combinedDistance = (xDistance + yDistance) * 0.8;
    return Math.max(xDistance, yDistance, combinedDistance);
  }

  distanceBetween(obj1, obj2) {
    if (!obj1 || !obj2) return 9999;
    if (obj1 == obj2) return 0;

    const rect1 = obj1.getRectPosition();
    const rect2 = obj2.getRectPosition();
    const defaultX = obj1.x - obj2.x;
    const defaultY = obj1.y - obj2.y;

    return this.distanceBetweenRects(rect1, rect2, defaultX, defaultY);
  }

  distanceBetweenCenters(obj1, obj2) {
    if (!obj1 || !obj2) return 9999;
    if (obj1 == obj2) return 0;

    const rect1 = obj1.getCenterPosition();
    const rect2 = obj2.getCenterPosition();
    const defaultX = obj1.x - obj2.x;
    const defaultY = obj1.y - obj2.y;

    return this.distanceBetweePoints(rect1, rect2, defaultX, defaultY);
  }

  distance(x1, y1, x2, y2) {
    return Math.abs(this.deltaX(x1, x2)) + Math.abs(this.deltaY(y1, y2));
  }

  mapToCanvasX(x) {
    const tileWidth = this.tileWidth() * this._zoom.x;
    const originX = this._displayX * tileWidth;

    const canvasX = Math.floor(x * tileWidth - originX);
    return canvasX;
  }

  mapToCanvasY(y) {
    const tileHeight = this.tileHeight() * this._zoom.y;
    const originY = this._displayY * tileHeight;
    const canvasY = Math.floor(y * tileHeight - originY);
    return canvasY;
  }

  mapToScreenX(x) {
    const tileWidth = this.tileWidth();

    const canvasX = Math.floor( (x - this._displayX) * tileWidth);
    return canvasX;
  }

  mapToScreenY(y) {
    const tileHeight = this.tileHeight();
    const canvasY = Math.floor((y - this._displayY) * tileHeight);
    return canvasY;
  }

  resizedCanvasToMapX(x) {
    return this.canvasToMapX(x * this.zoom.x);
  }

  resizedCanvasToMapY(y) {
    return this.canvasToMapY(y * this.zoom.y);
  }

  canvasToMapX(x, skipFlooring) {
    const tileWidth = this.tileWidth() * this._zoom.x;

    const originX = this._displayX * tileWidth;
    let mapX = (originX + x) / tileWidth;
    if (!skipFlooring) {
      mapX = Math.floor(mapX);
    }
    return this.roundX(mapX);
  }

  canvasToMapY(y, skipFlooring) {
    const tileHeight = this.tileHeight() * this._zoom.y;
    const originY = this._displayY * tileHeight;

    let mapY = (originY + y) / tileHeight;

    if (!skipFlooring) {
      mapY = Math.floor(mapY);
    }

    return this.roundY(mapY);
  }

  getBgmName() {
    if (Switches.startedTutorial && !Switches.finishedTutorial) {
      return 'title';
    }

    if (!Switches.finishedIntroEvent) return undefined;

    if (Switches.isPlayingDate) {
      // return 'date';
      return 'title';
    }

    if ($gameSystem.isPlayingCutscene() || $gameTemp.isCutsceneReserved()) {
      // return 'cutscene';
      return 'title';
    }

    let songName = 'spring';

    // if (Managers.Weather.isRaining()) {
    //   songName = 'rain';
    // } else
    if (Switches.isInsideFestival) {
      if (Switches.isInsideFestivalCompetition) {
        songName = 'competition';
      } else {
        songName = 'festival';
      }
    } else if (this.isCave()) {
      songName = 'cave';
    } else if (this.isCemetery()) {
      songName = 'graveyard';
    } else if (Managers.Time.hour >= 21 || Managers.Time.hour < 6) {
      songName = 'night';
    } else {
      switch(Managers.Time.mapMonth) {
        case Seasons.SUMMER :
          songName = 'summer';
          break;
        case Seasons.FALL :
          songName = 'fall';
          break;
        case Seasons.WINTER :
          songName = 'winter';
          break;
        default :
          songName = 'spring';
          break;
      }
    }

    return songName;
  }

  autoplay() {
    const songName = this.getBgmName();

    if (!songName) return;

    Audio.playBgm({name : songName, volume : 90}, 0, 1);
  }

  refreshIfNeeded() {
    if (this._needsRefresh) {
      this.refresh();
    }
  }

  refresh() {
    this.events().forEach(event => {
      event.refresh();
    });
    this._commonEvents.forEach(event => {
      event.refresh();
    });
    this.refreshTileEvents();
    this._needsRefresh = false;
  }

  refreshTileEvents() {
    this.tileEvents = this.events().filter(event => event.isTile());
  }

  eventsHardXy(x, y) {
    return this._events.filter(event => !!event && !event._erased && event._x === x && event._y === y);
  }

  eventsXy(x, y) {
    return this._events.filter(event => !!event && !event._erased && event.pos(x, y));
  }

  realEventsXy(x, y) {
    return this._events.filter(event => !!event && !event._erased && event.realPos(x, y));
  }

  farmObjectsXy(x, y) {
    const farmObjectsX = this._farmObjectsXy[x];
    if (farmObjectsX === undefined) return [];

    const farmObjectsY = farmObjectsX[y];
    if (farmObjectsY === undefined) return [];

    return farmObjectsY.filter(event => !!event && !event._erased);
  }

  farmObjectsXyEx(x, y) {
    const base = this.farmObjectsXy(x, y);
    const wide = this.farmObjectsXy(x - 1, y).filter(event => event._farmObjectData && event._farmObjectData.modelName && Managers.FarmObjects.isWideModel(event._farmObjectData.modelName));
    const tall = this.farmObjectsXy(x, y + 1).filter(event => event._farmObjectData && event._farmObjectData.modelName && Managers.FarmObjects.isTallModel(event._farmObjectData.modelName));

    return [].concat(base).concat(wide).concat(tall);
  }

  allEvents() {
    const events = this.events();
    const villagers = this.villagerEvents();
    const farmObjects = this.farmObjectEvents();
    const creatures = this.creatureEvents();

    return events.concat(villagers).concat(farmObjects).concat(creatures);
  }

  anythingXy(x, y) {
    const events = this.eventsXy(x, y);
    const villagers = this.villagersXy(x, y);
    const farmObjects = this.farmObjectsXy(x, y);
    const creatures = this.creaturesXy(x, y);

    return events.concat(villagers).concat(farmObjects).concat(creatures);
  }

  anythingTouchableXy(x, y) {
    const minX = Math.floor(x);
    const maxX = Math.ceil(x);
    const minY = Math.floor(y);
    const maxY = Math.ceil(y);

    let events = this._events.filter(event => !!event && !event._erased && event.realPosIn(minX, minY, maxX, maxY));

    const villagers = this.sortedVillagersXy(x, y);
    const farmObjects = this.farmObjectsXy(minX, minY);
    const creatures = this.creaturesXy(minX, minY);

    events = events.concat(villagers, farmObjects, creatures);

    const villagersUnder = this.sortedVillagersXy(minX, minY + 1);
    events = events.concat(villagersUnder);

    let i;
    const farmObjectsUnder = this.farmObjectsXy(minX, minY + 1);
    const farmObjectsLeft = this.farmObjectsXy(minX - 1, minY);
    const farmObjectsRight = this.farmObjectsXy(minX + 1, minY);
    const farmObjectsLeftUnder = this.farmObjectsXy(minX - 1, minY + 1);
    const farmObjectsRightUnder = this.farmObjectsXy(minX + 1, minY + 1);
    const farmObjectsUnder2 = this.farmObjectsXy(minX, minY + 2);
    const farmObjectsWayUnder = this.farmObjectsXy(minX, minY + 3);

    const extraFarmObjects = farmObjectsUnder.concat(farmObjectsLeft).concat(farmObjectsRight).concat(farmObjectsLeftUnder).concat(farmObjectsRightUnder).concat(farmObjectsUnder2).concat(farmObjectsWayUnder);

    if (extraFarmObjects.length > 0) {
      for (i = 0; i < extraFarmObjects.length; i++) {
        if (!extraFarmObjects[i]) continue;
        if (!extraFarmObjects[i].canBeTouchedAt(x, y)) continue;

        events.push(extraFarmObjects[i]);
      }
    }

    // Sort events by Z and Y levels
    return events.sort((event1, event2) => {
      let value1 = (x >= event1.left && x <= event1.right) ? 1 : 0;
      let value2 = (x >= event2.left && x < event2.right) ? 1 : 0;

      // If one event is better positioned under the mouse cursor, then pick it first
      if (value1 !== value2) {
        return value2 - value1;
      }

      value1 = (y >= event1.top && y < event1.bottom) ? 1 : 0;
      value2 = (y >= event2.top && y < event2.bottom) ? 1 : 0;
      if (value1 !== value2) {
        return value2 - value1;
      }

      const z1 = event1.screenZ();
      const z2 = event2.screenZ();

      if (z1 !== z2) {
        return z2 - z1;
      }

      return event2.screenY() - event1.screenY();
    });
  }

  anythingAliveXy(x, y) {
    const villagers = this.villagersXy(x, y);
    const creatures = this.creaturesXy(x, y);

    return villagers.concat(creatures);
  }

  anyCollisionXy(x, y, allowItems, allowSeeds, allowAnimals) {
    const events = this.eventsXyNt(x, y);
    if (events.length > 0) return true;

    const villagers = this.villagersXy(x, y);
    if (villagers.length > 0) return true;

    if (!allowAnimals) {
      const creatures = this.creaturesXy(x, y);
      if (creatures.length > 0) return true;
    }

    const farmObjects = this.farmObjectsXy(x, y);
    for (let i = 0; i < farmObjects.length; i++) {
      if (allowItems) {
        if (farmObjects[i]._farmObjectData.isItem()) continue;
      }

      if (farmObjects[i]._priorityType == EventPriorities.NORMAL) return true;
      if (farmObjects[i]._farmObjectData.hasCollision()) return true;

      if (allowSeeds === false && farmObjects[i]._farmObjectData.isCropOrTree) {
        return true;
      }
    }

    return false;
  }

  destroyAnythingAt(x, y) {
    const objects = this.anythingXy(x, y);

    const len = objects.length;
    for (let i = 0; i < len; i++) {
      objects[i].erase();
    }
  }

  livingBeingsXy(x, y) {
    const villagers = this.villagersXy(x, y);
    const creatures = this.creaturesXy(x, y);

    return villagers.concat(creatures);
  }

  villagersXy(x, y) {
    return this._villagers.filter(event => !!event && event.pos(x, y));
  }

  sortedVillagersXy(x, y) {
    const villagers = this.villagersXy(x, y);
    if (villagers.length < 2) {
      return villagers;
    }

    return villagers.sort((firstVillager, secondVillager) => {
      let value1 = (x >= firstVillager.left && x <= firstVillager.right) ? 1 : 0;
      let value2 = (x >= secondVillager.left && x < secondVillager.right) ? 1 : 0;

      if (value1 !== value2) {
        return value2 - value1;
      }

      value1 = (y >= firstVillager.top && y < firstVillager.bottom) ? 1 : 0;
      value2 = (y >= secondVillager.top && y < secondVillager.bottom) ? 1 : 0;
      return value2 - value1;
    });
  }

  creaturesXy(x, y) {
    return this._creatures.filter(event => !!event && !event._erased && !event._hidden && event.pos(x, y));
  }

  filteredCreatures(filterFn) {
    return this._creatures.filter(filterFn);
  }

  getIndexForNewCreature() {
    let index = 1;
    while (index < this._creatures.length && !!this._creatures[index]) {
      index++;
    }

    return index;
  }

  getIndexForNewEvent() {
    let index = 1;
    while (index < this._events.length && !!this._events[index]) {
      index++;
    }

    return index;
  }

  addFarmObjectEvent(eventData, classType) {
    if (classType === undefined) classType = Objects.Object;

    const gameEvent = new classType(this._mapId, -1, eventData);
    if (this._farmObjectsXy[eventData.x] === undefined) {
      this._farmObjectsXy[eventData.x] = {};
    }

    if (this._farmObjectsXy[eventData.x][eventData.y] === undefined) {
      this._farmObjectsXy[eventData.x][eventData.y] = [];
    }

    this._farmObjectsXy[eventData.x][eventData.y].push(gameEvent);
    this.addObjectSprite(gameEvent);

    return gameEvent;
  }

  addCreatureEvent(eventData, classType) {
    if (classType === undefined) classType = Objects.Creature;

    const index = this.getIndexForNewCreature();

    const gameEvent = new classType(this._mapId, index, eventData);
    this._creatures[index] = gameEvent;

    eventData.id = index;
    this.addEventSprite(gameEvent);

    return gameEvent;
  }

  addEventSprite(gameEvent) {
    if ((Managers.Scenes._scene instanceof GameScenes.Map || Managers.Scenes._scene instanceof GameScenes.GameMap) && Managers.Scenes._scene._spriteset !== undefined) {
      Managers.Scenes._scene._spriteset.addEventSprite(gameEvent);
    }
  }

  addObjectSprite(farmObjectEvent) {
    if ((Managers.Scenes._scene instanceof GameScenes.Map || Managers.Scenes._scene instanceof GameScenes.GameMap) && Managers.Scenes._scene._spriteset !== undefined) {
      Managers.Scenes._scene._spriteset.addObjectSprite(farmObjectEvent);
    }
  }

  addSoilSprite(farmObjectEvent) {
    if ((Managers.Scenes._scene instanceof GameScenes.Map || Managers.Scenes._scene instanceof GameScenes.GameMap) && Managers.Scenes._scene._spriteset !== undefined) {
      Managers.Scenes._scene._spriteset.addSoilSprite(farmObjectEvent);
    }
  }

  findAllObjectSprites(farmObject) {
    if ((Managers.Scenes._scene instanceof GameScenes.Map || Managers.Scenes._scene instanceof GameScenes.GameMap) && Managers.Scenes._scene._spriteset !== undefined) {
      return Managers.Scenes._scene._spriteset.findAllObjectSprites(farmObject);
    }

    return [];
  }

  addVillagerEvent(eventData, classType) {
    if (classType === undefined) {
      const villagerName = $gameActors.actorName(eventData.id);
      if (villagerName) {
        classType = Managers.Content.getVillagerObjectClass(villagerName);
      }

      if (!classType) {
        classType = Objects.Villager;
      }
    }

    const gameEvent = new classType(this._mapId, eventData.id, eventData);
    this._villagers[eventData.id] = gameEvent;

    if (eventData.isCutscene) {
      gameEvent.isCutscene = true;
    }

    this.addEventSprite(gameEvent);

    return gameEvent;
  }

  addEvent(eventData, temporary, classType, reservedId = undefined) {
    if (!classType) {
      classType = Objects.CustomEvent;
    }

    if (eventData instanceof CustomEventData) {
      eventData.endAllPages();
    }

    if (classType === Objects.Object || classType.prototype instanceof Objects.Object || classType.isObjectClass) {
      return this.addFarmObjectEvent(eventData, classType);
    }
    if (classType == Objects.Villager || classType.prototype instanceof Objects.Villager || classType.isVillagerClass) {
      return this.addVillagerEvent(eventData, classType);
    }
    if (classType == Objects.Creature || classType.prototype instanceof Objects.Creature || classType.isCreatureClass) {
      if (classType !== Objects.WildCreature) {
        return this.addCreatureEvent(eventData, classType);
      }
    }

    if (temporary === undefined) {
      temporary = false;
    }

    const index = reservedId || this.getIndexForNewEvent();

    eventData.id = index;
    const gameEvent = new classType(this._mapId, index, eventData);
    if (!reservedId) {
      $gameSystem.clearSelfSwitches(this._mapId, index);
    }

    this._events[index] = gameEvent;

    this.addEventSprite(gameEvent);

    if (temporary === false) {
      // $gameSystem.addCustomEvent(this._mapId, eventData);
    }

    return gameEvent;
  }

  positionHasEvents(x, y) {
    return this._events.filter(event => !!event && event.pos(x, y)).length > 0;
  }

  realEventsXyNt(x, y) {
    return this.events().filter(event => event.realPosNt(x, y));
  }

  eventsXyNt(x, y) {
    return this.events().filter(event => event.posNt(x, y));
  }

  tileEventsXy(x, y) {
    if (!this.tileEvents) this.refreshTileEvents();

    return this.tileEvents.filter(event => event.posNt(x, y));
  }

  eventIdXy(x, y) {
    const list = this.eventsXy(x, y);
    return list.length === 0 ? 0 : list[0].eventId();
  }

  scrollDown(distance) {
    if (this.isLoopVertical()) {
      this._displayY += distance;
      this._displayY %= $dataMap.height;
      if (this._parallaxLoopY) {
        this._parallaxY += distance;
      }
    } else if (this.height() >= this.screenTileY()) {
      const lastY = this._displayY;
      this._displayY = Math.min(this._displayY + distance, this.height() - this.screenTileY());
      this._parallaxY += this._displayY - lastY;
    }

    this.fixScrollBounds();
  }

  scrollLeft(distance) {
    if (this.isLoopHorizontal()) {
      this._displayX += $dataMap.width - distance;
      this._displayX %= $dataMap.width;
      if (this._parallaxLoopX) {
        this._parallaxX -= distance;
      }
    } else if (this.width() >= this.screenTileX()) {
      const lastX = this._displayX;
      this._displayX = Math.max(this._displayX - distance, 0);
      this._parallaxX += this._displayX - lastX;
    }

    this.fixScrollBounds();
  }

  scrollRight(distance) {
    if (this.isLoopHorizontal()) {
      this._displayX += distance;
      this._displayX %= $dataMap.width;
      if (this._parallaxLoopX) {
        this._parallaxX += distance;
      }
    } else if (this.width() >= this.screenTileX()) {
      const lastX = this._displayX;
      this._displayX = Math.min(this._displayX + distance, this.width() - this.screenTileX());
      this._parallaxX += this._displayX - lastX;
    }

    this.fixScrollBounds();
  }

  scrollUp(distance) {
    if (this.isLoopVertical()) {
      this._displayY += $dataMap.height - distance;
      this._displayY %= $dataMap.height;
      if (this._parallaxLoopY) {
        this._parallaxY -= distance;
      }
    } else if (this.height() >= this.screenTileY()) {
      const lastY = this._displayY;
      this._displayY = Math.max(this._displayY - distance, 0);
      this._parallaxY += this._displayY - lastY;
    }

    this.fixScrollBounds();
  }

  centerCameraOn(destX, destY, speed, instant) {
    const horzTiles = Graphics.width / this.tileWidth() / this.zoom.x;
    const vertTiles = Graphics.height / this.tileHeight() / this.zoom.y;

    destX -= horzTiles / 2;
    destY -= vertTiles / 2;

    if (destX < 0) destX = 0;
    if (destY < 0) destY = 0;
    if (destX > this.width() - horzTiles) destX = this.width() - horzTiles;
    if (destY > this.height() - vertTiles) destY = this.height() - vertTiles;

    this.scrollTo(destX, destY, speed, instant);
  }

  scrollTo(destX, destY, speed, instant) {
    const maxX = Math.floor(this.width() - (Graphics.width / (this.tileWidth() * this.zoom.x)));
    const maxY = Math.floor(this.height() - (Graphics.height / (this.tileHeight() * this.zoom.y)));

    const scrollX = Math.round(Math.max(0, Math.min(destX, maxX)));
    const scrollY = Math.round(Math.max(0, Math.min(destY, maxY)));

    if (instant) {
      this._displayX = scrollX;
      this._displayY = scrollY;
      this.fixScrollBounds();
      return;
    }

    this._scrollTargetX = scrollX;
    this._scrollTargetY = scrollY;
    this._scrollDirection = 2;
    this._scrollRest = 0;

    if (speed) {
      this._scrollSpeed = speed;
    }
  }

  restoreScroll(instant) {
    this.centerCameraOn(Math.round($gamePlayer.x), Math.round($gamePlayer.y), undefined, instant);
  }

  getPassabilityFromTileId(tileId, d) {
    const flags = this.tilesetFlags();
    if (d === undefined) d = 0;

    const flag = flags[tileId];
    if ((flag & 0x10) !== 0) return null;
    if (d > 0) {
      if ((flag & d) === 0) return true;
      if ((flag & d) === d) return false;
    } else {
      if ((flag & Direction.DOWN) === Direction.DOWN) return false;
      if ((flag & Direction.LEFT) === Direction.LEFT) return false;
      if ((flag & Direction.RIGHT) === Direction.RIGHT) return false;
      if ((flag & Direction.UP) === Direction.UP) return false;

      return true;
    }

    return null;
  }

  checkFishPassage(x, y, bit) {
    const flags = this.tilesetFlags();
    const tiles = this.layeredTiles(x, y);
    const regionId = this.tileId(x, y, 5);

    if (regionId == Region.BLOCK_FISHING) return false;
    if (regionId == Region.BRIDGE) return false;
    if (Managers.Map.mapMonth($gameMap._mapId) == Seasons.WINTER) {
      if (regionId == Region.BLOCK_FISHING_ON_WINTER) return false;
    }

    let isWater = false;

    for (let i = 0; i < tiles.length; i++) {
      if (tiles[i] === 0) continue;
      let flag = flags[tiles[i]];
      const tag = flag >> 12;

      if (tag == TerrainTags.WATER) {
        isWater = true;
        continue;
      }

      flag = flags[tiles[i]];

      if ((flag & 0x10) !== 0)  // [*] No effect on passage
        continue;
      if ((flag & bit) === 0)   // [o] Passable
        return isWater;
      if ((flag & bit) === bit) // [x] Impassable
        return false;
    }
    return isWater;
  }

  checkPassage(x, y, bit) {
    const flags = this.tilesetFlags();
    const tiles = this.allTiles(x, y);
    const regionId = this.tileId(x, y, 5);

    if (regionId == Region.BLOCKED_MOVEMENT) return false;
    if (regionId == Region.SUPER_BLOCKED_MOVEMENT) return false;

    for (let i = 0; i < tiles.length; i++) {
      const flag = flags[tiles[i]];
      if ((flag & 0x10) !== 0)  // [*] No effect on passage
        continue;
      if ((flag & bit) === 0)   // [o] Passable
        return true;
      if ((flag & bit) === bit) // [x] Impassable
        return false;
    }
    return false;
  }

  tileId(x, y, z) {
    const width = $dataMap.width;
    const height = $dataMap.height;
    return $dataMap.data[(z * height + y) * width + x] || 0;
  }

  layeredTiles(x, y) {
    const tiles = [];
    for (let i = 0; i < 4; i++) {
      tiles.push(this.tileId(x, y, 3 - i));
    }
    return tiles;
  }

  allTiles(x, y) {
    const tiles = this.tileEventsXy(x, y).map(event => event.tileId());
    return tiles.concat(this.layeredTiles(x, y));
  }

  autotileType(x, y, z) {
    const tileId = this.tileId(x, y, z);
    return tileId >= 2048 ? Math.floor((tileId - 2048) / 48) : -1;
  }

  isPassable(x, y, d) {
    return this.checkPassage(x, y, (1 << (d / 2 - 1)) & 0x0f);
  }

  isPassableForFish(x, y, d) {
    return this.checkFishPassage(x, y, (1 << (d / 2 - 1)) & 0x0f);
  }

  isTilePassable(x, y) {
    const regionId = this.tileId(x, y, 5);
    if (regionId == Region.BLOCKED_MOVEMENT) return false;
    if (regionId == Region.SUPER_BLOCKED_MOVEMENT) return false;

    const flags = this.tilesetFlags();
    const tiles = this.allTiles(x, y);

    for (let i = 0; i < tiles.length; i++) {
      const flag = flags[tiles[i]];
      if ((flag & 0x10) !== 0)
        continue;

      const left = flag & Direction.LEFT;
      const right = flag & Direction.RIGHT;
      const up = flag & Direction.UP;
      const down = flag & Direction.DOWN;

      if (left === Direction.LEFT && right === Direction.RIGHT && down === Direction.DOWN && up === Direction.UP) return false;
    }

    return true;
  }

  checkLayeredTilesFlags(x, y, bit) {
    const flags = this.tilesetFlags();
    const method = tileId => (flags[tileId] & bit) !== 0;

    for (let newX = x.floor(); newX <= x.ceil(); newX++) {
      for (let newY = y.floor(); newY <= y.ceil(); newY++) {
        const result = this.layeredTiles(newX, newY).some(method);

        if (result) return result;
      }
    }

    return false;
  }

  isLadder(x, y) {
    return this.isValid(x, y) && this.checkLayeredTilesFlags(x, y, 0x20);
  }

  isBush() /*x, y*/{
    return false;
  }

  isCounter(x, y) {
    return this.isValid(x, y) && this.checkLayeredTilesFlags(x, y, 0x80);
  }

  terrainTag(x, y) {
    if (this.isValid(x, y)) {
      const flags = this.tilesetFlags();
      const tiles = this.layeredTiles(x, y);
      for (let i = 0; i < tiles.length; i++) {
        const tag = flags[tiles[i]] >> 12;
        if (tag > 0) {
          return tag;
        }
      }
    }
    return 0;
  }

  regionId(x, y) {
    return this.isValid(x, y) ? this.tileId(x, y, 5) : 0;
  }

  startScroll(direction, distance, speed) {
    this._scrollTargetX = false;
    this._scrollTargetY = false;
    this._scrollDirection = direction;
    this._scrollRest = distance;
    this._scrollSpeed = speed;
  }

  isScrolling() {
    return this._scrollRest > 0 || (this._scrollTargetX !== false) || (this._scrollTargetY !== false);
  }

  update(sceneActive) {
    this.refreshIfNeeded();

    if (this._mouseDelay > 0) {
      this._mouseDelay--;
    }

    this.updateTimeouts();

    if (sceneActive) {
      this.updateInterpreter();
    }

    this.updateScroll();
    this.updateFarmObjects();
    this.updateVillagers();
    this.updateCreatures();
    this.updateEvents();
    this.updateBackgroundSounds();
    this.updateParallax();

    this.updateMapController();
    Managers.Content.update();
    Managers.Weather.updateWind();

    this.updateNotices();
  }

  updateMapController() {
    Managers.Map.update(this._mapId);
  }

  updateNotices() {
    if ($gameMessage.isBusy()) return;

    if ($gameTemp.hasUpdateNotice()) {
      const notice = $gameTemp.getNextUpdateNotice();

      $gameMessage.setPositionType(1);
      $gameMessage.addText(notice);
    }
  }

  updateScroll() {
    if (this.isScrolling()) {
      const lastX = this._displayX;
      const lastY = this._displayY;

      if (this._scrollRest > 0) {
        this.doScroll(this._scrollDirection, this.scrollDistance());
        if (this._displayX === lastX && this._displayY === lastY) {
          this._scrollRest = 0;
        } else {
          this._scrollRest -= this.scrollDistance();
        }
      } else {
        if (this._scrollTargetX !== false && this._scrollTargetX !== this._displayX) {
          if (this._scrollTargetX > this._displayX) {
            this.doScroll(Direction.RIGHT, this.scrollDistance());

            if (this._scrollTargetX < this._displayX) {
              this._displayX = this._scrollTargetX;
              this._scrollTargetX = false;
            }
          } else if (this._scrollTargetX < this._displayX) {
            this.doScroll(Direction.LEFT, this.scrollDistance());
            if (this._scrollTargetX > this._displayX) {
              this._displayX = this._scrollTargetX;
              this._scrollTargetX = false;
            }
          }
        } else {
          this._scrollTargetX = false;
        }

        if (this._scrollTargetY !== false && this._scrollTargetY !== this._displayY) {
          if (this._scrollTargetY > this._displayY) {
            this.doScroll(Direction.DOWN, this.scrollDistance());
            if (this._scrollTargetY < this._displayY) {
              this._displayY = this._scrollTargetY;
              this._scrollTargetY = false;
            }
          } else if (this._scrollTargetY < this._displayY) {
            this.doScroll(Direction.UP, this.scrollDistance());
            if (this._scrollTargetY > this._displayY) {
              this._displayY = this._scrollTargetY;
              this._scrollTargetY = false;
            }
          }
        } else {
          this._scrollTargetY = false;
        }
      }
    }

    // this.fixScrollBounds();
  }

  fixScrollBounds() {
    // this._displayX = Math.floor(this._displayX * $gameMap.tileWidth()) / $gameMap.tileWidth();
    // this._displayY = Math.floor(this._displayY * $gameMap.tileHeight()) / $gameMap.tileHeight();
  }

  scrollDistance() {
    const diff = 256 / Managers.Scenes._gameSpeed;
    return 2 ** this._scrollSpeed / diff;
  }

  doScroll(direction, distance) {
    switch (direction) {
      case 2:
        this.scrollDown(distance);
        break;
      case 4:
        this.scrollLeft(distance);
        break;
      case 6:
        this.scrollRight(distance);
        break;
      case 8:
        this.scrollUp(distance);
        break;
      default :
        break;
    }
  }

  farmObjects() {
    return Managers.FarmObjects.getMapFarmObjects(this._mapId);
  }

  updateTileEvents() /*x, y*/{

  }

  updateVillagers() {
    this._villagers.forEach(event => {
      if (!!event && !event._erased) {
        event.update();
      }
    });

    if (this._villagerUpdateDelay > 0) {
      this._villagerUpdateDelay--;
      return;
    }

    Managers.Map.updateVillagers();
    this._villagerUpdateDelay = Utils.getFrameCount(5);
  }

  refreshCreatures() {
    for (const creature of this._creatures) {
      if (!creature) continue;
      creature.erase();
    }

    Managers.Creatures.setupMapEvents();
    Managers.Map.afterRefreshCreatures();
  }

  updateCreatures() {
    // if (this._creatureUpdateDelay > 0) {
    //   this._creatureUpdateDelay--;
    //   return;
    // }

    this._creatures.forEach(event => {
      if (event) {
        event.update();
      }
    });

    // this._creatureUpdateDelay = Utils.getFrameCount(5);
  }

  updateFarmObjects() {
    this.updateButtonEvents();
    this.updateFarmObjectAnimations();
  }

  updateButtonEvents() {
    if (this._buttonUpdateDelay > 0) {
      this._buttonUpdateDelay--;
      return;
    }

    Managers.FarmObjects.updateButtonEvents();
    this._buttonUpdateDelay = Utils.getFrameCount(10);
  }

  updateFarmObjectAnimations() {
    // If there's any farmObject with stepAnime enabled, update it.
    // If there's any farmObject that inherites Objects.AnimatedObject, update it too.

    this.iterateFarmObjectEvents((obj) => {
      if (!obj) return;

      if (obj._stepAnime) {
        obj.updateAnimation();
      }

      if (obj instanceof Objects.AnimatedObject) {
        obj.updateEventAnimation();
      }
    });
  }

  requestGrassUpdate(event) {
    if (!this._grassEventsPendingUpdate) {
      this._grassEventsPendingUpdate = [];
    }

    this._grassEventsPendingUpdate.push(event);
  }

  setupCustomMapEvents() {

  }

  updateEvents() {
    this.events().forEach(event => {
      event.update();
    });
    this._commonEvents.forEach(event => {
      event.update();
    });
  }

  updateParallax() {
    if (this._parallaxLoopX) {
      this._parallaxX += this._parallaxSx / this.tileWidth() / 2;
    }
    if (this._parallaxLoopY) {
      this._parallaxY += this._parallaxSy / this.tileHeight() / 2;
    }
  }

  changeTileset(tilesetId) {
    this._tilesetId = tilesetId;
    this.refresh();
  }

  changeParallax(name, loopX, loopY, sx, sy) {
    this._parallaxName = name;
    this._parallaxZero = Managers.Images.isZeroParallax(this._parallaxName);
    if (this._parallaxLoopX && !loopX) {
      this._parallaxX = 0;
    }
    if (this._parallaxLoopY && !loopY) {
      this._parallaxY = 0;
    }
    this._parallaxLoopX = loopX;
    this._parallaxLoopY = loopY;
    this._parallaxSx = sx;
    this._parallaxSy = sy;
  }

  updateInterpreter() {
    for (;;) {
      this._interpreter.update();
      if (this._interpreter.isRunning()) {
        return;
      }

      if (this._interpreter.eventId() > 0) {
        this.unlockEvent(this._interpreter.eventId());
        $gameSystem._isPlayingDialogue = false;
        this._interpreter.clear();
      }

      if (!this.setupStartingEvent()) {
        return;
      }
    }
  }

  unlockEvent(eventId) {
    if (this._events[eventId] && this._events[eventId]._locked) {
      this._events[eventId].unlock();
    } else if (this._villagers[eventId] && this._villagers[eventId]._locked) {
      this._villagers[eventId].unlock();
    }
  }

  playCutscene(cutsceneName) {
    $gameTemp.reserveCutscene(cutsceneName);
    this.setupStartingEvent();
  }

  setupStartingEvent() {
    this.refreshIfNeeded();
    if (this._interpreter.setupReservedCutscene()) {
      return true;
    }

    if (this._interpreter.setupReservedDialogue()) {
      return true;
    }

    if (this._interpreter.setupReservedCommonEvent()) {
      return true;
    }

    if (this.setupStartingMapEvent()) {
      return true;
    }

    if (this.setupAutorunCommonEvent()) {
      return true;
    }

    return false;
  }

  setupVillagerEvent(eventObj, commonEventId) {
    eventObj.clearStartingFlag();

    const list = [];
    list.push({
      code: 117,
      parameters : [commonEventId],
      indent : 0
    });
    list.push({
      code : 0,
      indent : 0,
      parameters : []
    });

    this._interpreter.setup(list, eventObj.eventId());
  }

  setupStartingMapEvent() {
    const events = this.events();

    for (const event of events) {
      if (event.isStarting()) {
        event.clearStartingFlag();
        this._interpreter.setup(event.list(), event.eventId());
        return true;
      }
    }

    return false;
  }

  setupAutorunCommonEvent() {
    for (const event of $dataCommonEvents) {
      if (event && event.trigger === 1 && $gameSwitches.booleanValue(event.switchId)) {
        this._interpreter.setup(event.list);
        return true;
      }
    }

    return false;
  }

  isAnyEventStarting() {
    return this._eventsStarting > 0;
    // return this.events().some(function(event) {
    //   return event.isStarting();
    // });
  }

  createStaticIconEventAt(iconIndex, x, y, priority, temporary, classType, through) {
    const eventData = new CustomEventData();
    eventData.page.image.iconIndex = iconIndex;
    eventData.page.priorityType = priority;
    eventData.page.through = through;

    return this.addEventAt(eventData, x, y, temporary, classType);
  }

  createStoreObjectEventAt(itemId, x, y) {
    const eventData = new CustomEventData();
    eventData.page.image.iconIndex = Inventory.getItemIconIndex(itemId);
    eventData.page.priorityType = 1;
    eventData.page.through = true;

    const obj = this.addEventAt(eventData, x, y, true, Objects.StoreObject);
    if (!obj) return obj;

    obj.itemId = itemId;
    return obj;
  }

  createStoreSwitchObjectEventAt(switchId, itemId, x, y) {
    const eventData = new CustomEventData();
    eventData.page.image.iconIndex = Inventory.getItemIconIndex(itemId);
    eventData.page.priorityType = 1;
    eventData.page.through = true;

    eventData.changePage(1);
    eventData.page.conditions.switch1Id = switchId;
    eventData.page.conditions.switch1Valid = true;

    const obj = this.addEventAt(eventData, x, y, true, Objects.StoreSwitchObject);
    if (!obj) return obj;
    obj.switchId = switchId;
    obj._pageHandling = true;
    return obj;
  }

  createStaticTileEventAt(tileId, x, y, priority, temporary, classType, through) {
    const eventData = new CustomEventData();
    eventData.page.image.tileId = tileId;
    eventData.page.priorityType = priority;
    eventData.page.through = through;

    return this.addEventAt(eventData, x, y, temporary, classType || Objects.CustomStaticEvent);
  }

  createActionTileEventAt(tileId, x, y, scriptOrCommonEventId, priority, temporary, classType, through) {
    const eventData = new CustomEventData();
    eventData.page.image.tileId = tileId;
    eventData.page.priorityType = priority;
    eventData.page.through = through;

    if (scriptOrCommonEventId !== false) {
      eventData.page.callScriptOrCommonEvent(scriptOrCommonEventId);
    }

    return this.addEventAt(eventData, x, y, temporary, classType);
  }

  createActionTileEventCallAt(tileId, x, y, customEventName, isChild) {
    const eventData = new CustomEventData();
    eventData.page.image.tileId = tileId;
    eventData.page.priorityType = 1;

    if (customEventName !== false) {
      if (isChild) {
        eventData.page.includeChildEvent(customEventName);
      } else {
        eventData.page.playEvent(customEventName);
      }
    }

    return this.addEventAt(eventData, x, y, true);
  }

  createShadowEventAt(x, y) {
    return this.createStaticTileEventAt(381, x, y, 0, true, undefined, true);
  }

  createMultiSpriteActionEventAt(x, y, pages, customEventName = false, priority = 1, classType = undefined, reservedId = undefined) {
    const eventData = new CustomEventData();
    let pageIndex = 0;

    let characterName = '';
    let characterIndex = 0;
    let direction = 2;
    let pattern = 1;
    let dynamicHitbox = false;

    for (const page of pages) {
      if (page.characterName !== undefined) {
        characterName = page.characterName;
      }
      if (page.characterIndex !== undefined) {
        characterIndex = page.characterIndex;
      }
      if (page.direction !== undefined) {
        direction = page.direction;
      }
      if (page.pattern !== undefined) {
        pattern = page.pattern;
      }

      eventData.changePage(pageIndex);

      eventData.page.image.direction = direction;
      eventData.page.image.characterName = characterName;
      eventData.page.image.characterIndex = characterIndex;
      eventData.page.image.pattern = pattern;
      eventData.page.stepAnime = false;
      eventData.page.walkAnime = false;
      eventData.page.directionFix = true;
      eventData.page.priorityType = priority;

      if (page.hitboxWidth !== undefined) {
        eventData.page.hitboxWidth = page.hitboxWidth;
        dynamicHitbox = true;
      }

      if (page.condition) {
        if (page.condition.variableName) {
          const variableId = Variables['_' + page.condition.variableName];
          if (!variableId) {
            throw new Error('Invalid Variable: ' + page.condition.variableName);
          }
          eventData.page.conditions.addVariableCondition(variableId, page.condition.value || 0);
        }
        if (page.condition.switchName) {
          const switchId = Switches['_' + page.condition.switchName];
          if (!switchId) {
            throw new Error('Invalid Switch: ', + page.condition.switchName);
          }

          eventData.page.conditions.addSwitch1Condition(switchId);
        }
        if (page.condition.selfSwitchIndex) {
          eventData.page.conditions.addSelfSwitchCondition(page.condition.selfSwitchIndex);
        }
      }

      if (customEventName !== false) {
        eventData.page.playEvent(customEventName);
      }

      pageIndex++;
    }

    const event = this.addEventAt(eventData, x, y, true, classType, reservedId);

    event._pageHandling = true;
    event._lockPattern = true;
    eventData._dynamicHitbox = dynamicHitbox;
    return event;
  }

  createSimpleEventAt(characterName, x, y, {
    characterIndex,
    d,
    pattern,
    priority,
    temporary,
    classType,
    through,
    customEventName,
    lockPattern,
    directionFix,
    walkAnime,
    stepAnime,
    offsetX,
    offsetY,
    name,
  }) {
    const eventData = new CustomEventData();
    eventData.name = name;

    eventData.page.image.direction = d || 2;
    eventData.page.image.characterName = characterName;
    eventData.page.image.characterIndex = characterIndex || 0;
    eventData.page.image.pattern = pattern || 0;
    eventData.page.stepAnime = stepAnime || false;
    eventData.page.walkAnime = walkAnime || false;
    eventData.page.directionFix = directionFix !== false;
    eventData.page.priorityType = priority || 0;
    eventData.page.through = through === true;

    if (customEventName) {
      eventData.page.playEvent(customEventName);
    }

    const event = this.addEventAt(eventData, x, y, temporary !== false, classType || Objects.CustomStaticEvent);

    if (lockPattern) {
      event._lockPattern = lockPattern;
    }
    if (offsetY) {
      event.offsetY = offsetY;
    }
    if (offsetX) {
      event.offsetX = offsetX;
    }

    return event;
  }

  createStaticEventAt(characterName, characterIndex, x, y, d, pattern, priority, temporary, classType, through, customEventName) {
    return this.createSimpleEventAt(characterName, x, y, {
      characterIndex,
      d,
      pattern,
      priority,
      temporary,
      classType,
      through,
      customEventName,
      stepAnime: false,
      walkAnime: false,
      directionFix: true,
    });
  }

  createActionEventAt(x, y, priority, scriptOrCommonEventId, customEventName) {
    const eventData = new CustomEventData();
    eventData.page.priorityType = priority;

    if (scriptOrCommonEventId !== false) {
      eventData.page.callScriptOrCommonEvent(scriptOrCommonEventId);
    }
    if (customEventName) {
      eventData.page.includeChildEvent(customEventName);
    }

    return this.addEventAt(eventData, x, y, true);
  }

  createActionIconEventAt(iconIndex, x, y, customEventName, isChild) {
    const eventData = new CustomEventData();
    eventData.page.image.iconIndex = iconIndex;
    eventData.page.priorityType = 1;

    if (customEventName !== false) {
      if (isChild) {
        eventData.page.includeChildEvent(customEventName);
      } else {
        eventData.page.playEvent(customEventName);
      }
    }

    return this.addEventAt(eventData, x, y, true);
  }

  createActionItemEventAt(itemName, x, y, customEventName, isChild) {
    const itemData = Inventory.getItemData(itemName);
    if (!itemData) {
      return false;
    }

    return this.createActionIconEventAt(itemData.iconIndex, x, y, customEventName, isChild);
  }

  createEventCallAt(x, y, eventName, priority, isChild, classType) {
    const eventData = new CustomEventData();
    if (priority === undefined) priority = 1;
    eventData.page.priorityType = priority;
    eventData.page.through = true;

    if (isChild) {
      eventData.page.includeChildEvent(eventName);
    } else {
      eventData.page.playEvent(eventName);
    }

    return this.addEventAt(eventData, x, y, true, classType);
  }

  createPlayerImpersonatorEvent(characterName, characterIndex, d, stepAnime, walkAnime, pattern, lockPattern, classType) {
    const createdEvent = $gameMap.createNormalEventAt(characterName, characterIndex, $gamePlayer._realX, $gamePlayer._realY, d, false, true, classType);
    createdEvent.setPosition($gamePlayer._realX, $gamePlayer._realY);
    createdEvent._playerImpersonator = true;
    if (stepAnime !== undefined) createdEvent._stepAnime = stepAnime;
    if (walkAnime !== undefined) createdEvent._walkAnime = walkAnime;
    if (pattern !== undefined) createdEvent._pattern = pattern;
    if (lockPattern !== undefined) createdEvent._lockPattern = lockPattern;

    createdEvent._realX = $gamePlayer._realX;
    createdEvent._x = createdEvent._realX;
    createdEvent._realY = $gamePlayer._realY;
    createdEvent._y = createdEvent._realY;

    return createdEvent;
  }

  createNormalEventAt(characterName, characterIndex, x, y, d, scriptOrCommonEventId, temporary, classType) {
    const eventData = new CustomEventData();
    eventData.page.image.direction = d;
    eventData.page.image.characterName = characterName;
    eventData.page.image.characterIndex = characterIndex;

    if (scriptOrCommonEventId !== false) {
      eventData.page.callScriptOrCommonEvent(scriptOrCommonEventId);
    }

    return this.addEventAt(eventData, x, y, temporary, classType);
  }

  createSingleSpriteEventAt(characterName, x, y, priority, offsetX, offsetY, through, customEventName)  {
    const event = this.createStaticEventAt(characterName, 0, x, y, 2, 0, priority, true, Objects.CustomStaticEvent, through, customEventName || false);
    event._lockPattern = true;

    if (offsetY) {
      event.offsetY = offsetY;
    }

    if (offsetX) {
      event.offsetX = offsetX;
    }

    return event;
  }

  createCollisionArea(x, y, width, height) {
    const endX = x + width;
    const endY = y + height;

    const minX = Math.floor(x);
    const maxX = Math.floor(endX);
    const minY = Math.floor(y);
    const maxY = Math.floor(endY);

    for (let newX = minX; newX <= maxX; newX++) {
      const xOffset = newX < x ? x - minX : 0;
      let boxWidth = 1;
      if (newX === minX) {
        boxWidth = 1 - xOffset;
      } else if (newX === maxX) {
        boxWidth = endX - maxX;
      }

      if (boxWidth !== 1) {
        boxWidth = Math.floor(boxWidth * 4) / 4;
      }

      if (!boxWidth) {
        continue;
      }

      if (!this._collisionBoxesXy[newX]) {
        this._collisionBoxesXy[newX] = {};
      }

      for (let newY = minY; newY <= maxY; newY++) {
        const yOffset = newY < y ? y - minY : 0;

        let boxHeight = 1;
        if (newY === minY) {
          if (newY < maxY) {
            boxHeight = 1 - yOffset;
          } else {
            boxHeight = height;
          }
        } else if (newY === maxY) {
          boxHeight = endY - maxY;
        }

        if (boxHeight !== 1) {
          boxHeight = Math.floor(boxHeight * 4) / 4;
        }

        if (!boxHeight) {
          continue;
        }

        const box = {
          width: boxWidth,
          height: boxHeight,
          x: xOffset,
          y: yOffset,
        };

        if (this._collisionBoxesXy[newX][newY]) {
          this._collisionBoxesXy[newX][newY].push(box);
        } else {
          this._collisionBoxesXy[newX][newY] = [box];
        }
      }
    }
  }

  createDoorEvent(characterName, characterIndex, pattern, x, y, villagerName, offsetX, offsetY, commandPrefix = 'town') {
    const eventData = new CustomEventData();
    eventData.page.image.direction = 2;
    eventData.page.image.characterName = characterName;
    eventData.page.image.characterIndex = characterIndex;
    eventData.note = `<door:${ villagerName }>`;

    eventData.page.command(commandPrefix || 'town', 'door', villagerName);

    const event = this.addEventAt(eventData, x, y, true, Objects.CustomStaticEvent);

    event._priorityType = 1;
    event._pattern = pattern;
    event._lockPattern = true;
    event._directionFix = true;

    if (offsetY) {
      event.offsetY = offsetY;
    }

    if (offsetX) {
      event.offsetX = offsetX;
    }

    return event;
  }

  createVillagerByNameAt(villagerName, x, y, d, returnExisting, classType, eraseExisting = false) {
    if (returnExisting === undefined) returnExisting = true;

    villagerName = Managers.Villagers.getVillagerNameByAlias(villagerName);
    classType = classType || Managers.Content.getVillagerClassType(villagerName) || undefined;

    const actorId = $gameActors.findActorId(villagerName);
    if (actorId > 0) {
      const eventData = this.villagerEvent(actorId);
      if (eventData && !eventData._erased) {
        if (eraseExisting === true) {
          eventData.erase();
        } else if (returnExisting === false) {
          return false;
        }

        return eventData;
      }

      return this.createVillagerAt(actorId, x, y, d, classType);
    } else {
      return false;
    }
  }

  createSeagullEventAt(x, y, d) {
    const eventData = new CustomEventData();
    eventData.name = 'seagull';
    eventData.page.image.direction = d;
    eventData.page.image.characterName = 'animals/seagull';
    eventData.page.image.characterIndex = 0;
    eventData.page.priorityType = EventPriorities.FLYING;
    eventData.page.through = true;
    eventData.page.moveType = 3;
    eventData.page.moveSpeed = 3.5;
    eventData.page.moveFrequency = 5;
    eventData.page.moveRoute.list = [
      {
        code: d / 2,
        parameters: []
      },
      {
        code: 0,
        parameters: []
      }
    ];

    return this.addEventAt(eventData, x, y, true, Objects.ScenarioEvent);
  }

  createAnimalByTypeAt(animalTypeName, x, y, d, classType) {
    const typeData = Managers.Creatures.getCreatureType(animalTypeName, false);

    if (!typeData) return false;

    const eventData = this.createNormalEventAt(typeData._spriteName, typeData._spriteIndex, x, y, d, 0, true, classType);
    return eventData;
  }

  createVillagerAt(actorId, x, y, d, classType) {
    const actorData = $gameActors.actor(actorId);

    const eventData = new CustomEventData();
    eventData.name = actorData._name;
    eventData.id = actorId;
    eventData.x = x;
    eventData.y = y;
    eventData.page.image.direction = d || Direction.DOWN;
    eventData.page.image.characterName = actorData._characterName;
    eventData.page.image.characterIndex = actorData._characterIndex;
    eventData.page.moveSpeed = 2;
    eventData.endAllPages();

    const event = this.addVillagerEvent(eventData, classType);

    event._realX = x;
    event._realY = y;
    event._x = x;
    event._y = y;

    if (actorData.offsetY) event.offsetY = actorData.offsetY;
    if (actorData.offsetX) event.offsetX = actorData.offsetX;

    if (actorData.hitboxX) {
      event._hitboxX = actorData.hitboxX;
    } else {
      event._hitboxX = 0.25;
    }

    if (actorData.hitboxY) {
      event._hitboxY = actorData.hitboxY;
    } else {
      event._hitboxY = 0.45;
    }

    if (actorData.hitboxHeight) {
      event._hitboxHeight = actorData.hitboxHeight;
    } else {
      event._hitboxHeight = 0.4;
    }
    if (actorData.hitboxWidth) {
      event._hitboxWidth = actorData.hitboxWidth;
    } else {
      event._hitboxWidth = 0.5;
    }

    return event;
  }

  createFishAt(x, y, characterIndex) {
    if (characterIndex === undefined || isNaN(characterIndex)) characterIndex = 0;

    const eventData = new CustomEventData();
    eventData.x = x;
    eventData.y = y;
    eventData.page.image.characterName = 'fishes';
    eventData.page.image.characterIndex = characterIndex;
    eventData.endAllPages();

    return this.addEvent(eventData, true, Objects.Fish);
  }

  createLargeFishAt(x, y) {
    return this.createFishAt(x, y, 1);
  }

  createTinyFishAt(x, y) {
    return this.createFishAt(x, y, 2);
  }

  createActorAt(actorId, x, y, d, scriptOrCommonEventId, temporary) {
    this.createNormalEventAt($gameActors.actor(actorId).characterName(), $gameActors.actor(actorId).characterIndex(), x, y, d, scriptOrCommonEventId, temporary);
  }

  createFenceEventAt(x, y, tileId) {
    const eventData = new CustomEventData();
    eventData.x = x;
    eventData.y = y;
    eventData.page.image.tileId = tileId;
    eventData.endAllPages();

    return this.addEvent(eventData, true);
  }

  createParallelProcess(scriptOrCommonEventId, temporary, autoErase) {
    const eventData = new CustomEventData();
    eventData.page.trigger = EventTriggers.PARALLEL;
    eventData.page.priorityType = EventPriorities.UNDER_PLAYER;

    eventData.page.callScriptOrCommonEvent(scriptOrCommonEventId);

    if (autoErase === true) {
      eventData.page.addCommand({
        code: 214
      });
    }

    return this.addEventAt(eventData, temporary);
  }

  tileSections() {
    return 4;
  }

  defaultStepSize() {
    return Math.fix(1 / this.tileSections());
  }

  isValid(x, y) {
    return x >= 0 && x.ceil() < this.width() && y >= 0 && y.ceil() < this.height();
  }

  isInsideAnyHouse() {
    return HouseTilesets.includes(this._tilesetId);
  }

  isInside() {
    return InsideTilesets.includes(this._tilesetId);
  }

  isCave() {
    return this._tilesetId === Tilesets.CAVES;
  }

  isCemetery() {
    if (this._mapId !== Maps.ORANGE_TOWN) {
      return false;
    }

    if ($gamePlayer.x < 41 || $gamePlayer.x > 60)  {
      return false;
    }

    if ($gamePlayer.y < 57 || $gamePlayer.y > 70) {
      return false;
    }

    if ($gamePlayer.x > 54 && $gamePlayer.y > 66) {
      return false;
    }

    return true;
  }

  isFarm() {
    const farmMaps = [
      Maps.FARM,
      Maps.FARM_CENTER
    ];

    return farmMaps.includes(this._realMapId);
  }

  isMountain() {
    const mountainMaps = [
      Maps.MOUNTAIN,
      Maps.MOUNTAIN_TOP
    ];

    return mountainMaps.includes(this._realMapId);
  }

  isTown() {
    const townMaps = [
      Maps.ORANGE_TOWN
    ];

    return townMaps.includes(this._realMapId);
  }

  isMine() {
    const mineMaps = [
      Maps.MAIN_MINE,
      Maps.MINE_LEVEL_2,
      Maps.SMALL_CAVE,
      Maps.CAVE_FARM,
      Maps.GOLD_MINE,
      Maps.PATH_TO_RUINS
    ];

    return mineMaps.includes(this._realMapId);
  }

  isEternalSpring() {
    return this._tilesetId == Tilesets.ETERNAL_SPRING;
  }

  isEternalSummer() {
    return this._tilesetId == Tilesets.ETERNAL_SUMMER;
  }

  isEternalFall() {
    return this._tilesetId == Tilesets.ETERNAL_FALL;
  }

  isEternalWinter() {
    return this._tilesetId == Tilesets.ETERNAL_WINTER;
  }

  allowAnimations() {
    return Graphics.fps > 30;
  }

  openDoor(eventId) {
    const doorEvent = this.event(eventId);
    if (doorEvent) {
      doorEvent.doDoorAnimation();
    }
  }

  closeDoor(eventId) {
    const doorEvent = this.event(eventId);
    if (doorEvent) {
      doorEvent.doCloseDoorAnimation();
    }
  }

  reactToPlayerMovement(tilesAroundPlayer) {
    let len = tilesAroundPlayer.length;
    let livingBeings = [];
    let i = 0;
    for (i = 0; i < len; i++) {
      const tile = tilesAroundPlayer[i];
      const newBeings = this.livingBeingsXy(tile.x, tile.y);

      livingBeings = livingBeings.concat(newBeings);
    }

    len = livingBeings.length;
    for (i = 0; i < len; i++) {
      livingBeings[i].reactToPlayerMovement();
    }
  }

  checkCutscenes(mapId) {
    if (mapId == Maps.FARM && $gameMap.isHome()) {
      if (Managers.Visitors.checkVisitors()) {
        return true;
      }
    }

    return Managers.Content.checkCutscenes(mapId);
  }

  processTileTouch(x, y, realX, realY) {
    if (Managers.Content.processTileTouch(x, y)) return;
    if (Managers.Map.processTileTouch(x, y)) return;

    if (!$gamePlayer.canMove()) return;

    if (TouchInput.isPressed()) {
      if (this._mouseDelay > 0) {
        return;
      }

      const triggered = TouchInput.isTriggered();
      const playerHeight = $gamePlayer.getHeightLevel();
      if (Managers.Map.getHeightLevel(x, y) != playerHeight) return;

      const events = $gameMap.anythingTouchableXy(realX, realY);
      const lowPriority = [];

      for (let i = 0; i < events.length; i++) {
        if (!events[i]) continue;
        if (events[i]._erased) continue;

        if (!triggered) {
          // Creatures don't accept mouse down, only clicks
          if (events[i] instanceof Objects.Creature) {
            continue;
          }

          // Regular events can't be triggered by pressed down mouse button, only by clicks. With a few exceptions
          if (events[i] instanceof Objects.MapEvent) {
            if (!events[i].shouldAcceptMouseDown()) {
              continue;
            }
          }
        }

        const distance = events[i].distanceToPlayer();
        if (distance >= events[i].maxActivationDistance()) continue;

        if (!events[i].hasAnythingToRun(true)) {
          if (events[i].hasAnythingToRunOnLowPriority(true)) {
            lowPriority.push(events[i]);
          }
          continue;
        }

        $gamePlayer.turnTowardCharacter(events[i]);
        if (events[i].start(true)) {
          this.setMouseDelayIfTheresNone(20);
        }
        return;
      }

      if (triggered) {
        if ($gameParty.hasCompanion()) {
          const follower = $gamePlayer.follower();
          if (follower && follower.pos(x, y)) {
            follower.start();
            return true;
          }
        }
      }

      if (lowPriority.length > 0) {
        $gamePlayer.turnTowardCharacter(lowPriority[0]);
        if (lowPriority[0].start()) {
          this.setMouseDelayIfTheresNone(20);
        }
        return;
      }

      if (triggered) {
        if (Managers.FarmObjects.triggerProtectedTile(x, y)) {
          this.setMouseDelayIfTheresNone(10);
          return;
        }

        if (this.dropItem(x, y)) {
          return;
        }
      }
    }
  }

  dropItem(x, y) {
    const dropDistance = this.distance(x, y, $gamePlayer.x, $gamePlayer.y);
    if (dropDistance > 2) {
      return false;
    }

    if (!Inventory.isHoldingItem()) {
      return false;
    }

    if (!$gamePlayer.canChangeItems()) {
      return false;
    }

    if ($gamePlayer.isRiding()) {
      return false;
    }

    if (Inventory.dropItem(true, false, x, y)) {
      $gamePlayer.turnTowardPosition(x, y);
      this.addMouseDelay(20);
      return true;
    }

    return false;
  }

  processMapTouch() {
    const realX = $gameMap.canvasToMapX(TouchInput.x, true);
    const realY = $gameMap.canvasToMapY(TouchInput.y, true);

    this.processTileTouch(Math.floor(realX), Math.floor(realY), realX, realY);
  }

  getMapName() {
    return $dataMapInfos[$gameMap._realMapId].name;
  }

  isGrass(x, y) {
    return this.terrainTag(x, y) == TerrainTags.GRASS;
  }

  countCreaturesByEventType(eventType) {
    const len = this._creatures.length;
    let count = 0;
    for (let i = 0; i < len; i++) {
      if (!this._creatures[i]) continue;
      if (this._creatures[i]._erased) continue;

      if (this._creatures[i] instanceof eventType) {
        count++;
      }
    }

    return count;
  }

  isSea() {
    return false;
  }

  graphicsWidth() {
    return Graphics.width / this.zoom.x;
  }

  graphicsHeight() {
    return Graphics.height / this.zoom.y;
  }

  hideCreatures() {
    for (const event of this._creatures) {
      if (!event) continue;
      if (event._erased) continue;

      event.hide();
    }
  }

  showHiddenCreatures() {
    for (const event of this._creatures) {
      if (!event) continue;
      if (event._erased) continue;

      event.show();
    }
  }

  getDataForSaveFile() {
    const data = {
      _interpreter : this._interpreter,
      _mapId : this._mapId,
      _realMapId : this._realMapId,
      _tilesetId : this._tilesetId,
      _events : [],
      _farmObjectsXy : {},
      _villagers : [],
      _creatures : [],
      _commonEvents : [],
      _displayX : this._displayX,
      _displayY : this._displayY,
      _nameDisplay : this._nameDisplay,
      _scrollDirection : this._scrollDirection,
      _scrollRest : this._scrollRest,
      _scrollSpeed : this._scrollSpeed,
      _scrollTargetX : this._scrollTargetX,
      _scrollTargetY : this._scrollTargetY,
      _parallaxName : this._parallaxName,
      _parallaxZero : this._parallaxZero,
      _parallaxLoopX : this._parallaxLoopX,
      _parallaxLoopY : this._parallaxLoopY,
      _parallaxSx : this._parallaxSx,
      _parallaxSy : this._parallaxSy,
      _parallaxX : this._parallaxX,
      _parallaxY : this._parallaxY,
      _buttonUpdateDelay : 10,
      _grassUpdateDelay : 10,
      _fullyLoaded : false,
      _mouseDelay : 0,
      _grassEventsPendingUpdate : [],
      _eventsStarting : 0,
      _allowRain : this._allowRain
    };

    return data;
  }

  addMouseDelay(frames) {
    this._mouseDelay = Utils.getFrameCount(frames);
  }

  setMouseDelayIfTheresNone(frames) {
    if (this._mouseDelay === 0) {
      this._mouseDelay = Utils.getFrameCount(frames);
    }
  }

  refreshWholeMap() {
    $gamePlayer.requestMapReload();
    $gamePlayer.quickTransfer($gameMap._mapId, $gamePlayer._x, $gamePlayer._y, $gamePlayer._direction, 0, false, false);
    Managers.Villagers.killAllVillagerEvents();
    this.refreshTilemap();
  }

  refreshTilemap() {
    if (Managers.Scenes._scene instanceof GameScenes.Map) {
      Managers.Scenes._scene.refreshTilemap();
    }
  }

  isHome() {
    return Managers.Map.isHomeMap(this._mapId);
  }

  isInsideOwnHouse() {
    if (this.isHome()) return true;

    return false;
  }

  setTimeout(callback, frames, avoidLock) {
    $gameTemp.setTimeout(callback, frames, avoidLock);
  }

  updateTimeouts() {
    $gameTemp.updateTimeouts();
  }

  eventIsVillagerDoor(event, villagerName) {
    return !!event && !event._erased && event._villagerDoor === villagerName;
  }

  getVillagerDoorEvent(villagerName) {
    return this._events.find(event => this.eventIsVillagerDoor(event, villagerName));
  }

  openVillagerDoor(villagerName, playSound) {
    const event = this.getVillagerDoorEvent(villagerName);
    if (event) {
      event.doDoorAnimation(playSound);
    }
  }

  closeVillagerDoor(villagerName, playSound) {
    const event = this.getVillagerDoorEvent(villagerName);
    if (event) {
      event.doCloseDoorAnimation(playSound);
    }
  }

  toggleVillagerDoor(villagerName) {
    const event = this.getVillagerDoorEvent(villagerName);
    if (event) {
      if (event._direction === Direction.DOWN) {
        event.doDoorAnimation();
      } else if (event._direction === Direction.UP) {
        event.doCloseDoorAnimation();
      }
    }
  }

  openMapDoor(mapId, eventId) {
    if ($gameMap._mapId !== mapId) return;

    const doorEvent = $gameMap.event(eventId);
    if (doorEvent) {
      doorEvent.doDoorAnimation();
    }
  }

  closeMapDoor(mapId, eventId) {
    if ($gameMap._mapId !== mapId) return;

    const doorEvent = $gameMap.event(eventId);
    if (doorEvent) {
      doorEvent.doCloseDoorAnimation();
    }
  }

  closeEveryDoor() {
    const doors = this._events.filter(event => !!event && !event._erased && event._villagerDoor);
    if (!doors || !doors.length) {
      return;
    }

    for (const door of doors) {
      if (door && door._direction !== Direction.DOWN) {
        door._direction = Direction.DOWN;
      }
    }
  }

  updateDoorsState() {
    if (Switches.isSpaDoorOpen) {
      const spaDoor = this.getVillagerDoorEvent('SPA');
      if (spaDoor) {
        spaDoor._direction = Direction.UP;
      }
    }

    // #ToDo: open shop doors and call this automatically
  }

  updateBackgroundSounds() {
    if (!this._soundSources.length) {
      Engine.Audio.stopBgs();
      return;
    }

    if (this._soundDelay > 0) {
      this._soundDelay--;
      return;
    }
    this._soundDelay = Utils.getFrameCount(30);

    let closestSfx = false;
    let closestSfxPriority = Infinity;

    for (const sfx of this._soundSources) {
      const distance = $gamePlayer.distanceToPosition(sfx.x, sfx.y);
      const priority = distance > 0 ? distance / sfx.priority : 0;

      if (priority < closestSfxPriority) {
        closestSfxPriority = priority;
        closestSfx = sfx;
      }
    }

    if (!closestSfx) {
      return;
    }

    const volume = 100 - closestSfxPriority * 3;
    Engine.Audio.playBgs({
      name: closestSfx.name,
      volume: volume > 0 ? volume: 0.5,
    });
  }

  addSoundSource(x, y, name, priority = 1) {
    this._soundSources.push({
      x,
      y,
      name,
      priority,
    });
  }

  getEventByNameAt(name, x, y) {
    const events = this._events.filter(event => !!event && !event._erased && event._x === x && event._y === y && event.name === name);
    if (events && events.length) {
      return events[0];
    }
  }
};