const { Input } = require('engine/core/Input');
const { Inventory } = require('engine/managers/Inventory');
const { Audio } = require('engine/managers/Audio');

//-----------------------------------------------------------------------------
// Game_Interpreter
//
// The interpreter for running event commands.

Objects.Interpreter = class Interpreter {
  constructor() {
    this.initialize(...arguments);
  }

  initialize(depth) {
    this._depth = depth || 0;
    this.checkOverflow();
    this.clear();
    this._branch = {};
    this._params = [];
    this._indent = 0;
    this._frameCount = 0;
    this._freezeChecker = 0;
    this._reservedChoices = [];
  }

  checkOverflow() {
    if (this._depth >= 100) {
      throw new Error('Common event calls exceeded the limit');
    }
  }

  clear() {
    this._mapId = 0;
    this._eventId = 0;
    this._list = null;
    this._index = 0;
    this._waitCount = 0;
    this._waitMode = '';
    this._comments = '';
    this._character = null;
    this._childInterpreter = null;
    this._reservedChoices = [];
  }

  setup(list, eventId) {
    this.clear();
    this._mapId = $gameMap.mapId();
    this._eventId = eventId || 0;
    this._list = list;
  }

  eventId() {
    return this._eventId;
  }

  isOnCurrentMap() {
    return this._mapId === $gameMap.mapId();
  }

  setupReservedCutscene() {
    if (!$gameTemp.isCutsceneReserved()) {
      return false;
    }

    const cutsceneName = $gameTemp.reservedCutscene();

    const eventId = Managers.CommonEvent.findEventIdByName(cutsceneName);
    if (eventId > 0) {
      const list = MVC.deepClone($dataCommonEvents[eventId].list);

      $gameSystem.startCutscene();

      const end = list.pop();
      list.push({
        code : 355,
        indent : 0,
        parameters : ['$gameSystem.endCutscene()']
      });
      list.push(end);

      $gameTemp.clearCutscene();
      this.setup(list);

      Managers.History.registerCutscene(cutsceneName);
      return true;
    }

    return false;
  }

  setupReservedDialogue() {
    if ($gameTemp.isDialogueReserved()) {
      this.setup($gameTemp.reservedDialogue());
      $gameTemp.clearDialogue();
      return true;
    } else {
      return false;
    }
  }

  setupReservedCommonEvent() {
    if (!$gameTemp.isCommonEventReserved()) {
      return false;
    }

    this.setup($gameTemp.reservedCommonEvent().list);
    return true;
  }

  isRunning() {
    return !!this._list;
  }

  update() {
    while (this.isRunning()) {
      if (this.updateChild() || this.updateWait()) {
        break;
      }
      if (Managers.Scenes.isSceneChanging()) {
        break;
      }
      if (!this._waitMode) {
        this._character = null;
      }

      if (Managers.CommonEvent._skipped) {
        if (this == $gameMap._interpreter) {
          Managers.CommonEvent.doSkip();
          $gameMessage.clear();
          this._list = [];
          return;
        }
      }

      if (!this.executeCommand()) {
        break;
      }
      if (this.checkFreeze()) {
        break;
      }
    }
  }

  updateChild() {
    if (this._childInterpreter) {
      this._childInterpreter.update();

      // If the childInterpreter no longer exists, it means that the currently running event replaced itself.
      // This can happen for example when the player chooses "Just Chat" on a shop list and the game runs a new conversation from it
      if (!this._childInterpreter) return true;

      if (this._childInterpreter.isRunning()) {
        return true;
      } else {
        this._childInterpreter = null;
      }
    }
    return false;
  }

  updateWait() {
    return this.updateWaitCount() || this.updateWaitMode();
  }

  updateWaitCount() {
    if (this._waitCount > 0) {
      // Don't fast forward wait times, it can break things
      if (window.$gameTemp && $gameTemp.isFastForwardFrame()) {
        return true;
      }

      this._waitCount--;
      return true;
    }
    return false;
  }

  isBalloonPlaying() {
    if (this._waitMode !== 'balloon') return false;
    if (!this._character) return false;

    return this._character.isBalloonPlaying();
  }

  updateWaitMode() {
    let waiting = false;
    switch (this._waitMode) {
      case 'message':
        waiting = $gameMessage.isBusy();
        break;
      case 'transfer':
        waiting = $gamePlayer.isTransferring();
        break;
      case 'scroll':
        waiting = $gameMap.isScrolling();
        break;
      case 'route':
        waiting = this._character.isMoveRouteForcing();
        break;
      case 'villager':
        if (this._waitingFor) {
          if (this._waitingFor._xDestination !== undefined && this._waitingFor._yDestination !== undefined) {
            waiting = true;
          }
          if (this._waitingFor._toolId !== undefined) {
            waiting = true;
          }
          if (this._waitingFor.isMoving()) {
            waiting = true;
          }
          if (this._waitingFor.isJumping()) {
            waiting = true;
          }
          if (this._waitingFor.isBalloonPlaying()) {
            waiting = true;
          }
        }
        break;
      case 'animation':
        waiting = this._character.isAnimationPlaying();
        break;
      case 'balloon':
        waiting = this._character.isBalloonPlaying();
        if (!waiting) {
          this._character = null;
        }
        break;
      case 'gather':
        waiting = $gamePlayer.areFollowersGathering();
        break;
      case 'video':
        waiting = Graphics.isVideoPlaying();
        break;
      case 'image':
        waiting = !Managers.Images.isReady();
        break;
      default:
        waiting = false;
        break;
    }
    if (!waiting) {
      this._waitMode = '';
    }
    return waiting;
  }

  setWaitMode(waitMode) {
    this._waitMode = waitMode;
  }

  wait(duration) {
    this._waitCount = Utils.getFrameCount(duration);
  }

  waitFor(villagerName) {
    if (villagerName.toLowerCase() == 'camera') {
      this._waitMode = 'scroll';
      return;
    }

    this._waitingFor = Managers.Villagers.getVillagerData(villagerName);
    if (this._waitingFor) {
      this._waitMode = 'villager';
    }
  }

  fadeSpeed() {
    return 24;
  }

  executeCommand() {
    const command = this.currentCommand();
    if (command) {
      this._params = command.parameters;
      this._indent = command.indent;
      const methodName = `command${command.code}`;
      if (typeof this[methodName] === 'function') {
        if (!this[methodName]()) {
          return false;
        }
      }
      this._index++;
    } else {
      this.terminate();
    }
    return true;
  }

  checkFreeze() {
    if (this._frameCount !== Graphics.frameCount) {
      this._frameCount = Graphics.frameCount;
      this._freezeChecker = 0;
    }
    this._freezeChecker++;

    if (this._freezeChecker >= 100000) {
      return true;
    } else {
      return false;
    }
  }

  terminate() {
    this._list = null;
    this._comments = '';
  }

  skipBranch() {
    while (this._list[this._index + 1].indent > this._indent) {
      this._index++;
    }
  }

  currentCommand() {
    return this._list[this._index];
  }

  nextEventCode() {
    const command = this._list[this._index + 1];
    if (command) {
      return command.code;
    } else {
      return 0;
    }
  }

  iterateActorId(param, callback) {
    if (param === 0) {
      $gameParty.members().forEach(callback);
    } else {
      const actor = $gameActors.actor(param);
      if (actor) {
        callback(actor);
      }
    }
  }

  iterateActorEx(param1, param2, callback) {
    if (param1 === 0) {
      this.iterateActorId(param2, callback);
    } else {
      this.iterateActorId($gameVariables.value(param2), callback);
    }
  }

  iterateActorIndex(param, callback) {
    if (param < 0) {
      $gameParty.members().forEach(callback);
    } else {
      const actor = $gameParty.members()[param];
      if (actor) {
        callback(actor);
      }
    }
  }

  character(param) {
    switch (param) {
      case -1:
        return $gamePlayer;
      case -2:
        return $gamePlayer.followers[0];
      default:
        return $gameMap.event(param > 0 ? param : this._eventId);
    }
  }

  operateValue(operation, operandType, operand) {
    const value = operandType === 0 ? operand : $gameVariables.value(operand);
    return operation === 0 ? value : -value;
  }

  changeHp(target, value, allowDeath) {
    if (target.isAlive()) {
      if (!allowDeath && target.hp <= -value) {
        value = 1 - target.hp;
      }
      target.gainHp(value);
      if (target.isDead()) {
        target.performCollapse();
      }
    }
  }

  executeShowBalloon(target, balloonId, wait) {
    let character;
    if (typeof(target) == 'object') {
      character = target;
    } else if (typeof(target) == 'string' && isNaN(target)) {
      if (target.toUpperCase() == 'PLAYER') {
        character = $gamePlayer;
      } else {
        character = $gameMap.villagerEventByName(target);
      }
    } else {
      character = this.character(target);
    }

    if (character) {
      this._character = character;
      character.requestBalloon(balloonId);
      if (wait) {
        this.setWaitMode('balloon');
      }
    }

    return character;
  }

  executeCommandShowText(faceName, faceIndex, backgroundType, positionType, message) {
    if (!$gameMessage.isBusy()) {
      $gameMessage.setFaceImage(faceName, faceIndex);
      $gameMessage.setBackground(backgroundType);
      $gameMessage.setPositionType(positionType);

      if (!!message && message.length > 0) {
        $gameMessage.addText(message);
        this.setWaitMode('message');
      }
    }
  }

  // Show Text
  command101() {
    if (!$gameMessage.isBusy()) {
      this.executeCommandShowText(this._params[0], this._params[1], this._params[2], this._params[3]);

      while (this.nextEventCode() === 401) { // Text data
        this._index++;
        if (this._list[this._index].code === 401) {
          $gameMessage.addText(this.currentCommand().parameters[0]);
        }
        if ($gameMessage._texts.length >= $gameSystem.messageRows()) break;
      }

      switch (this.nextEventCode()) {
        case 102: // Show Choices
          this._index++;
          this.setupChoices(this.currentCommand().parameters);
          break;
        case 103: // Input Number
          this._index++;
          this.setupNumInput(this.currentCommand().parameters);
          break;
        default:
          break;
      }
      this._index++;
      this.setWaitMode('message');
    }
    return false;
  }

  isContinueMessageString() {
    if (this.nextEventCode() === 101 && $gameSystem.messageRows() > 4) {
      return true;
    } else {
      return this.nextEventCode() === 401;
    }
  }

  // Show Choices
  command102() {
    if (!$gameMessage.isBusy()) {
      this.setupChoices(this._params);
      this._index++;
      this.setWaitMode('message');
    }
    return false;
  }

  setupChoices(params) {
    const choices = params[0].clone();
    let cancelType = params[1];
    const defaultType = params.length > 2 ? params[2] : 0;
    const positionType = params.length > 3 ? params[3] : 2;
    const background = params.length > 4 ? params[4] : 0;
    if (cancelType >= choices.length) {
      cancelType = -2;
    }
    $gameMessage.setChoices(choices, defaultType, cancelType);
    $gameMessage.setChoiceBackground(background);
    $gameMessage.setChoicePositionType(positionType);
    $gameMessage.setChoiceCallback(n => {
      this._branch[this._indent] = n;
      window.responseIndex = n;
      if (choices.length > n && n >= 0) {
        window.response = choices[n];
      } else {
        window.response = '';
      }
      $gameTemp.setResponse(window.response);
    });
  }

  // When [**]
  command402() {
    if (this._branch[this._indent] !== this._params[0]) {
      this.skipBranch();
    }
    return true;
  }

  // When Cancel
  command403() {
    if (this._branch[this._indent] >= 0) {
      this.skipBranch();
    }
    return true;
  }

  // Input Number
  command103() {
    if (!$gameMessage.isBusy()) {
      this.setupNumInput(this._params);
      this._index++;
      this.setWaitMode('message');
    }
    return false;
  }

  setupNumInput(params) {
    if (params.length > 2 && params[2]) {
      $gameMessage.setVariableInput(params[0], params[1]);
    } else {
      $gameMessage.setNumberInput(params[0], params[1]);
    }
  }

  // Show Scrolling Text
  command105() {
    if (!$gameMessage.isBusy()) {
      $gameMessage.setScroll(this._params[0], this._params[1]);
      while (this.nextEventCode() === 405) {
        this._index++;
        $gameMessage.add(this.currentCommand().parameters[0]);
      }
      this._index++;
      this.setWaitMode('message');
    }
    return false;
  }

  // Comment
  command108() {
    this._comments = [this._params[0]];
    while (this.nextEventCode() === 408) {
      this._index++;
      this._comments.push(this.currentCommand().parameters[0]);
    }
    return true;
  }

  commandIf() {
    let result = false;
    try {
      const parser = new PluginConditionParser(this);
      const args = this._params[0].split(" ");
      const command = args.shift();

      result = parser.parse(command, args);
      if (result === undefined) {
        result = !!eval(this._params[0]); // jshint ignore:line
      }
    }
    catch(e) {
      throw new Error(`Failed to execute conditional script:<br/><br/>${this._params[1]}<br/><br/>parsed as:<br/><br/>${this._params[0]}<br/><br/>Error:<br/><br/>${e}`);
    }

    this._branch[this._indent] = result;
    if (this._branch[this._indent] === false) {
      this.skipBranch();
    }
    return true;
  }

  // Conditional Branch
  command111() {
    let result = false;
    switch (this._params[0]) {
      case 0: // Switch
        result = ($gameSwitches.booleanValue(this._params[1]) === (this._params[2] === 0));
        break;
      case 1: // Variable
        var value1 = $gameVariables.value(this._params[1]);
        var value2;
        if (this._params[2] === 0) {
          value2 = this._params[3];
        } else {
          value2 = $gameVariables.value(this._params[3]);
        }
        switch (this._params[4]) {
          case 0: // Equal to
            result = (value1 == value2);
            break;
          case 1: // Greater than or Equal to
            result = (value1 >= value2);
            break;
          case 2: // Less than or Equal to
            result = (value1 <= value2);
            break;
          case 3: // Greater than
            result = (value1 > value2);
            break;
          case 4: // Less than
            result = (value1 < value2);
            break;
          case 5: // Not Equal to
            result = (value1 !== value2);
            break;
          default:
            break;
        }
        break;
      case 2: // Self Switch
        if (this._eventId > 0) {
          const key = [this._mapId, this._eventId, this._params[1]];
          result = ($gameSelfSwitches.value(key) === (this._params[2] === 0));
        }
        break;
      case 3: // Timer
        if ($gameTimer.isWorking()) {
          if (this._params[2] === 0) {
            result = ($gameTimer.seconds() >= this._params[1]);
          } else {
            result = ($gameTimer.seconds() <= this._params[1]);
          }
        }
        break;
      case 4: // Actor
        var actor = $gameActors.actor(this._params[1]);
        if (actor) {
          const n = this._params[3];
          switch (this._params[2]) {
            case 0: // In the Party
              result = $gameParty.members().contains(actor);
              break;
            case 1: // Name
              result = (actor.name() === n);
              break;
            default: // State
              result = false;
              break;
          }
        }
        break;
      case 6: // Character
        var character = this.character(this._params[1]);
        if (character) {
          result = (character.direction() === this._params[2]);
        }
        break;
      case 7: // Gold
        switch (this._params[2]) {
          case 0: // Greater than or equal to
            result = (Inventory.gold >= this._params[1]);
            break;
          case 1: // Less than or equal to
            result = (Inventory.gold <= this._params[1]);
            break;
          case 2: // Less than
            result = (Inventory.gold < this._params[1]);
            break;
          default:
            break;
        }
        break;
      case 8: // Item
        result = Inventory.hasItem($dataItems[this._params[1]]);
        break;
      case 9: // Weapon
        result = false;
        break;
      case 10: // Armor
        result = false;
        break;
      case 11: // Button
        result = Input.isPressed(this._params[1]);
        break;
      case 12: // Script
        try {
          const parser = new PluginConditionParser(this);
          const args = this._params[1].split(" ");
          let command = args.shift();
          let reverse = false;

          if (command.toUpperCase() === 'NOT') {
            reverse = true;
            command = args.shift();
          }

          result = parser.parse(command, args);
          if (result === undefined) {
            result = !!eval(this._params[1]); // jshint ignore:line
          }

          if (reverse) {
            result = !result;
          }
        }
        catch(e) {
          console.log(...log(this._params[1]));
          throw e;
        }
        break;
      default:
        result = false;
        break;
    }
    this._branch[this._indent] = result;
    if (this._branch[this._indent] === false) {
      this.skipBranch();
    }
    return true;
  }

  // Else
  command411() {
    if (this._branch[this._indent] !== false) {
      this.skipBranch();
    }
    return true;
  }

  // Loop
  command112() {
    return true;
  }

  // Repeat Above
  command413() {
    do {
      this._index--;
    } while (this.currentCommand().indent !== this._indent);
    return true;
  }

  // Break Loop
  command113() {
    let depth = 0;
    while (this._index < this._list.length - 1) {
      this._index++;
      const command = this.currentCommand();

      if (command.code === 112)
        depth++;

      if (command.code === 413) {
        if (depth > 0)
          depth--;
        else
          break;
      }
    }
    return true;
  }

  // Exit Event Processing
  command115() {
    this._index = this._list.length;
    return true;
  }

  // Common Event
  command117() {
    const commonEvent = $dataCommonEvents[this._params[0]];
    if (commonEvent) {
      const eventId = this.isOnCurrentMap() ? this._eventId : 0;
      this.setupChild(commonEvent.list, eventId);
    }
    return true;
  }

  setupChild(list, eventId) {
    this._childInterpreter = new Objects.Interpreter(this._depth + 1);
    this._childInterpreter.setup(list, eventId);
  }

  includeChild(list) {
    const newList = MVC.deepClone(list);
    newList.splice(newList.length - 1, 1);

    for (let i = 0; i < newList.length; i++) {
      newList[i].indent += this._indent;
    }

    const args = [this._index + 1, 0].concat(newList);

    const oldList = MVC.shallowClone(this._list);
    Array.prototype.splice.apply(oldList, args);
    this._list = oldList;
    return true;
  }

  // Label
  command118() {
    return true;
  }

  // Jump to Label
  command119() {
    const labelName = this._params[0];
    for (let i = 0; i < this._list.length; i++) {
      const command = this._list[i];
      if (command.code === 118 && command.parameters[0] === labelName) {
        this.jumpTo(i);
        return undefined;
      }
    }
    return true;
  }

  jumpTo(index) {
    const lastIndex = this._index;
    const startIndex = Math.min(index, lastIndex);
    const endIndex = Math.max(index, lastIndex);
    let indent = this._indent;
    for (let i = startIndex; i <= endIndex; i++) {
      const newIndent = this._list[i].indent;
      if (newIndent !== indent) {
        this._branch[indent] = null;
        indent = newIndent;
      }
    }
    this._index = index;
  }

  // Control Switches
  command121() {
    for (let i = this._params[0]; i <= this._params[1]; i++) {
      $gameSwitches.setValue(i, this._params[2] === 0);
    }
    return true;
  }

  // Control Variables
  command122() {
    let value = 0;
    switch (this._params[3]) { // Operand
      case 0: // Constant
        value = this._params[4];
        break;
      case 1: // Variable
        value = $gameVariables.value(this._params[4]);
        break;
      case 2: // Random
        value = this._params[5] - this._params[4] + 1;
        for (let i = this._params[0]; i <= this._params[1]; i++) {
          this.operateVariable(i, this._params[2], this._params[4] + Math.randomInt(value));
        }
        return true;
      case 3: // Game Data
        value = this.gameDataOperand(this._params[4], this._params[5], this._params[6]);
        break;
      case 4: // Script
        value = MVC.safeEval(this._params[4]); // jshint ignore:line
        break;
      default:
        break;
    }
    for (let i = this._params[0]; i <= this._params[1]; i++) {
      this.operateVariable(i, this._params[2], value);
    }
    return true;
  }

  gameDataOperand(type, param1, param2) {
    switch (type) {
      case 0: // Item
        return Inventory.numItems($dataItems[param1]);
      case 1: // Weapon
        return 0;
      case 2: // Armor
        return 0;
      case 3: // Actor
        var actor = $gameActors.actor(param1);
        if (actor) {
          switch (param2) {
            case 0: // Level
              return actor.level;
            case 1: // EXP
              return actor.currentExp();
            case 2: // HP
              return actor.hp;
            case 3: // MP
              return actor.mp;
            default: // Parameter
              if (param2 >= 4 && param2 <= 11) {
                return actor.param(param2 - 4);
              }
          }
        }
        break;
      case 4: // Enemy
        break;
      case 5: // Character
        var character = this.character(param1);
        if (character) {
          switch (param2) {
            case 0: // Map X
              return character.x;
            case 1: // Map Y
              return character.y;
            case 2: // Direction
              return character.direction();
            case 3: // Screen X
              return character.screenX();
            case 4: // Screen Y
              return character.screenY();
            default:
              break;
          }
        }
        break;
      case 6: // Party
        actor = $gameParty.members()[param1];
        return actor ? actor.actorId() : 0;
      case 7: // Other
        switch (param1) {
          case 0: // Map ID
            return $gameMap.mapId();
          case 1: // Party Members
            return $gameParty.size();
          case 2: // Gold
            return Inventory.gold;
          case 3: // Steps
            return $gameParty.steps();
          case 4: // Play Time
            return $gameSystem.playtime();
          case 5: // Timer
            return $gameTimer.seconds();
          case 6: // Save Count
            return $gameSystem.saveCount();
          default:
            return 0;
        }
      default:
        break;
    }
    return 0;
  }

  operateVariable(variableId, operationType, value) {
    try {
      let oldValue = $gameVariables.value(variableId);
      switch (operationType) {
        case 0: // Set
          $gameVariables.setValue(variableId, oldValue = value);
          break;
        case 1: // Add
          $gameVariables.setValue(variableId, oldValue + value);
          break;
        case 2: // Sub
          $gameVariables.setValue(variableId, oldValue - value);
          break;
        case 3: // Mul
          $gameVariables.setValue(variableId, oldValue * value);
          break;
        case 4: // Div
          $gameVariables.setValue(variableId, oldValue / value);
          break;
        case 5: // Mod
          $gameVariables.setValue(variableId, oldValue % value);
          break;
        default:
          break;
      }
    } catch (e) {
      $gameVariables.setValue(variableId, 0);
    }
  }

  // Control Self Switch
  command123() {
    if (this._eventId > 0) {
      const key = [this._mapId, this._eventId, this._params[0]];
      $gameSelfSwitches.setValue(key, this._params[1] === 0);
    }
    return true;
  }

  // Control Timer
  command124() {
    if (this._params[0] === 0) { // Start
      $gameTimer.start(this._params[1] * 60);
    } else { // Stop
      $gameTimer.stop();
    }
    return true;
  }

  // Change Gold
  command125() {
    const value = this.operateValue(this._params[0], this._params[1], this._params[2]);
    Inventory.gainGold(value);
    return true;
  }

  // Change Items
  command126() {
    const value = this.operateValue(this._params[1], this._params[2], this._params[3]);
    Inventory.forceGainItem($dataItems[this._params[0]], 0, value);
    return true;
  }

  // Change Party Member
  command129() {
    const actor = $gameActors.actor(this._params[0]);
    if (actor) {
      if (this._params[1] === 0) { // Add
        if (this._params[2]) { // Initialize
          $gameActors.actor(this._params[0]).setup(this._params[0]);
        }
        $gameParty.addActor(this._params[0]);
      } else { // Remove
        $gameParty.removeActor(this._params[0]);
      }
    }
    return true;
  }

  // Change Save Access
  command134() {
    if (this._params[0] === 0) {
      $gameSystem.disableSave();
    } else {
      $gameSystem.enableSave();
    }
    return true;
  }

  // Change Menu Access
  command135() {
    if (this._params[0] === 0) {
      $gameSystem.disableMenu();
    } else {
      $gameSystem.enableMenu();
    }
    return true;
  }

  // Change Formation Access
  command137() {
    if (this._params[0] === 0) {
      $gameSystem.disableFormation();
    } else {
      $gameSystem.enableFormation();
    }
    return true;
  }

  // Change Window Color
  command138() {
    $gameSystem.setWindowTone(this._params[0]);
    return true;
  }

  // Transfer Player
  command201() {
    if (!$gameMessage.isBusy()) {
      let mapId;
      let x;
      let y;
      if (this._params[0] === 0) { // Direct designation
        mapId = this._params[1];
        x = this._params[2];
        y = this._params[3];
      } else { // Designation with variables
        mapId = $gameVariables.value(this._params[1]);
        x = $gameVariables.value(this._params[2]);
        y = $gameVariables.value(this._params[3]);
      }
      $gamePlayer.reserveTransfer(mapId, x, y, this._params[4], this._params[5]);
      this.setWaitMode('transfer');
      this._index++;
    }

    return false;
  }

  // Set Event Location
  command203() {
    const character = this.character(this._params[0]);
    if (character) {
      if (this._params[1] === 0) { // Direct designation
        character.locate(this._params[2], this._params[3]);
      } else if (this._params[1] === 1) { // Designation with variables
        const x = $gameVariables.value(this._params[2]);
        const y = $gameVariables.value(this._params[3]);
        character.locate(x, y);
      } else { // Exchange with another event
        const character2 = this.character(this._params[2]);
        if (character2) {
          character.swap(character2);
        }
      }
      if (this._params[4] > 0) {
        character.setDirection(this._params[4]);
      }
    }
    return true;
  }

  // Scroll Map
  command204() {
    if ($gameMap.isScrolling()) {
      this.setWaitMode('scroll');
      return false;
    }
    $gameMap.startScroll(this._params[0], this._params[1], this._params[2]);
    return true;
  }

  // Set Movement Route
  command205() {
    $gameMap.refreshIfNeeded();
    this._character = this.character(this._params[0]);
    if (this._character) {
      this._character.forceMoveRoute(this._params[1]);
      if (this._params[1].wait) {
        this.setWaitMode('route');
      }
    }
    return true;
  }

  // Show Balloon Icon
  command213() {
    this._character = this.executeShowBalloon(this._params[0], this._params[1], this._params[2]);
    return true;
  }

  // Erase Event
  command214() {
    if (this.isOnCurrentMap() && this._eventId > 0) {
      $gameMap.eraseEvent(this._eventId);
    }
    return true;
  }

  // Change Player Followers
  command216() {
    if (this._params[0] === 0) {
      $gamePlayer.showFollowers();
    } else {
      $gamePlayer.hideFollowers();
    }
    $gamePlayer.refresh();
    return true;
  }

  // Gather Followers
  command217() {
    $gamePlayer.gatherFollowers();
    this.setWaitMode('gather');
    return true;
  }

  // Fadeout Screen
  command221() {
    if (!$gameMessage.isBusy()) {
      $gameScreen.startFadeOut(this.fadeSpeed());
      this.wait(this.fadeSpeed());
      this._index++;
    }
    return false;
  }

  // Fadein Screen
  command222() {
    if (!$gameMessage.isBusy()) {
      $gameScreen.startFadeIn(this.fadeSpeed());
      this.wait(this.fadeSpeed());
      this._index++;
    }
    return false;
  }

  // Tint Screen
  command223() {
    $gameScreen.startTint(this._params[0], this._params[1]);
    if (this._params[2]) {
      this.wait(this._params[1]);
    }
    return true;
  }

  // Flash Screen
  command224() {
    $gameScreen.startFlash(this._params[0], this._params[1]);
    if (this._params[2]) {
      this.wait(this._params[1]);
    }
    return true;
  }

  // Shake Screen
  command225() {
    $gameScreen.startShake(this._params[0], this._params[1], this._params[2]);
    if (this._params[3]) {
      this.wait(this._params[2]);
    }
    return true;
  }

  // Wait
  command230() {
    this.wait(this._params[0]);
    return true;
  }

  commandWaitFor() {
    let characterName;
    if (this._params[0].toLowerCase() == 'js') {
      characterName = MVC.safeEval(this._params[1]); // jshint ignore:line
    } else {
      characterName = this._params[0];
    }

    this.waitFor(characterName);
    return true;
  }

  // Show Picture
  command231() {
    let x;
    let y;
    if (this._params[3] === 0) { // Direct designation
      x = this._params[4];
      y = this._params[5];
    } else { // Designation with variables
      x = $gameVariables.value(this._params[4]);
      y = $gameVariables.value(this._params[5]);
    }
    $gameScreen.showPicture(this._params[0], this._params[1], this._params[2], x, y, this._params[6], this._params[7], this._params[8], this._params[9]);
    return true;
  }

  // Move Picture
  command232() {
    let x;
    let y;
    if (this._params[3] === 0) { // Direct designation
      x = this._params[4];
      y = this._params[5];
    } else { // Designation with variables
      x = $gameVariables.value(this._params[4]);
      y = $gameVariables.value(this._params[5]);
    }
    $gameScreen.movePicture(this._params[0], this._params[2], x, y, this._params[6],
      this._params[7], this._params[8], this._params[9], this._params[10]);
    if (this._params[11]) {
      this.wait(this._params[10]);
    }
    return true;
  }

  // Rotate Picture
  command233() {
    $gameScreen.rotatePicture(this._params[0], this._params[1]);
    return true;
  }

  // Tint Picture
  command234() {
    $gameScreen.tintPicture(this._params[0], this._params[1], this._params[2]);
    if (this._params[3]) {
      this.wait(this._params[2]);
    }
    return true;
  }

  // Erase Picture
  command235() {
    $gameScreen.erasePicture(this._params[0]);
    return true;
  }

  // Set Weather Effect
  command236() {
    $gameScreen.changeWeather(this._params[0], this._params[1], this._params[2]);
    if (this._params[3]) {
      this.wait(this._params[2]);
    }
    return true;
  }

  // Play BGM
  command241() {
    Audio.playBgm(this._params[0]);
    return true;
  }

  // Fadeout BGM
  command242() {
    Audio.fadeOutBgm(this._params[0]);
    return true;
  }

  // Save BGM
  command243() {
    $gameSystem.saveBgm();
    return true;
  }

  // Resume BGM
  command244() {
    $gameSystem.replayBgm();
    return true;
  }

  // Play BGS
  command245() {
    Audio.playBgs(this._params[0]);
    return true;
  }

  // Fadeout BGS
  command246() {
    Audio.fadeOutBgs(this._params[0]);
    return true;
  }

  // Play ME
  command249() {
    Audio.playMe(this._params[0]);
    return true;
  }

  // Play SE
  command250() {
    Audio.playSe(this._params[0]);
    return true;
  }

  // Stop SE
  command251() {
    Audio.stopSe();
    return true;
  }

  // Play Movie
  command261() {
    if (!$gameMessage.isBusy()) {
      const name = this._params[0];
      if (name.length > 0) {
        const ext = this.videoFileExt();
        Graphics.playVideo(`movies/${name}${ext}`);
        this.setWaitMode('video');
      }
      this._index++;
    }
    return false;
  }

  videoFileExt() {
    if (Graphics.canPlayVideoType('video/webm') && !Utils.isMobileDevice()) {
      return '.webm';
    } else {
      return '.mp4';
    }
  }

  // Change Map Name Display
  command281() {
    if (this._params[0] === 0) {
      $gameMap.enableNameDisplay();
    } else {
      $gameMap.disableNameDisplay();
    }
    return true;
  }

  // Change Tileset
  command282() {
    const tileset = $dataTilesets[this._params[0]];
    const allReady = tileset.tilesetNames.map(tilesetName => Managers.Images.requestTileset(tilesetName, 0), this).every(bitmap => bitmap.isReady());

    if (allReady) {
      $gameMap.changeTileset(this._params[0]);

      return true;
    } else {
      return false;
    }
  }

  // Change Parallax
  command284() {
    $gameMap.changeParallax(this._params[0], this._params[1],
      this._params[2], this._params[3], this._params[4]);
    return true;
  }

  // Get Location Info
  command285() {
    let x;
    let y;
    let value;
    if (this._params[2] === 0) { // Direct designation
      x = this._params[3];
      y = this._params[4];
    } else { // Designation with variables
      x = $gameVariables.value(this._params[3]);
      y = $gameVariables.value(this._params[4]);
    }
    switch (this._params[1]) {
      case 0: // Terrain Tag
        value = $gameMap.terrainTag(x, y);
        break;
      case 1: // Event ID
        value = $gameMap.eventIdXy(x, y);
        break;
      case 2: // Tile ID (Layer 1)
      case 3: // Tile ID (Layer 2)
      case 4: // Tile ID (Layer 3)
      case 5: // Tile ID (Layer 4)
        value = $gameMap.tileId(x, y, this._params[1] - 2);
        break;
      default: // Region ID
        value = $gameMap.regionId(x, y);
        break;
    }
    $gameVariables.setValue(this._params[0], value);
    return true;
  }

  // Shop Processing
  command302() {
    const goods = [this._params];
    while (this.nextEventCode() === 605) {
      this._index++;
      goods.push(this.currentCommand().parameters);
    }
    Managers.Scenes.push(GameScenes.Shop);
    Managers.Scenes.prepareNextScene(goods, this._params[4]);

    return true;
  }

  // Name Input Processing
  command303() {
    if ($dataActors[this._params[0]]) {
      Managers.Scenes.push(GameScenes.Name);
      Managers.Scenes.prepareNextScene(this._params[0], this._params[1]);
    }
    return true;
  }

  // Change HP
  command311() {
    const value = this.operateValue(this._params[2], this._params[3], this._params[4]);
    this.iterateActorEx(this._params[0], this._params[1], actor => {
      this.changeHp(actor, value, this._params[5]);
    });
    return true;
  }

  // Recover All
  command314() {
    this.iterateActorEx(this._params[0], this._params[1], actor => {
      actor.recoverAll();
    });
    return true;
  }

  // Change EXP
  command315() {
    const value = this.operateValue(this._params[2], this._params[3], this._params[4]);
    this.iterateActorEx(this._params[0], this._params[1], actor => {
      actor.changeExp(actor.currentExp() + value, this._params[5]);
    });
    return true;
  }

  // Change Level
  command316() {
    const value = this.operateValue(this._params[2], this._params[3], this._params[4]);
    this.iterateActorEx(this._params[0], this._params[1], actor => {
      actor.changeLevel(actor.level + value, this._params[5]);
    });
    return true;
  }

  // Change Parameter
  command317() {
    const value = this.operateValue(this._params[3], this._params[4], this._params[5]);
    this.iterateActorEx(this._params[0], this._params[1], actor => {
      actor.addParam(this._params[2], value);
    });
    return true;
  }

  // Change Skill
  command318() {
    this.iterateActorEx(this._params[0], this._params[1], actor => {
      if (this._params[2] === 0) {
        actor.learnSkill(this._params[3]);
      } else {
        actor.forgetSkill(this._params[3]);
      }
    });
    return true;
  }

  // Change Equipment
  command319() {
    const actor = $gameActors.actor(this._params[0]);
    if (actor) {
      actor.changeEquipById(this._params[1], this._params[2]);
    }
    return true;
  }

  // Change Name
  command320() {
    const actor = $gameActors.actor(this._params[0]);
    if (actor) {
      actor.setName(this._params[1]);
    }
    return true;
  }

  // Change Class
  command321() {
    const actor = $gameActors.actor(this._params[0]);
    if (actor && $dataClasses[this._params[1]]) {
      actor.changeClass(this._params[1], false);
    }
    return true;
  }

  // Change Actor Images
  command322() {
    const actor = $gameActors.actor(this._params[0]);
    if (actor) {
      actor.setCharacterImage(this._params[1], this._params[2]);
      actor.setFaceImage(this._params[3], this._params[4]);
    }
    $gamePlayer.refresh();
    return true;
  }

  // Change Nickname
  command324() {
    const actor = $gameActors.actor(this._params[0]);
    if (actor) {
      actor.setNickname(this._params[1]);
    }
    return true;
  }

  // Open Menu Screen
  command351() {
    Managers.Scenes.push(GameScenes.Diary);
    return true;
  }

  // Open Save Screen
  command352() {
    Managers.Scenes.push(GameScenes.Diary);
    return true;
  }

  // Return to Title Screen
  command354() {
    Managers.Scenes.goToScene(GameScenes.Title);
    return true;
  }

  // Script
  command355() {
    let script = `${this.currentCommand().parameters[0]}\n`;
    while (this.nextEventCode() === 655) {
      this._index++;
      script += `${this.currentCommand().parameters[0]}\n`;
    }

    window.evalInterpreter = this;
    try {
      eval(script); // jshint ignore:line
    }
    catch(e) {
      console.log(...log('failed to eval script'));
      console.log(...log(script));
      throw e;
    }
    delete window.evalInterpreter;
    return true;
  }

  commandScript() {
    const script = `${this.currentCommand().parameters[0]}\n`;
    const original = this.currentCommand().parameters[1];

    try {
      eval(script); // jshint ignore:line
    }
    catch(e) {
      throw new Error(`Failed to execute script:<br/><br/>${original}<br/><br/>parsed as:<br/><br/>${script}<br/><br/>Error:<br/><br/>${e}`);
    }
    return true;
  }

  commandJS() {
    const method = this.currentCommand().parameters[0];
    method();
    return true;
  }

  // Plugin Command
  command356() {
    const args = this._params[0].split(" ");
    const command = args.shift();
    this.pluginCommand(command, args);
    return true;
  }

  checkClearPathCommand(command, args) {
    if (command.toUpperCase() !== 'CLEAR') return;
    if (args.length < 2) return;

    let nextIndex = 0;
    let eventId;

    if (args[0].toUpperCase() === 'EVENT') {
      eventId = parseInt(args[1], 10);
      nextIndex = 2;
    } else if (args[0].toUpperCase() === 'PLAYER') {
      eventId = -1;
      nextIndex = 1;
    } else if (args[0].toUpperCase() === 'THIS' && args[1].toUpperCase() === 'EVENT') {
      eventId = 0;
      nextIndex = 2;
    }

    if (args.length > nextIndex && args[nextIndex].toUpperCase() === 'PATH') {
      const character = this.character(eventId);
      if (character !== undefined && character !== null) {
        character.clearDestination();
      }
    }
  }

  checkNewMoveToFormat(command, args) {
    if (command.toUpperCase() !== 'MOVE') return;
    if (args.length <= 4) return;

    let eventId;
    let newEventId;
    let newX;
    let newY;
    let nextIndex = 0;
    let follow = false;
    let direction = 0;

    if (args[0].toUpperCase() === 'EVENT') {
      eventId = parseInt(args[1], 10);
      nextIndex = 2;
    } else if (args[0].toUpperCase() === 'PLAYER') {
      eventId = -1;
      nextIndex = 1;
    } else if (args[0].toUpperCase === 'THIS' && args[1].toUpperCase() === 'EVENT') {
      eventId = 0;
      nextIndex = 2;
    } else {
      return;
    }

    if (args[nextIndex].toUpperCase() !== 'TO') return;
    nextIndex++;

    if (args.length <= nextIndex) return;

    if (args[nextIndex].toUpperCase() === 'EVENT') {
      nextIndex++;

      if (args.length <= nextIndex) return;

      newEventId = parseInt(args[nextIndex], 10);
      nextIndex++;
    } else if (args[nextIndex].toUpperCase() === 'PLAYER') {
      newEventId = -1;
      nextIndex++;
    } else if (args[nextIndex].toUpperCase() === 'POSITION') {
      if (args.length <= nextIndex + 2) return;

      newX = parseFloat(args[nextIndex + 1]);
      newY = parseFloat(args[nextIndex + 2]);

      nextIndex += 3;
    }

    if (args.length > nextIndex +1) {
      if (args[nextIndex].toUpperCase() === 'AND') {
        nextIndex++;

        if (args[nextIndex].toUpperCase() === 'FOLLOW') {
          follow = true;
          nextIndex++;
        } else if (args[nextIndex].toUpperCase() === 'FACE' || args[nextIndex].toUpperCase() === 'TURN') {
          nextIndex++;

          if (args.length > nextIndex) {
            if (args[nextIndex].toUpperCase() === 'LEFT') {
              direction = 4;
            } else if (args[nextIndex].toUpperCase() === 'RIGHT') {
              direction = 6;
            } else if (args[nextIndex].toUpperCase() === 'UP') {
              direction = 8;
            } else if (args[nextIndex].toUpperCase() === 'DOWN') {
              direction = 2;
            } else {
              direction = parseInt(args[nextIndex], 10);
            }

            nextIndex++;
          }
        }
      }
    }

    const character = this.character(eventId);

    if (newEventId !== undefined) {
      const newCharacter = this.character(newEventId);
      if (newCharacter === undefined || newCharacter === null) return;

      character.setCharacterDestination(newCharacter, follow);
    } else {
      character.setDestination(newX, newY, direction);
    }
  }

  pluginCommand(command, args) {
    let name = args[0];
    const line = args.join(' ');

    //#ToDo: Move those commands to the command parser
    switch(command.toUpperCase()) {
      case 'DIALOGUE' :
        this.executeDialogueCommand(name, line);
        return;
      case 'WAITFOR' :
        if (name.toLowerCase() == 'js') {
          name = MVC.safeEval(line.replace('js ', '')); // jshint ignore:line
        }
        this.executeWaitForCommand(name);
        return;
      case 'CHOICE' :
        this.executeChoicesCommand(name, line);
        return;
    }

    const parser = new PluginCommandParser(this);
    if (parser.parse(command, args)) {
      return;
    }

    //#ToDo: Move those commands to the command parser
    this.checkNewMoveToFormat.call(this, command, args);
    this.checkClearPathCommand.call(this, command, args);
  }

  executeDialogueCommand(name, line) {
    const data = Managers.CommonEvent.generateDataForDialogueLine(name, line);

    this.processDialogueData(name, data);
  }

  executeWaitForCommand(name) {
    this.waitFor(name);
  }

  processDialogueData(name, data) {
    if (name.toLowerCase() == 'letter') {
      $gameSystem.setupLetter(data.message);
      return;
    }

    if (data.balloonId !== false && data.balloonId !== undefined) {
      this.executeShowBalloon(name, data.balloonId, data.waitForBalloon);
    }

    const eventData = Managers.Villagers.getVillagerData(name);
    $gameMessage.setMessageTarget(eventData);

    if (!!data.message && data.message !== '') {
      this.executeCommandShowText(data.faceName, data.faceIndex, data.backgroundType, data.positionType, data.message);
    }

    if (!!data.choices && data.choices.length > 0) {
      this.setupChoices([data.choices, data.cancelChoiceIndex, -1, data.positionType, data.backgroundType]);
    }

    if (!!data.input && data.input.length >= 2) {
      this.setupNumInput(data.input);
    }
  }

  executeChoicesCommand(name, line) {
    const data = Managers.CommonEvent.generateDataForChoiceLine(name, line, this._reservedChoices || []);
    if (data) {
      this.processDialogueData(name, data);
    }
  }

  doDoorAnimation(playSound) {
    if (this._eventId > 0) {
      const event = $gameMap.event(this._eventId);
      if (event) {
        event.doDoorAnimation(playSound);
      }
    }
  }

  clearReservedChoices() {
    this._reservedChoices = [];
  }

  addChoice(choice) {
    this._reservedChoices.push(choice);
  }

  hasChoices() {
    return this._reservedChoices && this._reservedChoices.length;
  }

  listFoodItems() {
    const interpreter = this;
    const foodItems = Inventory.container.items.filter(item => item && item.type == 'food');
    const foodItemIds = foodItems.map(item => item.id);

    foodItemIds.forEach(itemId => {
      interpreter.addChoice(`item-${itemId}`);
    });
  }

  listAvailableAutoTasks() {
    const tasks = Managers.Tasks.getAvailableCustomTasks();
    let count = 0;
    const interpreter = this;
    tasks.forEach(taskId => {
      const data = Managers.Tasks.getTaskData(taskId);
      if (!data) return;

      interpreter.addChoice(data.requester);
      Variables.autoTaskRequester = data.requester;
      count++;
    });

    Variables.autoTaskCount = count;
  }

  listCandyItems() {
    const interpreter = this;
    const candyItems = Inventory.container.items.filter(item => item && item.candy);
    const candyItemIds = candyItems.map(item => item.id);

    candyItemIds.forEach(itemId => {
      interpreter.addChoice(`item-${itemId}`);
    });
  }

};