//-----------------------------------------------------------------------------
// Game_Followers
//
// The wrapper class for a follower array.

Objects.FollowerHub = class FollowerHub {
  constructor() {
    this.initialize(...arguments);
  }

  initialize() {
    this._visible = $dataSystem.optFollowers;
    this._gathering = false;
    this._data = [];
    for (let i = 1; i < $gameParty.maxPartyMembers(); i++) {
      this._data.push(new Objects.Follower(i));
    }
    this._following = true;
  }

  restoreData(data) {
    const propertiesToSkip = ['_data'];
    Utils.assignProperties(this, data, propertiesToSkip);

    if (data._data) {
      for (let i = 0; i < this._data.length; i++) {
        const follower = this._data[i];
        const followerData = data._data[i];

        if (followerData) {
          Utils.assignProperties(follower, followerData);
        }
      }
    }
  }

  isVisible() {
    return this._visible;
  }

  stopFollowing() {
    this._following = false;
  }

  startFollowing() {
    this._following = true;
  }

  show() {
    this._visible = true;
  }

  hide() {
    this._visible = false;
  }

  follower(index) {
    return this._data[index];
  }

  forEach(callback, thisObject) {
    this._data.forEach(callback, thisObject);
  }

  reverseEach(callback, thisObject) {
    this._data.reverse();
    this._data.forEach(callback, thisObject);
    this._data.reverse();
  }

  refresh() {
    this.forEach(follower => follower.refresh(), this);
  }

  update() {
    if (this.areGathering()) {
      if (!this.areMoving()) {
        this.updateMove();
      }
      if (this.areGathered()) {
        this._gathering = false;
      }
    }
    this.forEach(follower => {
      follower.update();
    }, this);
  }

  updateMove() {
    if (this._following === false) return;

    for (let i = this._data.length - 1; i >= 0; i--) {
      const precedingCharacter = (i > 0 ? this._data[i - 1] : $gamePlayer);
      this._data[i].chaseCharacter(precedingCharacter);
    }
  }

  jumpAll() {
    if ($gamePlayer.isJumping()) {
      for (const follower of this._data) {
        const sx = $gamePlayer.deltaXFrom(follower.x);
        const sy = $gamePlayer.deltaYFrom(follower.y);
        follower.jump(sx, sy);
      }
    }
  }

  synchronize(x, y, d) {
    this.forEach(follower => {
      follower.locate(x, y);
      follower.setDirection(d);
    }, this);
  }

  gather() {
    this._gathering = true;
  }

  areGathering() {
    return this._gathering;
  }

  visibleFollowers() {
    return this._data.filter(follower => follower.isVisible(), this);
  }

  areMoving() {
    return this.visibleFollowers().some(follower => follower.isMoving(), this);
  }

  areGathered() {
    return this.visibleFollowers().every(follower => !follower.isMoving() && follower.pos($gamePlayer.x, $gamePlayer.y), this);
  }

  isSomeoneCollided(x, y) {
    return this.visibleFollowers().some(follower => follower.pos(x, y), this);
  }
};
