require('./Character');
//-----------------------------------------------------------------------------
// Game_Event
//
// The game object class for an event. It contains functionality for event page
// switching and running parallel process events.

/* globals insignificantValue */

Objects.Event = class Event extends Objects.Character {
  get hitboxX() {
    const realHitboxX = this.realHitboxX;
    const absCeil = realHitboxX.absCeil();
    if (absCeil >= this.realRight) {
      return realHitboxX.absFloor();
    }
    return absCeil;
  }
  set hitboxX(value) {
    this._hitboxX = value;
    this._canClearHitboxX = false;
  }

  getRealHitboxX() {
    return this._hitboxX || 0;
  }

  get realHitboxX() {
    return this.getRealHitboxX();
  }

  get hitboxY() {
    const realHitboxY = this.realHitboxY;
    const absCeil = realHitboxY.absCeil();
    if (absCeil >= this.realBottom) {
      return realHitboxY.absFloor();
    }
    return absCeil;
  }
  set hitboxY(value) {
    this._hitboxY = value;
    this._canClearHitboxY = false;
  }

  shouldDisableOwnBalloon() {
    return this._disableOwnBalloon || false;
  }

  get disableOwnBalloon() {
    return this.shouldDisableOwnBalloon();
  }

  getRealHitboxY() {
    return this._hitboxY || 0;
  }

  get realHitboxY() {
    return this.getRealHitboxY();
  }

  isSingleImage() {
    return this._singleImage || false;
  }

  get singleImage() {
    return this.isSingleImage();
  }

  get hitboxWidth() {
    return this.realHitboxWidth.absCeil();
  }
  set hitboxWidth(value) {
    this._hitboxWidth = value;
    this._canClearHitboxWidth = false;
  }

  getRealHitboxWidth() {
    return this._hitboxWidth || 1;
  }

  get realHitboxWidth() {
    return this.getRealHitboxWidth();
  }

  get hitboxHeight() {
    return this.realHitboxHeight.absCeil();
  }
  set hitboxHeight(value) {
    this._hitboxHeight = value;
    this._canClearHitboxHeight = false;
  }

  getRealHitboxHeight() {
    return this._hitboxHeight || 1;
  }

  get realHitboxHeight() {
    return this.getRealHitboxHeight();
  }

  get left() {
    return Math.fix(this._x + this.hitboxX);
  }
  get top() {
    return Math.fix(this._y + this.hitboxY);
  }
  get right() {
    return Math.fix(this.left + this.hitboxWidth - insignificantValue);
  }
  get bottom() {
    return Math.fix(this.top + this.hitboxHeight - insignificantValue);
  }

  get realLeft() {
    return Math.fix(this._x + this.realHitboxX);
  }
  get realTop() {
    return Math.fix(this._y + this.realHitboxY);
  }
  get realRight() {
    return Math.fix(this.realLeft + this.realHitboxWidth - insignificantValue);
  }
  get realBottom() {
    return Math.fix(this.realTop + this.realHitboxHeight - insignificantValue);
  }

  initialize(mapId, eventId) {
    super.initialize();
    this._mapId = mapId;
    this._eventId = eventId;

    this._note = this.event().note || '';
    this._noGraphic = this._note.includes('<nographic>');
    this._pageHandling = this._note.includes('<pages>');
    this._forceSequentialAnimation = this._note.includes('<123>');
    this._zVariation = this._note.includes('<z->') ? -1 : (this._note.includes('<z+>') ? 1 : 0);
    this._villagerDoor = this._note.includes('<door:') ? (MVC.getNoteValue(this._note, 'door') || false) : false;
    // this._forceCollision = this._note.include('<forceCollision>');
    this._useCustomTileId = false;
    this._forceEventSprite = false;
    this._fadingOut = false;
    this._fadingIn = false;

    this.locate(this.event().x, this.event().y);
    this.refresh();

    this.setEventTypeData();
  }

  initMembers() {
    super.initMembers();
    this._moveType = 0;
    this._trigger = 0;
    this._starting = false;
    this._erased = false;
    this._pageIndex = -2;
    this._originalPattern = 1;
    this._originalDirection = 2;
    this._prelockDirection = 0;
    this._locked = false;
  }

  setEventTypeData() {
  }

  eventId() {
    return this._eventId;
  }

  usesNoGraphic() {
    return this._noGraphic || (this._useCustomTileId && !this._forceEventSprite);
  }

  supportsPageHandling() {
    return this._pageHandling;
  }

  event() {
    return $dataMap.events[this._eventId];
  }

  page() {
    const eventData = this.event();
    if (!eventData) return;

    return eventData.pages[this._pageIndex];
  }

  list() {
    const page = this.page();
    if (!page) {
      return [];
    }

    return page.list;
  }

  isCollidedWithCharacters(x, y) {
    return (super.isCollidedWithCharacters(x, y) || this.isCollidedWithPlayerCharacters(x, y));
  }

  actor() {
    return undefined;
  }

  isCollidedWithEvents(x, y) {
    const events = $gameMap.eventsXyNt(x, y);
    return events.length > 0;
  }

  isCollidedWithPlayerCharacters(x, y) {
    return this.isNormalPriority() && $gamePlayer.isCollided(x, y);
  }

  lock() {
    if (this._directionFix) return;

    if (!this._locked) {
      this._prelockDirection = this.direction();
      this.turnTowardPlayer();
      this._locked = true;
    }
  }

  unlock() {
    if (this._directionFix) return;

    if (this._locked) {
      this.setDirection(this._prelockDirection);
      const that = this;
      $gameTemp.setTimeout(() => {
        that._locked = false;
      }, 2, true);
    }
  }

  updateStop() {
    if (this._locked) {
      this.resetStopCount();
    }
    super.updateStop();
    if (!this.isMoveRouteForcing()) {
      this.updateSelfMovement();
    }
  }

  updateSelfMovement() {
    if (this._moveType === 0) return;

    if (this.isNearTheScreen() && this.checkStop(this.stopCountThreshold())) {
      switch (this._moveType) {
        case 1:
          this.moveTypeRandom();
          break;
        case 2:
          this.moveTypeTowardPlayer();
          break;
        case 3:
          this.moveTypeCustom();
          break;
        case 4:
          this.moveTypeChicken();
          break;
        case 5:
          this.moveTypeRace();
          break;
        case 6:
          this.moveForward();
          if (this._forwardCount > 0) {
            this._forwardCount--;

            if (this._forwardCount === 0) {
              this.restoreMoveType();
            }
          }
          break;
        default :
          this.moveTypeUnknown(this._moveType);
          break;
      }
    }
  }

  stopCountThreshold() {
    return 30 * (5 - this.moveFrequency());
  }

  moveTypeRandom() {
    // No random movement on barns
    const mapId = Managers.Map.getRealMapId($gameMap._mapId);
    if (mapId === Maps.BARN || mapId === Maps.EXPANDED_BARN) {
      return;
    }

    switch (Math.randomInt(6)) {
      case 5:
        this.resetStopCount();
        this.stopForAWhile(60 + Math.randomInt(300));
        break;
      default:
        this.moveRandom();
        this._moveType = 6;
        this._moveFrequency = 5;
        this._forwardCount = 4 + Math.randomInt(30);
        break;
    }
  }

  moveTypeUnknown(moveType) {
  }

  moveTypeChicken() {
    switch (Math.randomInt(10)) {
      case 0: case 1: case 2: case 3:
        this.moveRandom();
        break;
      case 4:
        this.moveForward();
        break;
      case 5: case 6: case 7: case 8: case 9:
        this.stopForAWhile(Math.randomInt(60));
        break;
      default :
        break;
    }
  }

  moveTypeRace() {
    this.moveForward();
  }

  moveTypeTowardPlayer() {
    if (this.isNearThePlayer()) {
      switch (Math.randomInt(6)) {
        case 0: case 1: case 2: case 3:
          this.moveTowardPlayer();
          break;
        case 4:
          this.moveRandom();
          break;
        case 5:
          this.moveForward();
          break;
        default :
          break;
      }
    } else {
      this.moveRandom();
    }
  }

  isNearThePlayer(maxDistance = 4) {
    const sx = Math.abs(this.deltaXFrom($gamePlayer.x));
    const sy = Math.abs(this.deltaYFrom($gamePlayer.y));
    return sx + sy < maxDistance;
  }

  moveTypeCustom() {
    this.updateRoutineMove();
  }

  isStarting() {
    return this._starting;
  }

  clearStartingFlag() {
    if (!this._starting) return;

    $gameMap.decreaseStartingFlag();
    this._starting = false;
  }

  isTriggerIn(triggers) {
    return triggers.contains(this._trigger);
  }

  start(activatedByMouse = false) {
    if (this._doingDoorAnimation) return false;

    const list = this.list();
    if (list && list.length > 1) {
      this._starting = true;
      $gameMap.increaseStartingFlag();
      if (this.isTriggerIn([0,1,2])) {
        this.lock();
      }
      return true;
    }

    return false;
  }

  erase() {
    this._fadingOut = false;
    this._fadingIn = false;
    this._erased = true;
    this.refresh();
  }

  fadeIn() {
    this._opacity = 0;
    this._fadingIn = true;
  }

  fadeOutErase() {
    this._fadingOut = true;
  }

  refresh() {
    const newPageIndex = this._erased ? -1 : this.findProperPageIndex();
    if (this._pageIndex !== newPageIndex) {
      this._pageIndex = newPageIndex;
      this.setupPage();
    }
  }

  findProperPageIndex() {
    if (!this.supportsPageHandling()) return 0;

    const pages = this.event().pages;
    for (let i = pages.length - 1; i >= 0; i--) {
      const page = pages[i];
      if (this.meetsConditions(page)) {
        return i;
      }
    }
    return -1;
  }

  meetsConditions(page) {
    const c = page.conditions;
    if (c.switch1Valid) {
      if (!$gameSwitches.booleanValue(c.switch1Id)) {
        return false;
      }
    }
    if (c.switch2Valid) {
      if (!$gameSwitches.booleanValue(c.switch2Id)) {
        return false;
      }
    }
    if (c.variableValid) {
      if ($gameVariables.value(c.variableId) < c.variableValue) {
        return false;
      }
    }
    if (c.selfSwitchValid) {
      const key = [this._mapId, this._eventId, c.selfSwitchCh];
      if ($gameSelfSwitches.value(key) !== true) {
        return false;
      }
    }
    // if (c.itemValid) {
    //   var item = $dataItems[c.itemId];
    //   if (!Inventory.hasItem(item)) {
    //     return false;
    //   }
    // }
    // if (c.actorValid) {
    //   var actor = $gameActors.actor(c.actorId);
    //   if (!$gameParty.members().contains(actor)) {
    //     return false;
    //   }
    // }
    return true;
  }

  setupPage() {
    if (this._pageIndex >= 0) {
      this.setupPageSettings();
    } else {
      this.clearPageSettings();
    }

    if (this._canClearHitboxX === true) this._hitboxX = undefined;
    if (this._canClearHitboxY === true) this._hitboxY = undefined;
    if (this._canClearHitboxHeight === true) this._hitboxHeight = undefined;
    if (this._canClearHitboxWidth === true) this._hitboxWidth = undefined;

    this.clearStartingFlag();
    this.checkEventTriggerAuto();
  }

  clearPageSettings() {
    this.setImage('', 0);
    this._moveType = 0;
    this._trigger = null;
    this._interpreter = null;
    this.setThrough(true);
  }

  setupPageSettings() {
    const page = this.page();
    const image = page.image;
    if (image.iconIndex > 0) {
      this.setIconImage(image.iconIndex);
    } else if (image.tileId > 0) {
      this.setTileImage(image.tileId);
    } else {
      this.setImage(image.characterName, image.characterIndex);
    }

    let keepOldDirections = false;
    const eventData = this.event();
    if (eventData && eventData.meta && eventData.meta.keepOldDirections) {
      keepOldDirections = true;
    }

    // if (this._originalDirection !== image.direction) {
    if (!keepOldDirections) {
      this._originalDirection = image.direction;
      this._prelockDirection = 0;
      this.setDirectionFix(false);
      this.setDirection(image.direction);
    }
    // }

    if (this._originalPattern !== image.pattern) {
      this._originalPattern = image.pattern;
      this.setPattern(image.pattern);
    }
    this.setMoveSpeed(page.moveSpeed);
    this.setMoveFrequency(page.moveFrequency);
    this.setPriorityType(page.priorityType);
    this.setWalkAnime(page.walkAnime);
    this.setStepAnime(page.stepAnime);
    this.setDirectionFix(page.directionFix);
    this.setThrough(page.through);
    this.setMoveRoute(page.moveRoute);
    this._moveType = page.moveType;
    this._trigger = page.trigger;
    if (this._trigger === 4) {
      this._interpreter = new Objects.Interpreter();
    } else {
      this._interpreter = null;
    }
  }

  isOriginalPattern() {
    return this.pattern() === this._originalPattern;
  }

  resetPattern() {
    this.setPattern(this._originalPattern);
  }

  checkEventTriggerAuto() {
    if (this._trigger === 3) {
      this.start();
    }
  }

  update() {
    if (this._fadingOut) {
      this._fadingIn = false;
      if (this._opacity > 0) {
        this._opacity--;
      } else {
        this.erase();
        return;
      }
    } else if (this._fadingIn) {
      if (this._opacity < 255) {
        this._opacity++;
      }
    }

    super.update();
    this.checkEventTriggerAuto();
    this.updateParallel();
  }

  updateParallel() {
    if (this._interpreter) {
      if (!this._interpreter.isRunning()) {
        this._interpreter.setup(this.list(), this._eventId);
      }
      this._interpreter.update();
    }
  }

  locate(x, y) {
    super.locate(x, y);
    this._prelockDirection = 0;
  }

  forceMoveRoute(moveRoute) {
    super.forceMoveRoute(moveRoute);
    this._prelockDirection = 0;
  }

  shouldShowMouseIcon() {
    return true;
  }

  hasAnythingToRun(activatedByMouse = false) {
    const list = this.list();

    for (const idx in list) {
      const command = list[idx];

      // Comments
      if (command.code == 108 || command.code == 408) {
        continue;
      }

      // Label
      if (command.code == 118) {
        continue;
      }

      // End of List
      if (Number(command.code) === 0) {
        continue;
      }

      return true;
    }

    return false;
  }

  doDoorAnimation(playSound, callback) {
    const event = this;
    if (this._doingDoorAnimation) {
      $gameTemp.setTimeout(() => {
        event.doDoorAnimation(playSound);
      }, 6, true);
      return;
    }

    if (this._direction == 8) return;

    if (playSound !== false) {
      Managers.Sound.playDoorOpen(event.distanceToPlayer());
    }

    this._doingDoorAnimation = true;

    event._direction = 2;
    $gameTemp.setTimeout(() => {
      event._direction = 4;
      $gameTemp.setTimeout(() => {
        event._direction = 6;
        $gameTemp.setTimeout(() => {
          event._direction = 8;
          event._doingDoorAnimation = false;

          if (callback) {
            callback.call(event);
          }
        }, 6, true);
      }, 6, true);
    }, 6, true);
  }

  doCloseDoorAnimation(playSound) {
    const event = this;
    if (this._doingDoorAnimation) {
      $gameTemp.setTimeout(() => {
        event.doCloseDoorAnimation(playSound);
      }, 6, true);
      return;
    }

    if (this._direction == 2) return;

    if (playSound !== false) {
      Managers.Sound.playDoorClose();
    }

    this._doingDoorAnimation = true;
    event._direction = 8;
    $gameTemp.setTimeout(() => {
      event._direction = 6;
      $gameTemp.setTimeout(() => {
        event._direction = 4;
        $gameTemp.setTimeout(() => {
          event._doingDoorAnimation = false;
          event._direction = 2;
        }, 6, true);
      }, 6, true);
    }, 6, true);
  }


  // Adds a method that checks if the event is using the default hitbox,
  // in which case some methods don't need to be changed.
  isUsingDefaultHitbox() {
    return (this.hitboxX === 0 && this.hitboxY === 0 && this.hitboxWidth === 1 && this.hitboxHeight === 1);
  }

  pos(x, y) {
    if (this.isUsingDefaultHitbox()) {
      return this._x === x && this._y === y;
    } else {
      return (x >= this.left && x < this.right && y >= this.top && y < this.bottom);
    }
  }

  realPos(x, y) {
    if (this.isUsingDefaultHitbox()) {
      return this._x === x && this._y === y;
    } else {
      return (x >= this.realLeft && x < this.realRight && y >= this.realTop && y < this.realBottom);
    }
  }

  getHeightLevel() {
    return Managers.Map.getHeightLevel(this.realLeft, this.realTop);
  }

  realPosIn(left, top, right, bottom) {
    if (right < this.realLeft) return false;
    if (left > this.realRight) return false;
    if (bottom < this.realTop) return false;
    if (top > this.realBottom) return false;

    return true;
  }

  screenX() {
    const baseX = super.screenX();

    const metaOffsetX = this.getMetaOffsetX();
    if (metaOffsetX) {
      return Math.fix(baseX + metaOffsetX);
    }

    if (this.offsetX) {
      return Math.fix(baseX + this.offsetX);
    }

    return baseX;
  }

  getMetaOffsetX() {
    if (this._metaOffsetX !== undefined) {
      return this._metaOffsetX;
    }

    if (this._eventId > 0) {
      const event = $dataMap.events[this._eventId];
      if (event) {
        const meta = event.meta;
        if (meta.offsetX !== undefined) {
          this._metaOffsetX = Number(meta.offsetX);
          // return Number(meta.offsetX);
          return this._metaOffsetX;
        }
      }
    }

    this._metaOffsetX = false;
    return undefined;
  }

  getMetaOffsetY() {
    if (this._metaOffsetY !== undefined) {
      return this._metaOffsetY;
    }

    if (this._eventId > 0) {
      const event = $dataMap.events[this._eventId];
      if (event) {
        const meta = event.meta;
        if (meta.offsetY !== undefined) {
          this._metaOffsetY = Number(meta.offsetY);
          // return Number(meta.offsetY);
          return this._metaOffsetY;
        }
      }
    }

    this._metaOffsetY = false;
    return undefined;
  }

  screenY() {
    const baseY = super.screenY();

    const metaOffsetY = this.getMetaOffsetY();
    if (metaOffsetY) {
      return Math.fix(baseY + metaOffsetY);
    }

    if (this.offsetY) {
      return Math.fix(baseY + this.offsetY);
    }

    return baseY;
  }

  updateMove() {
    if (this._locked) {
      return;
    }
    super.updateMove();
  }

  updateAnimation() {
    if (this._locked) {
      this.resetPattern();
      this._animationCount = 0;
      return;
    }
    super.updateAnimation();
  }

  doAnimalDropReaction() {
    this.playAnimalSound();
  }

  playAnimalSound() {
  }

  setDirection(d) {
    if (this._locked) return;

    super.setDirection(d);
  }

  canBeActivatedFrom(d, x, y, playerUsedMouse) {
    // If the player is not even looking our way and didn't use the mouse, then we don't need to test anything
    if (!playerUsedMouse) {
      switch(d) {
        case Direction.LEFT:
          if (x < this.realLeft) return false;
          break;
        case Direction.RIGHT:
          if (x > this.realRight) return false;
          break;
        case Direction.DOWN:
          if (y > this.realBottom) return false;
          break;
        case Direction.UP:
          if (y < this.realTop) return false;
          break;
      }
    }

    if (this.getHeightLevel() != Managers.Map.getHeightLevel(x, y)) return false;

    // Make sure the event really can be activated from the current position
    if (d == Direction.DOWN || d == Direction.UP) {
      const w = this.realRight - this.realLeft;

      let leftLimit = x;
      let rightLimit = x;

      if (w < 1) {
        leftLimit -= (0.5 + w);
        rightLimit += (0.5 + w);
      }

      if (this.realRight < leftLimit || this.realLeft > rightLimit) {
        return false;
      }
    } else {
      const h = this.realBottom - this.realTop;
      let topLimit = y;
      let bottomLimit = y;

      if (h < 1) {
        topLimit -= (0.5 + h);
        bottomLimit += (0.5 + h);
      }

      if (this.realBottom < topLimit || this.realTop > bottomLimit) {
        return false;
      }
    }

    return true;
  }

  hasAnythingToRunOnLowPriority(activatedByMouse = false) {
    return false;
  }

  getScaleX() {
    return this._scaleX || 1;
  }

  getScaleY() {
    return this._scaleY || 1;
  }

  getSpriteClass() {
    return Sprite_Event;
  }

  getBalloonOffset() {
    return 0;
  }

  screenZ() {
    return super.screenZ() + (this._zVariation || 0);
  }

  isThrough() {
    // If it's a door, set it as "through" when it's open (direction other than 2)
    if (this._villagerDoor) {
      if (this._direction !== 2) {
        return true;
      }
    }

    return super.isThrough();
  }
};
