const { Audio } = require('engine/managers/Audio');

/* globals insignificantValue */

Objects.Character = class Character {
  constructor() {
    this.initialize(...arguments);
  }

  get x() {
    return this._x;
  }

  get y() {
    return this._y;
  }

  get asleep() {
    return this._asleep;
  }

  get top() {
    return this.getTop();
  }

  get left() {
    return this.getLeft();
  }

  get right() {
    return this.getRight();
  }

  get bottom() {
    return this.getBottom();
  }

  initialize() {
    this.initMembers();
  }

  initMembers() {
    this._x = 0;
    this._y = 0;
    this._realX = 0;
    this._realY = 0;
    this._lostX = 0;
    this._lostY = 0;
    this._moveSpeed = 4;
    this._moveFrequency = 6;
    this._opacity = 255;
    this._blendMode = 0;
    this._direction = 2;

    this._horzDirection = 0;
    this._vertDirection = 0;
    this._pattern = 1;
    this._priorityType = 1;
    this._tileId = 0;
    this._iconIndex = 0;
    this._toolId = undefined;
    this._toolState = undefined;
    this._toolDirection = 2;
    this._characterName = '';
    this._characterIndex = 0;
    this._walkAnime = true;
    this._stepAnime = false;
    this._directionFix = false;
    this._through = false;
    this._balloonId = 0;
    this._balloonPlaying = false;
    this._asleep = false;
    this._stopCount = 0;
    this._jumpCount = 0;
    this._jumpPeak = 0;
    this._jumpCollision = false;
    this._movementSuccess = true;
    this._doingToolAnimation = false;
    this._doingJumpAnimation = false;
    this._doingDelayAnimation = false;
    this._animationId = 0;
    this._animationPlaying = false;
    this._animationCount = 0;

    this._moveRouteForcing = false;
    this._moveRoute = null;
    this._moveRouteIndex = 0;
    this._originalMoveRoute = null;
    this._originalMoveRouteIndex = 0;
    this._waitCount = 0;  
    this._hidden = false;
  }

  pos(x, y) {
    return this._x === x && this._y === y;
  }

  floatPos(x, y) {
    return (x >= this.left && x < this.right && y >= this.top && y < this.bottom);
  }

  realPos(x, y) {
    return this._x === x && this._y === y;
  }

  realPosIn(left, top, right, bottom) {
    if (right < this.left) return false;
    if (left > this.right) return false;
    if (bottom < this.top) return false;
    if (top > this.bottom) return false;

    return true;
  }

  posNt(x, y) {
    // No through
    return !this.isThrough() && this.pos(x, y);
  }

  realPosNt(x, y) {
    // No through
    return !this.isThrough() && this.realPos(x, y);
  }

  getTop() {
    return this.y;
  }

  getLeft() {
    return this.x;
  }

  getRight() {
    return this.left + 1 - insignificantValue;
  }

  getBottom() {
    return this.top + 1 - insignificantValue;
  }

  isTouchingTile(x, y) {
    const left = Math.floor(this.left);
    const right = Math.ceil(this.right);
    const top = Math.floor(this.top);
    const bottom = Math.ceil(this.bottom);

    if (x < left || x > right) {
      return false;
    }

    if (y < top || y > bottom) {
      return false;
    }

    return true;
  }

  moveSpeed() {
    return this._moveSpeed;
  }

  setMoveSpeed(moveSpeed) {
    this._moveSpeed = moveSpeed;
  }

  moveFrequency() {
    return this._moveFrequency;
  }

  setMoveFrequency(moveFrequency) {
    this._moveFrequency = moveFrequency;
  }

  opacity() {
    return this._opacity;
  }

  setOpacity(opacity) {
    this._opacity = opacity;
  }

  blendMode() {
    return this._blendMode;
  }

  setBlendMode(blendMode) {
    this._blendMode = blendMode;
  }

  isNormalPriority() {
    return this._priorityType === 1;
  }

  setPriorityType(priorityType) {
    this._priorityType = priorityType;
  }

  isAboutToStop() {
    if (!this.isMoving()) return false;

    const distancePerFrame = this.distancePerFrame();

    if (this._y < this._realY) {
      const smallerY = this._realY - distancePerFrame;

      if (smallerY > this._y) return false;
    } else if (this._y > this._realY) {
      const biggerY = this._realY + distancePerFrame;
      
      if (biggerY < this._y) return false;
    }

    if (this._x < this._realX) {
      const smallerX = this._realX - distancePerFrame;
      
      if (smallerX > this._x) return false;
    } else if (this._x > this._realX) {
      const biggerX = this._realX + distancePerFrame;
      
      if (biggerX < this._x) return false;
    }

    return true;
  }

  isMoving() {
    return this._realX !== this._x || this._realY !== this._y;
  }

  isJumping() {
    return this._jumpCount > 0;
  }

  jumpHeight() {
    return (this._jumpPeak * this._jumpPeak - Math.abs(this._jumpCount - this._jumpPeak) ** 2) / 2;
  }

  isStopping() {
    return !this.isMoving() && !this.isJumping();
  }

  checkStop(threshold) {
    return this._stopCount > threshold;
  }

  resetStopCount() {
    this._stopCount = 0;
  }

  stopForAWhile(numberOfFrames) {
    this._stopCount = -numberOfFrames;
  }

  characterMoveSpeed() {
    return this._moveSpeed;
  }

  dashSpeedBoost() {
    return 1;
  }

  getHeightLevel() {
    return Managers.Map.getHeightLevel(this.left, this.top);
  }

  realMoveSpeed() {
    // #ToDo: When diagonal movement, multiply the speed by 0.851

    const baseSpeed = this.characterMoveSpeed() + (this.isDashing() ? this.dashSpeedBoost() : 0);
    return baseSpeed;
  }

  isDashing() {
    return false;
  }

  isRunning() {
    return this.isDashing() && this.isMoving();
  }

  distancePerFrame() {
    return (2 ** this.realMoveSpeed() / 256) * Managers.Scenes._gameSpeed;
  }

  isDebugThrough() {
    return false;
  }

  straighten() {
    if (this._lockPattern) return;

    if (this.hasWalkAnime() || this.hasStepAnime()) {
      this._pattern = 1;
    }
    this._animationCount = 0;
  }

  reverseDir(d) {
    return 10 - d;
  }

  isValid(x, y) {
    return $gameMap.isValid(x, y);
  }

  isCollidedWithPlayerCharacters() /*x, y*/{
    return false;
  }

  canPassTile(x, y, d) {
    const x2 = $gameMap.roundXWithDirection(x, d);
    const y2 = $gameMap.roundYWithDirection(y, d);
    if (!this.isValid(x2, y2)) {
      return false;
    }
    if (this.isThrough() || this.isDebugThrough()) {
      const regionId = $gameMap.regionId(x, y);
      if (regionId == Region.SUPER_BLOCKED_MOVEMENT) return false;

      return true;
    }
    if (!this.isMapPassable(x, y, d)) {
      return false;
    }
    if (this.isCollidedWithCharacters(x2, y2)) {
      return false;
    }
    if (this.isCollidedWithCollisionBoxes(x2, y2)) {
      return false;
    }
    return true;
  }

  isMovementBlocked() {
    if (this.canPass(this._x, this._y, Direction.DOWN)) {
      return false;
    }
    if (this.canPass(this._x, this._y, Direction.UP)) {
      return false;
    }
    if (this.canPass(this._x, this._y, Direction.LEFT)) {
      return false;
    }
    if (this.canPass(this._x, this._y, Direction.RIGHT)) {
      return false;
    }

    return true;
  }

  canPass(x, y, d) {
    return this.canPassTile(x, y, d);
  }

  canPassTileDiagonally(x, y, horz, vert) {
    const x2 = $gameMap.roundXWithDirection(x, horz);
    const y2 = $gameMap.roundYWithDirection(y, vert);

    if (!this.canPass(x, y, vert)) return false;
    if (!this.canPass(x, y, horz)) return false;
    if (!this.canPass(x, y2, horz)) return false;
    if (!this.canPass(x2, y, vert)) return false;
    return true;
  }

  canPassDiagonally(x, y, horz, vert) {
    return this.canPassTileDiagonally(x, y, horz, vert);
  }

  isMapTilePassable(x, y, d) {
    const x2 = $gameMap.roundXWithDirection(x, d);
    const y2 = $gameMap.roundYWithDirection(y, d);
    const d2 = this.reverseDir(d);

    return this.checkMapPassable(x, y, d) && this.checkMapPassable(x2, y2, d2);
  }

  isMapPassable(x, y, d) {
    return this.isMapTilePassable(x, y, d);
  }

  checkMapPassable(x, y, d) {
    return $gameMap.isPassable(x, y, d);
  }

  isMapTilePassableForFish(x, y, d) {
    const x2 = $gameMap.roundXWithDirection(x, d);
    const y2 = $gameMap.roundYWithDirection(y, d);
    const d2 = this.reverseDir(d);

    return $gameMap.isPassableForFish(x, y, d) && $gameMap.isPassableForFish(x2, y2, d2);
  }

  isMapPassableForFish(x, y, d) {
    return this.isMapTilePassableForFish(x, y, d);
  }

  isCollidedWithCharacters(x, y) {
    return this.isCollidedWithEvents(x, y) || this.isCollidedWithVillagers(x, y) || this.isCollidedWithFarmObjects(x, y);
  }

  isCollidedWithVillagers(x, y) {
    const villagers = $gameMap.villagersXy(x, y);
    return villagers.some(event => !event.isMoving());
  }

  isCollidedWithEvents(x, y) {
    const events = $gameMap.eventsXyNt(x, y);
    return events.some(event => event.isNormalPriority());
  }

  isCollidedWithCollisionBoxes(x, y) {
    const boxes = $gameMap.collisionBoxesXy(x, y);
    return boxes && boxes.length > 0;
  }

  setPosition(x, y) {
    this._x = Math.round(x);
    this._y = Math.round(y);
    this._realX = x;
    this._realY = y;
    this._lostX = 0;
    this._lostY = 0;
  }

  copyPosition(character) {
    this._x = character._x;
    this._y = character._y;
    this._realX = character._realX;
    this._realY = character._realY;
    this._lostX = 0;
    this._lostY = 0;
    this._direction = character._direction;
  }

  locate(x, y) {
    this.setPosition(x, y);
    this.straighten();
  }

  direction() {
    return this._direction;
  }

  leftDirection() {
    return DirectionHelper.leftSide(this._direction);
  }

  rightDirection() {
    return DirectionHelper.rightSide(this._direction);
  }

  backDirection() {
    return DirectionHelper.backSide(this._direction);
  }

  toolDirection() {
    return this._toolDirection;
  }

  prepareToolAnimation(toolId, toolState) {
    this._toolId = toolId;
    this.setToolState(toolState);
    this._toolDirection = Direction.DOWN;  
    this._toolResetCount = 5;
  }

  doToolAnimation(toolId, times, delay, toolState, endCallback, soundName) {
    if (this._doingToolAnimation) return;
    const character = this;

    if (endCallback === undefined) {
      endCallback = () => {
        if (soundName) {
          Audio.playSe(soundName);
        }
        character.endToolAnimation();
      };
    }

    if (times === undefined) times = 1;
    if (delay === undefined) delay = 120;

    this._toolId = toolId;
    this.setToolState(toolState);
    this._toolDirection = Direction.DOWN;
    this._doingToolAnimation = true;
    this._doingJumpAnimation = false;
    this._doingDelayAnimation = false;

    let intervalHandler;
    let timesLeft = times;

    const nextDirectionFn = () => {
      if (character._toolDirection == Direction.UP) {
        timesLeft--;
        if (timesLeft <= 0) {
          clearInterval(intervalHandler);
          endCallback();
        } else {
          character._toolDirection = Direction.DOWN;
          if (soundName && character._toolId) {
            Audio.playSe(soundName);
          }
        }
      } else {
        character._toolDirection += 2;
      }
    };

    intervalHandler = setInterval(nextDirectionFn, delay);
  }

  endToolAnimation() {
    this._toolId = undefined;
    this.setToolState(undefined);
    this._doingToolAnimation = false;
    this._usingTool = false;

    this.addToolDelay(5);

    this.checkTiredAnimations();
  }

  addToolDelay() /*delay*/{
  }

  checkTiredAnimations() {
  }

  doToolDelayAnimation(toolId, delay, toolState) {
    if (this._toolId !== undefined) return;
    if (delay === undefined) delay = 40;

    // this._toolState = toolState;
    this.setToolState(toolState);
    this._toolDirection = this._direction;
    this._doingDelayAnimation = true;
    this._doingToolAnimation = false;
    this._doingJumpAnimation = false;

    const character = this;

    setTimeout(() => {
      character.endToolAnimation();
    }, delay);
  }

  doToolJumpAnimation(toolId, delay, toolState) {
    if (this._toolId !== undefined) return;

    if (delay === undefined) delay = 80;

    // this._toolState = toolState;
    this.setToolState(toolState);
    this._toolId = toolId;
    this._toolDirection = this._direction;
    this._doingJumpAnimation = true;
    this._doingToolAnimation = false;
    this._doingDelayAnimation = false;

    const character = this;
    let intervalHandler;
    let timesTurned = 0;

    const nextDirectionFn = () => {
      if (timesTurned >= 4) {
        clearInterval(intervalHandler);
        character.endToolAnimation();
      } else {
        switch(character._toolDirection) {
          case Direction.UP :
            character._toolDirection = Direction.RIGHT;
            break;
          case Direction.RIGHT :
            character._toolDirection = Direction.DOWN;
            break;
          case Direction.DOWN :
            character._toolDirection = Direction.LEFT;
            break;
          default :
            character._toolDirection = Direction.UP;
            break;
        }

        timesTurned++;
      }
    };

    intervalHandler = setInterval(nextDirectionFn, delay);
  }

  setDirection(d) {
    if (!this.isDirectionFixed() && d) {
      this._direction = d;
    }
    this.resetStopCount();
  }

  setDiagonalDirection(horz, vert) {
    if (!this.isDirectionFixed()) {
      this._horzDirection = horz;
      this._vertDirection = vert;

      if (this._direction === this.reverseDir(horz)) {
        this.setDirection(horz);
      }
      if (this._direction === this.reverseDir(vert)) {
        this.setDirection(vert);
      }
    }
    
    this.resetStopCount();
  }

  isTile() {
    return this._tileId > 0 && this._priorityType === 0;
  }

  scrolledX() {
    return $gameMap.adjustX(this._realX);
  }

  scrolledY() {
    return $gameMap.adjustY(this._realY);
  }

  screenX() {
    const tw = $gameMap.tileWidth();
    const screenX = Math.fix(this.scrolledX() * tw + tw / 2);

    return screenX;
  }

  zoomedScreenX() {
    return this.screenX() * $gameMap.zoom.x;
  }

  screenY() {
    const th = $gameMap.tileHeight();
    return Math.fix(this.scrolledY() * th + th - this.jumpHeight());
  }

  zoomedScreenY() {
    return this.screenY() * $gameMap.zoom.y;
  }

  screenZ() {
    return (this._priorityType || 0) * 2 + 1;
  }

  isNearTheScreen() {
    const sx = this.screenX();
    const sy = this.screenY();

    return $gameScreen.isPositionNearTheScreen(sx, sy);
  }

  update() {
    if (this.isStopping()) {
      this.updateStop();
    }
    if (this.isJumping()) {
      this.updateJump();
    } else if (this.isMoving()) {
      this.updateMove();
    }
    this.updateAnimation();
  }

  erase() {
    console.error('Tried to erase a Game_Character');
  }

  checkDestination() {
    let direction;
    let distance;

    if (this._movementDelay !== undefined && this._movementDelay > 0) {
      this._movementDelay--;
      return false;
    }

    if (this._xDestination !== undefined && this._yDestination !== undefined) {
      if (this._xDestination == this._x && this._yDestination == this._y) {
        // If the character reached the destination, check if there's a direction to face
        if (this._dDestination !== undefined && this._dDestination !== 0) {
          if (this.isMoving()) {
            return false;
          }
          
          if (this._dDestination === 'remove') {
            this.erase();
          } else {
            this.setDirection(this._dDestination);
          }
        }

        this.clearDestination();
      }

      if (this._xDestination !== undefined) {
        if (!this.isMoving()) {
          const xDistance = this._x - this._xDestination;
          const yDistance = this._y - this._yDestination;
          
          // Check if there's any additional partial tile to walk
          if (Math.abs(xDistance) < 1 && Math.abs(yDistance) < 1) {
            if (xDistance < 0) {
              this.setDirection(6);
              this._x = this._xDestination;
              return true;
            } else if (yDistance < 0) {
              this.setDirection(2);
              this._y = this._yDestination;
              return true;
            } else if (xDistance > 0) {
              this.setDirection(4);
              this._x = this._xDestination;
              return true;
            } else if (yDistance > 0) {
              this._y = this._yDestination;
              this.setDirection(8);
              return true;
            }
          } else {
            //Check if there's any partial position to fix before start walking
            if (this._x - Math.floor(this._x) > 0) {
              this.setDirection(4);
              this._x = Math.floor(this._x);

              return true;
            }

            if (this._y - Math.floor(this._y) > 0) {
              this.setDirection(8);
              this._y = Math.floor(this._y);

              return true;
            }
          }

          direction = this.findDirectionTo(Math.floor(this._xDestination), Math.floor(this._yDestination));

          if (direction !== undefined && direction > 0) {
            this.moveStraight(direction);
            if (!this.isMovementSucceeded()) {
              this._movementDelay = Constants.FAILED_MOVEMENT_DELAY;
              
              if (this._isRandomDestination) {
                this.clearDestination();
              }

              return false;
            } else {
              return true;
            }
          } else {
            if (this._isRandomDestination) {
              this.clearDestination();
              return false;
            }
          }
        }
      }
    }

    if (this._destinationCharacter !== undefined) {
      if (this._destinationCharacter._x === this._x && this._destinationCharacter._y == this._y) {
        //If the stalker reached the character, check if it needs to keep following it
        if (this._followCharacter !== true) {
          this.clearDestination();
        } else {
          return false;
        }
      }

      if (this._destinationCharacter !== undefined) {
        if (!this.isMoving()) {
          direction = this.findDirectionTo(this._destinationCharacter._x, this._destinationCharacter._y);

          if (direction > 0) {
            this.moveStraight(direction);
            if (!this.isMovementSucceeded()) {

              //If failed to move, and it's not set to follow the character and distance is less than 1 tile, stop moving.
              if (this._followCharacter !== true) {
                distance = Math.abs(this._x - this._destinationCharacter._x) + Math.abs(this._y - this._destinationCharacter._y);
                if (distance <= 1) {
                  this.clearDestination();
                  return false;
                }                
              }

              this._movementDelay = Constants.FAILED_MOVEMENT_DELAY;
            }

            return false;
          } else {
            this.clearDestination();
            return false;
          }
        }
      }
    }

    return false;
  }

  setRandomDestination() {
    if (this._xDestination !== undefined && this._yDestination !== undefined) return;

    const width = $gameMap.width();
    const height = $gameMap.height();
    const rangeX = 5;
    const rangeY = 5;

    const minX = Math.max(0, this._x - rangeX);
    const minY = Math.max(0, this._y - rangeY);
    const maxX = Math.min(width - 1, this._x + rangeX);
    const maxY = Math.min(height - 1, this._y + rangeY);

    const diffX = maxX - minX;
    const diffY = maxY - minY;

    this._isRandomDestination = true;
    this._xDestination = minX + Math.floor(Math.random() * diffX);
    this._yDestination = minY + Math.floor(Math.random() * diffY);
  }

  setDestination(x, y, d/*, expectedX, expectedY*/) {
    if (this._x != x || this._y != y || this.isMoving()) {
      // this.fixPixelPositions();

      this._xDestination = x;
      this._yDestination = y;
      
      if (d !== undefined) {
        this._dDestination = d;
      }
    } else if (d !== undefined && d !== 0) {
      this.setDirection(d);
    }
  }

  setCharacterDestination(character, follow) {
    if (follow === undefined) follow = false;

    if (typeof(character) == "number") {
      character = $gameMap._interpreter.character(character);
    }

    if (character === undefined) return;

    if (follow === true) {
      this._destinationCharacter = character;
      this._followCharacter = true;
    } else {
      if (this._x != character._x || this._y != character._y || this.isMoving()) {
        this._destinationCharacter = character;
        this._followCharacter = false;
      }
    }
  }

  clearDestination() {
    this._xDestination = undefined;
    this._yDestination = undefined;
    this._dDestination = undefined;
    this._destinationCharacter = undefined;
    this._followCharacter = false;

    if (this._isRandomDestination) {
      this._movementDelay = Constants.FAILED_MOVEMENT_DELAY * (Math.random() * 10);
      this._isRandomDestination = false;
    }
  }

  updateStop() {
    if (this._stopCount > 0) {
      this._lostX = 0;
      this._lostY = 0;
    }

    if (!this.checkDestination()) {
      this._stopCount++;
    }
    if (this._moveRouteForcing) {
      this.updateRoutineMove();
    }  
  }

  updateJump() {
    this._jumpCount--;
    this._realX = (this._realX * this._jumpCount + this._x) / (this._jumpCount + 1.0);
    this._realY = (this._realY * this._jumpCount + this._y) / (this._jumpCount + 1.0);

    if (this._jumpCount === 0) {
      this._realX = this._x = $gameMap.roundX(this._x);
      this._realY = this._y = $gameMap.roundY(this._y);
    }

    this._lostX = 0;
    this._lostY = 0;
  }

  updateMove() {
    const distancePerFrame = this.distancePerFrame();
    let newDirection = this._direction;
    let vertDirection = 0;
    let horzDirection = 0;

    if (this._y < this._realY) {
      const smallerY = this._realY - distancePerFrame;
      this._realY = Math.max(smallerY, this._y);
      newDirection = Direction.UP;
      vertDirection = Direction.UP;

      this._lostY = smallerY - this._realY;
    } else if (this._y > this._realY) {
      const biggerY = this._realY + distancePerFrame;
      this._realY = Math.min(biggerY, this._y);
      newDirection = Direction.DOWN;
      vertDirection = Direction.DOWN;

      this._lostY = biggerY - this._realY;
    }

    if (this._x < this._realX) {
      const smallerX = this._realX - distancePerFrame;
      this._realX = Math.max(smallerX, this._x);
      newDirection = Direction.LEFT;
      horzDirection = Direction.LEFT;

      this._lostX = smallerX - this._realX;
    } else if (this._x > this._realX) {
      const biggerX = this._realX + distancePerFrame;
      this._realX = Math.min(biggerX, this._x);
      newDirection = Direction.RIGHT;
      horzDirection = Direction.RIGHT;

      this._lostX = biggerX - this._realX;
    }

    if (this.isMoving()) {
      if (!this._directionFix && newDirection !== this._direction) {
        if (this._direction !== vertDirection && this._direction !== horzDirection) {
          this.setDirectionAfterMoving(newDirection);
        }
      }
    }
  }

  updateAnimation() {
    this.updateAnimationCount();
    if (this._animationCount >= this.animationWait()) {
      this.updatePattern();
      this._animationCount = 0;
    }
  }

  animationWait() {
    return Utils.getFrameCount((9 - this.realMoveSpeed()) * 3);
  }

  updateAnimationCount() {
    if (this.isMoving() && this.hasWalkAnime()) {
      this._animationCount += 1.5;
    } else if (this.hasStepAnime() || !this.isOriginalPattern()) {
      this._animationCount++;
    }
  }

  updatePattern() {
    if (this._lockPattern) return;
    
    if (!this.hasStepAnime() && this._stopCount > 0) {
      this.resetPattern();
    } else {
      let maxPattern = this.maxPattern();
      if (!this._forceSequentialAnimation) {
        //For compatibility with RM's stupid way of handling frames, if maxPattern is 3, then it's actually 4.
        if (maxPattern == 3) maxPattern = 4;
      }

      this._pattern = (this._pattern + 1) % maxPattern;
    }
  }

  maxPattern() {
    if (!this._characterName) return 3;

    const regex = /\[(\d*)\]/;
    const match = this._characterName.match(regex);

    if (match && match.length >= 2) {
      return parseInt(match[1]);
    } else {
      return 3;
    }
  }

  pattern() {
    const maxPattern = this.maxPattern();

    if (!this._forceSequentialAnimation) {
      //If maxPattern is 3, use the stupid RM way
      //If it's anything else, make a sequence
      if (maxPattern == 3 && this._pattern == 3) return 1;
    }

    return this._pattern < maxPattern ? this._pattern : 0;
  }

  setPattern(pattern) {
    this._pattern = pattern;
  }

  isOriginalPattern() {
    return this.pattern() === 1;
  }

  resetPattern() {
    if (this._lockPattern) return;

    this.setPattern(1);
  }

  isOnLadder() {
    return $gameMap.isLadder(this._x, this._y);
  }

  isOnBush() {
    return $gameMap.isBush(this._x, this._y);
  }

  terrainTag() {
    return $gameMap.terrainTag(this._x, this._y);
  }

  regionId() {
    return $gameMap.regionId(this._x, this._y);
  }

  increaseSteps() {
    // if (this.isOnLadder()) {
    //   this.setDirection(8);
    // }
    this.resetStopCount();
  }

  tileId() {
    return this._tileId;
  }

  iconIndex() {
    return this._iconIndex;
  }

  characterName() {
    if (this._hidden) {
      return '';
    } else {
      return this._characterName;
    }
  }

  toolId() {
    if (this._toolId == 'none') return undefined;
    
    return this._toolId;
  }

  setToolState(newState) {
    this._toolState = newState;
  }

  toolState() {
    return this._toolState;
  }

  characterIndex() {
    return this._characterIndex;
  }

  setImage(characterName, characterIndex) {
    this._tileId = 0;
    this._iconIndex = 0;
    this._characterName = characterName;
    this._characterIndex = characterIndex;
  }

  setTileImage(tileId) {
    this._tileId = tileId;
    this._iconIndex = 0;
    this._characterName = '';
    this._characterIndex = 0;
  }

  setIconImage(iconIndex) {
    this._iconIndex = iconIndex;
    this._tileId = 0;
    this._characterName = '';
    this._characterIndex = 0;
  }

  isMovementSucceeded() {
    return this._movementSuccess;
  }

  setMovementSuccess(success) {
    this._movementSuccess = success;
  }

  beforeMove() /*d, maxStepSize*/{
    this._previousX = this._x;
    this._previousY = this._y;
  }

  afterMove() {
    this.increaseSteps();
  }

  setDirectionAfterMoving(d) {
    this.setDirection(d);
  }

  forceForwardMovement(stepSize=1) {
    switch(this._direction) {
      case Direction.LEFT :
        this._x -= stepSize;
        break;
      case Direction.RIGHT :
        this._x += stepSize;
        break;
      case Direction.DOWN :
        this._y += stepSize;
        break;
      case Direction.UP :
        this._y -= stepSize;
        break;
    }
  }

  setMovingPattern() {
    // if (this._x == this._realX && this._y == this._realY) {
    //   //Avoid starting the animation on the idle frame
    //   if (this.hasWalkAnime()) {
    //     this._pattern = 0;
    //     this._animationCount = 0;
    //   }
    // }
  }

  beforeSuccessfulMoveStraight() /*d*/{
    this.setMovingPattern();
  }

  moveStraight(d) {
    this.beforeMove(d, 1);
    this.setMovementSuccess(this.canPass(this._x, this._y, d));
    this.setDiagonalDirection(0, 0);

    if (this.isMovementSucceeded()) {
      this.beforeSuccessfulMoveStraight(d);

      this.setDirectionAfterMoving(d);
      this._x = $gameMap.roundXWithDirection(this._x, d);
      this._y = $gameMap.roundYWithDirection(this._y, d);
      this._realX = $gameMap.xWithDirection(this._x, this.reverseDir(d));
      this._realY = $gameMap.yWithDirection(this._y, this.reverseDir(d));
      this._lostX = 0;
      this._lostY = 0;

      this.afterMove();
    } else {
      this.setDirection(d);
    }
  }

  beforeSuccessfulMoveDiagonally() /*horz, vert*/{
    this.setMovingPattern();
  }

  moveDiagonally(horz, vert) {
    this.setMovementSuccess(this.canPassDiagonally(this._x, this._y, horz, vert));
    if (this.isMovementSucceeded()) {
      this.beforeSuccessfulMoveDiagonally(horz, vert);

      this._x = $gameMap.roundXWithDirection(this._x, horz);
      this._y = $gameMap.roundYWithDirection(this._y, vert);
      this._realX = $gameMap.xWithDirection(this._x, this.reverseDir(horz));
      this._realY = $gameMap.yWithDirection(this._y, this.reverseDir(vert));
      this._lostX = 0;
      this._lostY = 0;
      this.afterMove();
    }
    this.setDiagonalDirection(horz, vert);
  }

  lockedJump(xPlus, yPlus) {
    this.jump(xPlus, yPlus, true);
  }

  jump(xPlus, yPlus, lockDirection, validateCollision) {
    if (lockDirection !== true) {
      if (Math.abs(xPlus) > Math.abs(yPlus)) {
        if (xPlus !== 0) {
          this.setDirection(xPlus < 0 ? 4 : 6);
        }
      } else {
        if (yPlus !== 0) {
          this.setDirection(yPlus < 0 ? 8 : 2);
        }
      }
    }
    
    this._x += xPlus;
    this._y += yPlus;
    const distance = Math.round(Math.sqrt(xPlus * xPlus + yPlus * yPlus));
    this._jumpPeak = 10 + distance - this._moveSpeed;
    this._jumpCount = this._jumpPeak * 2;
    this._jumpCollision = !!validateCollision;
    this.resetStopCount();
    this.straighten();
  }

  jumpTo(x, y) {
    this.jump(x - this.x, y - this.y);
  }

  hasWalkAnime() {
    return this._walkAnime;
  }

  setWalkAnime(walkAnime) {
    this._walkAnime = walkAnime;
  }

  hasStepAnime() {
    return this._stepAnime;
  }

  setStepAnime(stepAnime) {
    this._stepAnime = stepAnime;
  }

  isDirectionFixed() {
    return this._directionFix;
  }

  setDirectionFix(directionFix) {
    this._directionFix = directionFix;
  }

  isThrough() {
    return this._through;
  }

  setThrough(through) {
    this._through = through;
  }

  requestBalloon(balloonId) {
    if (this._balloonId == balloonId) {
      return;
    }

    if (this._balloonPlaying && this._lastBalloonId == balloonId) {
      return;
    }

    this._balloonId = balloonId;
    this._lastBalloonId = balloonId;
  }

  balloonId() {
    return this._balloonId;
  }

  startBalloon() {
    this._balloonId = 0;
    this._balloonPlaying = true;
  }

  isBalloonPlaying() {
    return this._balloonId > 0 || this._balloonPlaying;
  }

  endBalloon() {
    this._balloonPlaying = false;
  }

  holdingFlashlight() {
    return false;
  }

  holdingLantern() {
    return false;
  }

  memorizeMoveRoute() {
    this._originalMoveRoute = this._moveRoute;
    this._originalMoveRouteIndex = this._moveRouteIndex;
  }

  restoreMoveRoute() {
    this._moveRoute = this._originalMoveRoute;
    this._moveRouteIndex = this._originalMoveRouteIndex;
    this._originalMoveRoute = null;
  }

  isMoveRouteForcing() {
    return this._moveRouteForcing;
  }

  setMoveRoute(moveRoute) {
    this.clearDestination();
    this._moveRoute = moveRoute;
    this._moveRouteIndex = 0;
    this._moveRouteForcing = false;
  }

  forceMoveRoute(moveRoute) {
    if (!this._originalMoveRoute) {
      this.memorizeMoveRoute();
    }
    this._moveRoute = moveRoute;
    this._moveRouteIndex = 0;
    this._moveRouteForcing = true;
    this._waitCount = 0;
  }

  updateRoutineMove() {
    if (this._waitCount > 0) {
      this._waitCount--;
    } else {
      this.setMovementSuccess(true);
      const command = this._moveRoute.list[this._moveRouteIndex];
      if (command) {
        this.processMoveCommand(command);
        this.advanceMoveRouteIndex();
      }
    }
  }

  processMoveCommand(command) {
    const params = command.parameters;
    switch (command.code) {
      case Routes.ROUTE_END:
        this.processRouteEnd();
        break;
      case Routes.ROUTE_MOVE_DOWN:
        this.moveStraight(2);
        break;
      case Routes.ROUTE_MOVE_LEFT:
        this.moveStraight(4);
        break;
      case Routes.ROUTE_MOVE_RIGHT:
        this.moveStraight(6);
        break;
      case Routes.ROUTE_MOVE_UP:
        this.moveStraight(8);
        break;
      case Routes.ROUTE_MOVE_LOWER_L:
        this.moveDiagonally(4, 2);
        break;
      case Routes.ROUTE_MOVE_LOWER_R:
        this.moveDiagonally(6, 2);
        break;
      case Routes.ROUTE_MOVE_UPPER_L:
        this.moveDiagonally(4, 8);
        break;
      case Routes.ROUTE_MOVE_UPPER_R:
        this.moveDiagonally(6, 8);
        break;
      case Routes.ROUTE_MOVE_RANDOM:
        this.moveRandom();
        break;
      case Routes.ROUTE_MOVE_TOWARD:
        this.moveTowardPlayer();
        break;
      case Routes.ROUTE_MOVE_AWAY:
        this.moveAwayFromPlayer();
        break;
      case Routes.ROUTE_MOVE_FORWARD:
        this.moveForward();
        break;
      case Routes.ROUTE_MOVE_BACKWARD:
        this.moveBackward();
        break;
      case Routes.ROUTE_JUMP:
        this.jump(params[0], params[1]);
        break;
      case Routes.ROUTE_WAIT:
        this._waitCount = Utils.getFrameCount(params[0] - 1);
        break;
      case Routes.ROUTE_TURN_DOWN:
        this.setDirection(2);
        break;
      case Routes.ROUTE_TURN_LEFT:
        this.setDirection(4);
        break;
      case Routes.ROUTE_TURN_RIGHT:
        this.setDirection(6);
        break;
      case Routes.ROUTE_TURN_UP:
        this.setDirection(8);
        break;
      case Routes.ROUTE_TURN_90D_R:
        this.turnRight90();
        break;
      case Routes.ROUTE_TURN_90D_L:
        this.turnLeft90();
        break;
      case Routes.ROUTE_TURN_180D:
        this.turn180();
        break;
      case Routes.ROUTE_TURN_90D_R_L:
        this.turnRightOrLeft90();
        break;
      case Routes.ROUTE_TURN_RANDOM:
        this.turnRandom();
        break;
      case Routes.ROUTE_TURN_TOWARD:
        this.turnTowardPlayer();
        break;
      case Routes.ROUTE_TURN_AWAY:
        this.turnAwayFromPlayer();
        break;
      case Routes.ROUTE_SWITCH_ON:
        $gameSwitches.setValue(params[0], true);
        break;
      case Routes.ROUTE_SWITCH_OFF:
        $gameSwitches.setValue(params[0], false);
        break;
      case Routes.ROUTE_CHANGE_SPEED:
        this.setMoveSpeed(params[0]);
        break;
      case Routes.ROUTE_CHANGE_FREQ:
        this.setMoveFrequency(params[0]);
        break;
      case Routes.ROUTE_WALK_ANIME_ON:
        this.setWalkAnime(true);
        break;
      case Routes.ROUTE_WALK_ANIME_OFF:
        this.setWalkAnime(false);
        break;
      case Routes.ROUTE_STEP_ANIME_ON:
        this.setStepAnime(true);
        break;
      case Routes.ROUTE_STEP_ANIME_OFF:
        this.setStepAnime(false);
        break;
      case Routes.ROUTE_DIR_FIX_ON:
        this.setDirectionFix(true);
        break;
      case Routes.ROUTE_DIR_FIX_OFF:
        this.setDirectionFix(false);
        break;
      case Routes.ROUTE_THROUGH_ON:
        this.setThrough(true);
        break;
      case Routes.ROUTE_THROUGH_OFF:
        this.setThrough(false);
        break;
      case Routes.ROUTE_CHANGE_IMAGE:
        this.setImage(params[0], params[1]);
        break;
      case Routes.ROUTE_CHANGE_OPACITY:
        this.setOpacity(params[0]);
        break;
      case Routes.ROUTE_CHANGE_BLEND_MODE:
        this.setBlendMode(params[0]);
        break;
      case Routes.ROUTE_PLAY_SE:
        Audio.playSe(params[0]);
        break;
      case Routes.ROUTE_SCRIPT:
        eval(params[0]); // jshint ignore:line
        break;
      default :
        break;
    }
  }

  deltaRealXFrom(x) {
    return $gameMap.deltaX(this._realX, x);
  }

  deltaRealYFrom(y) {
    return $gameMap.deltaY(this._realY, y);
  }

  deltaXFrom(x) {
    return $gameMap.deltaX(this.x, x);
  }

  deltaYFrom(y) {
    return $gameMap.deltaY(this.y, y);
  }

  moveRandom(forceRandom) {
    if (forceRandom !== true) {
      if (this._horzDirection && this._vertDirection) {
        if (Math.randomInt(2) == 1) {
          this.moveStraight(this._horzDirection);
        } else {
          this.moveStraight(this._vertDirection);
        }
        return;
      }
    }

    const d = 2 + Math.randomInt(4) * 2;
    if (this.canPass(this.x, this.y, d)) {
      this.moveStraight(d);
    }
  }

  canMoveToDirection(d) {
    return this.canPass(this.x, this.y, d);
  }

  moveTowardCharacter(character) {
    const sx = this.deltaXFrom(character.x);
    const sy = this.deltaYFrom(character.y);
    if (Math.abs(sx) > Math.abs(sy)) {
      this.moveStraight(sx > 0 ? 4 : 6);
      if (!this.isMovementSucceeded() && sy !== 0) {
        this.moveStraight(sy > 0 ? 8 : 2);
      }
    } else if (sy !== 0) {
      this.moveStraight(sy > 0 ? 8 : 2);
      if (!this.isMovementSucceeded() && sx !== 0) {
        this.moveStraight(sx > 0 ? 4 : 6);
      }
    }
  }

  moveAwayFromCharacter(character) {
    const sx = this.deltaXFrom(character.x);
    const sy = this.deltaYFrom(character.y);
    if (Math.abs(sx) > Math.abs(sy)) {
      this.moveStraight(sx > 0 ? 6 : 4);
      if (!this.isMovementSucceeded() && sy !== 0) {
        this.moveStraight(sy > 0 ? 2 : 8);
      }
    } else if (sy !== 0) {
      this.moveStraight(sy > 0 ? 2 : 8);
      if (!this.isMovementSucceeded() && sx !== 0) {
        this.moveStraight(sx > 0 ? 6 : 4);
      }
    }
  }

  directionToPosition(x, y) {
    const sx = this.deltaXFrom(x);
    const sy = this.deltaYFrom(y);
    if (Math.abs(sx) > Math.abs(sy)) {
      return sx > 0 ? 4 : 6;
    } else if (sy !== 0) {
      return sy > 0 ? 8 : 2;
    }
  }

  turnTowardPosition(x, y) {
    const direction = this.directionToPosition(x, y);

    this.setDirection(direction);
  }

  turnTowardCharacter(character) {
    this.turnTowardPosition(character.x, character.y);
  }

  turnAwayFromCharacter(character) {
    const sx = this.deltaXFrom(character.x);
    const sy = this.deltaYFrom(character.y);
    if (Math.abs(sx) > Math.abs(sy)) {
      this.setDirection(sx > 0 ? 6 : 4);
    } else if (sy !== 0) {
      this.setDirection(sy > 0 ? 2 : 8);
    }
  }

  turnTowardPlayer() {
    const sx = this.deltaRealXFrom($gamePlayer._realX);
    const sy = this.deltaRealYFrom($gamePlayer._realY);

    if (sx.abs() < 0 && sy.abs() < 0) {
      this.setDirection(10 - $gamePlayer.direction);
    } else {
      if (sx.abs() > sy.abs()) {
        this.setDirection(sx > 0 ? 4 : 6);
      } else if (sy !== 0) {
        this.setDirection(sy > 0 ? 8 : 2);
      }
    }
  }

  getDirectionToPlayer() {
    const sx = this.deltaXFrom($gamePlayer.x);
    const sy = this.deltaYFrom($gamePlayer.y);

    if (sx.abs() < 1 && sy.abs() < 1) {
      return 10 - $gamePlayer.direction;
    } else {
      if (sx.abs() > sy.abs()) {
        return sx > 0 ? 4 : 6;
      } else if (sy !== 0) {
        return sy > 0 ? 8 : 2;
      }
    }

    return 0;
  }

  isTurnedTowardPlayer() {
    const direction = this.getDirectionToPlayer();
    return this._direction === direction;
  }

  turnAwayFromPlayer() {
    const sx = this.deltaXFrom($gamePlayer.floatX);
    const sy = this.deltaYFrom($gamePlayer.floatY);

    if (sx.abs() < 1 && sy.abs() < 1) {
      this.setDirection($gamePlayer.direction);
    } else {
      if (sx.abs() > sy.abs()) {
        this.setDirection(sx > 0 ? 7 : 4);
      } else if (sy !== 0) {
        this.setDirection(sy > 0 ? 2 : 8);
      }
    }
  }

  moveTowardPlayer() {
    this.moveTowardCharacter($gamePlayer);
  }

  moveAwayFromPlayer() {
    this.moveAwayFromCharacter($gamePlayer);
  }

  makeRunAwayRoute(duration) {
    const list = [];

    const speed = this._moveSpeed;
    list.push({
      code: Routes.ROUTE_CHANGE_SPEED,
      parameters: [speed < 3 ? 3.5 : speed + 1]
    });

    for (let i = 0; i <= duration; i++) {
      list.push({
        code: Routes.ROUTE_MOVE_AWAY,
        parameters: []
      });
    }

    list.push({
      code: Routes.ROUTE_CHANGE_SPEED,
      parameters: [speed]
    });

    return list;
  }

  runAwayFromPlayer(duration) {
    const list = this.makeRunAwayRoute(duration);

    list.push({
      code: Routes.ROUTE_END,
      params: []
    });

    this.requestBalloon(Balloons.EXCLAMATION);
    this.forceMoveRoute({
      list,
      repeat: false,
      skippable: true
    });
  }

  moveForward() {
    if (this._horzDirection && this._vertDirection) {
      this.moveDiagonally(this._horzDirection, this._vertDirection);
    } else {
      this.moveStraight(this.direction());
    }
  }

  moveBackward() {
    const lastDirectionFix = this.isDirectionFixed();
    this.setDirectionFix(true);
    this.moveStraight(this.reverseDir(this.direction()));
    this.setDirectionFix(lastDirectionFix);
  }

  processRouteEnd() {
    if (this._moveRoute.repeat) {
      this._moveRouteIndex = -1;
    } else if (this._moveRouteForcing) {
      this._moveRouteForcing = false;
      this.restoreMoveRoute();
    }
  }

  advanceMoveRouteIndex() {
    if (this._xDestination === undefined && this._yDestination === undefined && this._destinationCharacter === undefined) {
      const moveRoute = this._moveRoute;
      if (moveRoute && (this.isMovementSucceeded() || moveRoute.skippable)) {
        const numCommands = moveRoute.list.length - 1;
        this._moveRouteIndex++;
        if (moveRoute.repeat && this._moveRouteIndex >= numCommands) {
          this._moveRouteIndex = 0;
        }
      }
    }
  }

  turnRight90() {
    switch (this.direction()) {
      case 2:
        this.setDirection(4);
        break;
      case 4:
        this.setDirection(8);
        break;
      case 6:
        this.setDirection(2);
        break;
      case 8:
        this.setDirection(6);
        break;
      default :
        break;
    }
  }

  turnLeft90() {
    switch (this.direction()) {
      case 2:
        this.setDirection(6);
        break;
      case 4:
        this.setDirection(2);
        break;
      case 6:
        this.setDirection(8);
        break;
      case 8:
        this.setDirection(4);
        break;
      default :
        break;
    }
  }

  turn180() {
    this.setDirection(this.reverseDir(this.direction()));
  }

  turnRightOrLeft90() {
    switch (Math.randomInt(2)) {
      case 0:
        this.turnRight90();
        break;
      case 1:
        this.turnLeft90();
        break;
      default :
        break;
    }
  }

  turnRandom() {
    this.setDirection(2 + Math.randomInt(4) * 2);
  }

  swap(character) {
    const newX = character.x;
    const newY = character.y;
    character.locate(this.x, this.y);
    this.locate(newX, newY);
  }

  getDirectionNode(start, goalX, goalY) {
    const searchLimit = this.searchLimit();
    const mapWidth = $gameMap.width();
    const nodeList = [];
    const openList = [];
    const closedList = [];
    let best = start;

    if (this.x === goalX && this.y === goalY) {
      return undefined;
    }

    nodeList.push(start);
    openList.push(start.y * mapWidth + start.x);

    while (nodeList.length > 0) {
      let bestIndex = 0;
      for (let i = 0; i < nodeList.length; i++) {
        if (nodeList[i].f < nodeList[bestIndex].f) {
          bestIndex = i;
        }
      }

      const current = nodeList[bestIndex];
      const x1 = current.x;
      const y1 = current.y;
      const pos1 = y1 * mapWidth + x1;
      const g1 = current.g;

      nodeList.splice(bestIndex, 1);
      openList.splice(openList.indexOf(pos1), 1);
      closedList.push(pos1);

      if (current.x === goalX && current.y === goalY) {
        best = current;
        break;
      }

      if (g1 >= searchLimit) {
        continue;
      }

      for (let j = 0; j < 4; j++) {
        const direction = 2 + j * 2;

        let x2;
        let y2;

        x2 = $gameMap.roundXWithDirection(x1, direction);
        y2 = $gameMap.roundYWithDirection(y1, direction);

        const pos2 = y2 * mapWidth + x2;

        if (closedList.contains(pos2)) {
          continue;
        }

        if (x1.floor() == goalX && y1.floor() == goalY) {
          break;
        }

        if (!this.canPass(x1, y1, direction)) {
          continue;
        }

        let g2 = g1;
        g2 += 1;

        const index2 = openList.indexOf(pos2);

        if (index2 < 0 || g2 < nodeList[index2].g) {
          let neighbor;
          if (index2 >= 0) {
            neighbor = nodeList[index2];
          } else {
            neighbor = {};
            nodeList.push(neighbor);
            openList.push(pos2);
          }
          neighbor.parent = current;
          neighbor.x = x2;
          neighbor.y = y2;
          neighbor.g = g2;
          neighbor.f = g2 + $gameMap.distance(x2, y2, goalX, goalY);
          if (!best || neighbor.f - neighbor.g < best.f - best.g) {
            best = neighbor;
          }
        }
      }
    }

    return best;
  }

  clearCachedNode() {
    this.setCachedNode();
  }

  setCachedNode(node, goalX, goalY) {
    this._cachedNode = node;
    this._cachedGoalX = goalX;
    this._cachedGoalY = goalY;
  }

  findDirectionTo(goalX, goalY) {
    if (this.x === goalX && this.y === goalY) {
      return 0;
    }

    if (this._cachedGoalX !== goalX || this._cachedGoalY !== goalY) {
      this.clearCachedNode();
    }

    let node = this._cachedNode;

    const start = {};
    start.parent = null;
    start.x = this.x;
    start.y = this.y;
    start.g = 0;
    start.f = $gameMap.distance(start.x, start.y, goalX, goalY);

    let canRetry = true;
    if (node === undefined) {
      node = this.getDirectionNode(start, goalX, goalY);
      this.setCachedNode(node, goalX, goalY);
      if (node === undefined) {
        return 0;
      }
      canRetry = false;
    }

    if (node.x !== start.x || node.y !== start.y) {
      while (node.parent && (node.parent.x !== start.x || node.parent.y !== start.y)) {
        node = node.parent;
      }

      if (!node.parent) {
        this.clearCachedNode();
        if (canRetry) {
          node = this.getDirectionNode(start, goalX, goalY);
          this.setCachedNode(node, goalX, goalY);
          if (node === undefined) {
            return 0;
          }
        }
      }
    }

    const deltaX1 = $gameMap.deltaX(node.x, start.x);
    const deltaY1 = $gameMap.deltaY(node.y, start.y);

    if (deltaY1 > 0) {
      return 2;
    } else if (deltaX1 < 0) {
      return 4;
    } else if (deltaX1 > 0) {
      return 6;
    } else if (deltaY1 < 0) {
      return 8;
    }

    const deltaX2 = this.deltaXFrom(goalX);
    const deltaY2 = this.deltaYFrom(goalY);
    let direction = 0;

    if (Math.abs(deltaX2) > Math.abs(deltaY2)) {
      direction = deltaX2 > 0 ? 4 : 6;
    } else if (deltaY2 !== 0) {
      direction = deltaY2 > 0 ? 8 : 2;
    }

    if (direction > 0) {
      if (!this.canPass(this._x, this._y, direction)) {
        this.clearCachedNode();
        direction *= -1;
      }
    }

    return direction;
  }

  searchLimit() {
    return 50;
  }

  roundXWithDirection(x, direction) {
    return $gameMap.roundXWithDirection(x, direction);
  }

  roundYWithDirection(y, direction) {
    return $gameMap.roundYWithDirection(y, direction);
  }

  requestAnimation(animationId) {
    this._animationId = animationId;
  }

  animationId() {
    return this._animationId;
  }

  startAnimation() {
    this._animationId = 0;
    this._animationPlaying = true;
  }

  isAnimationPlaying() {
    return this._animationId > 0 || this._animationPlaying;
  }

  endAnimation() {
    this._animationPlaying = false;
  }

  fixPixelPositions() {
    this._x = Math.floor(this._x);
    this._y = Math.floor(this._y);
  }

  myDefaultStepSize() {
    return $gameMap.defaultStepSize();
  }

  getSpriteClass() {
    return Sprite_Character;
  }

  distanceToPlayer() {
    const myRect = this.getRealRectPosition();
    const playerCenter = $gamePlayer.getCenterPosition();

    return $gameMap.distanceBetweenRectAndCenter(myRect, playerCenter);
  }

  maxActivationDistance() {
    return Constants.MAX_EVENT_DISTANCE_MOUSE_OVER;
  }

  centerDistanceToPlayer() {
    return $gameMap.distanceBetweenCenters(this, $gamePlayer);
  }

  distanceToPosition(x, y) {
    const left1 = this.left;
    const right1 = this.right;
    const top1 = this.top;
    const bottom1 = this.bottom;

    const xDistance = Math.fix(Math.min(Math.abs(right1 - x), Math.abs(left1 - x)));
    const yDistance = Math.fix(Math.min(Math.abs(top1 - y), Math.abs(bottom1 - y)));

    return Math.abs(xDistance) + Math.abs(yDistance);
  }

  getRectPosition() {
    return new PIXI.Rectangle(this.left, this.top, this.right - this.left, this.bottom - this.top);
  }

  getRealRectPosition() {
    return new PIXI.Rectangle(this.realLeft, this.realTop, this.realRight - this.realLeft, this.realBottom - this.realTop);
  }

  getCenterPosition() {
    const w = this.right - this.left;
    const h = this.bottom - this.top;

    return new PIXI.Point(this.left + w / 2, this.top + h / 2);
  }

  distanceToTile(x, y) {
    const rect1 = this.getRealRectPosition();
    
    // var rect2 = new PIXI.Rectangle(x, y, 1, 1);
    // return $gameMap.distanceBetweenRects(rect1, rect2, this.x - x, this.y - y);

    const tileCenter = new PIXI.Point(x + 0.5, y + 0.5);
    return $gameMap.distanceBetweenRectAndCenter(rect1, tileCenter);
  }

  distanceFromCenterToTile(x, y) {
    const center = this.getCenterPosition();

    return $gameMap.distanceBetweenPoints(center, new PIXI.Point(x, y));
  }

  hide() {
    this._hidden = true;
  }

  show() {
    this._hidden = false;
  }

  isHidden() {
    return this._hidden;
  }

  restoreSpeed() {
  }

  isHoldingItem() {
    const item = this.itemToDisplay();
    return !!item;
  }

  itemToDisplay() {
    return null;
  }

  isRiding() {
    return false;
  }

  getCurrentAnimalSpriteFrameCount() {
    return 3;
  }

  jumpBack(distance) {
    if (distance === undefined) distance = 1;

    if (distance > 1 || this.canMoveToDirection(this._direction)) {
      switch(this._direction) {
        case Direction.LEFT :
          this.jump(distance, 0, true);
          break;
        case Direction.RIGHT :
          this.jump(-distance, 0, true);
          break;
        case Direction.DOWN :
          this.jump(0, -distance, true);
          break;
        case Direction.UP :
          this.jump(0, distance, true);
          break;
      }
    }
  }

  jumpInPlace() {
    this.jump(0, 0, true);
  }

  isCollidedWithFarmObjects(x, y) {
    if (Managers.FarmObjects.isTilePassable($gameMap._mapId, x, y)) {
      return false;
    }

    if (Managers.FarmObjects.isTileProtected($gameMap._mapId, x, y)) {
      return true;
    }

    const right = x + 1 - insignificantValue;
    const bottom = y + 1 - insignificantValue;
    const character = this;

    const eventFilterFn = event => {
      if (!event) return false;
      if (event._erased) return false;
      if (event._hidden) return false;
      if (event.isThrough()) return false;
      if (!event.isNormalPriority()) return false;
      if (event._farmObjectData) {
        if (!event._farmObjectData.hasCollision()) return false;
      }

      if (!event.realPosIn(x, y, right, bottom)) return false;
      return true;
    };

    const allEvents = $gameMap.farmObjectEvents(eventFilterFn);
    const collided = allEvents.some(event => character.isCollidedWithThisEvent(event, x, y, right, bottom));

    if (collided) return collided;

    return false;
  }

  isCollidedWithThisEvent(event, left, top, right, bottom) {
    if (!event) return false;
    if (event._erased) return false;
    if (event._hidden) return false;
    if (event.isThrough()) return false;
    if (!event.isNormalPriority()) return false;
    if (!event.realPosIn(left, top, right, bottom)) return false;
    if (event._farmObjectData) {
      if (!event._farmObjectData.hasCollision()) return false;
    }

    //If the player is "inside" it, then this event won't be considered,
    //because if it did, the player would be locked on it.
    //this shouldn't be possible on normal conditions, but still happens.
    // Example: When you get off a horse, you may be placed on the same position as the horse sprite
    if (this.isTouchingEvent(event)) return false;

    return true;
  }

  isTouchingEvent(event) {
    if (event.realPosIn(this.left, this.top, this.right, this.bottom)) return true;

    return false;
  }

  animate(steps, { defaultStepLength = 10, loop = false, avoidLock = false }) {
    // Preload sprites
    for (const step of steps) {
      if (step.fileName) {
        Managers.Images.loadCharacter(step.fileName);
      }
    }

    let currentStep = -1;
    const playNextStep = () => {
      currentStep++;
      if (currentStep >= steps.length) {
        if (loop) {
          currentStep = 0;
        } else {
          return;
        }
      }

      if (this._erased) {
        return;
      }

      const step = steps[currentStep];
      if (step.fileName) {
        this._characterName = step.fileName;
      }

      if (step.index || step.index === 0) {
        this._characterIndex = step.index;
      }
      if (step.direction) {
        this._direction = step.direction;
      }
      if (step.pattern || step.pattern === 0) {
        this._pattern = step.pattern;
      }
      if (step.lockPattern !== undefined) {
        this._lockPattern = step.lockPattern;
      }
      if (step.stepAnime !== undefined) {
        this._stepAnime = step.stepAnime;
      }
      if (step.walkAnime !== undefined) {
        this._walkAnime = step.walkAnime;
      }

      const character = this._playerImpersonator ? $gamePlayer : this;

      if (step.offsetY || step.offsetY === 0) {
        character.offsetY = step.offsetY;
      }
      if (step.offsetX || step.offsetX === 0) {
        character.offsetX = step.offsetX;
      }

      if (step.y) {
        character._y += step.y;
        character._realY += step.y;
      }

      if (step.x) {
        character._x += step.x;
        character._realX += step.x;
      }

      if (step.eventName) {
        Managers.CommonEvent.playEventAsync(step.eventName);
      }

      if (step.erase) {
        this.erase();
      }

      if (step.callback) {
        step.callback(this);
      }

      const length = step.frameLength || defaultStepLength || 10;
      $gameTemp.setTimeout(() => {
        playNextStep();
      }, length, avoidLock);
    };

    playNextStep();
  }
};
