require('./Character');

//-----------------------------------------------------------------------------
// Game_ActorCharacter
//
// The superclass of Game_Player and Game_Follower, but not of Game_Event

/* globals insignificantValue */

Objects.ActorCharacter = class ActorCharacter extends Objects.Character {

};

function addPixelMovement($) {
  $.prototype.shouldUseTileMovement = function() {
    if (Managers.Config.tileMovement) return true;
    if (this._xDestination !== undefined && this._yDestination !== undefined) return true;

    return false;
  };

  var oldMyDefaultStepSize = $.prototype.myDefaultStepSize;
  $.prototype.myDefaultStepSize = function() {
    if (this.shouldUseTileMovement()) {
      return 1;
    }

    return oldMyDefaultStepSize.call(this);
  };

  // This method adds or subtracts the step_size to an X position, based on the direction
  $.prototype.fractionXWithDirection = function(x, d, stepSize) {
    if (stepSize === undefined) {
      stepSize = this.myDefaultStepSize();
    }

    if (DirectionHelper.goesLeft(d)) {
      return x - stepSize;
    } else if (DirectionHelper.goesRight(d)) {
      return x + stepSize;
    } else {
      return x;
    }
  };

  // This method adds or subtracts the step_size to a Y position, based on the direction
  $.prototype.fractionYWithDirection = function(y, d, stepSize) {
    if (stepSize === undefined) {
      stepSize = this.myDefaultStepSize();
    }

    if (DirectionHelper.goesDown(d)) {
      return y + stepSize;
    } else if (DirectionHelper.goesUp(d)) {
      return y - stepSize;
    } else {
      return y;
    }
  };

  // When using horizontally looped maps, this method gets the real X position
  $.prototype.roundFractionXWithDirection = function(x, d, stepSize) {
    return $gameMap.roundX(this.fractionXWithDirection(x, d, stepSize));
  };

  // When using vertically looped maps, this method gets the real Y position
  $.prototype.roundFractionYWithDirection = function(y, d, stepSize) {
    return $gameMap.roundY(this.fractionYWithDirection(y, d, stepSize));
  };

  $.prototype.beforeMove = function(/*d, maxStepSize*/) {
    this._previousX = this._x;
    this._previousY = this._y;
  };

  $.prototype.setHitboxX = function(value) {
    this._hitboxX = value;
  };

  $.prototype.getHitboxX = function() {
    if (this._hitboxX === undefined) {
      var actor = this.actor();

      if (actor === undefined) return 0;

      if (actor.hitboxX !== undefined) {
        this._hitboxX = actor.hitboxX;
      } else {
        this._hitboxX = 0;
      }
    }

    return this._hitboxX;
  };

  $.prototype.setHitboxY = function(value) {
    this._hitboxY = value;
  };

  $.prototype.getHitboxY = function() {
    if (this._hitboxY === undefined) {
      var actor = this.actor();

      if (actor === undefined) return 0;

      if (actor.hitboxY !== undefined) {
        this._hitboxY = actor.hitboxY;
      } else {
        this._hitboxY = 0;
      }
    }

    if (this.shouldUseTileMovement()) {
      //#ToDo
      return 0;
    }

    return this._hitboxY;
  };

  $.prototype.setHitboxWidth = function(value) {
    this._hitboxWidth = value;
  };

  $.prototype.getHitboxWidth = function() {
    if (this._hitboxWidth === undefined) {
      var actor = this.actor();

      if (actor === undefined) return $gameMap.tileWidth();

      if (actor.hitboxWidth !== undefined) {
        this._hitboxWidth = actor.hitboxWidth;
      } else {
        this._hitboxWidth = $gameMap.tileWidth();
      }
    }

    if (this.shouldUseTileMovement()) {
      //#ToDo
      return $gameMap.tileWidth();
    }

    return this._hitboxWidth;
  };

  $.prototype.setHitboxHeight = function(value) {
    this._hitboxHeight = value;
  };

  $.prototype.getHitboxHeight = function() {
    if (this._hitboxHeight === undefined) {
      var actor = this.actor();

      if (actor === undefined) return $gameMap.tileHeight();

      if (actor.hitboxHeight !== undefined) {
        this._hitboxHeight = actor.hitboxHeight;
      } else {
        this._hitboxHeight = $gameMap.tileHeight();
      }
    }

    if (this.shouldUseTileMovement()) {
      //#ToDo
      return $gameMap.tileHeight();
    }

    return this._hitboxHeight;
  };

  $.prototype.getOffsetX = function() {
    if (this._offsetX === undefined) {
      var actor = this.actor();
      if (actor === undefined) return 0;

      if (actor.offsetX !== undefined) {
        this._offsetX = actor.offsetX;
      } else {
        this._offsetX = 0;
      }
    }

    if (this.shouldUseTileMovement()) {
      //#ToDo
      return 0;
    }

    return this._offsetX;
  };

  $.prototype.setOffsetX = function(value) {
    this._offsetX = value;
  };

  $.prototype.getOffsetY = function() {
    if (this._offsetY === undefined) {
      var actor = this.actor();
      if (actor === undefined) return 0;

      if (actor.offsetY !== undefined) {
        this._offsetY = actor.offsetY;
      } else {
        this._offsetY = 0;
      }
    }

    if (this.shouldUseTileMovement()) {
      //#ToDo
      return 0;
    }

    return this._offsetY;
  };

  $.prototype.setOffsetY = function(value) {
    this._offsetY = value;
  };

  $.prototype.getHitboxXSize = function() {
    return Math.fix(this.hitboxX / $gameMap.tileWidth());
  };

  $.prototype.getHitboxYSize = function() {
    return Math.fix(this.hitboxY / $gameMap.tileHeight());
  };

  $.prototype.getHitboxWidthSize = function() {
    return Math.fix(this.hitboxWidth / $gameMap.tileWidth());
  };

  $.prototype.getHitboxHeightSize = function() {
    return Math.fix(this.hitboxHeight / $gameMap.tileHeight());
  };

  $.prototype.getTop = function() {
    return Math.fix(this._y + this.hitboxYSize);
  };

  $.prototype.getRealTop = function() {
    return Math.fix(this._realY + this.hitboxYSize);
  };

  $.prototype.getLeft = function() {
    return Math.fix(this._x + this.hitboxXSize);
  };

  $.prototype.getRealLeft = function() {
    return Math.fix(this._realX + this.hitboxXSize);
  };

  $.prototype.getRight = function() {
    return Math.fix(this._x + this.hitboxXSize + this.hitboxWidthSize - insignificantValue);
  };

  $.prototype.getRealRight = function() {
    return Math.fix(this._realX + this.hitboxXSize + this.hitboxWidthSize - insignificantValue);
  };

  $.prototype.getBottom = function() {
    return Math.fix(this._y + this.hitboxYSize + this.hitboxHeightSize - insignificantValue);
  };

  $.prototype.getRealBottom = function() {
    return Math.fix(this._realY + this.hitboxYSize + this.hitboxHeightSize - insignificantValue);
  };

  // X and Y position of the character hitbox (in pixels)
  MVC.accessor($.prototype, 'hitboxX', $.prototype.setHitboxX, $.prototype.getHitboxX);
  MVC.accessor($.prototype, 'hitboxY', $.prototype.setHitboxY, $.prototype.getHitboxY);

  // Width and Height of the character hitbox (in pixels)
  MVC.accessor($.prototype, 'hitboxWidth', $.prototype.setHitboxWidth, $.prototype.getHitboxWidth);
  MVC.accessor($.prototype, 'hitboxHeight', $.prototype.setHitboxHeight, $.prototype.getHitboxHeight);


  // X position of the character hitbox (in tiles)
  MVC.reader($.prototype, 'hitboxXSize', $.prototype.getHitboxXSize);

  // Y position of the character hitbox (in tiles)
  MVC.reader($.prototype, 'hitboxYSize', $.prototype.getHitboxYSize);

  // Width of the character hitbox (in tiles)
  MVC.reader($.prototype, 'hitboxWidthSize', $.prototype.getHitboxWidthSize);

  // Height of the character hitbox (in tiles)
  MVC.reader($.prototype, 'hitboxHeightSize', $.prototype.getHitboxHeightSize);

  // Gets the top Y position adjusted with the hitbox
  MVC.reader($.prototype, 'top', $.prototype.getTop);

  // Gets the top realY position adjusted with the hitbox
  MVC.reader($.prototype, 'realTop', $.prototype.getRealTop );

  // Gets the left X position adjusted with the hitbox
  MVC.reader($.prototype, 'left', $.prototype.getLeft);

  // Gets the left realX position adjusted with the hitbox
  MVC.reader($.prototype, 'realLeft', $.prototype.getRealLeft);

  // Gets the right X position adjusted with the hitbox
  MVC.reader($.prototype, 'right', $.prototype.getRight);

  // Gets the right realX position adjusted with the hitbox
  MVC.reader($.prototype, 'realRight', $.prototype.getRealRight);

  // Gets the bottom Y position adjusted with the hitbox
  MVC.reader($.prototype, 'bottom', $.prototype.getBottom);

  // Gets the bottom realY position adjusted with the hitbox
  MVC.reader($.prototype, 'realBottom', $.prototype.getRealBottom);

  MVC.accessor($.prototype, 'offsetX', $.prototype.setOffsetX, $.prototype.getOffsetX);
  MVC.accessor($.prototype, 'offsetY', $.prototype.setOffsetY, $.prototype.getOffsetY);

  $.prototype.deltaXFrom = function(x) {
    return $gameMap.deltaX(this._x, x);
  };

  $.prototype.deltaYFrom = function(y) {
    return $gameMap.deltaY(this._y, y);
  };

  $.prototype.roundXWithDirection = function(x, direction) {
    if (this.shouldUseTileMovement()) {
      return $gameMap.roundXWithDirection(x, direction);
    }

    return this.roundFractionXWithDirection(x, direction, this.myStepSize());
  };

  $.prototype.roundYWithDirection = function(y, direction) {
    if (this.shouldUseTileMovement()) {
      return $gameMap.roundYWithDirection(y, direction);
    }

    return this.roundFractionYWithDirection(y, direction, this.myStepSize());
  };

  // Method that checks if the character can move in a specified direction
  $.prototype.canGoTo = function(x, y, d) {
    switch (d) {
      case Direction.UP:
        return this.canGoUp(x, y);
      case Direction.DOWN:
        return this.canGoDown(x, y);
      case Direction.LEFT:
        return this.canGoLeft(x, y);
      case Direction.RIGHT:
        return this.canGoRight(x, y);
      default:
        return false;
    }
  };

  $.prototype.checkUpPassage = function(newX, theY, destinationY/*, minX, maxX*/) {
    var theYCeil = theY.ceil();
    var theYFloor = Math.floor(theY);
    var destinationYFloor = Math.floor(destinationY);

    if (theY + this.hitboxHeightSize > theYCeil) {
      if (!this.checkMapPassable(newX, theYCeil, Direction.UP)) {
        return false;
      }
    } else if (theY - this.hitboxHeightSize < theYFloor) {
      var auxDestinationY = Math.floor(destinationY - this.hitboxHeightSize);
      if (auxDestinationY < theYFloor) {
        if (!this.checkMapPassable(newX, destinationYFloor, Direction.UP)) {
          return false;
        }
      } else {
        if (!this.checkMapPassable(newX, theYFloor, Direction.UP)) {
          return false;
        }
      }
    }

    if (destinationY + this.hitboxHeightSize >= (destinationYFloor + 1)) {
      if (!this.checkMapPassable(newX, destinationYFloor, Direction.DOWN)) {
        return false;
      }
    }

    var destinationYCeil = Math.ceil(destinationY);

    if (destinationYCeil < theYFloor) {
      if (!this.checkMapPassable(newX, destinationYFloor, Direction.DOWN)) {
        return false;
      }
    }

    return null;
  };


  // Method that checks if the character can move up
  $.prototype.canGoUp = function(x, y) {
    // Variables theX and theY hold the true position, considering the hitbox configuration
    var theX = Math.fix(x + this.hitboxXSize);
    var theY = Math.fix(y + this.hitboxYSize);

    // Variable endX hold the right position, considering the hitbox configuration
    // The script decreases an insignificant value from it to ensure that this position doesn't pass to the next integer value unless the character is actually on the next tile.
    var endX = Math.fix(theX + this.hitboxWidthSize - insignificantValue);

    // destinationY is the top position where the player is moving to, considering the hitbox
    var destinationY = Math.fix(theY - this.myIntendedStepSize());

    // Run the collission check for every X tile the character is touching
    for (var newX = Math.floor(theX); newX <= Math.floor(endX); newX++) {
      if (this.checkUpPassage(newX, theY, destinationY, theX, endX) === false) {
        return false;
      }
    }

    return true;
  };

  $.prototype.checkDownPassage = function(newX, endY, destinationEndY/*, minX, maxX*/) {
    if (Math.floor(destinationEndY) > Math.floor(endY)) {
      if (!this.checkMapPassable(newX, Math.floor(endY), Direction.DOWN)) {
        return false;
      }

      if (!this.checkMapPassable(newX, Math.floor(destinationEndY), Direction.UP)) {
        return false;
      }
    }

    return null;
  };

  // Method that checks if the character can move down
  $.prototype.canGoDown = function(x, y) {
    // Variables theX and theY hold the true position, considering the hitbox configuration
    var theX = Math.fix(x + this.hitboxXSize);
    var theY = Math.fix(y + this.hitboxYSize);

    // Variables endX and endY hold the right and bottom position, considering the hitbox configuration
    // The script decreases an insignificant value from it to ensure that this position doesn't pass to the next integer value unless the character is actually on the next tile.
    var endX = Math.fix(theX + this.hitboxWidthSize - insignificantValue);
    var endY = Math.fix(theY + this.hitboxHeightSize - insignificantValue);

    // destinationEndY is the bottom position where the player is moving to, considering the hitbox
    var destinationEndY = Math.fix(endY + this.myIntendedStepSize());

    // Run the collission check for every X tile the character is touching
    for (var newX = Math.floor(theX); newX <= Math.floor(endX); newX++) {
      if (this.checkDownPassage(newX, endY, destinationEndY, theX, endX) === false) {
        return false;
      }
    }

    return true;
  };

  $.prototype.checkLeftPassage = function(theX, newY, destinationX) {
    var ceilX = Math.ceil(theX);

    // If the character is still touching another tile to the right, check if it let's the character move left
    if (theX + this.hitboxWidthSize > ceilX) {
      if (!this.checkMapPassable(ceilX, newY, Direction.LEFT, Direction.LEFT)) {
        return false;
      }
    } else {
      var floorX = Math.floor(theX);

      if (Math.floor(destinationX) < floorX) {
        // Check if the current tile allows moving to a new one
        if (!this.checkMapPassable(floorX, newY, Direction.LEFT, Direction.LEFT)) {
          return false;
        }

        // If the character is moving to a new tile, check if the tile it was on allows the movement
        // This will probably only trigger for events that have a step size bigger than it's hitbox
        if (Math.floor(destinationX + this.hitboxWidthSize) < floorX) {
          // if (!this.checkMapPassable(floorX, newY, Direction.LEFT, Direction.LEFT)) {
          //   return false;
          // }

          if (!this.checkMapPassable(floorX - 1, newY, Direction.RIGHT)) {
            return false;
          }
        }

      }
    }

    //If the character will still be touching two horizontal tiles, check if the left tile let they move right
    if (destinationX + this.hitboxWidthSize >= (Math.floor(destinationX) + 1)) {
      if (!this.checkMapPassable(Math.floor(destinationX), newY, Direction.RIGHT)) {
        return false;
      }
    }

    return null;
  };

  // Method that checks if the character can move left
  $.prototype.canGoLeft = function(x, y) {
    // Variables theX and theY hold the true position, considering the hitbox configuration
    var theX = x + this.hitboxXSize;
    var theY = y + this.hitboxYSize;

    // Variable endY hold the bottom position, considering the hitbox configuration
    // The script decreases an insignificant value from it to ensure that this position doesn't pass to the next integer value unless the character is actually on the next tile.
    var endY = theY + this.hitboxHeightSize - insignificantValue;

    // destinationX is the left position where the player is moving to, considering the hitbox
    var destinationX = theX - this.myIntendedStepSize();

    // Run the collission check for every Y tile the character is touching
    for (var newY = Math.floor(theY); newY <= Math.floor(endY); newY++) {
      if (this.checkLeftPassage(theX, newY, destinationX) === false) {
        return false;
      }
    }

    return true;
  };

  $.prototype.checkRightPassage = function(endX, newY, destinationEndX) {
    if (Math.floor(destinationEndX) > Math.floor(endX)) {
      if (!this.checkMapPassable(Math.floor(endX), newY, Direction.RIGHT)) {
        return false;
      }

      if (!this.checkMapPassable(Math.floor(destinationEndX), newY, Direction.LEFT)) {
        return false;
      }
    }

    return null;
  };

  // Method that checks if the character can move right
  $.prototype.canGoRight = function(x, y) {
    // Variables theX and theY hold the true position, considering the hitbox configuration
    var theX = Math.fix(x + this.hitboxXSize);
    var theY = Math.fix(y + this.hitboxYSize);

    // Variables endX and endY hold the right and bottom position, considering the hitbox configuration
    // The script decreases an insignificant value from it to ensure that this position doesn't pass to the next integer value unless the character is actually on the next tile.
    var endX = Math.fix(theX + this.hitboxWidthSize - insignificantValue);
    var endY = Math.fix(theY + this.hitboxHeightSize - insignificantValue);

    // destinationEndX is the right position where the player is moving to, considering the hitbox
    var destinationEndX = Math.fix(endX + this.myIntendedStepSize());

    // Run the collission check for every Y tile the character is touching
    for (var newY = Math.floor(theY); newY <= Math.floor(endY); newY++) {
      if (this.checkRightPassage(endX, newY, destinationEndX) === false) {
        return false;
      }
    }

    return true;
  };

  $.prototype.myIntendedStepSize = function() {
    return this.myDefaultStepSize();
  };

  $.prototype.myStepSize = function(maxStepSize) {
    var result;
    if (this._moveRoute !== undefined && this._moveRoute !== null && !!this._moveRoute._list && this._moveRoute._list.length > 1) {
      result = 1;
    } else {
      result = this.myIntendedStepSize();
    }

    if (maxStepSize !== undefined && maxStepSize < result) {
      return maxStepSize;
    }

    return result;
  };

  $.prototype.isTilesetPassable = function(x, y, d, stepSize) {
    stepSize = stepSize || this.myStepSize();
    var x2 = this.roundFractionXWithDirection(x, d, stepSize);
    var y2 = this.roundFractionYWithDirection(y, d, stepSize);

    if (!this.isValid(x2, y2)) {
      return false;
    }

    if (this.isThrough() || this.isDebugThrough()) {
      return true;
    }

    if (!this.isMapPassable(x, y, d)) {
      return false;
    }

    if (!this.isMapPassable(x2, y2, this.reverseDir(d))) {
      return false;
    }

    return true;
  };

  // Replaces the original canPass method to consider the step size when getting the new x and Y position
  $.prototype.canPass = function(x, y, d) {
    this._lastCollisionCheck = null;

    if (this.shouldUseTileMovement()) {
      this._lastCollisionCheck = 'tile';
      return this.canPassTile(x, y, d);
    }

    var stepSize = this.myStepSize();
    this._lastCollisionCheck = 'tileset';
    if (!this.isTilesetPassable(x, y, d, stepSize)) {
      return false;
    }

    if (this.isThrough() || this.isDebugThrough()) {
      return true;
    }

    var x2 = this.roundFractionXWithDirection(x, d, stepSize);
    var y2 = this.roundFractionYWithDirection(y, d, stepSize);

    this._lastCollisionCheck = 'characters';
    if (this.isCollidedWithCharacters(x2, y2)) {
      return false;
    }

    this._lastCollisionCheck = 'boxes';
    if (this.isCollidedWithCollisionBoxes(x2, y2)) {
      return false;
    }

    this._lastCollisionCheck = 'success';
    return true;
  };

  // Replaces the original canPassDiagonally method to consider the step size when getting the new x and Y position
  $.prototype.canPassDiagonally = function(x, y, horz, vert) {
    if (this.shouldUseTileMovement()) {
      return this.canPassTileDiagonally(x, y, horz, vert);
    }

    var stepSize = this.myStepSize();
    var x2 = this.roundFractionXWithDirection(x, horz, stepSize);
    var y2 = this.roundFractionYWithDirection(y, vert, stepSize);

    if (this.canPass(x, y, vert) && this.canPass(x, y, horz) && this.canPass(x, y2, horz) && this.canPass(x2, y, vert)) {
      return true;
    }
    return false;
  };

  // Replaces the original isMapPassable method, changing the way the collision is checked to consider fractional position
  $.prototype.isMapPassable = function(x, y, d) {
    if (this.shouldUseTileMovement()) {
      return this.isMapTilePassable(x, y, d);
    }

    if (DirectionHelper.goesUp(d)) {
      if (!this.canGoUp(x, y)) {
        return false;
      }
    } else if (DirectionHelper.goesDown(d)) {
      if (!this.canGoDown(x, y)) {
        return false;
      }
    }

    if (DirectionHelper.goesLeft(d)) {
      if (!this.canGoLeft(x, y)) {
        return false;
      }
    } else if (DirectionHelper.goesRight(d)) {
      if (!this.canGoRight(x, y)) {
        return false;
      }
    }

    return true;
  };

  //Good regions ignore the regions that the character is just slightly touching (usually only by the transparent part of the sprite)
  $.prototype.getGoodRegions = function() {
    return this.getAllRegions(true, true);
  };

  $.prototype.getAllRegions = function(useXRounding, useYRounding) {
    if (useXRounding !== true) useXRounding = false;
    if (useYRounding !== true) useYRounding = false;

    var regions = [];

    this.runForAllPositions(this._x, this._y, function(x, y){
      var regionId = $gameMap.regionId(x, y);

      if (regionId > 0) {
        if (regions.indexOf(regionId) < 0) {
          regions.push(regionId);
        }
      }
    }, useXRounding, useYRounding);

    return regions;
  };

  $.prototype.isTouchingOnlyRegion = function(regionIds) {
    var result = true;
    var touchedAny = false;

    this.runForAllPositions(this._x, this._y, function(x, y){
      var regionId = $gameMap.regionId(x, y);

      if (regionId <= 0) {
        result = false;
        return false;
      }

      if (regionIds.indexOf(regionId) < 0) {
        result = false;
        return false;
      }

      touchedAny = true;
    });

    return result === true && touchedAny === true;
  };

  $.prototype.isTouchingRegion = function(regionIds, useXRounding, useYRounding) {
    var touching = this.getAllRegions(useXRounding, useYRounding);

    if (regionIds instanceof Array) {
      for (var i = 0; i < regionIds.length; i++) {
        if (touching.indexOf(regionIds[i]) >= 0) {
          return true;
        }
      }
    } else {
      if (touching.indexOf(regionIds) >= 0) {
        return true;
      }
    }

    return false;
  };

  $.prototype.isTouchingTile = function(x, y) {
    var left = Math.floor(this.left);
    var right = Math.floor(this.right);
    var top = Math.floor(this.top);
    var bottom = Math.floor(this.bottom);

    if (x < left || x > right) {
      return false;
    }

    if (y < top || y > bottom) {
      return false;
    }

    if (this.shouldUseTileMovement()) {
      if (x == right || y == bottom) return false;
    }

    return true;
  };

  // $.prototype.isTouchingEvent = function(event) {
  //   if (event.realPosIn(this.left, this.top, this.right, this.bottom)) return true;

  //   return false;
  // };

  $.prototype.pos = function(x, y) {
    return this.isTouchingTile(x, y);
  };

  // Run the callback method for all tiles touched by the character on the informed position
  $.prototype.runForAllPositions = function(x, y, callback, useXRounding, useYRounding) {
    if (useXRounding !== true) useXRounding = false;
    if (useYRounding !== true) useYRounding = false;

    var first_x, first_y, last_x, last_y;
    var useTileMovement = this.shouldUseTileMovement();

    if (useTileMovement) {
      first_x = Math.fix(x + this.hitboxXSize);
      first_y = Math.fix(y + this.hitboxYSize);
      last_x = Math.floor(Math.floor(first_x) + this.hitboxWidthSize - insignificantValue);
      last_y = Math.floor(Math.floor(first_y) + this.hitboxHeightSize - insignificantValue);
    } else {
      first_x = Math.fix(x + this.hitboxXSize);
      first_y = Math.fix(y + this.hitboxYSize);
      last_x = Math.floor(first_x + this.hitboxWidthSize - insignificantValue);
      last_y = Math.floor(first_y + this.hitboxHeightSize - insignificantValue);
    }

    if (useXRounding) {
      var diffX = first_x - Math.floor(first_x);
      if (diffX <= 0.6) {
        first_x = Math.floor(first_x);
      } else {
        first_x = Math.ceil(first_x);
      }
    } else {
      first_x = Math.floor(first_x);
    }

    if (useYRounding) {
      var diffY = first_y - Math.floor(first_y);
      if (diffY <= 0.6) {
        first_y = Math.floor(first_y);
      } else {
        first_y = Math.ceil(first_y);
      }
    } else {
      first_y = Math.floor(first_y);
    }

    for (var newX = first_x; newX <= last_x; newX++) {
      for (var newY = first_y; newY <= last_y; newY++) {
        if (callback.call(this, newX, newY) === true) {
          return true;
        }
      }
    }

    return false;
  };

  $.prototype.isCollidedWithCharacters = function(x, y) {
    const left = x + this.hitboxXSize;
    const right = left + this.hitboxWidthSize - insignificantValue;
    const top = y + this.hitboxYSize;
    const bottom = top + this.hitboxHeightSize - insignificantValue;
    const character = this;

    var tileCollided = this.runForAllPositions(x, y, function(block_x, block_y) {
      if (this.isCollidedWithFarmObjects(x, y, block_x, block_y)) {
        return true;
      }

      if (this.isCollidedWithPlayerCharacters(block_x, block_y)) {
        return true;
      }
    });

    if (tileCollided) return true;

    const villagerCollided = $gameMap._villagers.some(function(event){
      return event && !event.isMoving() && character.isCollidedWithThisEvent(event, left, top, right, bottom);
    });

    if (villagerCollided) return true;


    return this.checkOnlyIfIsCollidedWithEvents(x, y);
  };

  $.prototype.isCollidedWithCollisionBoxes = function(x, y) {
    const left = x + this.hitboxXSize;
    const right = left + this.hitboxWidthSize - insignificantValue;
    const top = y + this.hitboxYSize;
    const bottom = top + this.hitboxHeightSize - insignificantValue;

    return this.runForAllPositions(x, y, function(blockX, blockY) {
      const boxes = $gameMap.collisionBoxesXy(blockX, blockY);
      if (!boxes || !boxes.length) {
        return false;
      }

      for (const box of boxes) {
        if (right < blockX + box.x) continue;
        if (left > blockX + box.x + box.width) continue;
        if (bottom < blockY + box.y) continue;
        if (top > blockY + box.y + box.height) continue;
        return true;
      }
    });
  };


  // $.prototype.isCollidedWithThisEvent = function(event, left, top, right, bottom) {
  //   if (!event) return false;
  //   if (event._erased) return false;
  //   if (event._hidden) return false;
  //   if (event.isThrough()) return false;
  //   if (!event.isNormalPriority()) return false;
  //   if (!event.realPosIn(left, top, right, bottom)) return false;
  //   if (event._farmObjectData) {
  //     if (!event._farmObjectData.hasCollision()) return false;
  //   }

  //   //If the player is "inside" it, then this event won't be considered,
  //   //because if it did, the player would be locked on it.
  //   //this shouldn't be possible on normal conditions, but still happens.
  //   // Example: When you get off a horse, you may be placed on the same position as the horse sprite
  //   if (this.isTouchingEvent(event)) return false;

  //   return true;
  // };

  $.prototype.checkOnlyIfIsCollidedWithEvents = function(x, y) {
    var left = x + this.hitboxXSize;
    var right = left + this.hitboxWidthSize - insignificantValue;
    var top = y + this.hitboxYSize;
    var bottom = top + this.hitboxHeightSize - insignificantValue;

    var character = this;

    var collided = $gameMap._events.some(function(event){
      return character.isCollidedWithThisEvent(event, left, top, right, bottom);
    });

    if (collided) return collided;

    return $gameMap._creatures.some(function(event){
      if (event._letPlayerWalkThroughThisEvent) return false;

      return character.isCollidedWithThisEvent(event, left, top, right, bottom);
    });
  };

  // Replaces the method that checks if the position would collide with an event, because fractional positions should test more than one tile
  $.prototype.isCollidedWithEvents = function(x, y) {
    var tileCollided = this.runForAllPositions(x, y, function(block_x, block_y) {

      // if (this.isTouchingTile(block_x, block_y)) {
      //   return false;
      // }

      if (this.isCollidedWithFarmObjects(x, y, block_x, block_y)) {
        return true;
      }
    });

    if (tileCollided) return true;

    return this.checkOnlyIfIsCollidedWithEvents(x, y);
  };

  $.prototype.setMovementBlockingReason = function(data) {
    if (this._lastCollisionCheck !== 'characters') {
      return;
    }

    if (data.reason == 'object') {
      this._lastCollisionCheck = 'object';
    }
  };

  $.prototype.isCollidedWithFarmObjects = function(x, y, blockX, blockY) {
    //Let's make sure a random object won't bug the cutscenes
    if ($gameSystem.isPlayingCutscene()) return false;

    if (Managers.FarmObjects.isTilePassable($gameMap._mapId, blockX, blockY)) {
      return false;
    }

    this.setMovementBlockingReason({
      reason: '!tilePassable',
      x,
      y,
      blockX,
      blockY
    });

    if (Managers.FarmObjects.isTileProtected($gameMap._mapId, blockX, blockY)) {
      this.setMovementBlockingReason({
        reason: 'tileProtected',
        x,
        y,
        blockX,
        blockY
      });
      return true;
    }

    const left = x + this.hitboxXSize;
    const right = left + this.hitboxWidthSize - insignificantValue;
    const top = y + this.hitboxYSize;
    const bottom = top + this.hitboxHeightSize - insignificantValue;

    const character = this;

    const eventFilterFn = function(event) {
      if (!event) return false;
      if (event._erased) return false;
      if (event._hidden) return false;
      if (event.isThrough()) return false;
      if (!event.isNormalPriority()) return false;
      if (event._farmObjectData) {
        if (!event._farmObjectData.hasCollision()) return false;
      }

      if (!event.realPosIn(left, top, right, bottom)) return false;

      return true;
    };

    const allEvents = $gameMap.farmObjectEvents(eventFilterFn);
    const collided = allEvents.some((event) => {
      if (character.isCollidedWithThisEvent(event, left, top, right, bottom)) {
        this.setMovementBlockingReason({
          reason: 'object',
          x,
          y,
          blockX,
          blockY,
          event,
          left,
          top,
          right,
          bottom
        });
        return true;
      }

      return false;
    });

    if (collided) return collided;

    return false;
  };

  $.prototype.isCollidedWithVillagers = function(x, y) {
    return this.runForAllPositions(x, y, function(block_x, block_y) {
      //If the player is "inside" it, then this villager won't be considered,
      //because if it did, the player would be locked on it
      //this will only happen when the villager walks over the player.

      if (this.isTouchingTile(block_x, block_y)) {
        return false;
      }

      if (this.isCollidedWithFarmObjects(x, y, block_x, block_y)) {
        return true;
      }

      var villagers = $gameMap.villagersXy(block_x, block_y);
      return villagers.some(function(event){
        return !event.isMoving();
      });
    });
  };

  // Replaces the original moveStraight method, changing the calculation of the new position to consider the step_size
  $.prototype.moveStraight = function(d, maxStepSize) {
    this.beforeMove(d, maxStepSize);
    this.setDiagonalDirection(0, 0);
    this.setMovementSuccess(this.canPass(this._x, this._y, d));
    var stepSize = this.myStepSize(maxStepSize);

    if (this.isMovementSucceeded()) {
      this.beforeSuccessfulMoveStraight(d, maxStepSize);

      this.setDirection(d);
      this._x = this.roundFractionXWithDirection(this._x, d, stepSize);
      this._y = this.roundFractionYWithDirection(this._y, d, stepSize);

      if ((this._x > this._realX && this._lostX > 0) || (this._x < this._realX && this._lostX < 0)) {
        this._realX += this._lostX;
      }

      if ((this._y > this._realY && this._lostY > 0) || (this._y < this._realY && this._lostY < 0)) {
        this._realY += this._lostY;
      }

      this._lostX = 0;
      this._lostY = 0;
      this.afterMove();
    } else {
      this.setDirection(d);
    }

  };

  $.prototype.onMoveFurther = function() {
    return false;
  };

  // Replaces the original moveDiagonally method, changing the calculation of the new position to consider the step_size
  $.prototype.moveDiagonally = function(horz, vert) {
    this.setMovementSuccess(this.canPassDiagonally(this._x, this._y, horz, vert));
    var stepSize = this.myStepSize();

    if (this.isMovementSucceeded()) {
      this.beforeSuccessfulMoveDiagonally(horz, vert);

      this._x = this.roundFractionXWithDirection(this._x, horz, stepSize);
      this._y = this.roundFractionYWithDirection(this._y, vert, stepSize);

      this._lostX = 0;
      this._lostY = 0;
      this.afterMove();
    }

    if (this._direction === this.reverseDir(horz)) {
      this.setDirection(horz);
    }
    if (this._direction === this.reverseDir(vert)) {
      this.setDirection(vert);
    }
  };

  $.prototype.setPosition = function(x, y) {
    if (this.shouldUseTileMovement()) {
      x = Math.floor(x);
      y = Math.floor(y);
    }

    this._x = x;
    this._y = y;
    this._realX = x;
    this._realY = y;
    this._lostX = 0;
    this._lostY = 0;
  };

  $.prototype.screenY = function() {
    var baseY = Objects.Character.prototype.screenY.call(this);

    if (this.offsetY) {
      return baseY + this.offsetY;
    }

    return baseY;
  };

  $.prototype.screenX = function() {
    var baseX = Objects.Character.prototype.screenX.call(this);

    if (this.offsetX) {
      return Math.fix(baseX + this.offsetX);
    }

    return baseX;
  };

  $.prototype.getHeightLevel = function() {
    return Managers.Map.getHeightLevel(this.realLeft, this.realTop);
  };
}

addPixelMovement(Objects.ActorCharacter);