//-----------------------------------------------------------------------------
// StorageManager
//
// The static class that manages storage for saving game data.

Managers.Storage = class Storage {
  constructor() {
    throw new Error('This is a static class');
  }

  static backupLocalSaveFile(saveFilePath) {
    if (window.fs.existsSync(saveFilePath)) {
      const backupFilePath = `${ saveFilePath }.backup`;

      if (window.fs.existsSync(backupFilePath)) {
        window.fs.unlinkSync(backupFilePath);
      }

      const content = this.loadSaveFile(saveFilePath);
      this.saveToFile(backupFilePath, content);
    }
  }

  static storageKeyExists(saveFilePath) {
    return !!localStorage.getItem(saveFilePath);
  }

  static backupStorageSaveFile(saveFilePath) {
    // #Test
    const data = this.loadStorageFile(saveFilePath);
    if (!data) {
      return;
    }

    const key = `${saveFilePath}.backup`;
    localStorage.setItem(key, data);
  }

  static backupSaveFile(saveFilePath) {
    if (this.isLocalMode()) {
      this.backupLocalSaveFile(saveFilePath);
    } else {
      this.backupStorageSaveFile(saveFilePath);
    }
  }

  static restoreLocalSaveFile(saveFilePath) {
    const backupFilePath = `${ saveFilePath }.backup`;
    if (window.fs.existsSync(backupFilePath)) {
      const content = this.loadSaveFile(backupFilePath);
      this.saveToFile(saveFilePath, content);
    }
  }

  static restoreStorageSaveFile(saveFilePath) {
    // #Test
    const backupKey = `${saveFilePath}.backup`;
    const data = this.loadStorageFile(backupKey);
    if (!data) {
      return;
    }

    localStorage.setItem(saveFilePath, data);
  }

  static restoreSaveFile(saveFilePath) {
    if (this.isLocalMode()) {
      this.restoreLocalSaveFile();
    } else {
      this.restoreStorageSaveFile();
    }
  }

  static getSavesFromFolder(fullPath, saveList, fileNamesToSkip) {
    const allFiles = window.fs.readdirSync(fullPath);
    const existingFileNames = saveList.map((fullFilePath) => Utils.getFileNameFromPath(fullFilePath));

    allFiles.forEach((fileName) => {
      if (fileNamesToSkip && fileNamesToSkip.indexOf(fileName) >= 0) {
        return;
      }

      const filePath = Wrapper.joinFileNames(fullPath, fileName);
      if (Wrapper.isDirectory(filePath)) {
        return;
      }

      if (!fileName.endsWith('.ffsave')) {
        return;
      }

      // If a file with this name was already on the list, skip this one
      if (existingFileNames.indexOf(fileName) >= 0) {
        return;
      }

      existingFileNames.push(fileName);
      saveList.push(filePath);
    });
  }

  static getAllExistingLocalSaves(fileNamesToSkip, ignoreHomeFolder = false) {
    let homeFolder = false;
    let docsFolder = false;

    try {
      docsFolder = this.localFileDirectoryPath(true, true);
    }  
    catch(e) {
      docsFolder = false;
    }

    try {
      homeFolder = this.localFileDirectoryPath(false);
    }
    catch(e) {
      homeFolder = false;
    }
    
    const allSaves = [];

    if (docsFolder) {
      this.getSavesFromFolder(docsFolder, allSaves, fileNamesToSkip);
    }

    if (ignoreHomeFolder !== true && homeFolder !== docsFolder && homeFolder) {
      this.getSavesFromFolder(homeFolder, allSaves, fileNamesToSkip);
    }

    return allSaves;
  }

  static getAllExistingStorageSaves(fileNamesToSkip) {
    // #Test
    const allSaves = [];

    for (var key in localStorage) {
      if (key.endsWith('.ffsave')) {
        if (fileNamesToSkip && fileNamesToSkip.indexOf(key) >= 0) {
          continue;
        }

        allSaves.push(key);
      }
    }
    
    return allSaves;
  }

  static getAllExistingSaves(fileNamesToSkip) {
    if (this.isLocalMode()) {
      return this.getAllExistingLocalSaves(fileNamesToSkip);
    } else {
      return this.getAllExistingStorageSaves(fileNamesToSkip);
    }
  }

  static isLocalMode() {
    return Boolean(window.fs);
  }

  static getUserHome() {
    if (!Utils.isNwjs()) {
      return '';
    }

    if (process.platform == 'win32') {
      return process.env.USERPROFILE;
    } else {
      return process.env.HOME;
    }
  }

  static getUserDocs() {
    if (!Utils.isNwjs()) {
      return '';
    }

    if (process.platform == 'win32') {
      if (process.env.USER_MYDOCS) return process.env.USER_MYDOCS;

      const userProfile = process.env.USERPROFILE;
      const docs = Wrapper.joinFileNames(userProfile, 'Documents');
      if (Wrapper.existsSync(docs)) return docs;

      return userProfile;
    } else {
      const home = process.env.HOME;
      const unixDocs = Wrapper.joinFileNames(home, 'Documents');
      if (Wrapper.existsSync(unixDocs)) return unixDocs;

      return home;
    }
  }

  static loadFileFromLocalGameFolder(fileName) {
    const folderPath = this.localFileDirectoryPath(true);
    const fullPath = Wrapper.joinFileNames(folderPath, fileName);

    if (!Wrapper.existsSync(fullPath)) {
      const alternativePath = this.localFileDirectoryPath(false);
      if (alternativePath !== folderPath) {
        const fullAltPath = Wrapper.joinFileNames(alternativePath, fileName);
        if (Wrapper.existsSync(fullAltPath)) {
          return this.loadSaveFile(fullAltPath);
        }
      }
    }

    return this.loadSaveFile(fullPath);
  }

  static loadFileFromStorageGameFolder(fileName) {
    // #Test
    return this.loadStorageFile(fileName);
  }

  static loadFileFromGameFolder(fileName) {
    if (this.isLocalMode()) {
      return this.loadFileFromLocalGameFolder(fileName);
    } else {
      return this.loadFileFromStorageGameFolder(fileName);
    }
  }

  static saveFileToLocalGameFolder(fileName, content, compress, asyncCallback) {
    const folderPath = this.localFileDirectoryPath(true, true);
    const fullPath = Wrapper.joinFileNames(folderPath, fileName);

    try {
      if (!Wrapper.existsSync(folderPath)) {
        window.fs.mkdirSync(folderPath);
      }

      this.saveToFile(fullPath, content, compress, asyncCallback);
    }
    catch(error) {
      // const message = t('Something prevented the game from saving a file to your computer. Please make sure the game is able to write files to this directory.');
      console.error(fullPath);
      console.error(new Error('save-file-failed', fullPath));

      return {
        error,
        fullPath
      };
    }
  }

  static saveFileToStorageGameFolder(fileName, content, compress, asyncCallback) {
    // #Test
    try {
      this.saveToFile(fileName, content, compress, asyncCallback);
    }
    catch(e) {
      console.error(fileName);
      console.error(new Error('save-file-failed', fileName));

      return e;
    }
  }

  static saveFileToGameFolder(fileName, content, compress, asyncCallback) {
    if (this.isLocalMode()) {
      return this.saveFileToLocalGameFolder(fileName, content, compress, asyncCallback);
    } else {
      throw new Error("Can't save to local file if not in local mode.");
    }
  }

  static removeLocalSave(filePath) {
    if (window.fs.existsSync(filePath)) {
      window.fs.unlinkSync(filePath);
    }
  }

  static removeStorageSave(filePath) {
    // #Test
    localStorage.removeItem(filePath);
  }

  static removeSave(filePath) {
    if (this.isLocalMode()) {
      this.removeLocalSave(filePath);
    } else {
      this.removeStorageSave(filePath);
    }
  }

  static writeLocalFile(filePath, data, asyncCallback) {
    if (!asyncCallback) {
      window.fs.writeFileSync(filePath, data);
      return;
    }

    window.fs.writeFile(filePath, data, asyncCallback);
  }

  static writeStorageFile(filePath, data, asyncCallback) {
    // #Test
    localStorage.setItem(filePath, data);
    if (asyncCallback) {
      asyncCallback(data);
    }
  }

  static writeFile(filePath, data, asyncCallback) {
    if (this.isLocalMode()) {
      this.writeLocalFile(filePath, data, asyncCallback);
    } else {
      this.writeStorageFile(filePath, data, asyncCallback);
    }
  }

  static saveToFile(filePath, contents, compress, asyncCallback) {
    let data;

    const json = typeof contents === 'string' ? contents : JsonEx.stringify(contents, undefined, 2);

    if (compress) {
      data = LZString.compressToBase64(json);
    } else {
      data = json;
    }

    this.writeFile(filePath, data, asyncCallback);

    if (filePath.endsWith('.ffsave')) {
      Managers.Steam.saveFileOnCloud(filePath, json);
    }
  }

  static loadLocalFile(filePath, callback) {
    if (callback) {
      Wrapper.readFile(filePath, callback);
    } else {
      let data = null;
      if (Wrapper.existsSync(filePath)) {
        data = Wrapper.readFileSync(filePath);
      }

      return data;
    }
  }

  static loadStorageFile(filePath) {
    // #Test
    return localStorage.getItem(filePath);
  }

  static loadFile(filePath) {
    if (this.isLocalMode()) {
      return this.loadLocalFile(filePath);
    }

    return this.loadStorageFile(filePath);
  }

  static loadSaveFile(filePath) {
    const data = this.loadFile(filePath);

    try {
      if (data.startsWith('{')) return data;

      return LZString.decompressFromBase64(data);
    }
    catch(e) {
      return data;
    }
  }

  static loadSaveFileData(filePath) {
    const json = this.loadSaveFile(filePath);
    return JsonEx.parse(json);
  }

  static getCustomSavePath() {
    const project = Utils.getProjectFolder();
    const pathFile = path.join(project, 'path.txt');

    if (!Wrapper.existsSync(pathFile)) {
      return false;
    }

    const customPath = Wrapper.readFileSync(pathFile);

    if (Wrapper.existsSync(customPath)) {
      return customPath;
    }
  }

  static saveCustomSavePath(savePath) {
    const project = Utils.getProjectFolder();
    const pathFile = path.join(project, 'path.txt');

    this.writeLocalFile(pathFile, savePath);
  }

  static localFileDirectoryPath(useDocs, forceValidPath) {
    const customPath = this.getCustomSavePath();
    let dirPath;
    if (customPath) {
      dirPath = customPath;
    } else if (useDocs !== false) {
      dirPath = path.join(this.getUserDocs(), 'OrangeSeason');
    } else {
      dirPath = path.join(this.getUserHome(), 'OrangeSeason');
    }

    try {
      if (!Wrapper.existsSync(dirPath)) {
        window.fs.mkdirSync(dirPath);
      }
    }
    catch(e) {
      if (e.code != "EEXIST") {

        if (forceValidPath && !customPath) {
          try {
            dirPath = path.join(Utils.getProjectFolder(), 'saves');
            if (!Wrapper.existsSync(dirPath)) {
              window.fs.mkdirSync(dirPath);
            }

            this.saveCustomSavePath(dirPath);
            return dirPath;
          } catch(e2) {
            console.error(e2);
            throw new Error("Unable to create custom save file path");
          }
        }

        console.error(e);
        throw e;
      }
    }

    return dirPath;
  }

  static directoryPath() {
    if (this.isLocalMode()) {
      return this.localFileDirectoryPath(true);
    }

    return '';
  }

  static loadDataFile(fullFilePath, callback) {
    this.loadLocalFile(fullFilePath, (data) => {
      let jsonData;
      try {
        jsonData = JsonEx.parse(data);
      }
      catch(e) {
        console.error(`Failed to parse file ${ fullFilePath }`);
        throw e;
      }
      if (callback) {
        callback(jsonData);
      }
    });
  }

  static loadYamlFile(fullFilePath, callback) {
    this.loadLocalFile(fullFilePath, (data) => {
      const jsonData = YAML.parse(data);
      callback(jsonData);
    });
  }

  static getNewLocalSaveFilePathForPlayer(playerName) {
    const folderPath = this.localFileDirectoryPath(true);
    if (Wrapper.existsSync(folderPath)) {
      let index = 0;
      while (true) {
        const fileName = index > 0 ? `${playerName}${ index.padZero(3) }` : playerName;

        const fullPath = Wrapper.joinFileNames(folderPath, `${ fileName }.ffsave`);
        if (Wrapper.existsSync(fullPath)) {
          index++;
          if (index >= 1000) {
            return Wrapper.joinFileNames(folderPath, 'fullStorage.ffsave');
          }
          continue;
        }

        return fullPath;
      }
    } else {
      window.fs.mkdirSync(folderPath);
      return Wrapper.joinFileNames(folderPath, `${ playerName }.ffsave`);
    }
  }

  static getNewStorageSaveFilePathForPlayer(playerName) {
    // #Test
    let index = 0;
    while (true) {
      const fileName = index > 0 ? `${ playerName }${ index.padZero(3) }` : playerName;
      const fullPath = `${ fileName }.ffsave`;
      if (this.storageKeyExists(fullPath)) {
        index++;
        if (index >= 1000) {
          return 'fullStorage.ffsave';
        }
        continue;
      }

      return fullPath;
    }

  }

  static getNewSaveFilePathForPlayer(playerName) {
    if (this.isLocalMode()) {
      return this.getNewLocalSaveFilePathForPlayer(playerName);
    } else {
      return this.getNewStorageSaveFilePathForPlayer(playerName);
    }
  }

  static getPathForNewSave() {
    const playerName = (() => {
      try {
        const baseName = ($gameActors.actor(1).name()).toLowerCase();
        if (baseName && Utils.isValidFileName(baseName)) {
          return baseName;
        }
      }
      catch(e) {
        console.error('Failed to validate player name');
        console.error(e);
      }

      return 'orange';
    })();

    return this.getNewSaveFilePathForPlayer(playerName);
  }
};
