const { Utils } = require('engine/core/Utils');
const { Input } = require('engine/core/Input');
const { Data } = require('engine/managers/Data');
const { Audio } = require('engine/managers/Audio');

//-----------------------------------------------------------------------------
// SceneManager
//
// The static class that manages scene transitions.

Managers.Scenes = class Scenes {
  constructor() {
    throw new Error('This is a static class');
  }

  /*
   * Gets the current time in ms without on iOS Safari.
   * @private
   */
  static run(sceneClass) {
    try {
      this.initialize();
      this.goToScene(sceneClass);
      this.requestUpdate();
    } catch (e) {
      this.catchException(e);
    }
  }

  static initialize() {
    this.initGraphics();
    this.checkFileAccess();
    this.initAudio();
    this.initInput();
    this.initNwjs();
    this.setupErrorHandlers();
  }

  static initGraphics() {
    const type = this.preferableRendererType();
    Graphics.initialize(this._screenWidth, this._screenHeight, type);
    Graphics.boxWidth = this._boxWidth;
    Graphics.boxHeight = this._boxHeight;
    // Graphics.setLoadingImage('img/system/Loading.png');
    if (Utils.isOptionValid('showfps')) {
      Graphics.showFps();
    }
    if (type === 'webgl') {
      this.checkWebGL();
    }
  }

  static preferableRendererType() {
    if (!!Managers.Config && Managers.Config.renderer) {
      return Managers.Config.renderer;
    }

    if (Utils.isOptionValid('canvas')) {
      return 'canvas';
    } else if (Utils.isOptionValid('webgl')) {
      return 'webgl';
    } else if (this.shouldUseCanvasRenderer()) {
      return 'canvas';
    } else {
      return 'auto';
    }
  }

  static getTimeInMs() {
    return performance.now();
  }

  static updateGameSpeed(newGameSpeed) {
    this._gameSpeed = newGameSpeed;
    this._fps = Utils.getFrameCount(60);
    this._idealFrameLength = 1.0 / this._fps;
  }

  static shouldUseCanvasRenderer() {
    return Utils.isMobileDevice();
  }

  static checkWebGL() {
    if (!Graphics.hasWebGL()) {
      throw new Error('Your browser does not support WebGL.');
    }
  }

  static checkFileAccess() {
    if (!Utils.canReadGameFiles()) {
      throw new Error('Your browser does not allow to read local files.');
    }
  }

  static initAudio() {
    const noAudio = Utils.isOptionValid('noaudio');
    if (!WebAudio.initialize(noAudio) && !noAudio) {
      throw new Error('Your browser does not support Web Audio API.');
    }
  }

  static initInput() {
    Input.initialize();
    TouchInput.initialize();
  }

  static initNwjs() {
    if (Utils.isNwjs()) {
      const gui = require('nw.gui');
      const win = gui.Window.get();
      if (process.platform === 'darwin' && !win.menu) {
        const menubar = new gui.Menu({ type: 'menubar' });
        const option = { hideEdit: true, hideWindow: true };
        menubar.createMacBuiltin('Game', option);
        win.menu = menubar;
      }
    }
  }

  static checkModErrors() {
    Managers.Mods.checkErrors();
  }

  static setupErrorHandlers() {
    window.addEventListener('error', this.onError.bind(this));
    document.addEventListener('keydown', this.onKeyDown.bind(this));
  }

  static requestUpdate() {
    if (this._updateRequested) return;
    this._updateRequested = true;

    if (!this._stopped) {
      this._lastRequest = this._currentTime;
      if (Managers.Config.useScreenRefreshRate !== false) {
        requestAnimationFrame(this.update.bind(this));
      } else {
        const now = this.getTimeInMs();
        const timeSpent = now - this._lastRequest;
        const calibratedFps = Math.max(30, this._fps + Managers.Config.fpsCalibration);
        const timeout = Math.max(1, Math.floor((1000 / calibratedFps) - timeSpent));
        setTimeout(this.update.bind(this), timeout);
      }
    }
  }

  static update() {
    this._updateRequested = false;

    try {
      if (Managers.Config.useScreenRefreshRate !== false) {
        this.requestUpdate();
      }
      this.determineGameSpeedSkip();

      if (this.skipNextFrame) {
        if (Managers.Config.useScreenRefreshRate === false) {
          this.requestUpdate();
        }
        return;
      }

      this._then = this._now;
      this.tickStart();

      if (Utils.isMobileSafari()) {
        this.updateInputData();
      }

      if ($gameTemp && $gameTemp.isProfileRequested()) {
        $gameTemp.clearProfileRequest();
        console.profile();
        setTimeout(() => {
          console.profileEnd();
        }, 1000);
      }

      const ticks = this._ticks ? this._ticks : 0;

      this._ticks = 0;

      this.updateManagers(ticks);
      this.updateMain();

      if (!this._updateRequested) {
        this.requestUpdate();
      }

      if (Graphics.frameDuration > 100 && window.$gameTemp && window.$gameTemp.isDsv()) {
        console.log('[DEBUG] Frame Duration: ', Graphics.frameDuration);
      }

      this.tickEnd();
    } catch (e) {
      this.catchException(e);
    }
  }

  static terminate() {
    window.close();
  }

  static onError(e) {
    try {
      console.error(e.message);
      console.error(e.filename, e.lineno);
      console.log(...log(e.error.stack));
    }
    catch(e2) {
      // even console failed :O
    }

    this.catchException(e);
  }

  static determineGameSpeedSkip() {
    if (Managers.Config.useScreenRefreshRate === false) return;

    this.skipNextFrame = false;
    const fpsInterval = 1000 / Managers.Config.gameFps;
    let averageFrameLength = this._averageFrameLength || fpsInterval;
    const calibratedFps = Math.max(30, Managers.Config.gameFps + Managers.Config.fpsCalibration);
    const acceptableInterval = 1000 / (calibratedFps * 1.2);

    if (!this._then) {
      this._then = Date.now();
    }

    this._now = Date.now();
    const elapsed = (this._now - this._then);
    averageFrameLength = (averageFrameLength + elapsed) / 2;

    if (averageFrameLength < acceptableInterval) {
      this.skipNextFrame = true;
    }

    this._averageFrameLength = averageFrameLength;
  }

  static onKeyDown(event) {
    if (!event.ctrlKey && !event.altKey) {
      switch (event.keyCode) {
        case 119: // F8
          // if (Utils.isNwjs() && !!window.$gameTemp && $gameTemp.isPlaytest()) {
          if (!$dataSystem) {
            if (Utils._isTestMode) {
              require('nw.gui').Window.get().showDevTools();
            }

            return;
          }

          if (!$dataSystem.isDemoMode) {
            require('nw.gui').Window.get().showDevTools();
          }
          // }
          break;
        case 123: //F12
          event.preventDefault();
          break;
        default :
          break;
      }
    }
  }

  static catchException(e) {
    try {
      this.stop();

      console.error(e);
      console.log(...window.log(e.stack));

      if (e.message == 'save-file-failed') {
        if (e.filename) {
          Graphics.printError(t('Failed to Save File'), t('Something prevented the game from saving a file to your computer.<br/>Please make sure the game is able to write files to this directory.') + "<br/>" + (e.filename || ''));
        } else {
          Graphics.printError(t('Failed to Save File'), t('Something prevented the game from saving a file to your computer.<br/>Please make sure the game is able to write files to your save path.'));
        }
      } else {
        Graphics.printError('Fatal Error', t('Please Report this to the Developer') + "<br/>" + e.message + "<br/>" + (e.filename || '') + "<br/>" + (e.lineno || ''));
      }

      Audio.stopAll();
    }
    catch(e2) {
      // welp
    }
  }

  static showExplicitException(title, message) {
    Graphics.printError(title, message);
    Audio.stopAll();
    this.stop();
  }

  static tickStart() {
    if (this.skipNextFrame) return;
    Graphics.tickStart();
  }

  static tickEnd() {
    if (this.skipNextFrame) return;
    Graphics.tickEnd();
  }

  static updateInputData() {
    Input.update();
    TouchInput.update();

    if ($gameTemp) {
      $gameTemp.skipSelectableWindowHandling(false);
    }
  }

  static checkMainInput() {
    if (!this._scene || !this._scene.isReady()) return;

    if (Input.isTriggered('quicksave')) {
      if (Input.isPressed('altKey')) {
        Data.autoLoad('quick');
      } else {
        Data.quickSave();
      }
    }
  }

  static updateMain() {
    let maxAutoFrameSkip = Number(Managers.Config.maxAutoFrameSkip);
    if (isNaN(maxAutoFrameSkip)) maxAutoFrameSkip = 0;

    maxAutoFrameSkip = Math.ceil(maxAutoFrameSkip / (Managers.Config.gameFps / 60));

    if (Utils.isMobileSafari()) {
      this.changeScene();
      this.updateScene();
    } else {
      const newTime = this.getTimeInMs();
      var actualTime = (newTime - this._currentTime) / 1000;

      this._currentTime = newTime;
      this._forceNewFrame = false;

      this._ticks++;
      this.updateInputData();

      if (this.isSceneChanging() && !this.isCurrentSceneBusy()) {
        this.changeScene();
      } else {
        this.updateScene();
        if (this._sceneStarted) {
          this.checkMainInput();
        }
      }
    }

    if (this._forceNewFrame || actualTime < this._idealFrameLength || this._skipCount >= maxAutoFrameSkip) {
      this._skipCount = 0;
      this.renderScene();
    } else {
      this._skipCount++;
    }

    if (Managers.Config.useScreenRefreshRate === false) {
      this.requestUpdate();
    }
  }

  static updateManagers(ticks) {
    Managers.Images.update();

    Managers.Images.cache.update(ticks);
  }

  static changeScene() {
    if (this.isSceneChanging() && !this.isCurrentSceneBusy()) {
      if (this._scene) {
        this._scene.terminate();
        this._previousClass = this._scene.constructor;
      }
      this._scene = this._nextScene;
      if (this._scene) {
        document.body.className = '';

        this._scene.create();
        this._nextScene = null;
        this._sceneStarted = false;
        this.onSceneCreate();
      }
      if (this._exiting) {
        this.terminate();
      }
    }
  }

  static updateScene() {
    if (this._scene) {
      if (!this._sceneStarted && this._scene.isReady()) {
        this._scene.start();
        this._sceneStarted = true;
        this.onSceneStart();
      } else {
        if (this._sceneStarted && !this._scene.isFullyLoaded()) {
          // this._scene.continueLoading();
        }
      }

      if (this.isCurrentSceneStarted()) {
        this._scene.update();
      }
    }
  }

  static renderScene() {
    if (this.isCurrentSceneStarted()) {
      Graphics.render(this._scene);
    } else if (this._scene) {
      this.onSceneLoading();
    }
  }

  static onSceneCreate() {
    Graphics.startLoading();
  }

  static onSceneStart() {
    Graphics.endLoading();
  }

  static onSceneLoading() {
    Graphics.updateLoading();
  }

  static isSceneChanging() {
    return this._exiting || !!this._nextScene;
  }

  static isCurrentSceneBusy() {
    return this._scene && this._scene.isBusy();
  }

  static isScene(sceneClass) {
    return this._scene && this._scene instanceof sceneClass;
  }

  static isCurrentSceneStarted() {
    return this._scene && this._sceneStarted && this._scene.isFullyLoaded();
  }

  static isNextScene(sceneClass) {
    return this._nextScene && this._nextScene.constructor === sceneClass;
  }

  static isPreviousScene(sceneClass) {
    return this._previousClass === sceneClass;
  }

  static goToScene(sceneClass) {
    if (sceneClass) {
      this._nextScene = new sceneClass();
    }
    if (this._scene) {
      this._scene.stop();
    }
  }

  static push(sceneClass) {
    this._stack.push(this._scene.constructor);
    this.goToScene(sceneClass);
  }

  static pop() {
    if (this._stack.length > 0) {
      this.goToScene(this._stack.pop());
    } else {
      this.exit();
    }
  }

  static popTo(scene) {
    this.clearStack();
    this.goToScene(scene);
  }

  static exit() {
    this.goToScene(null);
    this._exiting = true;
  }

  static clearStack() {
    this._stack = [];
  }

  static stop() {
    this._stopped = true;
  }

  static prepareNextScene() {
    this._nextScene.prepare(...arguments);
  }

  static snap() {
    return Bitmap.snap(this._scene);
  }

  static snapForBackground() {
    try {
      this._backgroundBitmap = this.snap();
    }
    catch(e) {
      console.error(e);
      this._backgroundBitmap = null;
    }
  }

  static clearBackgroundBitmap() {
    this._backgroundBitmap = null;
  }

  static backgroundBitmap() {
    return this._backgroundBitmap;
  }

  static resume() {
    this._stopped = false;
    this.requestUpdate();

    this._currentTime = this.getTimeInMs();
    this._accumulator = 0;
  }

  static shouldPreloadSprites() {
    if (this._scene instanceof GameScenes.Boot) return false;
    if (this._scene instanceof GameScenes.Title) return false;
    if (this._scene instanceof GameScenes.NewGame) return false;

    return true;
  }
};

Managers.Scenes._scene = null;
Managers.Scenes._nextScene = null;
Managers.Scenes._stack = [];
Managers.Scenes._stopped = false;
Managers.Scenes._sceneStarted = false;
Managers.Scenes._exiting = false;
Managers.Scenes._previousClass = null;
Managers.Scenes._backgroundBitmap = null;
Managers.Scenes._screenWidth = 1280;
Managers.Scenes._screenHeight = 720;
Managers.Scenes._boxWidth = 1280;
Managers.Scenes._boxHeight = 720;
Managers.Scenes._forceNewFrame = false;
Managers.Scenes._skipCount = 0;

Managers.Scenes._t = 0.0;
Managers.Scenes._gameSpeed = 2;
Managers.Scenes._fps = Utils.getFrameCount(60);
Managers.Scenes._idealFrameLength = 1.0 / Managers.Scenes._fps;
Managers.Scenes._currentTime = Managers.Scenes.getTimeInMs();
Managers.Scenes._accumulator = 0.0;
