const { ItemContainer } = require('engine/core/ItemContainer');
const { Items } = require('engine/managers/Items');

class Inventory extends Items {
  static get container() {
    return this._container;
  }
  static get storageContainer() {
    return this._storageContainer;
  }
  static get trashContainer() {
    return this._trashContainer;
  }
  static get mailboxContainer() {
    return this._mailboxContainer;
  }
  static get waiting() {
    return this._waiting;
  }
  static get gold() {
    return this._gold;
  }
  static set gold(value) {
    this._gold = value;
  }
  static get itemBarIndex() {
    return this._itemBarIndex;
  }
  static set itemBarIndex(value) {
    this._itemBarIndex = value;
  }
  static get toolBarIndex() {
    return this._toolBarIndex;
  }
  static set toolBarIndex(value) {
    this._toolBarIndex = value;
  }

  static get itemIndex() {
    return this._itemIndex;
  }
  static set itemIndex(value) {
    this._itemIndex = value;
  }

  static get specialItem() {
    return this._specialItem;
  }
  static set specialItem(value) {
    this._specialItem = value;
    this._itemInfo = {};
  }

  static get displayItemId() {
    return this._displayItemId;
  }
  static set displayItemId(value) {
    this._displayItemId = value;
  }

  static get selectedItem() {
    if (this._specialItem) {
      return this._specialItem;
    }

    if (this._itemIndex >= 0) {
      const trueIndex = (this.itemBarIndex * 10) + this._itemIndex;
      return this._container._items[trueIndex] || undefined;
    }

    return undefined;
  }

  static get itemToDisplay() {
    if (this._displayItemId) {
      if (this._displayItemId == 'none') return undefined;
      return this.getItemData(this._displayItemId);
    }

    return this.selectedItem;
  }

  static getCustomContainer(containerName) {
    if (!this._customContainers[containerName]) {
      this._customContainers[containerName] = new ItemContainer();
    }

    return this._customContainers[containerName];
  }

  static getAllTrashContainers() {
    const list = [
      this.trashContainer,
    ];

    for (const containerName in this._customContainers) {
      if (!containerName) {
        continue;
      }

      if (containerName.toLowerCase().startsWith('trash_')) {
        list.push(this._customContainers[containerName]);
      }
    }

    return list;
  }

  static getTrashContainer(containerName) {
    if (!containerName || containerName.toLowerCase() === 'default') {
      return this.trashContainer;
    }

    return this.getCustomContainer(`trash_${ containerName.toLowerCase() }`);
  }

  static getCustomContainersData() {
    const data = {};

    for (const containerName in this._customContainers) {
      if (containerName && this._customContainers[containerName]) {
        data[containerName] = this._customContainers[containerName].getData();
      }
    }

    return data;
  }

  static setCustomContainersData(data) {
    for (const containerName in this._customContainers) {
      if (containerName && this._customContainers[containerName]) {
        this._customContainers[containerName].clear();
      }
    }

    for (const containerName in data) {
      if (!containerName) {
        continue;
      }

      if (!this._customContainers[containerName]) {
        this._customContainers[containerName] = new ItemContainer();
      }

      this._customContainers[containerName].setData(data[containerName]);
    }
  }

  static clear() {
    if (!this._container) {
      this._container = new ItemContainer();
    }
    if (!this._storageContainer) {
      this._storageContainer = new ItemContainer();
    }
    if (!this._trashContainer) {
      this._trashContainer = new ItemContainer();
    }
    if (!this._mailboxContainer) {
      this._mailboxContainer = new ItemContainer();
    }
    if (!this._customContainers) {
      this._customContainers = {};
    }

    this._container.clear();
    this._storageContainer.clear();
    this._trashContainer.clear();
    this._mailboxContainer.clear();

    for (const containerName in this._customContainers) {
      if (containerName && this._customContainers[containerName]) {
        this._customContainers[containerName].clear();
      }
    }

    this._lastItem = undefined;
    this._specialItem = undefined;
    this._displayItemId = undefined;
    this._itemInfo = {};
    this._waiting = false;
    this._gold = 500;
    this._itemBarIndex = 0;
    this._toolBarIndex = 0;
    this._itemIndex = -1;

    this._container.setMaxItemTypes(this.maxItemTypes());

    this.getTrashContainer('farm');
    this.getTrashContainer('bridge');
    this.getTrashContainer('mountain');
    this.getTrashContainer('benji');
    this.getTrashContainer('lake');
    this.getTrashContainer('cemetery');
    this.getTrashContainer('festival-left');
    this.getTrashContainer('festival-right');
  }

  static items() {
    return this.container.items.clone();
  }

  static allItems() {
    return this.items().concat(this.tools());
  }

  static numItems(item) {
    if (item.type == 'seed-box') {
      return Inventory.numItemsById(item.id.replace('seed-box', 'seeds'));
    }

    const contentNum = Managers.Content.numItems(item);
    if (contentNum !== undefined) return contentNum;

    const containerAmount = this.container.numItems(item);
    const holding = 0;

    return containerAmount + holding;
  }

  static numItemsById(itemId) {
    const itemData = this.getItemData(itemId);

    return this.numItems(itemData);
  }

  static getItemIdIngredients(itemId) {
    const itemData = this.getItemData(itemId);

    return this.getItemIngredients(itemData);
  }

  static getItemIngredients(item) {
    if (!item) return false;

    return item.ingredients;
  }

  static maxItemTypes() {
    if (!window.$gameSwitches) return 999;

    if (Switches.gotSecondBag) {
      return 30;
    } else if (Switches.gotFirstBag) {
      return 20;
    } else {
      return 10;
    }
  }

  static maxItemBarIndex() {
    if (!window.$gameSwitches) return 0;

    if (Switches.gotSecondBag) {
      return 2;
    } else if (Switches.gotFirstBag) {
      return 1;
    } else {
      return 0;
    }
  }

  static maxToolBarIndex() {
    return 1;
  }

  static hasMaxItems(item) {
    return this.numItems(item) >= this.maxItems(item);
  }

  static hasItem(item) {
    return this.container.hasItem(item);
  }

  static hasItemById(itemId) {
    return this.numItemsById(itemId) > 0;
  }

  static spaceForItem(item) {
    return this.container.spaceForItem(item);
  }

  static canGainItem(item) {
    if (Managers.Content.canGainItem(item) === true) {
      return true;
    }

    if (this._specialItem !== undefined) {
      if (!this.isHoldingItem(item.id) || !this.itemTypeIsStackable(item.type)) {
        return false;
      }
    }

    return this.container.canGainItem(item);
  }

  static hasFreeSlots() {
    return this.container.hasFreeSlots();
  }

  static canGainItemId(itemId) {
    const itemData = this.getItemData(itemId);
    return this.canGainItem(itemData);
  }

  static tryPickItemId(itemId, amount) {
    if (!this.canGainItemId(itemId)) return false;

    return this.pickItemId(itemId, amount);
  }

  static canBuyItem(itemId, price) {
    const itemData = this.getItemData(itemId);
    const max = this.maxItems(itemData) - this.numItems(itemData);
    if (price === undefined) {
      price = this.getDiscountedBuyPrice(itemData);
    }

    let maxPerPrice = max;
    if (price > 0) {
      maxPerPrice = Math.min(max, Math.floor(this.gold / price));
    }

    if (maxPerPrice <= 0) return false;

    if (!this.checkIfRequiredExperienceIsMet(itemData)) {
      return false;
    }

    const maxPerIngredients = this.getMaxCraftsAvailableForItem(itemData);
    if (maxPerIngredients <= 0) return false;

    const maxBuy = Math.min(maxPerPrice, maxPerIngredients);

    if (maxBuy > 0) {
      return maxBuy;
    }

    return false;
  }

  static buyItem(itemId, amount, price) {
    const itemData = this.getItemData(itemId);
    if (!itemData) return false;
    if (!amount) amount = 1;

    if (price === undefined) {
      price = this.getDiscountedBuyPrice(itemData);
    }

    const canBuy = this.canBuyItem(itemId, price);
    if (canBuy === false) return false;
    if (canBuy < amount) return false;

    const ingredients = itemData.ingredients;
    if (ingredients && ingredients.length) {
      for (let i = 0; i < ingredients.length; i++) {
        this.loseItemId(ingredients[i], amount, true);
      }
    }

    this.loseGold(amount * price);
    this.pickBoughtItem(itemData, amount);
    return true;
  }

  static pickBoughtItem(item, amount) {
    if (Managers.Content.pickBoughtItem(item, amount)) {
      return;
    }

    this.pickItem(item, amount, true);
  }

  static pickItem(item, amount, mailboxIfFull) {
    if (!item) return false;

    if ($gamePlayer.isRiding()) {
      return false;
    }

    if (this._waiting) {
      return false;
    }

    amount = amount || item.amount || 1;
    if (Managers.Content.pickItem(item, amount, mailboxIfFull) === true) {
      this.onInventoryChange();
      this.requestHudRefresh();
      return true;
    }

    let playAnimation = true;
    let ranGetEvent = false;

    if (this._specialItem !== undefined) {
      if (this.isHoldingItem(item.id) && this.itemTypeIsStackable(item.type)) {
        this._specialItem.amount += amount;
        playAnimation = false;
      } else {
        $gamePlayer.requestBalloon(Balloons.FAILURE);
        Managers.Sound.playBuzzer();
        return false;
      }
    } else {
      if (!this.itemTypeCanBeStored(item.type)) {
        this._specialItem = MVC.shallowClone(item);
        return true;
      }

      if (!this.hasFreeSlots() && this.numItems(item) === 0) {
        if (mailboxIfFull === true) {
          Switches.bagWasFullWhileBuying = true;
          Inventory.mailboxContainer.gainItem(item, amount);
          return true;
        }

        $gamePlayer.requestBalloon(Balloons.FULL_BACKPACK);
        Managers.Sound.playBuzzer();
        return false;
      }

      if (this._displayItemId !== undefined && this.isHoldingItem()) {
        playAnimation = false;
      }

      item.amount = amount;
      this.gainItem(item);
      ranGetEvent = true;
    }

    if (playAnimation) {
      this._waiting = true;
      this._displayItemId = item.id;
      this.requestHudRefresh();

      setTimeout(() => {
        this._displayItemId = undefined;
        this._waiting = false;
      }, 300);
    }

    this.requestHudRefresh();
    Managers.Sound.playPickItem();

    this.runEvent(`pick-${item.id}`, item.amount || 1);
    if (!ranGetEvent) {
      this.runEvent(`get-${item.id}`, item.amount || 1);
    }
    this.onInventoryChange();
    return true;
  }

  static isHoldingItem(itemId) {
    if (itemId === undefined) {
      return this.selectedItem !== undefined;
    }

    return this.selectedItem !== undefined && this.selectedItem.id === itemId;
  }

  static isHoldingSpecialItem() {
    return (this.specialItem !== undefined);
  }

  static isHoldingCreature(creatureId) {
    if (!this._specialItem) return false;
    const itemId = this._specialItem.id;

    if (!!itemId && !!Managers.Creatures.creatureList[itemId]) {
      const animalData = this.getItemInfo('animalData');
      if (animalData) {
        if (animalData.id == creatureId) return true;
      }
    }

    return false;
  }

  static isHoldingAnimal(race) {
    if (!this.isHoldingSpecialItem()) return false;
    if (this.specialItem.type !== 'animal') return false;

    const animalData = this.getItemInfo('animalData');
    if (!animalData) return false;

    if (race !== undefined) {
      if (!Managers.Creatures.isSameRace(race, animalData.type)) return false;
    }

    return true;
  }

  static changeItemIndex(reverse) {
    if (this._specialItem) return;

    if (reverse) {
      this._itemIndex--;
    } else {
      this._itemIndex++;
    }

    if (this._itemIndex >= 10) {
      this._itemIndex = -1;
    } else if (this._itemIndex < -1) {
      this._itemIndex = 9;
    }

    if (this.isHoldingItem()) {
      Managers.Sound.playPickItem();
    } else {
      Managers.Sound.playCursor();
    }
    return true;
  }

  static switchBackpack() {
    const maxIndex = this.maxItemBarIndex();
    if (this._itemBarIndex >= maxIndex) {
      this._itemBarIndex = 0;
    } else {
      this._itemBarIndex++;
    }

    Managers.Sound.playCursor();
    this.requestHudRefresh();
  }

  static changeTool(reverse) {
    if (this._specialItem) return;

    const tools = this.tools();
    const len = tools.length;
    let index = -1;

    if (Managers.Tools.toolId !== undefined) {
      for (let i = 0; i < len; i++) {
        if (tools[i].id == Managers.Tools.toolId) {
          index = i;
          break;
        }
      }
    }

    if (reverse) {
      index++;
    } else {
      index--;
    }

    if (index < -1) {
      index = tools.length - 1;
    }
    if (index >= tools.length) {
      index = -1;
    }

    if (index < 0) {
      Managers.Tools.setTool(undefined);
      Managers.Sound.playEquip();
    } else {
      Managers.Tools.setTool(tools[index].id);
      this._itemIndex = -1;
    }

    this.requestHudRefresh();
  }

  static requestHudRefresh() {
    if (this.waiting) return;

    if (Managers.Scenes._scene instanceof GameScenes.GameMap) {
      if (Managers.Scenes._scene._toolHudWindow) {
        Managers.Scenes._scene._toolHudWindow.requestUpdate();
      }
    }

    this._container.setMaxItemTypes(this.maxItemTypes());
  }

  static canUseItem(item) {
    if (!item) return false;
    if (item.type == 'animal') return false;

    if (item.type == 'seed') {
      return this.canUseSeed(item);
    }
    return true;
  }

  static canUseSeed(item) {
    if (!item) return false;
    if (item.type !== 'seed') return false;

    const tool = Managers.Tools.getTool(item.id);
    if (!tool) return false;

    return true;
  }

  static useToolItem(item, targetX = null, targetY = null) {
    if (!item) return false;
    if (!this.itemTypeIsStackable(item.type)) return false;

    const tool = Managers.Tools.getTool(item.id);
    if (tool !== undefined) {
      const tile = !isNaN(Number(targetX)) && !isNaN(Number(targetY)) ? new TileHelper.Tile(targetX, targetY) : null;
      return tool.use(0, tile);
    }

    return false;
  }

  static getSelectedItemId() {
    const selectedItem = this.selectedItem;
    if (selectedItem) {
      return selectedItem.id;
    }

    return 'none';
  }

  static itemCanBeUsedWithTheMouse(itemId) {
    const itemData = this.findItem(itemId);
    if (!itemData) return false;

    if (itemData.type == 'food') return false;
    if (itemData.type == 'medicine') return false;

    return true;
  }

  static selectedItemCanBeUsedWithTheMouse() {
    if (!this.selectedItem) return false;

    return this.itemCanBeUsedWithTheMouse(this.selectedItem.id);
  }

  static useSelectedItem(targetX = null, targetY = null) {
    if (!this.selectedItem) return false;
    if (!this.canUseItem(this.selectedItem)) {
      $gamePlayer.requestBalloon(Balloons.FAILURE);
      Managers.Sound.playBuzzer();
      return false;
    }

    if (this.selectedItem.type == 'seed' || this.selectedItem.type == 'object' || this.selectedItem.type == 'tool-item') {
      this.useToolItem(this.selectedItem, targetX, targetY);
    } else {
      this.consumeItem(this.selectedItem);
    }

    this.requestHudRefresh();
    return true;
  }

  static takeGift() {
    if (!this.isHoldingItem()) return false;

    const villagerName = Managers.Relationship.currentGiftReceiver;

    Managers.Relationship.registerGift(villagerName, this.selectedItem.id);
    this.loseUnitOfSelectedItem(1);
  }

  static rejectGift() {
    const villagerName = Managers.Relationship.currentGiftReceiver;
    Managers.Relationship.decreaseFriendship(villagerName, 300);
  }

  static loseUnitOfSpecialItem(amount, instant) {
    if (!this.specialItem) return;
    if (amount === undefined) amount = 1;
    if (this.specialItem.amount === undefined) this.specialItem.amount = 1;

    if (this.specialItem.amount > amount) {
      this.specialItem.amount -= amount;
    } else {
      this._specialItem = undefined;
    }

    if (instant) {
      this.requestHudRefresh();
      this.onInventoryChange();
    } else {
      this._waiting = true;
      this.displayItemId = 'none';

      setTimeout(() => {
        this.displayItemId = undefined;
        this._waiting = false;
        this.requestHudRefresh();
        this.onInventoryChange();
      }, 200);
    }

    return amount;
  }

  static loseUnitOfSelectedItem(amount, instant) {
    if (!this.selectedItem) return;
    if (amount === undefined) amount = 1;

    if (this.specialItem) {
      return this.loseUnitOfSpecialItem(amount, instant);
    }

    return this.loseItem(this.selectedItem, amount);
  }

  static forcePickItemId(itemId, amount) {
    if (this.tryPickItemId(itemId, amount)) return true;

    return this.storageContainer.gainItemId(itemId, amount);
  }

  static forceGainItemId(itemId, amount) {
    if (this.canGainItemId(itemId, amount)) {
      this.gainItemId(itemId, amount);
      return;
    }

    return this.storageContainer.gainItemId(itemId, amount);
  }

  static forceGainItem(item, amount) {
    if (this.canGainItem(item, amount)) {
      this.gainItem(item, amount);
      return;
    }

    return this.storageContainer.gainItem(item, amount);
  }

  static pickItemId(itemId, amount, mailboxIfFull) {
    if (Managers.Content.pickItemId(itemId, amount, mailboxIfFull) === true) return true;

    const itemData = this.getItemData(itemId);
    if (!itemData) {
      console.log(...log('invalid-item-id', itemId));
      return;
    }

    return this.pickItem(itemData, amount || 1, mailboxIfFull);
  }

  static gainItem(item, amount, reverse, isNew) {
    if (item === undefined) return false;

    amount = this.selectAmount(item, amount);
    if (Managers.Content.gainItem(item, amount, reverse, isNew) === true) {
      this.onInventoryChange();
      this.requestHudRefresh();
      return true;
    }

    if (isNew === undefined) isNew = true;

    this.container.gainItem(item, amount, reverse);

    if (isNew === true) {
      this.runEvent(`get-${item.id}`, amount);
    }

    if (isNew === true) this.runEvent(`gain-${item.id}`, amount);
    this.runEvent(`store-${item.id}`, [amount]);

    $gameMap.requestRefresh();
    this.requestHudRefresh();
    return true;
  }

  static itemIdExists(itemId) {
    for (let i = 0; i < $dataItems.length; i++) {
      if ($dataItems[i].id === itemId) {
        return true;
      }
    }

    return false;
  }

  static gainItemId(itemId, amount, isNew) {
    this.gainItem(this.getItemData(itemId), amount, undefined, isNew);
  }

  static loseItemId(itemId, amount, instant) {
    this.loseItem(this.getItemData(itemId), amount, instant);
  }

  static deliverItemId(itemId, amount) {
    const price = this.getBuyPrice(itemId);
    this.sellItemId(itemId, amount, price);
  }

  static sellItemId(itemId, amount, price) {
    if (amount === undefined) amount = 1;
    if (price === undefined) price = this.getSellPrice(itemId);

    this.loseItemId(itemId, amount);
    this.gainGold(amount * price);
  }

  static loseItem(item, amount, instant) {
    if (!item) return 0;

    if (amount === undefined) amount = 1;
    let amountRemoved = 0;

    if (!!this.specialItem && item.id == this.specialItem.id) {
      amountRemoved = this.loseUnitOfSpecialItem(amount, instant);
      if (amountRemoved >= amount) {
        $gameMap.requestRefresh();
        this.requestHudRefresh();
        return amountRemoved;
      }
    }

    amountRemoved += this.container.loseItem(item, amount - amountRemoved);

    $gameMap.requestRefresh();
    this.requestHudRefresh();

    return amountRemoved;
  }

  static clearList() {
    this.container.clearList();
  }

  static doItemEffect(item) {
    return false;
  }

  static itemIdIsConsumable(itemId) {
    const item = this.getItemData(itemId);

    return item && item.consumable;
  }

  static consumeItem(item) {
    if (!item) return false;

    if (item.type === "tool") {
      Managers.Tools.setTool(item.id);
    } else if (item.type == "seed") {
      // nothing to lose
    } else if (item.type == "food" || item.type == "medicine") {
      if (item.consumable) {
        return this.eatItem(item);
      }
    }

    if (this.doItemEffect(item)) {
      if (item.consumable) {
        this.loseItem(item, 1);
      }
      return true;
    }

    return false;

  }

  static checkSpecialItemDrop(targetX, targetY) {
    const toolObjects = [
      'fence',
      'gate',
      'sprinkler',
      'super-sprinkler',
      'hay',
      'chicken-feed',
      'chest',
      'coop',
      'barn'
    ];

    if (toolObjects.includes(this.selectedItem.id)) {
      return this.useSelectedItem(targetX, targetY);
    }

    // Keyboard / gamepad only
    if (!TouchInput.isPressed()) {
      if (this.selectedItem.type == 'seed') {
        return this.useSelectedItem(targetX, targetY);
      }
    }

    if (Managers.Content.checkSpecialItemDrop(targetX, targetY)) {
      return true;
    }

    return false;
  }

  static reserveItemDrop(item, amount) {
    if (item === undefined) return false;

    if (amount === undefined) amount = item.amount;

    const tile = TileHelper.getSmartFrontTile(1);

    if (!tile || !TileHelper.checkIfTileCanGetObject(tile.x, tile.y, true, item.type, false, true)) {
      $gamePlayer.requestBalloon(Balloons.FAILURE);
      Managers.Sound.playBuzzer();
      return false;
    }

    amount = this.loseItem(item, amount);
    Managers.FarmObjects.createItemAt($gameMap._mapId, tile.x, tile.y, item.id, amount);

    $gamePlayer.leaveTile(tile.x, tile.y);
    Managers.Sound.playDropItem();
    return true;
  }

  static dropItem(blockSound, forceDrop, intendedX, intendedY) {
    if (this.selectedItem === undefined) return false;
    if (this._waiting) return false;

    let tile;
    let leaveTile = true;

    if (intendedX !== undefined && intendedY !== undefined) {
      tile = TileHelper.newTile(intendedX, intendedY);
      leaveTile = false;
    } else {
      tile = TileHelper.getSmartFrontTile(1);
    }

    if (tile) {
      const heightLevel = $gamePlayer.getHeightLevel();
      if (Managers.Map.getHeightLevel(tile.x, tile.y) != heightLevel) return false;
    }

    if (!tile || !TileHelper.checkIfTileCanGetObject(tile.x, tile.y, true, this.selectedItem.type, false)) {
      if (!forceDrop) {
        if (this.checkSpecialItemDrop(tile.x, tile.y)) {
          return true;
        }
      }

      if (!blockSound) {
        $gamePlayer.requestBalloon(Balloons.FAILURE);
        Managers.Sound.playBuzzer();
      }
      return false;
    }

    if (!forceDrop) {
      if (this.checkSpecialItemDrop(tile.x, tile.y)) {
        return true;
      }
    }

    if (this.selectedItem.type === 'animal') {
      const animalData = this.getItemInfo('animalData');

      if (Managers.Content.dropAnimalAt($gameMap._mapId, tile.x, tile.y, this.selectedItem.id, animalData)) {
        this._onDropItem(tile.x, tile.y, this.selectedItem.id, 1, leaveTile, false);
        return true;
      }

      if (Managers.Creatures.dropAnimalAt($gameMap._mapId, tile.x, tile.y, this.selectedItem.id, animalData)) {
        this._onDropItem(tile.x, tile.y, this.selectedItem.id, 1, leaveTile, false);
        return true;
      }

      if (!blockSound) {
        $gamePlayer.requestBalloon(Balloons.FAILURE);
        Managers.Sound.playBuzzer();
      }
      return false;
    }

    // if (Managers.Map.canDropItemAt(this.selectedItem.id, tile.x, tile.y)) {
    //   this._createDropedItem(tile.x, tile.y, this.selectedItem.id, this.selectedItem.amount, leaveTile);
    //   this.loseItem(this.selectedItem, this.selectedItem.amount);
    // }
    return true;
  }

  static _createDropedItem(x, y, id, amount, leaveTile) {
    Managers.FarmObjects.createItemAt($gameMap._mapId, x, y, id, amount);
    this._onDropItem(x, y, id, amount, leaveTile, true);
  }

  static _onDropItem(x, y, id, amount, leaveTile, playSound) {
    this._specialItem = undefined;
    this._itemInfo = {};
    this.requestHudRefresh();
    if (leaveTile) {
      $gamePlayer.leaveTile(x, y);
    }

    if (playSound !== false) {
      Managers.Sound.playDropItem();
    }

    Managers.Map.onDropItem(x, y, id, amount);
    Managers.Content.onDropItem(x, y, id, amount);
  }

  static canUse(item) {
    return true;
  }

  static lastItem() {
    return this._lastItem;
  }

  static setLastItem(item) {
    this._lastItem = item;
  }

  static gainGold(amount, silent, register) {
    this.gold = (this.gold + amount).clamp(0, this.maxGold());
    if (amount > 0) {
      // if (silent !== true) {
      //   Managers.Sound.playShop();
      // }

      if (register !== false) {
        Managers.History.registerMoneyEarned(amount);
      }
    } else if (amount < 0) {
      if (register !== false) {
        Managers.History.registerMoneySpent(-amount);
      }
    }
  }

  static loseGold(amount) {
    this.gainGold(-amount);
  }

  static loseMoney(amount) {
    this.loseGold(Math.min(amount, this.gold));
  }

  static maxGold() {
    return 999999999999999;
  }

  static findBrokenAxe() {
    Managers.CommonEvent.playEvent('find_broken_axe');
  }

  static findIronHammer() {
    Managers.CommonEvent.playEvent('find_iron_hammer');

    Managers.Steam.activateAchievement(Achievements.GET_IRON_HAMMER);

    setTimeout(() => {
      Switches.hasIronHammer = true;
      this.requestHudRefresh();
    }, 300);
  }

  static findIronSickle() {
    Managers.CommonEvent.playEvent('find_iron_sickle');
    Managers.Steam.activateAchievement(Achievements.GET_IRON_SICKLE);
    setTimeout(() => {
      Switches.hasIronSickle = true;
      this.requestHudRefresh();
    }, 300);
  }

  static eatItem(item) {
    if ($gameSystem.isPlayingCutscene()) return false;

    if (this.loseItem(item, 1) !== 1) {
      return false;
    }

    if (item.restores) {
      Managers.Health.restoreHealth(item.restores);
    }

    if (item.type == 'food') {
      if (item.drink) {
        Managers.Sound.playDrink();
      } else {
        Managers.Sound.playEat();
      }
    }
    return true;
  }

  static eat(itemId) {
    const data = this.getItemData(itemId);
    if (!data) return false;

    return this.eatItem(data);
  }

  static doStorageEvent() {
    if ($gameSystem.isPlayingCutscene()) return;

    Managers.Map.openStorage(true, () => {
      if (this.isHoldingItem()) {
        const item = this.selectedItem;

        if (this.storeSelectedItemAt(this.storageContainer, item.amount)) {
          Managers.Sound.playDropItem();
          Managers.Map.closeStorage();
          return;
        }
      }

      this.openStorage();
      Managers.Map.closeStorage();
    });
  }

  static doTrashCanEvent(containerName) {
    if ($gameSystem.isPlayingCutscene()) return;

    if (this.canSelectedItemBeTrashed()) {
      const container = this.getTrashContainer(containerName);

      if (this.storeSelectedItemAt(container, this.selectedItem.amount)) {
        Managers.Sound.playDropItem();
        return;
      }
    }

    this.openTrashCan(containerName);
  }

  static doShippingBinEvent() {
    if ($gameSystem.isPlayingCutscene()) return;

    this._waiting = true;
    const farmMap = Managers.Map.getClass(Maps.FARM);
    farmMap.openShippingBin(!this.isHoldingItem(), () => {
      this._waiting = false;

      if (this.isHoldingItem()) {
        const item = this.selectedItem;
        const price = this.getSellPrice(item.id);

        if (price > 0) {
          if (this.storeSelectedItemAt(Managers.ShippingBin.container, item.amount)) {
            Managers.Sound.playDropItem();
            farmMap.closeShippingBin();
            return;
          }
        }
      }

      this.openShippingBin();
      farmMap.closeShippingBin();
    });
  }

  static itemIdHasPrice(itemId) {
    const price = this.getSellPrice(itemId);
    return price > 0;
  }

  static storeSelectedItemAt(newContainer, amount) {
    const item = this.selectedItem;
    if (!newContainer.canGainItem(item)) return false;

    const amountToAdd = this.loseItem(item, amount);

    if (amountToAdd > 0) {
      newContainer.gainItem(item, amountToAdd);
      return true;
    }

    return false;
  }

  static openShippingBin() {
    Managers.Scenes.snapForBackground();
    Managers.Scenes.push(GameScenes.ShippingBin);
  }

  static openStorage() {
    Managers.Scenes.snapForBackground();
    Managers.Scenes.push(GameScenes.Storage);
  }

  static openTrashCan(containerName) {
    Managers.Scenes.snapForBackground();
    Managers.Scenes.push(GameScenes.TrashCan);
    Managers.Scenes.prepareNextScene(containerName);
  }

  static openMailbox() {
    Managers.Scenes.snapForBackground();
    Managers.Scenes.push(GameScenes.Mailbox);
  }

  static getItemIcon(itemId) {
    const data = this.getItemData(itemId, true);

    if (!data) return 0;

    return data.iconIndex;
  }

  static canSelectedItemBeTrashed() {
    return this.isHoldingItem() && this.canItemIdBeTrashed(this.selectedItem.id);
  }

  static canItemBeTrashed(item) {
    return this.canItemIdBeTrashed(item.id);
  }

  static canItemIdBeTrashed(itemId) {
    const itemData = this.getItemData(itemId);
    return itemData.canTrash === true;
  }

  static getSellPrice(itemId) {
    const itemData = this.getItemData(itemId);
    return itemData.sellPrice;
  }

  static getTaskPrice(itemId) {
    const itemData = this.getItemData(itemId);
    return itemData.taskPrice;
  }

  static getBuyPrice(itemId) {
    const itemData = this.getItemData(itemId);
    return itemData.buyPrice;
  }

  static getPurchaseDiscount(itemId) {
    let contentDiscount = Managers.Content.getPurchaseDiscount(itemId);

    if (isNaN(contentDiscount)) {
      contentDiscount = 0;
    }

    return contentDiscount || 0;
  }

  static getDiscountedBuyPrice(itemId) {
    const basePrice = this.getBuyPrice(itemId);
    const sellPrice = this.getSellPrice(itemId);

    //Blocks discounts on items with fixed prices
    if (basePrice == sellPrice) return basePrice;

    let discount = this.getPurchaseDiscount(itemId);

    if (discount > 0 && discount <= 100) {
      discount = discount / 100;
      const multiplier = 1 - discount;

      return Math.floor(basePrice * multiplier);
    }

    return basePrice;
  }

  static assignEvent(itemName, eventName, actionType) {
    if (actionType === undefined) actionType = 'get';

    const callback = function(){
      Managers.CommonEvent.playEvent(eventName);
      this.un(`${actionType}-${itemName}`, callback);
    };

    this.on(`${actionType}-${itemName}`, callback);
  }

  static getItemIconIndex(itemId) {
    const len = $dataItems.length;
    let i = 0;

    for (i = 0; i < len; i++) {
      const data = $dataItems[i];

      if (data.id == itemId) {
        return data.iconIndex;
      }
    }

    return -1;
  }

  static getItemName(itemId) {
    return Managers.Text.item(itemId);
  }

  static addItemInfo(infoName, infoData) {
    if (!this._itemInfo) this._itemInfo = {};

    this._itemInfo[infoName] = infoData;
  }

  static getItemInfo(infoName) {
    if (!this._itemInfo) return undefined;

    return this._itemInfo[infoName];
  }

  static isHoldingChicken() {
    return this.selectedItem !== undefined && this.selectedItem.id.includes('chicken');
  }

  static isHoldingFood() {
    return this.selectedItem !== undefined && this.selectedItem.type == "food";
  }

  static hasAnyFood() {
    return this.container.items.some(item => item && item.type == 'food');
  }

  static getSeedCropData(seedId) {
    //#ToDo: Move this to the SeedBoxes content class somehow
    if (!seedId.endsWith('-seeds') && !seedId.endsWith('seed-box')) {
      return false;
    }

    const cropName = seedId.replace('-seeds', '').replace('-seed-box', '');
    const cropData = this.getItemData(cropName);

    if (!cropData) return false;

    return cropData;
  }

  static canGiftItem(itemId) {
    const data = this.getItemData(itemId);

    if (!data) return false;
    return data.possibleGift;
  }

  static isOutOfSeason(itemId, mapId) {
    const data = this.getItemData(itemId);
    if (!data) return false;
    if (!data.seasons) return false;

    let month = Managers.Time.month;
    if (mapId) {
      month = Managers.Map.mapMonth(mapId);
    }

    switch (month) {
      case Seasons.SPRING :
        return !data.seasons.includes('Spring');
      case Seasons.SUMMER :
        return !data.seasons.includes('Summer');
      case Seasons.FALL :
        return !data.seasons.includes('Fall');
      case Seasons.WINTER :
        return !data.seasons.includes('Winter');
      default :
        return false;
    }
  }

  static explainSelectedItem() {
    const itemId = this.getSelectedItemId();
    if (!itemId) return false;

    const animalData = this.getItemInfo('animalData');

    if (animalData) {
      Variables.animalName = animalData.creatureName;
      Managers.CommonEvent.playEvent('tell_animal_name');
    } else {
      Variables.itemName = Managers.Text.item(itemId);
      Variables.itemDescription = Managers.Text.itemDescription(itemId);

      Managers.CommonEvent.playEvent('tell_item_details');
    }
  }

  static explainItem(itemId) {
    Variables.itemName = Managers.Text.item(itemId);
    Variables.itemDescription = Managers.Text.itemDescription(itemId);

    Managers.CommonEvent.playEvent('tell_item_details');
  }

  static checkIfRequiredExperienceIsMet(itemData) {
    if (!itemData.expRequired) return true;

    for (const expType in itemData.expRequired) {
      const attrName = `${ expType }Exp`;
      const needed = itemData.expRequired[expType] || 1000;
      const current = Managers.Player[attrName] || 0;

      if (needed > current) {
        return false;
      }
    }

    return true;
  }

  static getMaxCraftsAvailableForItemId(itemId) {
    const itemData = this.getItemData(itemId);
    return this.getMaxCraftsAvailableForItem(itemData);
  }

  // Check the ingredients to see how many of this item the player can craft -- doesn't check the crafting price.
  static getMaxCraftsAvailableForItem(itemData) {
    if (!itemData.ingredients) return Infinity;
    if (!itemData.ingredients.length) return Infinity;

    const ingredientList = [];
    let maxCount = Infinity;
    for (let i = 0; i < itemData.ingredients.length; i++) {
      const ingredientId = itemData.ingredients[i];
      if (ingredientList.includes(ingredientId)) continue;

      let requiredAmount = 1;
      for (let j = i + 1; j < itemData.ingredients.length; j++) {
        if (itemData.ingredients[j] == ingredientId) {
          requiredAmount++;
        }
      }

      const numItems = this.numItemsById(ingredientId);
      const maxCountForIngredient = Math.floor(numItems / requiredAmount);

      if (maxCountForIngredient < maxCount) {
        maxCount = maxCountForIngredient;
      }
    }

    return maxCount;
  }

  static pickShinyItem() {
    const allItems = ['coin', 'silver-coin', 'broken-glass', 'key', 'old-key'];
    const badItems = ['coin', 'broken-glass'];

    let itemId;
    if (Utils.chance(30)) {
      itemId = allItems[Math.randomInt(allItems.length)];
    } else {
      itemId = badItems[Math.randomInt(badItems.length)];
    }

    return this.pickItemId(itemId);
  }

  static itemIdIsFish(itemId) {
    const data = this.getItemData(itemId);

    return !!data.fish;
  }

  static countItems() {
    return this.container.countItems();
  }

  static hasAnyFish() {
    const items = this.container._items;
    for (const item of items) {
      if (!item) {
        continue;
      }

      if (this.itemIdIsFish(item.id)) {
        return true;
      }
    }

    return false;

  }
}

Inventory.clear();

module.exports = {
  Inventory,
};
