require('../core/CacheMap');
require('../core/ImageCache');
require('../core/RequestQueue');

const Window_Base = require('../windows/Window_Base');

//-----------------------------------------------------------------------------
// ImageManager
//
// The static class that loads images, creates bitmap objects and retains them.

Managers.Images = class Images {
  constructor() {
    throw new Error('This is a static class');
  }

  static initialize() {
    this.cache = new Core.CacheMap(this);

    this._imageCache = new ImageCache();
    this._requestQueue = new RequestQueue();
    this._minCustomIndex = 1000;
    this._customIconIndex = this._minCustomIndex;
  }

  static _generateCacheKey(cachePath, hue) {
    return  `${cachePath}:${hue}`;
  }

  static requestAnimation(filename, hue) {
    return this.requestBitmap('img/animations/', filename, hue, true);
  }

  static loadAnimation(filename, hue) {
    return this.loadBitmap('img/animations/', filename, hue, true);
  }

  static _imgIsInvalid(filename, folder) {
    if (!Managers.Extension.imgIsInvalid(filename, folder)) {
      return false;
    }

    if (Utils.isNwjs()) {
      const base = Utils.getProjectFolder();
      const fullPath = Wrapper.joinFileNames(base, 'img', folder, `${filename}.png`);

      return !Wrapper.existsSync(fullPath);
    } else {
      return false;
    }
  }

  static characterIsInvalid(filename) {
    return Managers.Images._imgIsInvalid(filename, 'characters');
  }

  static portraitIsInvalid(filename) {
    return Managers.Images._imgIsInvalid(filename, 'portraits');
  }

  static pictureIsInvalid(filename) {
    return Managers.Images._imgIsInvalid(filename, 'pictures');
  }

  static loadCharacter(filename, hue) {
    if (Managers.Palletes.isPalleteBasedCharacter(filename)) {
      return Managers.Palletes.loadCharacter(filename, hue);
    }

    if (Managers.PaperDolls.isPaperDollCharacter(filename)) {
      return Managers.PaperDolls.loadCharacter(filename, hue);
    }

    return this.loadRegularCharacter(filename, hue);
  }

  static releaseCharacter(filename, hue) {
    return this.releaseBitmap('img/characters/', filename, hue, false);
  }

  static loadRegularCharacter(filename, hue) {
    return this.loadBitmap('img/characters/', filename, hue, false);
  }

  static loadRegularPortrait(filename, hue) {
    return this.loadBitmap('img/portraits/', filename, hue, false);
  }

  static loadRegularPicture(filename, hue) {
    return this.loadBitmap('img/pictures/', filename, hue, false);
  }

  static requestParallax(filename, hue) {
    return this.requestBitmap('img/parallaxes/', filename, hue, false);
  }

  static loadParallax(filename, hue) {
    return this.loadBitmap('img/parallaxes/', filename, hue, false);
  }

  static requestPicture(filename, hue) {
    if (filename.startsWith('menu/villagers/')) {
      return this.requestBitmap(`img/portraits/${ filename.replace('menu/villagers/', 'mini/').toLowerCase() }`, hue, false);
    }

    return this.requestBitmap('img/pictures/', filename, hue, false);
  }

  static loadPicture(filename, hue) {
    if (filename.startsWith('menu/villagers/')) {
      return this.loadPortrait(filename.replace('menu/villagers/', 'mini/').toLowerCase(), hue);
    }

    if (Managers.Palletes.isPalleteBasedPicture(filename)) {
      return Managers.Palletes.loadPicture(filename, hue);
    }
  
    return this.loadRegularPicture(filename, hue);
  }

  static releasePicture(filename, hue) {
    if (filename.startsWith('menu/villagers/')) {
      return this.releaseBitmap(`img/portraits/${ filename.replace('menu/villagers/', 'mini/').toLowerCase() }`, hue, false);
    }

    return this.releaseBitmap('img/pictures/', filename, hue, false);
  }

  static loadLight(filename) {
    return this.loadBitmap('img/lights/', filename, 0, false);
  }

  static createIconFromImage(fullImagePath, hue) {
    const bitmap = this.loadNormalBitmap(fullImagePath, hue || 0);
    bitmap.smooth = false;

    const newIndex = ++this._customIconIndex;
    const key = `icon-${newIndex}`;

    //Icons created from images should never be removed from cache, as they would never be reloaded
    this.cache.setItem(key, bitmap, Infinity);

    return newIndex;
  }

  static requestPortrait(filename, hue) {
    return this.requestBitmap('img/portraits/', filename, hue, false);
  }

  static loadPortrait(filename, hue) {
    if (Managers.Palletes.isPalleteBasedPortrait(filename)) {
      return Managers.Palletes.loadPortrait(filename, hue);
    }
  
    return this.loadRegularPortrait(filename, hue);
  }

  static requestChannel(filename, hue) {
    return this.requestBitmap('img/channel/', filename, hue, false);
  }

  static loadChannel(filename, hue) {
    return this.loadBitmap('img/channels/', filename, hue, false);
  }

  static requestSystem(filename, hue) {
    return this.requestBitmap('img/system/', filename, hue, false);
  }

  static loadSystem(filename, hue) {
    return this.loadBitmap('img/system/', filename, hue, false);
  }

  static requestTileset(filename, hue) {
    return this.requestBitmap('img/tilesets/', filename, hue, false);
  }

  static loadTileset(filename, hue) {
    return this.loadBitmap('img/tilesets/', filename, hue, false);
  }

  static requestTitle1(filename, hue) {
    return this.requestBitmap('img/titles1/', filename, hue, true);
  }

  static loadTitle1(filename, hue) {
    return this.loadBitmap('img/titles1/', filename, hue, true);
  }

  static loadBackground(filename, hue) {
    return this.loadBitmap('img/backgrounds/', filename, hue, false);
  }

  static releaseBackground(filename, hue) {
    return this.releaseBitmap('img/backgrounds/', filename, hue, false);
  }

  static requestTitle2(filename, hue) {
    return this.requestBitmap('img/titles2/', filename, hue, true);
  }

  static loadTitle2(filename, hue) {
    return this.loadBitmap('img/titles2/', filename, hue, true);
  }

  static loadBitmap(folder, filename, hue, smooth) {
    const modBitmap = Managers.Extension.loadBitmap(folder, filename, hue, smooth);
    if (modBitmap) return modBitmap;

    if (filename) {
      const filenameParts = filename.split('/');
      if (filenameParts.length > 1) {
        filename = filenameParts.pop();
        folder += `${filenameParts.join('/')}/`;
      }

      const bitmapPath = `${folder + encodeURIComponent(filename)}.png`;
      const bitmap = this.loadNormalBitmap(bitmapPath, hue || 0);
      bitmap.smooth = smooth;
      return bitmap;
    } else {
      return this.loadEmptyBitmap();
    }
  }

  static releaseBitmap(folder, filename, hue) {
    if (filename) {
      const filenameParts = filename.split('/');
      if (filenameParts.length > 1) {
        filename = filenameParts.pop();
        folder += `${filenameParts.join('/')}/`;
      }

      const bitmapPath = `${folder + encodeURIComponent(filename)}.png`;
      this.releaseNormalBitmap(bitmapPath, hue || 0);
    }
  }

  /**
  * Managers.Images.loadImage(filePath, hue, smooth)
  * @param filePath The path of the image to load
  * @param hue The hue change to perform on the image
  * @param smooth The smooth boolean for the bitmap
  * @return A new bitmap from the filePath specified
  */
  static loadImage(filePath, hue, smooth) {
    const folder = filePath.substring(0, filePath.lastIndexOf("/") + 1);
    const file = filePath.substring(folder.length);
    return this.loadBitmap(folder, file, hue, smooth);
  }

  static loadResizedCustomIcon(iconIndex, w, h) {
    const key = `resized-icon-${iconIndex}`;
    let newBitmap = this.cache.getItem(key);

    if (!newBitmap) {
      const iconBitmap = this.loadIcon(iconIndex);

      newBitmap = new Bitmap(w, h);
      newBitmap.blt(iconBitmap, 0, 0, iconBitmap.width, iconBitmap.height, 0, 0, w, h);
      this.cache.setItem(key, newBitmap, 10);
    } else {
      newBitmap.touch();
    }

    return newBitmap;
  }

  static loadResizedIcon(index, w, h) {
    const key = `resized-icon-${w}-${h}-${index}`;
    let bitmap = this.cache.getItem(key);

    if (!bitmap) {
      if (index >= this._minCustomIndex) {
        return this.loadResizedCustomIcon(index, w, h);
      }

      const itemIcons = this.loadSystem("ItemIcons");
      if (!itemIcons.isReady()) {
        console.warn("Tried to draw an icon before the Icon Set was loaded.");
        return new Bitmap(w, h);
      }

      const pw = Window_Base._iconWidth;
      const ph = Window_Base._iconHeight;
      const sx = index % 16 * pw;
      const sy = Math.floor(index / 16) * ph;

      bitmap = new Bitmap(w, h);
      bitmap.bltBitmap(itemIcons, sx, sy, pw, ph, 0, 0, w, h);
      this.cache.setItem(key, bitmap, 10);
    } else {
      bitmap.touch();
    }

    return bitmap;
  }

  /**
  * Managers.Images.loadIcon(index)
  * @param index The icon index to return as a single bitmap
  * @return A new bitmap containing only the icon specified
  */
  static loadIcon(index) {
    const key = `icon-${index}`;
    let bitmap = this.cache.getItem(key);

    if (!bitmap) {
      const itemIcons = this.loadSystem("ItemIcons");
      const pw = Window_Base._iconWidth;
      const ph = Window_Base._iconHeight;
      const sx = index % 16 * pw;
      const sy = Math.floor(index / 16) * ph;
      bitmap = itemIcons.copySection(sx, sy, pw, ph);

      this.cache.setItem(key, bitmap, 20);
    } else {
      bitmap.touch();
    }

    return bitmap;
  }

  static loadIconWithShadow(index) {
    const key = `shadow-icon-${index}`;
    let bitmap = this.cache.getItem(key);

    if (!bitmap) {
      const itemIcons = this.loadSystem("ItemIcons");

      const pw = Window_Base._iconWidth;
      const ph = Window_Base._iconHeight;
      const shadowX = Constants.SHADOW_ICON % 16 * pw;
      const shadowY = Math.floor(Constants.SHADOW_ICON / 16) * ph;

      bitmap = itemIcons.copySection(shadowX, shadowY, pw, ph);

      const sourceX = index % 16 * pw;
      const sourceY = Math.floor(index / 16) * ph;
      bitmap.blt(itemIcons, sourceX, sourceY, pw, ph, 0, 0, pw, ph);

      this.cache.setItem(key, bitmap, 20);
    } else {
      bitmap.touch();
    }

    return bitmap;
  }

  static loadTileId(tileId, tileset) {
    const setNumber = 5 + Math.floor(tileId / 256);
    const tilesetName = tileset.tilesetNames[setNumber];

    const key = `tile-${ tilesetName}-${ tileId }`;
    let bitmap = this.cache.getItem(key);

    if (!bitmap) {
      const tilesetBitmap = this.loadTileset(tilesetName);
      const size = Constants.TILE_SIZE;
      const sx = (Math.floor(tileId / 128) % 2 * 8 + tileId % 8) * size;
      const sy = Math.floor(tileId % 256 / 8) % 16 * size;

      bitmap = tilesetBitmap.copySection(sx, sy, size, size);

      this.cache.setItem(key, bitmap, 20);
    } else {
      bitmap.touch();
    }

    return bitmap;
  }

  static loadEmptyBitmap() {
    let empty = this.cache.getItem('empty');
    if (!empty) {
      empty = new Bitmap();
      this.cache.setItem('empty', empty);
    } else {
      empty.touch();
    }
    return empty;
  }

  static loadItemBadge(iconIndex) {
    return this.loadIcon(iconIndex);
  }

  static loadCharacterIcon(characterName, characterIndex, zoom, direction) {
    if (zoom === undefined) zoom = 1;
    if (direction === undefined) direction = 2;

    // var key = 'character-' + characterName + '-' + zoom;
    // var bitmap = this.cache.getItem(key);
    let bitmap;

    if (!bitmap) {
      const characterBitmap = this.loadCharacter(characterName);
      if (!characterBitmap.isReady()) {
        bitmap = new Bitmap();
        bitmap._image = new Image();
        bitmap._loadingState = 'pending';
        return bitmap;
      }

      const regex = /\[(\d*)\]/;
      const match = characterName.match(regex);
      let frameCount = 3;
      if (match && match.length >= 2) {
        frameCount = parseInt(match[1]);
      }

      const big = this.isBigCharacter(characterName);
      const isRow = this.isCharacterRow(characterName);

      const pw = characterBitmap.width / (big ? frameCount : frameCount * 4);
      const ph = isRow ? characterBitmap.height : (characterBitmap.height / (big ? 4 : 8));
      const n = characterIndex;
      const sx = (n % 4 * frameCount + 1) * pw;
      let sy = (Math.floor(n / 4) * 4) * ph;
      sy += (direction - 2) / 2 * ph;

      const dw = pw * zoom;
      const dh = ph * zoom;

      bitmap = new Bitmap(dw, dh);
      bitmap.bltBitmap(characterBitmap, sx, sy, pw, ph, 0, 0, dw, dh);

      // this.cache.setItem(key, bitmap);
    }

    return bitmap;
  }

  static loadNormalBitmap(bitmapPath, hue, cachePath) {
    const key = this._generateCacheKey(cachePath || bitmapPath, hue);
    let bitmap = this._imageCache.get(key);
    if (!bitmap) {
      bitmap = Bitmap.load(decodeURIComponent(bitmapPath));
      bitmap.addLoadListener(() => {
        bitmap.rotateHue(hue);
      });
      this._imageCache.add(key, bitmap);
    } else if (!bitmap.isReady()) {
      bitmap.decode();
    }

    return bitmap;
  }

  static releaseNormalBitmap(bitmapPath, hue, cachePath) {
    const key = this._generateCacheKey(cachePath || bitmapPath, hue);
    this._imageCache.free(key);
  }

  static clear() {
    // this._cache.clear();
    this._imageCache = new ImageCache();
  }

  static isReady() {
    return this._imageCache.isReady();
  }

  static _findSymbol(filename, symbol) {
    if (!filename) {
      return false;
    }

    const names = filename.split(/[/\\]/);

    for (let name of names) {
      if (name.substring(0, 1) == symbol) return true;
      if (name.substring(1, 2) == symbol) return true;
    }

    return false;
  }

  static isBigCharacter(filename) {
    return this._findSymbol(filename, '$');
  }

  static isCharacterRow(filename) {
    return filename.includes('$$');
  }

  static isSingleImage(filename) {
    return filename.includes('$$$') || filename.includes('{sprite}') || this._findSymbol(filename, '-');
  }

  static isZeroParallax(filename) {
    return filename.charAt(0) === '!';
  }

  static requestBitmap(folder, filename, hue, smooth) {
    const modBitmap = Managers.Extension.requestBitmap(folder, filename, hue, smooth);
    if (modBitmap) return modBitmap;

    if (filename) {
      const filenameParts = filename.split('/');
      if (filenameParts.length > 1) {
        filename = filenameParts.pop();
        folder += `${filenameParts.join('/')}/`;
      }

      const bitmapPath = `${folder + encodeURIComponent(filename)}.png`;
      const bitmap = this.requestNormalBitmap(bitmapPath, hue || 0);
      bitmap.smooth = smooth;
      return bitmap;
    } else {
      return this.loadEmptyBitmap();
    }
  }

  static requestNormalBitmap(bitmapPath, hue) {
    const key = this._generateCacheKey(bitmapPath, hue);
    let bitmap = this._imageCache.get(key);
    if(!bitmap){
      bitmap = Bitmap.request(bitmapPath);
      bitmap.addLoadListener(() => {
        bitmap.rotateHue(hue);
      });
      this._imageCache.add(key, bitmap);
      this._requestQueue.enqueue(key, bitmap);
    } else {
      this._requestQueue.raisePriority(key);
    }

    return bitmap;
  }

  static update() {
    this._requestQueue.update();
  }

  static clearRequest() {
    this._requestQueue.clear();
  }

  static requestCurrentTilesets() {
    if (!$gameMap) return;

    const tileset = $gameMap.tileset();
    if (tileset) {
      const tilesetNames = tileset.tilesetNames;
      for (let i = 0; i < tilesetNames.length; i++) {
        this.requestTileset(tilesetNames[i]);
      }
    }
  }
};

Managers.Images.initialize();