/* eslint no-global-assign: 0 */

// #ToDo: Remove those hard-dependencies
require('game/managers/Villagers');
require('game/managers/Items');
const { Inventory } = require('engine/managers/Inventory');
const { Input } = require('engine/core/Input');

/* eslint no-unused-vars: 0 */
var $dataActors = null;
var $dataItems = null;
var $dataAnimations = null;
var $dataTilesets = null;
var $dataCommonEvents = [];
var $dataSystem = null;
var $dataMapInfos = null;
var $dataMap = null;
var $dataPreload = null;
var $dataTasks = {};
var $dataLanguages = {};
var $dataLetters = {};
var $dataVisitors = {};
var $dataStrings = {};
var $dataPalletes = null;
var $dataAchievements = null;
var $dataRecipes = null;
var $dataProfiles = null;

var $gameTemp = null;
var $gameSystem = null;
var $gameScreen = null;
var $gameTimer = null;
var $gameMessage = null;
var $gameSwitches = null;
var $gameVariables = null;
var $gameSelfSwitches = null;
var $gameActors = null;
var $gameParty = null;
var $gameMap = null;
var $gamePlayer = null;

//-----------------------------------------------------------------------------
// DataManager
//
// The static class that manages the database and game objects.

class Data {
  constructor() {
    throw new Error('This is a static class');
  }

  static restoreInfoFromOldSaveFile(fullData) {
    const { playerName, currentDateTime, gold } = fullData;

    const info = {
      playerName,
      currentDateTime,
      gold,
      timestamp: 0
    };

    return info;
  }

  static maybeBuildSaveFileCache() {
    if (!this._dataCache) {
      this.buildSaveFileCache();
    }
  }

  static getCacheableDataFromFullData(fullData) {
    return {
      info: fullData.info,
      currentDateTime: fullData.currentDateTime,
      fileName: fullData.fileName,
      gold: fullData.gold,
      health: fullData.health,
      party: fullData.party,
      player: fullData.player,
      playerData: fullData.playerData,
      playerName: fullData.playerName,
      system: fullData.system,
      weather: fullData.weather,
      creatures: fullData.creatures,
      relationship: fullData.relationship
    };
  }

  static cacheSaveData(saveFilePath, fullData) {
    if (!this._dataCache) return;

    if (fullData) {
      this._dataCache[saveFilePath] = this.getCacheableDataFromFullData(fullData);
      this._dataCache[saveFilePath].fileName = Utils.getFileNameFromPath(saveFilePath).replace('.ffsave', '');
    } else {
      delete this._dataCache[saveFilePath];
    }

    // Rebuilds the save list
    this._allSaves = Managers.Storage.getAllExistingSaves([]);
  }

  static buildSaveFileCache() {
    this._dataCache = {};
    this._allSaves = Managers.Storage.getAllExistingSaves([]);
    this.migrateOldSaveData(this._allSaves);

    for (const i in this._allSaves) {
      const saveFilePath = this._allSaves[i];

      if (!this._dataCache[saveFilePath]) {
        const fullData = Managers.Storage.loadSaveFileData(saveFilePath);
        if (fullData) {
          this.cacheSaveData(saveFilePath, fullData);
        }
      } else {
        this._dataCache[saveFilePath].fileName = Utils.getFileNameFromPath(saveFilePath).replace('.ffsave', '');
      }
    }
  }

  static getCachedSaveData(saveFilePath) {
    if (!this._dataCache) {
      this.buildSaveFileCache();
    }

    return this._dataCache[saveFilePath];
  }

  static loadDatabase() {
    const projectFolder = Utils.getProjectFolder();
    const dataFolder = Wrapper.joinFileNames(projectFolder, 'data');

    for (let i = 0; i < this._databaseFiles.length; i++) {
      const name = this._databaseFiles[i].name;
      const src = this._databaseFiles[i].src;
      const callback = this._databaseFiles[i].callback;
      const isYaml = src.toLowerCase().endsWith('.yaml');
      const logName = src.replace(/(\.json)|(\.yaml)$/gi, '');

      const dataFile = Wrapper.joinFileNames(dataFolder, src);

      const fn = (isYaml ? Managers.Storage.loadYamlFile : Managers.Storage.loadDataFile).bind(Managers.Storage);

      fn(dataFile, (data) => { // jshint ignore:line
        if (!Utils._isAutoTestMode) {
          if (data) {
            if (Array.isArray(data)) {
              console.log(...log(`[DataManager] Loaded ${data.length} ${logName}`));
            } else {
              console.log(...log(`[DataManager] Loaded ${Object.keys(data).length} ${logName}`));
            }
          } else {
            console.log(...log(`[DataManager] Loaded no ${logName}`));
          }
        }

        window[name] = data;
        if (callback) {
          callback.call(this, data);
        }
        this.onLoad(data);
      });
    }
  }

  static loadDataFile(name, src, callback) {
    if (Utils.isOptionValid('test')) {
      src += `?${new Date().getTime()}`;
    }

    const xhr = new XMLHttpRequest();
    const url = `data/${src}`;
    xhr.open('GET', url);
    xhr.overrideMimeType('application/json');
    xhr.onload = function() {
      if (xhr.status < 400) {
        try {
          window[name] = JSON.parse(xhr.responseText);
        }
        catch(e) {
          console.log(...log('Failed to parse JSON', name));
          throw e;
        }
        Data.onLoad(window[name]);

        if (callback) {
          callback.call(this);
        }
      }
    };
    xhr.onerror = this._mapLoader || (() => {
      Data._errorUrl = Data._errorUrl || url;
    });
    window[name] = null;
    xhr.send();
  }

  static loadPluginDataFile(pluginName, globalName, src, callback) {
    const projectFolder = Utils.getProjectFolder();
    src = Wrapper.joinFileNames(projectFolder, 'plugins', pluginName, src);

    if (Utils.isOptionValid('test')) {
      src += `?${new Date().getTime()}`;
    }

    const xhr = new XMLHttpRequest();
    xhr.open('GET', src);
    xhr.overrideMimeType('application/json');
    xhr.onload = function() {
      if (xhr.status < 400) {
        if (callback) {
          callback.call(this, xhr.responseText);
        }
      }
    };
    xhr.onerror = () => {
      Data._errorUrl = Data._errorUrl || src;
    };
    xhr.send();
  }

  static isDatabaseLoaded() {
    this.checkError();
    for (let i = 0; i < this._databaseFiles.length; i++) {
      if (!window[this._databaseFiles[i].name]) {
        return false;
      }
    }
    return true;
  }

  static loadMapData(mapId) {
    if (mapId > 0) {
      if (mapId >= 1000) {
        mapId = Math.floor(mapId / 1000);
      }

      const fileName = 'Map%1.json'.format(mapId.padZero(3));
      this._mapLoader = ResourceHandler.createLoader(`data/${fileName}`, this.loadDataFile.bind(this, '$dataMap', fileName));
      this.loadDataFile('$dataMap', fileName);
    } else {
      this.makeEmptyMap();
    }
  }

  static loadMapDataSync(mapId) {
    try {
      if (mapId > 0) {
        if (mapId >= 1000) {
          mapId = Math.floor(mapId / 1000);
        }

        const fileName = 'data/Map%1.json'.format(mapId.padZero(3));
        const data = Managers.Storage.loadFile(fileName);

        return JsonEx.parse(data);
      }
    } catch (e) {
      console.error(e);
      return false;
    }

    return false;
  }

  static makeEmptyMap() {
    $dataMap = {};
    $dataMap.data = [];
    $dataMap.events = [];
    $dataMap.width = 100;
    $dataMap.height = 100;
    $dataMap.scrollType = 3;
  }

  static isMapLoaded() {
    this.checkError();
    return !!$dataMap;
  }

  static onLoad(object) {
    let array;
    if (object === $dataMap) {
      this.extractMetadata(object);
      array = object.events;
    } else {
      array = object;
    }
    if (Array.isArray(array)) {
      for (const data of array) {
        if (data && data.note !== undefined) {
          this.extractMetadata(data);
        }
      }
    }

    if (object === $dataSystem) {
      window.isDemo = !!object.isDemoMode;
      GameScenes.Boot.loadSystemImages();
    }
  }

  static extractMetadata(data) {
    const re = /<([^<>:]+)(:?)([^>]*)>/g;
    data.meta = {};
    for (;;) {
      const match = re.exec(data.note);
      if (match) {
        if (match[2] === ':') {
          data.meta[match[1]] = match[3];
        } else {
          data.meta[match[1]] = true;
        }
      } else {
        break;
      }
    }
  }

  static checkError() {
    if (this._errorUrl) {
      throw new Error(`Failed to load: ${this._errorUrl}`);
    }
  }

  static createGameObjects() {
    $gameTemp = new Objects.Temp();
    $gameSystem = new Objects.System();
    $gameScreen = new Objects.Screen();
    $gameTimer = new Objects.Timer();
    $gameMessage = new Objects.Message();
    $gameSwitches = new Objects.Switches();
    $gameVariables = new Objects.Variables();
    $gameSelfSwitches = new Objects.SelfSwitches();
    $gameActors = new Objects.ActorHub();
    $gameParty = new Objects.Party();
    $gameMap = new Objects.Map();
    $gamePlayer = new Objects.Player();

    Managers.Lighting.clear();
    Managers.Player.clear();
    Managers.Tasks.clear();

    if (Managers.Steam.isSteamRunning()) {
      $gameActors.actor(1).setName(Managers.Steam.getScreenName());
    }
  }

  static addInitialItems() {
  }

  static setupEventTest() {
    this.createGameObjects();
    $gameParty.setupStartingMembers();
    $gamePlayer.reserveTransfer(-1, 8, 6, undefined, undefined, false);
  }

  static defaultGlobalInfo() {
    return {
      fullscreen : false,
      feedbackCount: 0,
    };
  }

  static loadGlobalInfo() {
    let json;
    try {
      json = Managers.Storage.loadFileFromGameFolder('global.ffdata');
    } catch (e) {
      console.error(e);
      return this.defaultGlobalInfo();
    }

    if (json) {
      return JSON.parse(json);
    } else {
      return this.defaultGlobalInfo();
    }
  }

  static saveGlobalInfo(info) {
    try {
      const result = Managers.Storage.saveFileToGameFolder('global.ffdata', JSON.stringify(info, undefined, 2));
      if (result && result.error) {
        throw result.error;
      }
    } catch(e) {
      console.log('Failed to save global info');
      console.error(e);
    }
  }

  static migrateOldSaveData(allSaves) {
    // If there are no saves found, skip this
    if (!allSaves || !allSaves.length) return;

    // Check if the global data still exists - if it does, then move the data to the save files
    const globalInfo = this.loadGlobalInfo();

    if (!globalInfo) return;
    if (globalInfo.migrated) return;

    let anyFailures = false;

    for (let key in globalInfo) {
      const info = globalInfo[key];
      if (info && typeof info == 'object' && info.version) {
        const fileName = key == 'auto' ? 'auto.ffsave' : 'file%1.ffsave'.format(key);
        let success = false;

        allSaves.forEach((filePath) => {
          if (!filePath.endsWith(fileName)) return;

          Managers.Storage.backupSaveFile(filePath);
          try {
            const json = Managers.Storage.loadSaveFile(filePath);
            if (!json) return;
            const data = JsonEx.parse(json);
            if (!data) return;
            if (data.info) return;

            // If the save file has a player name, override the one from the info object
            if (data.playerName) {
              info.playerName = data.playerName;
            }

            data.info = info;

            Managers.Storage.saveToFile(filePath, data);

            if (this._dataCache) {
              this._dataCache[filePath] = this.getCacheableDataFromFullData(data);
            }

            success = true;
          }
          catch(e) {
            anyFailures = true;
            Managers.Storage.restoreSaveFile(filePath);
            return;
          }
        });

        if (success) {
          delete globalInfo[key];
        }
      }
    }

    globalInfo.migrated = !anyFailures;
    this.saveGlobalInfo(globalInfo);
  }

  static anySaveExists() {
    if (!this._dataCache) {
      this.buildSaveFileCache();
    }

    return this._allSaves.length > 0;    
  }

  static getAllExistingSaves(fileNamesToSkip) {
    if (!this._dataCache) {
      this.buildSaveFileCache();
    }

    const saves = this._allSaves.filter((saveFilePath) => {
      for (let i in fileNamesToSkip) {
        if (saveFilePath.endsWith(fileNamesToSkip[i])) {
          return false;
        }
      }

      return true;
    });

    return saves;
  }

  static maxSavefiles() {
    return 10;
  }

  static saveGame(saveFilePath) {
    try {
      const status = this.saveGameWithoutRescue(saveFilePath);

      if (status && !$gameTemp.getLoadedGamePath()) {
        $gameTemp.setLoadedGamePath(saveFilePath);
      }

      return status;
    } catch (e) {
      console.error(e);
      return false;
    }
  }

  static canSaveGracefully() {
    if (!Managers.Scenes) return false;
    if (!Managers.Scenes._scene) return false;
    if (Switches.isInsideFestival) return false;

    const scene = Managers.Scenes._scene;

    if (scene instanceof GameScenes.Map || scene instanceof GameScenes.Diary) {
      if (!$gameSystem.isSaveEnabled()) return false;
      if (!!$gameMap && !!$gameMap._interpreter) {
        if (!!$gameMap._interpreter._list && $gameMap._interpreter._list.length > 0) return 'event';
        if ($gameMap._interpreter._character) return 'event';
        if (!!$gameMap._interpreter._eventId && $gameMap._interpreter._eventId > 0) return 'event';
        if ($gameMap._interpreter._waitMode) return 'event';
      }

      return true;
    }

    return false;
  }

  static quickSave() {
    if (this.canSaveGracefully() === true) {
      this.autoSave('quick');
      Managers.Sound.playSave();
    } else {
      console.log(...log("can't save gracefully"));
      if (window.$gameMap && $gameMap._interpreter) {
        console.log(...log($gameMap._interpreter._waitMode, $gameMap._interpreter._list, $gameMap._interpreter._character, $gameMap._interpreter._eventId));
      }
      Managers.Sound.playBuzzer();
    }
  }

  static onCrashAutoSave() {
    const canSave = this.canSaveGracefully();

    if (canSave === false) return;
    if (canSave === 'event') {
      $gameMap._interpreter._list = [];
      $gameMap._interpreter._character = null;
      $gameMap._interpreter._eventId = 0;
      $gameMap._interpreter._freezeChecker = 0;
      $gameMap._interpreter._waitMode = null;
    }

    this.autoSave('crash');
  }

  static prepareToSave() {
    $gameSystem.onBeforeSave();
    if (!!$gameMap && $gameMap._interpreter) {
      $gameMap._interpreter._freezeChecker = 0;
    }
  }

  static performAutoSave(slotName) {
    try {
      const saveFilePath = Wrapper.joinFileNames(Managers.Storage.directoryPath(), `${ slotName }.ffsave`);

      if (this.saveGameWithoutRescue(saveFilePath)) {
        return true;
      }
    } catch (e) {
      console.log(...log(e, 'autoSave'));
      if (e.message == 'save-file-failed') {
        throw e;
      }
      if (e.code == 'EPERM') {
        throw e;
      }
    }

    return false;
  }

  static autoSaveLoadedGame() {
    const loadedPath = $gameTemp.getLoadedGamePath();
    if (!loadedPath) {
      return false;
    }

    this.prepareToSave();

    try {
      return this.saveGameWithoutRescue(loadedPath);
    } catch (e) {
      console.log(...log(e, 'autoSaveLoadedGame'));
      if (e.message == 'save-file-failed') {
        throw e;
      }
    }
  }

  static autoSave(slotName) {
    if (slotName === undefined) slotName = 'auto';

    this.prepareToSave();
    return this.performAutoSave(slotName);
  }

  static getLatestSaveFile() {
    let slotName = 'quick';

    if (!this._dataCache) {
      return slotName;
    }

    if (!this._allSaves) {
      return slotName;
    }

    let largestTimestamp = 0;
    for (let saveFilePath in this._dataCache) {
      const data = this._dataCache[saveFilePath];
      if (!data || !data.info || !data.info.timestamp) {
        continue;
      }

      if (data.info.timestamp > largestTimestamp) {
        largestTimestamp = data.info.timestamp;
        slotName = data.fileName;
      }
    }

    return slotName;
  }

  static _executeAutoLoad(slotName) {
    if (slotName === '<latest>') {
      slotName = this.getLatestSaveFile();
    }

    const saveFilePath = Wrapper.joinFileNames(Managers.Storage.directoryPath(), `${ slotName }.ffsave`);

    if (this.loadGame(saveFilePath)) {
      Managers.Scenes._scene.fadeOutAll();
      $gameSystem.onAfterLoad();

      Managers.Sound.playLoad();

      $gamePlayer.reserveTransfer($gameMap.mapId(), $gamePlayer.x, $gamePlayer.y, undefined, undefined, false);
      $gamePlayer.requestMapReload();

      Managers.Scenes.goToScene(GameScenes.PreloadSave);
      $gameSystem.unlockSave();
    } else {
      Managers.Scenes.pop();
      $gameSystem.unlockSave();
    }
  }

  static autoLoad(slotName) {
    $gameSystem.lockSave();
    Managers.Scenes.push(GameScenes.AutoLoad);
    Managers.Scenes.prepareNextScene(slotName);
  }

  static autoNewGame() {
    Managers.Scenes.push(GameScenes.NewGame);
  }

  static loadGame(savefilePath) {
    try {
      return this.loadGameWithoutRescue(savefilePath);
    } catch (e) {
      if (e instanceof Utils.UpdateError) {
        throw e;
      }

      console.error(e);
      console.log(...log(e, 'loadGame'));
      return false;
    }
  }

  static deleteSave(saveFilePath) {
    try {
      Managers.Storage.removeSave(saveFilePath);
      this.cacheSaveData(saveFilePath, null);

      Managers.Steam.deleteFileOnCloud(saveFilePath);
      Managers.Sound.playDeleteFile();
      return true;
    } catch(e) {
      console.error(e);
      console.log(...log(e, 'deleteSave'));
      return false;
    }
  }

  static saveGameWithoutRescue(saveFilePath) {
    const contents = this.makeSaveContents();

    Managers.Storage.saveToFile(saveFilePath, contents);

    this.cacheSaveData(saveFilePath, contents);
    this.updateGlobalInfo();

    return true;
  }

  static updateGlobalInfo() {
    const globalInfo = this.loadGlobalInfo() || {};
    globalInfo.weather = Managers.Weather.currentWeather;
    globalInfo.feedbackCount = window.feedbackCount || 0;
    this.saveGlobalInfo(globalInfo);
  }

  static validateLoadedData(contents) {
    if (!contents.info) {
      // Let's make some assumptions
      contents.info = {
        version: 0.4
      };
    }

    if (!contents.farmObjects && contents.farmObjectList) {
      contents.farmObjects = {
        farmObjects : contents.farmObjectList
      };
    }
  }

  static loadGameWithoutRescue(saveFilePath) {
    const json = Managers.Storage.loadSaveFile(saveFilePath);
    const contents = JsonEx.parse(json);

    this.validateLoadedData(contents);

    if (!Managers.Updates.isVersionAccepted(contents.info.version, contents.saveFormatVersion)) {
      return false;
    }

    this.createGameObjects();
    this.extractSaveContents(contents);

    if (Managers.Updates.runUpdates(contents.info.version, contents) === false) {
      return false;
    }

    $gameTemp.setLoadedGamePath(saveFilePath);
    return true;
  }

  static makeSavefileInfo() {
    const info = {};
    info.globalId = this._globalId;
    info.title = $dataSystem.gameTitle;
    info.characters = $gameParty.charactersForSavefile();
    info.faces = $gameParty.facesForSavefile();
    info.playtime = $gameSystem.playtimeText();
    info.timestamp = Date.now();
    info.currentDateTime = Managers.Time.getDateTime();
    info.gold = Inventory.gold;
    info.version = Managers.Updates.currentVersion;
    info.saveFormatVersion = Managers.Updates.saveFormatVersion;
    info.playerName = $gameActors.actor(1).name();

    return info;
  }

  static makeSaveContents() {
    const contents = {};
    const modContents = {};

    Managers.Content.makeSaveContents(modContents);

    contents.mods = modContents;
    contents.info = this.makeSavefileInfo();
    contents.screen = $gameScreen.getDataForSaveFile();
    contents.timer = $gameTimer.getDataForSaveFile();
    contents.switches = $gameSwitches.getDataForSaveFile();
    contents.variables = $gameVariables.getDataForSaveFile();
    contents.selfSwitches = $gameSelfSwitches.getDataForSaveFile();
    contents.customSwitches = Switches.getCustomSwitches();
    contents.customVariables = Variables.getCustomVariables();
    contents.party = $gameParty.getDataForSaveFile();
    contents.specialItem = Inventory._specialItem;
    contents.specialItemInfo = Inventory._itemInfo;
    contents.itemBarIndex = Inventory.itemBarIndex;
    contents.toolBarIndex = Inventory.toolBarIndex;
    contents.items = Inventory.container.getData();
    contents.storage = Inventory.storageContainer.getData();
    contents.trash = Inventory.trashContainer.getData();
    contents.customContainers = Inventory.getCustomContainersData();
    contents.mailbox = Inventory.mailboxContainer.getData();
    contents.creatures = Managers.Creatures.getCreaturesData();
    contents.player = $gamePlayer.getDataForSaveFile();
    contents.currentDateTime = Managers.Time.getDateTime();
    contents.timesystemCallbacks = Managers.Time.getCallbacks();
    contents.farmObjects = Managers.FarmObjects.getData();
    contents.toolId = Managers.Tools.toolId;
    contents.water = Managers.Tools.water;
    contents.gold = Inventory.gold;
    contents.health = Managers.Health.getHealthData();
    contents.shipping = Managers.ShippingBin.container.getData();
    contents.tasks = Managers.Tasks.taskData;
    contents.customTasks = Managers.Tasks.customTasks;
    contents.weather = Managers.Weather.getWeatherData();
    contents.relationship = Managers.Relationship.getData();
    contents.mail = Managers.Mailbox.getData();
    contents.playerData = Managers.Player.getData();
    contents.playerName = $gameActors.actor(1).name();
    contents.system = $gameSystem.getDataForSaveFile();
    contents.history = Managers.History.getData();
    contents.achievements = Managers.Achievements.getData();
    contents.cooking = Managers.Cooking.getData();

    contents.mapData = $gameMap.getDataForSaveFile();

    return contents;
  }

  static extractSaveContents(contents) {
    this.createGameObjects();

    Utils.assignProperties($gameSystem, contents.system);
    Utils.assignProperties($gameScreen, contents.screen);
    Utils.assignProperties($gameTimer, contents.timer);
    Utils.assignProperties($gameSwitches, contents.switches);
    Utils.assignProperties($gameVariables, contents.variables);
    Utils.assignProperties($gameSelfSwitches, contents.selfSwitches);
    $gameParty.restoreData(contents.party);
    $gamePlayer.restoreData(contents.player);
    $gameMap.restoreData(contents.mapData || contents.map);

    $gamePlayer.restoreSpeed();
    $gamePlayer.clearTempData();

    Managers.Weather.setWeatherData(contents.weather);
    Managers.Mailbox.setData(contents.mail);
    Managers.Relationship.setData(contents.relationship);
    Managers.FarmObjects.setData(contents.farmObjects);
    Managers.Player.setData(contents.playerData, $gamePlayer);
    Managers.History.setData(contents.history);
    Managers.Achievements.setData(contents.achievements);
    Managers.Cooking.setData(contents.cooking);

    Inventory.gold = contents.gold;

    if (contents.items !== undefined) {
      Inventory.container.setData(contents.items);
    }
    if (contents.itemBarIndex !== undefined) {
      Inventory.itemBarIndex = contents.itemBarIndex;
    } else {
      Inventory.itemBarIndex = 0;
    }
    if (contents.toolBarIndex !== undefined) {
      Inventory.toolBarIndex = contents.toolBarIndex;
    } else {
      Inventory.toolBarIndex = 0;
    }

    if (contents.storage !== undefined) {
      Inventory.storageContainer.setData(contents.storage);
    }

    if (contents.trash !== undefined) {
      Inventory.trashContainer.setData(contents.trash);
    }
    if (contents.mailbox !== undefined) {
      Inventory.mailboxContainer.setData(contents.mailbox);
    }
    if (contents.customContainers !== undefined) {
      Inventory.setCustomContainersData(contents.customContainers);
    }

    if (contents.specialItem !== undefined) {
      Inventory._specialItem = contents.specialItem;
    } else {
      Inventory._specialItem = undefined;
    }

    if (contents.specialItemInfo !== undefined) {
      Inventory._itemInfo = contents.specialItemInfo;
    } else {
      Inventory._itemInfo = {};
    }

    if (contents.shipping !== undefined) {
      Managers.ShippingBin.container.setData(contents.shipping);
    }

    if (contents.creatures !== undefined) {
      Managers.Creatures.setCreaturesData(contents.creatures);
    }

    if (contents.currentDateTime !== undefined) {
      Managers.Time.setDateTime(contents.currentDateTime);
      Managers.Time.updateTime(false);
    }

    if (contents.timesystemCallbacks !== undefined) {
      Managers.Time.setCallbacks(contents.timesystemCallbacks);
    }

    if (contents.toolId !== undefined) {
      Managers.Tools.toolId = contents.toolId;
    }

    if (contents.health !== undefined) {
      Managers.Health.setHealthData(contents.health);
    }

    if (contents.tasks !== undefined) {
      Managers.Tasks.taskData = contents.tasks;
    }

    if (contents.customTasks !== undefined) {
      Managers.Tasks.customTasks = contents.customTasks;
    }

    if (contents.water !== undefined) {
      Managers.Tools.water = contents.water;
    }

    if (contents.playerName) {
      $gameActors.actor(1).setName(contents.playerName);
    }

    Managers.Tools.toolId = undefined;
    Switches.deleteCustomSwitches();
    if (contents.customSwitches) {
      Switches.setCustomSwitches(contents.customSwitches);
    }
    Variables.deleteCustomVariables();
    if (contents.customVariables) {
      Variables.setCustomVariables(contents.customVariables);
    }

    if (contents.mods) {
      Managers.Content.extractSaveContents(contents.mods);
    }

    $gameSystem.checkData();
  }

  static setupNewGame() {
    this.createGameObjects();
    $gameParty.setupStartingMembers();
    // $gamePlayer.reserveTransfer(Maps.FARM, 44, 0, 2, 0, false);
    Graphics.frameCount = 0;
    Managers.Tools.water = 0;
    $gameSystem.startGame();

    Managers.FarmObjects.clear();
    Managers.Creatures.clear();
    Inventory.clear();
    Managers.Health.clear();
    Managers.Tasks.clear();
    Managers.ShippingBin.clear();
    Managers.Mailbox.clear();
    Managers.Weather.clear();
    Managers.Relationship.clear();
    Managers.History.clear();
    Managers.Cooking.clear();

    Switches.deleteCustomSwitches();
    Variables.deleteCustomVariables();

    Variables.dogName = 'Spike';
    Variables.farmName = 'Orange Farm';

    Managers.Tools.setTool('empty');

    Managers.Time.setInitialDateTime();
    Managers.Time.enableTime();

    Switches.reset();
    $gameSystem.startCutscene();
    this.addInitialItems();
    Managers.Content.setupNewGame();
    $gameTemp.reserveCutscene('intro');
  }

  static calculateGameCompletionRate(summaryLines) {
    if (!summaryLines) {
      summaryLines = Managers.Content.getCustomDatabaseSummaryLines();
    }

    let total = 0;
    let current = 0;

    if (summaryLines) {
      for (let i = 0; i < summaryLines.length; i++) {
        if (!summaryLines[i]) continue;
        if (!summaryLines[i].dataFn) continue;

        const weight = summaryLines[i].weight || 0;
        if (!weight) continue;

        const data = summaryLines[i].dataFn(true);
        if (!data || data.length < 2) continue;

        const value = data[0];
        const max = data[1];

        current += Math.min(weight, Math.floor(value * weight / max * 100) / 100);
        total += weight;
      }
    }

    if (total > 0) {
      return Number(current.toFixed(2));
      // return Number((current * 100 / total).toFixed(2));
    }

    return 100;
  }

  static onSyncCloud({ successful, failed, ignored }) {
    this._finishedSyncing = true;
  }

  static syncCloudSaves() {
    this._finishedSyncing = false;

    if (!Managers.Steam.syncAllSaves(this.onSyncCloud.bind(this))) {
      this._finishedSyncing = true;
      return;
    }
  }

  static finishedSyncingCloudSaves() {
    if (!Managers.Steam.isSteamRunning()) {
      return true;
    }

    return this._finishedSyncing;
  }
}

Data._globalId = 'OrangeSeason';
Data._lastAccessedId = 1;
Data._errorUrl = null;
Data._versionId = "6c8e3f6ab0fc67da7f0e5bf3dbeb31a1".split(/(.{2})/).filter(Boolean);

Data._databaseFiles = [
  { name: '$dataActors', src: 'Characters.json', callback : Managers.Villagers.onLoadCharacters.bind(Managers.Villagers)},
  { name: '$dataItems', src: 'Objects.json', callback : Inventory.onLoadItems.bind(Inventory)},
  { name: '$dataRecipes', src: 'Recipes.json'},
  { name: '$dataAnimations', src: 'Animations.json'},
  { name: '$dataTilesets', src: 'Tilesets.json'},
  { name: '$dataAchievements', src: 'Achievements.json'},
  // { name: '$dataCommonEvents', src: 'CommonEvents.json', callback : Managers.CommonEvent.makeConditionTable},
  { name: '$dataSystem', src: 'System.json'},
  { name: '$dataMapInfos', src: 'MapInfos.json'},
  { name: '$dataPalletes', src: 'Palletes.json'},
  { name: '$dataProfiles', src: 'Profiles.yaml', callback: Input.onLoadProfiles.bind(Input)},
];

module.exports = {
  Data,
};
