//-----------------------------------------------------------------------------
// AudioManager
//
// The static class that handles BGM, BGS, ME and SE.

/* globals Html5Audio */

class Audio {
  constructor() {
    throw new Error('This is a static class');
  }

  static get masterVolume() {
    return this._masterVolume;
  }
  static set masterVolume(value) {
    this._masterVolume = value;
    WebAudio.setMasterVolume(this._masterVolume);
    Graphics.setVideoVolume(this._masterVolume);
  }

  static get bgmVolume() {
    return this._bgmVolume;
  }

  static set bgmVolume(value) {
    this._bgmVolume = value;
    this.updateBgmParameters(this._currentBgm);
  }

  static get bgsVolume() {
    return this._bgsVolume;
  }

  static set bgsVolume(value) {
    this._bgsVolume = value;
    this.updateBgsParameters(this._currentBgs);
  }

  static get meVolume() {
    return this._meVolume;
  }

  static set meVolume(value) {
    this._meVolume = value;
    this.updateMeParameters(this._currentMe);
  }

  static get seVolume() {
    return this._seVolume;
  }

  static set seVolume(value) {
    this._seVolume = value;
  }

  static _playBgm(bgm, pos, duration) {
    if (bgm.name) { 
      this._bgmBuffer = this.createBuffer('bgm', bgm.name);

      if (duration) {
        this._bgmBuffer.fadeIn(duration);
      }

      this.updateBgmParameters(bgm);
      if (!this._meBuffer) {
        this._bgmBuffer.play(true, pos || 0);
      }
    }

    this.updateCurrentBgm(bgm, pos);
  }

  static playBgm(bgm, pos, duration) {
    if (this.isCurrentBgm(bgm)) {
      this.updateBgmParameters(bgm);
      this.updateCurrentBgm(bgm, pos);
      return;
    }

    if (this.hasBgmPlaying()) {
      Audio.fadeOutBgm(0.5);
      setTimeout(() => {
        Audio.stopBgm();
        Audio._playBgm(bgm, pos, duration);
      }, 600);

      return;
    }

    Audio._playBgm(bgm, pos, duration);
  }

  static playBgmName(name, volume, pos, duration) {
    Audio.playBgm({name, volume : volume || 90, pitch: 100}, pos || 0, duration || 1);
  }

  static replayBgm(bgm) {
    if (this.isCurrentBgm(bgm)) {
      this.updateBgmParameters(bgm);
    } else {
      this.playBgm(bgm, bgm.pos);
      if (this._bgmBuffer) {
        this._bgmBuffer.fadeIn(this._replayFadeTime);
      }
    }
  }

  static isCurrentBgm(bgm) {
    return (this._currentBgm && this._bgmBuffer &&
        this._currentBgm.name === bgm.name);
  }

  static updateBgmParameters(bgm) {
    this.updateBufferParameters(this._bgmBuffer, this._bgmVolume * 0.8, bgm);
  }

  static updateCurrentBgm(bgm, pos) {
    this._currentBgm = {
      name: bgm.name,
      volume: bgm.volume,
      pitch: bgm.pitch,
      pan: bgm.pan,
      pos
    };
  }

  static hasBgmPlaying() {
    return this._bgmBuffer && this._currentBgm;
  }

  static stopBgm() {
    if (this._bgmBuffer) {
      this._bgmBuffer.stop();
      this._bgmBuffer = null;
      this._currentBgm = null;
    }
  }

  static fadeOutBgm(duration) {
    if (this._bgmBuffer) {
      if (this._currentBgm) {
        this._bgmBuffer.fadeOut(duration);
        this._currentBgm = null;
      } else {
        this.stopBgm();
      }
    }
  }

  static fadeInBgm(duration) {
    if (this._bgmBuffer && this._currentBgm) {
      this._bgmBuffer.fadeIn(duration);
    }
  }

  static playBgs(bgs, pos) {
    if (this.isCurrentBgs(bgs)) {
      this.updateBgsParameters(bgs);
    } else {
      this.stopBgs();
      if (bgs.name) {
        this._bgsBuffer = this.createBuffer('bgs', bgs.name);
        this.updateBgsParameters(bgs);
        this._bgsBuffer.play(true, pos || 0);
      }
    }
    this.updateCurrentBgs(bgs, pos);
  }

  static replayBgs(bgs) {
    if (this.isCurrentBgs(bgs)) {
      this.updateBgsParameters(bgs);
    } else {
      this.playBgs(bgs, bgs.pos);
      if (this._bgsBuffer) {
        this._bgsBuffer.fadeIn(this._replayFadeTime);
      }
    }
  }

  static isCurrentBgs(bgs) {
    return (this._currentBgs && this._bgsBuffer && this._currentBgs.name === bgs.name);
  }

  static updateBgsParameters(bgs) {
    this.updateBufferParameters(this._bgsBuffer, this._bgsVolume, bgs);
  }

  static updateCurrentBgs(bgs, pos) {
    this._currentBgs = {
      name: bgs.name,
      volume: bgs.volume,
      pitch: bgs.pitch,
      pan: bgs.pan,
      pos
    };
  }

  static stopBgs() {
    if (this._bgsBuffer) {
      this._bgsBuffer.stop();
      this._bgsBuffer = null;
      this._currentBgs = null;
    }
  }

  static fadeOutBgs(duration) {
    if (this._bgsBuffer && this._currentBgs) {
      this._bgsBuffer.fadeOut(duration);
      this._currentBgs = null;
    }
  }

  static fadeInBgs(duration) {
    if (this._bgsBuffer && this._currentBgs) {
      this._bgsBuffer.fadeIn(duration);
    }
  }

  static playMe(me) {
    if (typeof(me) === "string") {
      me = {
        name : me,
        pitch : 100,
        pan : 0,
        volume : 90
      };
    }

    this.stopMe();
    if (me.name) {
      if (this._bgmBuffer && this._currentBgm) {
        this._currentBgm.pos = this._bgmBuffer.seek();
        this._bgmBuffer.stop();
      }
      this._meBuffer = this.createBuffer('me', me.name);
      this.updateMeParameters(me);
      this._meBuffer.play(false);
      this._meBuffer.addStopListener(this.stopMe.bind(this));
    }
  }

  static updateMeParameters(me) {
    this.updateBufferParameters(this._meBuffer, this._meVolume, me);
  }

  static fadeOutMe(duration) {
    if (this._meBuffer) {
      this._meBuffer.fadeOut(duration);
    }
  }

  static stopMe() {
    if (this._meBuffer) {
      this._meBuffer.stop();
      this._meBuffer = null;
      if (this._bgmBuffer && this._currentBgm && !this._bgmBuffer.isPlaying()) {
        this._bgmBuffer.play(true, this._currentBgm.pos);
        this._bgmBuffer.fadeIn(this._replayFadeTime);
      }
    }
  }

  static seStringToObject(se, pitch) {
    if (typeof(se) === "string") {
      return {
        name : se,
        pitch : pitch || 100,
        pan : 0,
        volume : 90
      };
    }

    return se;
  }

  static playDynamicSe(se, distance, pitch) {
    const data = Audio.seStringToObject(se, pitch);
    data.volume = 5 + Math.floor(120 - (distance > 20 ? 20 : distance) * 6);

    return Audio.playSe(data);
  }

  static playSe(se, pitch) {
    se = Audio.seStringToObject(se, pitch);

    if (se.name) {
      this._seBuffers = this._seBuffers.filter(audio => audio.isPlaying());
      const buffer = this.createBuffer('se', se.name);
      this.updateSeParameters(buffer, se);
      buffer.play(false);
      this._seBuffers.push(buffer);
    }
  }

  static updateSeParameters(buffer, se) {
    this.updateBufferParameters(buffer, this._seVolume, se);
  }

  static stopSe() {
    this._seBuffers.forEach(buffer => {
      buffer.stop();
    });
    this._seBuffers = [];
  }

  static playStaticSe(se) {
    if (se.name) {
      this.loadStaticSe(se);

      for (const buffer of this._staticBuffers) {
        if (buffer._reservedSeName === se.name) {
          buffer.stop();
          this.updateSeParameters(buffer, se);
          buffer.play(false);
          break;
        }
      }
    }
  }

  static loadStaticSe(se) {
    if (se.name && !this.isStaticSe(se)) {
      const buffer = this.createBuffer('se', se.name);
      buffer._reservedSeName = se.name;
      this._staticBuffers.push(buffer);
      if (this.shouldUseHtml5Audio()) {
        Html5Audio.setStaticSe(buffer._url);
      }
    }
  }

  static isStaticSe(se) {
    for (const buffer of this._staticBuffers) {
      if (buffer._reservedSeName === se.name) {
        return true;
      }
    }

    return false;
  }

  static stopAll() {
    this.stopMe();
    this.stopBgm();
    this.stopBgs();
    this.stopSe();
  }

  static saveBgm() {
    if (this._currentBgm) {
      const bgm = this._currentBgm;
      return {
        name: bgm.name,
        volume: bgm.volume,
        pitch: bgm.pitch,
        pan: bgm.pan,
        pos: this._bgmBuffer ? this._bgmBuffer.seek() : 0
      };
    } else {
      return this.makeEmptyAudioObject();
    }
  }

  static saveBgs() {
    if (this._currentBgs) {
      const bgs = this._currentBgs;
      return {
        name: bgs.name,
        volume: bgs.volume,
        pitch: bgs.pitch,
        pan: bgs.pan,
        pos: this._bgsBuffer ? this._bgsBuffer.seek() : 0
      };
    } else {
      return this.makeEmptyAudioObject();
    }
  }

  static makeEmptyAudioObject() {
    return { name: '', volume: 0, pitch: 0 };
  }

  static createBuffer(folder, name) {
    const ext = this.audioFileExt();
    const url = `${this._path + folder}/${encodeURIComponent(name)}${ext}`;
    if (this.shouldUseHtml5Audio() && folder === 'bgm') {
      if (this._blobUrl) {
        Html5Audio.setup(this._blobUrl);
      }
      else {
        Html5Audio.setup(url);
      } 
      
      return Html5Audio;
    } else {
      return new WebAudio(url);
    }
  }

  static updateBufferParameters(buffer, configVolume, audio) {
    if (buffer && audio) {
      buffer.volume = configVolume * (audio.volume || 0) / 10000;
      buffer.pitch = (audio.pitch || 100) / 100;
      buffer.pan = (audio.pan || 0) / 100;
    }
  }

  static audioFileExt() {
    if (WebAudio.canPlayOgg() && !Utils.isMobileDevice()) {
      return '.ogg';
    } else {
      return '.m4a';
    }
  }

  static shouldUseHtml5Audio() {
    // We use HTML5 Audio to play BGM instead of Web Audio API
    // because decodeAudioData() is very slow on Android Chrome.
    return Utils.isAndroidChrome();
  }

  static checkErrors() {
    this.checkWebAudioError(this._bgmBuffer);
    this.checkWebAudioError(this._bgsBuffer);
    this.checkWebAudioError(this._meBuffer);
    this._seBuffers.forEach(buffer => {
      this.checkWebAudioError(buffer);
    });
    this._staticBuffers.forEach(buffer => {
      this.checkWebAudioError(buffer);
    });
  }

  static checkWebAudioError(webAudio) {
    if (webAudio && webAudio.isError()) {
      throw new Error(`Failed to load: ${webAudio.url}`);
    }
  }
}

Audio._masterVolume = 0.2;   // (min: 0, max: 1)
WebAudio.setMasterVolume(Audio._masterVolume);
Graphics.setVideoVolume(Audio._masterVolume);
Audio._bgmVolume = 60;
Audio._bgsVolume = 60;
Audio._meVolume = 60;
Audio._seVolume = 60;
Audio._currentBgm = null;
Audio._currentBgs = null;
Audio._bgmBuffer = null;
Audio._bgsBuffer = null;
Audio._meBuffer = null;
Audio._seBuffers = [];
Audio._staticBuffers = [];
Audio._replayFadeTime = 0.5;
Audio._path = 'audio/';
Audio._blobUrl = null;

module.exports = {
  Audio,
};