//-----------------------------------------------------------------------------
/**
 * The window in the game.
 *
 * @class Window
 * @constructor
 */
class Window extends PIXI.Container {
  constructor() {
    super(...arguments);
    this.initialize(...arguments);
  }

  initialize() {
    /**
     * The origin point of the window for scrolling.
     *
     * @property origin
     * @type Point
     */
    this.origin = new Point();

    this._isWindow = true;
    this._windowskin = null;
    this._width = 0;
    this._height = 0;
    this._animationCount = 0;

    this._padding = this.standardPadding();
    this._paddingV = this.verticalPadding();
    this._paddingH = this.horizontalPadding();
    this._margin = 4;

    this._createAllParts();
  }

  /**
   * The image used as a window skin.
   *
   * @property windowskin
   * @type Bitmap
   */
  get windowskin() {
    return this._windowskin;
  }

  set windowskin(value) {
    if (this._windowskin !== value) {
      this._windowskin = value;
      this._windowskin.addLoadListener(this._onWindowskinLoad.bind(this));
    }
  }

  /**
   * The bitmap used for the window contents.
   *
   * @property contents
   * @type Bitmap
   */
  get contents() {
    return this._windowContentsSprite.bitmap;
  }

  set contents(value) {
    this._windowContentsSprite.bitmap = value;
  }

  /**
   * The width of the window in pixels.
   *
   * @property width
   * @type Number
   */
  get width() {
    return this._width;
  }

  set width(value) {
    this._width = value;
    this._refreshAllParts();
  }

  /**
   * The height of the window in pixels.
   *
   * @property height
   * @type Number
   */
  get height() {
    return this._height;
  }

  set height(value) {
    this._height = value;
    this._refreshAllParts();
  }

  /**
   * The size of the padding between the frame and contents.
   *
   * @property padding
   * @type Number
   */
  get padding() {
    return this._padding;
  }

  set padding(value) {
    this._padding = value;
    this._paddingV = value;
    this._paddingH = value;
    this._refreshAllParts();
  }

  get paddingV() {
    return this._paddingV;
  }
  set paddingV(value) {
    this._paddingV = value;
    this._refreshAllParts();
  }

  get paddingH() {
    return this._paddingH;
  }
  set paddingH(value) {
    this._paddingH = value;
    this._refreshAllParts();
  }

  /**
   * The size of the margin for the window background.
   *
   * @property margin
   * @type Number
   */
  get margin() {
    return this._margin;
  }

  set margin(value) {
    this._margin = value;
    this._refreshAllParts();
  }

  /**
   * Updates the window for each frame.
   *
   * @method update
   */
  update() {
    if (this.active) {
      this._animationCount += Managers.Scenes._gameSpeed;
    }
    this.children.forEach(child => {
      if (child.update) {
        child.update();
      }
    });
  }

  /**
   * Sets the x, y, width, and height all at once.
   *
   * @method move
   * @param {Number} x The x coordinate of the window
   * @param {Number} y The y coordinate of the window
   * @param {Number} width The width of the window
   * @param {Number} height The height of the window
   */
  move(x, y, width, height) {
    this.x = x || 0;
    this.y = y || 0;

    if (this._width !== width || this._height !== height) {
      this._width = width || 0;
      this._height = height || 0;
      this._refreshAllParts();
    }
  }

  /**
   * @method updateTransform
   * @private
   */
  updateTransform() {
    this._updateContents();
    PIXI.Container.prototype.updateTransform.call(this);
  }

  /**
   * @method _createAllParts
   * @private
   */
  _createAllParts() {
    this._windowContentsSprite = new Sprite();
    this.addChild(this._windowContentsSprite);
  }

  /**
   * @method _onWindowskinLoad
   * @private
   */
  _onWindowskinLoad() {
    this._refreshAllParts();
  }

  /**
   * @method _refreshAllParts
   * @private
   */
  _refreshAllParts() {
    this._refreshContents();
  }

  /**
   * @method _refreshContents
   * @private
   */
  _refreshContents() {
    this._windowContentsSprite.move(this.paddingH, this.paddingV);
  }

  /**
   * @method _updateContents
   * @private
   */
  _updateContents() {
    const w = this._width - this._paddingH * 2;
    const h = this._height - this._paddingV * 2;
    if (w > 0 && h > 0) {
      this._windowContentsSprite.setFrame(this.origin.x, this.origin.y, w, h);
      this._windowContentsSprite.visible = this.isOpen();
    } else {
      this._windowContentsSprite.visible = false;
    }
  }

  standardPadding() {
    return 0;
  }

  horizontalPadding() {
    return this.standardPadding();
  }

  verticalPadding() {
    return this.standardPadding();
  }

  isOpen() {
    return true;
  }

  // The important members from Pixi.js

  /**
   * The visibility of the window.
   *
   * @property visible
   * @type Boolean
   */

  /**
   * The x coordinate of the window.
   *
   * @property x
   * @type Number
   */

  /**
   * The y coordinate of the window.
   *
   * @property y
   * @type Number
   */

  /**
   * [read-only] The array of children of the window.
   *
   * @property children
   * @type Array
   */

  /**
   * [read-only] The object that contains the window.
   *
   * @property parent
   * @type Object
   */

  /**
   * Adds a child to the container.
   *
   * @method addChild
   * @param {Object} child The child to add
   * @return {Object} The child that was added
   */

  /**
   * Adds a child to the container at a specified index.
   *
   * @method addChildAt
   * @param {Object} child The child to add
   * @param {Number} index The index to place the child in
   * @return {Object} The child that was added
   */

  /**
   * Removes a child from the container.
   *
   * @method removeChild
   * @param {Object} child The child to remove
   * @return {Object} The child that was removed
   */

  /**
   * Removes a child from the specified index position.
   *
   * @method removeChildAt
   * @param {Number} index The index to get the child from
   * @return {Object} The child that was removed
   */
}

Core.Window = Window;
module.exports = Window;
