require('./JsExtensions');

//-----------------------------------------------------------------------------
/**
 * The static class that defines utility methods.
 *
 * @class Utils
 */
function Utils() {
  throw new Error('This is a static class');
}

/**
 * The name of the RPG Maker. 'MV' in the current version.
 *
 * @static
 * @property RPGMAKER_NAME
 * @type String
 * @final
 */
Utils.RPGMAKER_NAME = 'MV';

/**
 * Checks whether the option is in the query string.
 *
 * @static
 * @method isOptionValid
 * @param {String} name The option name
 * @return {Boolean} True if the option is in the query string
 */
Utils.isOptionValid = function(name) {
  if (location.search.slice(1).split('&').contains(name)) {
    return true;
  }

  if (typeof nw !== "undefined") {
    if (nw.App.argv.length > 0) {
      if (nw.App.argv[0].split('$').contains(name)) {
        return true;
      }
    }
  }

  if (window.gui && window.gui.App && window.gui.App.argv) {
    if (window.gui.App.argv.indexOf(`--${ name }`) >= 0) {
      return true;
    }
  }

  return false;
};

Utils._isTestMode = Utils.isOptionValid('test');
Utils._isAutoTestMode = Utils.isOptionValid('auto-test');

/**
 * Checks whether the platform is NW.js.
 *
 * @static
 * @method isNwjs
 * @return {Boolean} True if the platform is NW.js
 */
Utils.isNwjs = function() {
  return typeof require === 'function' && typeof process === 'object';
};

Utils.isCordova = function() {
  return false;
};

/**
 * Checks whether the platform is a mobile device.
 *
 * @static
 * @method isMobileDevice
 * @return {Boolean} True if the platform is a mobile device
 */
Utils.isMobileDevice = function() {
  var r = /Android|webOS|iPhone|iPad|iPod|BlackBerry|IEMobile|Opera Mini/i;
  return !!navigator.userAgent.match(r);
};

/**
 * Checks whether the browser is Mobile Safari.
 *
 * @static
 * @method isMobileSafari
 * @return {Boolean} True if the browser is Mobile Safari
 */
Utils.isMobileSafari = function() {
  if (this.__isMobileSafari === undefined) {
    var agent = navigator.userAgent;
    this.__isMobileSafari = !!(agent.match(/iPhone|iPad|iPod/) && agent.match(/AppleWebKit/) && !agent.match('CriOS'));
  }

  return this.__isMobileSafari;
};

/**
 * Checks whether the browser is Android Chrome.
 *
 * @static
 * @method isAndroidChrome
 * @return {Boolean} True if the browser is Android Chrome
 */
Utils.isAndroidChrome = function() {
  var agent = navigator.userAgent;
  return !!(agent.match(/Android/) && agent.match(/Chrome/));
};

/**
 * Checks whether the browser is IE Mobile.
 *
 * @static
 * @method isIEMobile
 * @return {Boolean} True if the browser is IE Mobile
 */
Utils.isIEMobile = function() {
  var agent = navigator.userAgent;
  return !!(agent.match(/IEMobile/i));
};


/**
 * Checks whether the browser can read files in the game folder.
 *
 * @static
 * @method canReadGameFiles
 * @return {Boolean} True if the browser can read files in the game folder
 */
Utils.canReadGameFiles = function() {
  var scripts = document.getElementsByTagName('script');
  var lastScript = scripts[scripts.length - 1];
  var xhr = new XMLHttpRequest();
  try {
    xhr.open('GET', lastScript.src);
    xhr.overrideMimeType('text/javascript');
    xhr.send();
    return true;
  } catch (e) {
    return false;
  }
};

/**
 * Makes a CSS color string from RGB values.
 *
 * @static
 * @method rgbToCssColor
 * @param {Number} r The red value in the range (0, 255)
 * @param {Number} g The green value in the range (0, 255)
 * @param {Number} b The blue value in the range (0, 255)
 * @return {String} CSS color string
 */
Utils.rgbToCssColor = function(r, g, b) {
  r = Math.round(r);
  g = Math.round(g);
  b = Math.round(b);
  return 'rgb(' + r + ',' + g + ',' + b + ')';
};

Utils.hexToRgb = function(hex) {
  var result = /^#?([a-f\d]{2})([a-f\d]{2})([a-f\d]{2})$/i.exec(hex);
  return result ? {
    red : parseInt(result[1], 16),
    green : parseInt(result[2], 16),
    blue : parseInt(result[3], 16)
  } : null;
};

Utils.uniqueId = function() {
  return (Date.now()).toString() + ':' + Utils.fakeGuid();
};

Utils.fakeGuid = function(){
  // var d0 = Math.random()*0xffffffff|0;
  // var d1 = Math.random()*0xffffffff|0;
  // var d2 = Math.random()*0xffffffff|0;
  // var d3 = Math.random()*0xffffffff|0;
  // return lut[d0&0xff]+lut[d0>>8&0xff]+lut[d0>>16&0xff]+lut[d0>>24&0xff]+'-'+ lut[d1&0xff]+lut[d1>>8&0xff]+'-'+lut[d1>>16&0x0f|0x40]+lut[d1>>24&0xff]+'-'+ lut[d2&0x3f|0x80]+lut[d2>>8&0xff]+'-'+lut[d2>>16&0xff]+lut[d2>>24&0xff]+ lut[d3&0xff]+lut[d3>>8&0xff]+lut[d3>>16&0xff]+lut[d3>>24&0xff];

  function s4() {
    return Math.floor((1 + Math.random()) * 0x10000).toString(16).substring(1);
  }

  return s4() + s4() + '-' + s4() + '-' + s4() + '-' + s4() + '-' + s4() + s4() + s4();
};

Utils.enableLargeMapshots = function() {
  $gameTemp._enableMapshots = true;
};

Utils.disableLargeMapshots = function() {
  $gameTemp._enableMapshots = false;
};

Utils.getCountryCode = function() {
  if (!Utils.isNwjs()) return 0;

  try {
    var child = require('child_process');
    var output = child.execSync('wmic os get countrycode').toString();

    var code = output.match(/[0-9]+/gi);

    if (code.length > 0) {
      return parseInt(code[0], 10);
    }

    return 0;
  }
  catch(e) {
    return 0;
  }
};

// The insignificantValue is used to decrease from the 'right' and 'bottom' positions of the hitboxes, so that those position do not "flow" to the next integer value
// Example:  Left  = 10, Top = 15, Right = 10.999999, Bottom = 15.999999 instead of Right = 11 and Bottom = 16
window.insignificantValue = 0.000001;

Utils._id = 1;
Utils.generateRuntimeId = function(){
  return Utils._id++;
};

Utils._supportPassiveEvent = null;
/**
 * Test this browser support passive event feature
 *
 * @static
 * @method isSupportPassiveEvent
 * @return {Boolean} this browser support passive event or not
 */
Utils.isSupportPassiveEvent = function() {
  if (typeof Utils._supportPassiveEvent === "boolean") {
    return Utils._supportPassiveEvent;
  }
  // test support passive event
  // https://github.com/WICG/EventListenerOptions/blob/gh-pages/explainer.md#feature-detection
  var passive = false;
  var options = Object.defineProperty({}, "passive", {
    get: function() { passive = true; return true; }
  });
  window.addEventListener("test", null, options);
  Utils._supportPassiveEvent = passive;
  return passive;
};


Utils.evaluatesToTrue = function(value) {
  if (!value) return false;

  if (typeof value == 'string') {
    value = value.trim().toUpperCase();

    if (value == 'ON') return true;
    if (value == 'OFF') return false;
    if (value == 'YES') return true;
    if (value == 'NO') return false;
    if (value == 'TRUE') return true;
    if (value == 'T') return true;
    if (value == 'Y') return true;
    if (value == '1') return true;
  }

  try {
    if (eval(value)) return true; //jshint ignore:line
  }
  catch(e) {
    return false;
  }

  return false;
};

Utils.isNaN = function(value) {
  if (isNaN(value)) return true;

  var valueType = typeof value;
  if (valueType !== 'number' && valueType !== 'string') return true;

  if (valueType == 'string') {
    if (value.trim() === '') return true;
  }

  return false;
};

Utils.isNumber = function(value) {
  return !Utils.isNaN(value);
};

Utils.assignProperties = function(receiver, properties, propertiesToSkip) {
  for (var key in properties) {
    if (!properties.hasOwnProperty(key)) continue;

    if (propertiesToSkip && propertiesToSkip.indexOf(key) >= 0) continue;

    receiver[key] = properties[key];
  }
};

Utils.mergeObjects = function(...args) {
  const mergedObj = {};

  for (let i in args) {
    const obj = args[i];
    if (!obj) continue;

    for (let key in obj) {
      if (!obj.hasOwnProperty(key)) continue;

      mergedObj[key] = obj[key];
    }
  }

  return mergedObj;
};

Utils.chance = function(chance) {
  var luck = Math.randomInt(100);
  return Boolean(luck < chance);
};

Utils.whatever = function() {
  return Utils.chance(Math.randomInt(100));
};

Utils.getResolutionZoomLevel = function(resolutionName, zoomType) {
  var zoomLevel = 2;

  switch(resolutionName) {
    case '8k' :
      zoomLevel = zoomType === 'out' ? 8 : 12;
      break;
    case '4k' :
      zoomLevel = zoomType === 'out' ? 4 : 6;
      break;
    case '2880' :
      zoomLevel = zoomType === 'out' ? 3 : 5;
      break;
    case '2560' :
      zoomLevel = zoomType === 'out' ? 2 : 4;
      break;
    case 'full-hd' :
    case 'mac-full-hd' :
      zoomLevel = zoomType === 'out' ? 2 : 3;
      break;
    case '1366' :
    case '1360' :
      zoomLevel = 2;
      break;
    case 'medium' :
    case '1600' :
      zoomLevel = zoomType === 'out' ? 2 : 3;
      break;
    case 'mobile' :
    case 'small' :
      zoomLevel = 1;
      break;
  }

  return zoomLevel;
};

Utils.getResolutionHudZoomLevel = function(resolutionName, smallHud) {
  switch(resolutionName) {
    case '8k' :
      return smallHud ? 8 : 12;
    case '4k' :
      return smallHud ? 4 : 6;
    case '2880' :
      return smallHud ? 3 : 5;
    case '2560' :
      return smallHud ? 2 : 4;
    case 'full-hd' :
    case 'mac-full-hd' :
      return smallHud ? 2 : 3;
    case '1366' :
    case '1360' :
      return smallHud ? 1 : 2;
    case 'medium' :
      return smallHud ? 2 : 3;
    case 'mobile' :
    case 'small' :
      return 1;
    default :
      return smallHud ? 1 : 2;
  }
};

Utils.getResolutionWidth = function(resolutionName) {
  switch(resolutionName) {
    case '8k' :
      return 7680;
    case '4k' :
      return 3840;
    case '2560' :
      return 2560;
    case 'full-hd' :
    case 'mac-full-hd' :
      return 1920;
    case '1366' :
      return 1366;
    case '1360' :
      return 1360;
    case '1600' :
      return 1600;
    case 'mobile' :
      return 864;
    case 'small' :
      return 864;
    case 'disabled' :
    case 'medium' :
      return 1280;
    case '2880' :
      return 2880;
    default :
      return 1280;
  }
};

Utils.getResolutionHeight = function(resolutionName) {
  switch(resolutionName) {
    case '8k' :
      return 4320;
    case '4k' :
      return 2160;
    case '2560' :
    case 'full-hd' :
      return 1080;
    case 'mac-full-hd' :
      return 1000;
    case '1366' :
    case '1360' :
      return 768;
    case '1600' :
      return 900;
    case 'mobile' :
      return 486;
    case 'small' :
      return 486;
    case 'medium' :
      return 960;
    case '2880' :
      return 1800;
    case 'disabled' :
      return 720;
    default :
      return 720;
  }
};

Utils.windowSizeFitsScreenAndResolution = function(windowSize, selectedResolution) {
  var width = Utils.getResolutionWidth(windowSize);
  var height = Utils.getResolutionHeight(windowSize);
  var availWidth = screen.availWidth * (window.devicePixelRatio || 1);
  var availHeight = screen.availHeight * (window.devicePixelRatio || 1);

  if (width > availWidth) return false;
  if (height > availHeight) return false;

  if (selectedResolution) {
    var resWidth = Utils.getResolutionWidth(selectedResolution);
    var resHeight = Utils.getResolutionHeight(selectedResolution);

    if (width < resWidth) return false;
    if (height < resHeight) return false;
  }

  return true;
};

Utils.getResolutionList = function() {
  if (Utils.isMobileDevice()) {
    return ['mobile'];
  }

  const completeWidth = Math.max(screen.width, screen.width * (window.devicePixelRatio || 1));

  var list = [];

  list.push('hd');

  if (completeWidth >= 1920) {
    list.push('full-hd');
  }

  // if (completeWidth >= 1600) {
  //   list.push('1600');
  // }

  if (completeWidth >= 3840) {
    list.push('4k');
  }

  if (completeWidth >= 7680) {
    list.push('8k');
  }

  return list;
};

Utils.getWindowSizeList = function() {
  if (Utils.isMobileDevice()) {
    return ['disabled'];
  }

  var resolutions = [];
  var list = [];
  const completeWidth = screen.width * (window.devicePixelRatio || 1);

  resolutions.push('hd');
  if (completeWidth >= 1920) {
    resolutions.push('full-hd');
  }

  // if (completeWidth >= 1600) {
  //   resolutions.push('1600');
  // }

  if (completeWidth >= 3840) {
    list.push('4k');
  }

  if (completeWidth >= 7680) {
    list.push('8k');
  }

  resolutions.forEach(function(r){
    if (Utils.windowSizeFitsScreenAndResolution(r, Managers.Config.resolution)) {
      list.push(r);
    }
  });

  if (!list.length) {
    return ['disabled'];
  }
  return list;
};

Utils.getNextOnList = function(list, currentItem) {
  var idx = list.indexOf(currentItem) + 1;

  if (idx >= list.length) idx = 0;
  return list[idx];
};

Utils.getPreviousOnList = function(list, currentItem) {
  var idx = list.indexOf(currentItem);

  if (idx < 0) {
    idx = 0;
  } else if (idx === 0) {
    idx = list.length -1;
  } else {
    idx--;
  }

  return list[idx];
};

Utils.getNextResolution = function(currentResolution) {
  var list = Utils.getResolutionList();
  return Utils.getNextOnList(list, currentResolution);
};

Utils.getPreviousResolution = function(currentResolution) {
  var list = Utils.getResolutionList();
  return Utils.getPreviousOnList(list, currentResolution);
};

Utils.getNextWindowSize = function(currentWindowSize) {
  var list = Utils.getWindowSizeList();
  if (list.length) {
    return Utils.getNextOnList(list, currentWindowSize);
  } else {
    return 'default';
  }
};

Utils.getPreviousWindowSize = function(currentWindowSize) {
  var list = Utils.getWindowSizeList();
  if (list.length) {
    return Utils.getPreviousOnList(list, currentWindowSize);
  } else {
    return 'default';
  }
};

Utils.getFrameCount = function(frames) {
  return Math.ceil(frames / Managers.Scenes._gameSpeed);
};

Utils.getProjectFolder = function() {
  if (!Utils.isNwjs()) {
    return '';
  }

  return Wrapper.dirname(process.mainModule.filename).replace(/%20/, ' ');
};

Utils.getFileNameFromPath = function(filePath) {
  return Wrapper.basename(filePath);
};

Utils.isValidFileName = function(fileName) {
  if (!fileName) return false;
  if (fileName.length > 255) return false;

  const regex = /[<>:"/\\|?*\x00-\x1F]/g;
  const forbiddenNames = /^(con|prn|aux|nul|com[0-9]|lpt[0-9])$/i;

  if (regex.test(fileName)) {
    return false;
  }
  if (forbiddenNames.test(fileName)) {
    return false;
  }

  if (/^\.\.?$/.test(fileName)) {
    return false;
  }

  return true;
};

Utils.convertFontSizeToZoom3 = function(originalFontSize) {
  const result = 5 + Math.round(originalFontSize / 3 * 4);
  return result;
};

Utils.convertFontSizeToZoom1 = function(originalFontSize) {
  const result = Math.round(originalFontSize / 3 * 2);
  return result;
};

Utils.convertFontSizeToZoom4 = function(originalFontSize) {
  const result = 5 + Math.round(originalFontSize / 3 * 5);
  return result;
};

Utils.convertFontSizeToZoom5 = function(originalFontSize) {
  const result = 5 + Math.round(originalFontSize / 3 * 6);
  return result;
};

Utils.convertFontSizeToZoom6 = function(originalFontSize) {
  const result = 5 + Math.round(originalFontSize / 3 * 7);
  return result;
};

Utils.convertFontSizeToZoom12 = function(originalFontSize) {
  const result = 5 + Math.round(originalFontSize / 3 * 13);
  return result;
};

Utils.convertFontSize = function(originalFontSize, zoomLevel) {
  if (!zoomLevel) return originalFontSize;
  // Default font sizes are programmed for 2x zoom.

  switch(zoomLevel) {
    case 1:
      return Utils.convertFontSizeToZoom1(originalFontSize);
    case 2:
      return originalFontSize;
    case 3:
      return Utils.convertFontSizeToZoom3(originalFontSize);
    case 4:
      return Utils.convertFontSizeToZoom4(originalFontSize);
    case 5:
      return Utils.convertFontSizeToZoom5(originalFontSize);
    case 6:
      return Utils.convertFontSizeToZoom6(originalFontSize);
    case 12:
      return Utils.convertFontSizeToZoom12(originalFontSize);
    default:
      return originalFontSize;
  }
};

Utils.padNumber = function(value, length) {
  let paddedValue = typeof value == 'string' ? value : value.toString();
  while (paddedValue.length < length) {
    paddedValue = '0' + paddedValue;
  }

  return paddedValue;
};

Utils.compareValues = function(value1, value2, operation) {
  switch(operation) {
    case '==':
    case '=':
      return value1 == value2 || `"${ value1 }"` == value2 || `'${ value1 }'` == value2;
    case '!=':
    case '<>':
      return value1 != value2;
    case '>=':
      return value1 >= parseFloat(value2, 10);
    case '<=':
      return value1 <= parseFloat(value2, 10);
    case '<':
      return value1 < parseFloat(value2, 10);
    case '>':
      return value1 > parseFloat(value2, 10);
    case 'includes':
      return String(value1).includes(value2);
    case 'startsWith':
      return String(value1).startsWith(value2);
    case 'endsWith':
      return String(value1).endsWith(value2);
  }

  throw new Error(`Invalid Variable Operation: ${ operation }`);
};


class UpdateError extends Error {
  constructor(versionNumber, message) {
    super(message);
    this.versionNumber = versionNumber;
  }

  versionNumberStr() {
    if (!this.versionNumber) {
      return '';
    }

    try {
      let str = '';
      const floor = Math.floor(this.versionNumber);
      if (floor < 1) {
        str += '0.';
      } else {
        str += String(floor) + '.';
      }

      let remaining = (this.versionNumber - floor) * 10;
      const major = Math.floor(remaining);
      str += String(major) + '.';

      remaining = (remaining - major) * 10;
      const minor = Math.floor(remaining);
      str += String(minor) + '.';

      const build = Math.floor((remaining - minor) * 100);
      str += build.padZero(2);

      return str;
    } catch(e) {
      return String(this.versionNumber);
    }
  }
}

Utils.UpdateError = UpdateError;

module.exports = {
  Utils,
};