//-----------------------------------------------------------------------------
/**
 * The static class that handles input data from the mouse and touchscreen.
 *
 * @class TouchInput
 */
function TouchInput() {
  throw new Error('This is a static class');
}

/**
 * Initializes the touch system.
 *
 * @static
 * @method initialize
 */
TouchInput.initialize = function() {
  this.clear();
  this._setupEventHandlers();
};

/**
 * The wait time of the pseudo key repeat in frames.
 *
 * @static
 * @property keyRepeatWait
 * @type Number
 */
TouchInput.keyRepeatWait = 24;

/**
 * The interval of the pseudo key repeat in frames.
 *
 * @static
 * @property keyRepeatInterval
 * @type Number
 */
TouchInput.keyRepeatInterval = 6;

TouchInput._mouseCorrectionX = 6;
TouchInput._mouseCorrectionY = 0;

/**
 * Clears all the touch data.
 *
 * @static
 * @method clear
 */
TouchInput.clear = function() {
  this._mouseUsed = false;
  this._mouseClickedRecently = false;
  this._mousePressed = false;
  this._screenPressed = false;
  this._pressedTime = 0;
  this._events = {};
  this._events.triggered = false;
  this._events.cancelled = false;
  this._events.moved = false;
  this._events.released = false;
  this._events.wheelX = 0;
  this._events.wheelY = 0;
  this._events.rightTriggered = false;
  this._events.rightReleased = true;
  this._triggered = false;
  this._rightTriggered = false;
  this._cancelled = false;
  this._moved = false;
  this._released = false;
  this._rightReleased = false;
  this._wheelX = 0;
  this._wheelY = 0;
  this._x = -1000;
  this._y = -1000;
  this._rightX = -1000;
  this._rightY = -1000;
  this._mouseX = -1000;
  this._mouseY = -1000;
  this._date = 0;
};

TouchInput.isMouseInsideWindow = function() {
  var style = getComputedStyle( document.body );
  var cursor = style.getPropertyValue('cursor');

  if (cursor.indexOf('cursor-active') < 0 && cursor.indexOf('default') < 0) {
    return false;
  }

  return true;
};


/**
 * Updates the touch data.
 *
 * @static
 * @method update
 */
TouchInput.update = function() {

  if (this._mousePressed || this._rightMousePressed) {
    var style = getComputedStyle( document.body );
    var cursor = style.getPropertyValue('cursor');

    if (cursor.indexOf('cursor-active') < 0 && cursor.indexOf('default') < 0) {
      this._mousePressed = false;
      this._rightMousePressed = false;
    }
  }

  this._triggered = this._events.triggered;
  this._rightTriggered = this._events.rightTriggered;
  this._cancelled = this._events.cancelled;
  this._moved = this._events.moved;
  this._released = this._events.released;
  this._rightReleased = this._events.rightReleased;
  this._wheelX = this._events.wheelX;
  this._wheelY = this._events.wheelY;
  this._events.triggered = false;
  this._events.cancelled = false;
  this._events.rightTriggered = false;
  this._events.moved = false;
  this._events.released = false;
  this._events.rightReleased = false;
  this._events.wheelX = 0;
  this._events.wheelY = 0;

  if (this.isPressed()) {
    this._pressedTime++;
  }

  if (this.isRightPressed()) {
    this._rightPressedTime++;
  }
};

/**
 * Checks whether the mouse button or touchscreen is currently pressed down.
 *
 * @static
 * @method isPressed
 * @return {Boolean} True if the mouse button or touchscreen is pressed
 */
TouchInput.isPressed = function() {
  return this._mousePressed || this._screenPressed;
};

TouchInput.isRightPressed = function() {
  return this._rightMousePressed;
};

/**
 * Checks whether the left mouse button or touchscreen is just pressed.
 *
 * @static
 * @method isTriggered
 * @return {Boolean} True if the mouse button or touchscreen is triggered
 */
TouchInput.isTriggered = function() {
  return this._triggered;
};

TouchInput.isRightTriggered = function() {
  return this._rightTriggered;
};

/**
 * Checks whether the left mouse button or touchscreen is just pressed
 * or a pseudo key repeat occurred.
 *
 * @static
 * @method isRepeated
 * @return {Boolean} True if the mouse button or touchscreen is repeated
 */
TouchInput.isRepeated = function() {
  return (this.isPressed() &&
            (
              this._triggered ||
              (this._pressedTime >= this.keyRepeatWait && this._pressedTime % this.keyRepeatInterval === 0)
            )
  );
};

TouchInput.isRightRepeated = function() {
  return (this.isRightPressed() && (this._rightTriggered || this._rightPressedTime >= this.keyRepeatWait && this._rightPressedTime % this.keyRepeatInterval === 0));
};

/**
 * Checks whether the left mouse button or touchscreen is kept depressed.
 *
 * @static
 * @method isLongPressed
 * @return {Boolean} True if the left mouse button or touchscreen is long-pressed
 */
TouchInput.isLongPressed = function() {
  return this.isPressed() && this._pressedTime >= this.keyRepeatWait;
};

TouchInput.isRightLongPressed = function() {
  return this.isRightPressed() && this._rightPressedTime >= this.keyRepeatWait;
};

/**
 * Checks whether the right mouse button is just pressed.
 *
 * @static
 * @method isCancelled
 * @return {Boolean} True if the right mouse button is just pressed
 */
TouchInput.isCancelled = function() {
  return this._cancelled;
};

/**
 * Checks whether the mouse or a finger on the touchscreen is moved.
 *
 * @static
 * @method isMoved
 * @return {Boolean} True if the mouse or a finger on the touchscreen is moved
 */
TouchInput.isMoved = function() {
  return this._moved;
};

/**
 * Checks whether the left mouse button or touchscreen is released.
 *
 * @static
 * @method isReleased
 * @return {Boolean} True if the mouse button or touchscreen is released
 */
TouchInput.isReleased = function() {
  return this._released;
};

TouchInput.isRightReleased = function() {
  return this._rightReleased;
};

/**
 * [read-only] The horizontal scroll amount.
 *
 * @static
 * @property wheelX
 * @type Number
 */
Object.defineProperty(TouchInput, 'wheelX', {
  get: function() {
    return this._wheelX;
  },
  configurable: true
});

/**
 * [read-only] The vertical scroll amount.
 *
 * @static
 * @property wheelY
 * @type Number
 */
Object.defineProperty(TouchInput, 'wheelY', {
  get: function() {
    return this._wheelY;
  },
  configurable: true
});

/**
 * [read-only] The x coordinate on the canvas area of the latest touch event.
 *
 * @static
 * @property x
 * @type Number
 */
Object.defineProperty(TouchInput, 'x', {
  get: function() {
    if (!this._mouseUsed) return -1000;
    return this._x + TouchInput._mouseCorrectionX;
  },
  configurable: true
});

/**
 * [read-only] The y coordinate on the canvas area of the latest touch event.
 *
 * @static
 * @property y
 * @type Number
 */
Object.defineProperty(TouchInput, 'y', {
  get: function() {
    if (!this._mouseUsed) return -1000;
    return this._y + TouchInput._mouseCorrectionY;
  },
  configurable: true
});


Object.defineProperty(TouchInput, 'mouseX', {
  get: function() {
    if (!this._mouseUsed) return -1000;
    return this._mouseX + TouchInput._mouseCorrectionX;
  },
  configurable: true
});

Object.defineProperty(TouchInput, 'mouseY', {
  get: function() {
    if (!this._mouseUsed) return -1000;
    return this._mouseY + TouchInput._mouseCorrectionY;
  },
  configurable: true
});

Object.defineProperty(TouchInput, 'rightX', {
  get: function() {
    if (!this._mouseUsed) return -1000;
    return this._rightX + TouchInput._mouseCorrectionX;
  },
  configurable: true
});

Object.defineProperty(TouchInput, 'rightY', {
  get: function() {
    if (!this._mouseUsed) return -1000;
    return this._rightY + TouchInput._mouseCorrectionY;
  },
  configurable: true
});

/**
* [read-only] The mouse button that was pressed.
*/
Object.defineProperty(TouchInput, 'button', {
  get: function() {
    return this._button;
  },
  configurable: true
});

/**
 * [read-only] The time of the last input in milliseconds.
 *
 * @static
 * @property date
 * @type Number
 */
Object.defineProperty(TouchInput, 'date', {
  get: function() {
    return this._date;
  },
  configurable: true
});

/**
 * @static
 * @method _setupEventHandlers
 * @private
 */
TouchInput._setupEventHandlers = function() {
  var isSupportPassive = Utils.isSupportPassiveEvent();
  
  document.addEventListener('mousedown', this._onMouseDown.bind(this));
  document.addEventListener('mousemove', this._onMouseMove.bind(this));
  document.addEventListener('mouseup', this._onMouseUp.bind(this));
  document.addEventListener('wheel', this._onWheel.bind(this));
  document.addEventListener('touchstart', this._onTouchStart.bind(this), isSupportPassive ? {passive: false} : false);
  document.addEventListener('touchmove', this._onTouchMove.bind(this), isSupportPassive ? {passive: false} : false);
  document.addEventListener('touchend', this._onTouchEnd.bind(this));
  document.addEventListener('touchcancel', this._onTouchCancel.bind(this));
  document.addEventListener('pointerdown', this._onPointerDown.bind(this));
  
  window.addEventListener('blur', this._onWindowBlur.bind(this));
};

/**
 * @static
 * @method _onMouseDown
 * @param {MouseEvent} event
 * @private
 */
TouchInput._onMouseDown = function(event) {
  if (!Managers.Config.enableMouse) return;

  this.setMouseUsed();
  this.setMouseClicked();
  Graphics.updateBodyClass();

  if (event.button === 0) {
    this._onLeftButtonDown(event);
  } else if (event.button === 1) {
    this._onMiddleButtonDown(event);
  } else if (event.button === 2) {
    this._onRightButtonDown(event);
  }
};

TouchInput._onWindowBlur = function(event) {
  this._mousePressed = false;
};

/**
 * @static
 * @method _onLeftButtonDown
 * @param {MouseEvent} event
 * @private
 */
TouchInput._onLeftButtonDown = function(event) {
  var x = Graphics.pageToCanvasX(event.pageX);
  var y = Graphics.pageToCanvasY(event.pageY);
  if (Graphics.isInsideCanvas(x, y)) {
    this._mousePressed = true;
    this._pressedTime = 0;
    this._onTrigger(x, y, event.button);
  }
};

/**
 * @static
 * @method _onMiddleButtonDown
 * @param {MouseEvent} event
 * @private
 */
TouchInput._onMiddleButtonDown = function(event) {
};

/**
 * @static
 * @method _onRightButtonDown
 * @param {MouseEvent} event
 * @private
 */
TouchInput._onRightButtonDown = function(event) {
  var x = Graphics.pageToCanvasX(event.pageX);
  var y = Graphics.pageToCanvasY(event.pageY);
  
  if (Graphics.isInsideCanvas(x, y)) {
    this._rightMousePressed = true;
    this._rightPressedTime = 0;
    this._onRightTrigger(x, y, event.button);
    // this._onCancel(x, y);
  }
};

TouchInput.wasMouseUsed = function() {
  return this._mouseUsed;
};

TouchInput.wasMouseClickedRecently = function() {
  return this._mouseClickedRecently;
};

TouchInput.setMouseClicked = function() {
  this._mouseClickedRecently = true;
  if (this._mouseClickedRecentlyTimeout) {
    clearTimeout(this._mouseClickedRecentlyTimeout);
  }

  this._mouseClickedRecentlyTimeout = setTimeout(function(){
    TouchInput._mouseClickedRecently = false;
    TouchInput._mouseClickedRecentlyTimeout = false;
  }, 1000);
};

TouchInput.setMouseUsed = function() {
  if (!TouchInput._mouseUsed) {
    TouchInput._mouseUsed = true;
    Graphics.updateBodyClass();
  }

  if (TouchInput._mouseUsedTimeout) {
    clearInterval(TouchInput._mouseUsedTimeout);
  }

  TouchInput._mouseUsedTimeout = setInterval(() => {
    if (!TouchInput.isPressed()) {
      clearInterval(TouchInput._mouseUsedTimeout);
      TouchInput._mouseUsedTimeout = false;

      TouchInput._mouseUsed = false;
      Graphics.updateBodyClass();
    }
  }, 3000);
};

/**
 * @static
 * @method _onMouseMove
 * @param {MouseEvent} event
 * @private
 */
TouchInput._onMouseMove = function(event) {
  if (!Managers.Config.enableMouse) return;

  var x = Graphics.pageToCanvasX(event.pageX);
  var y = Graphics.pageToCanvasY(event.pageY);

  if (TouchInput._mouseX >= 0 || TouchInput._mouseY >= 0) {
    if (x !== TouchInput._mouseX || y !== TouchInput._mouseY) {
      this.setMouseUsed();
    }
  }
  
  this.setMouseUsed();
  
  if (this._mousePressed) {
    this._onMove(x, y);
  }

  this._mouseX = x;
  this._mouseY = y;
};

TouchInput.clearMouseUsed = function() {
  this._mouseUsed = false;
  this._mouseX = -1000;
  this._mouseY = -1000;

  Graphics.updateBodyClass();
};

/**
 * @static
 * @method _onMouseUp
 * @param {MouseEvent} event
 * @private
 */
TouchInput._onMouseUp = function(event) {
  var x = Graphics.pageToCanvasX(event.pageX);
  var y = Graphics.pageToCanvasY(event.pageY);

  if (event.button === 0) {
    this._mousePressed = false;
    this._onRelease(x, y);
  } else if (event.button === 2) {
    this._rightMousePressed = false;
    this._onRightRelease(x, y);
  }
};

/**
 * @static
 * @method _onWheel
 * @param {WheelEvent} event
 * @private
 */
TouchInput._onWheel = function(event) {
  if (!Managers.Config.enableMouse) return;

  this._events.wheelX += event.deltaX;
  this._events.wheelY += event.deltaY;
  event.preventDefault();
};

/**
 * @static
 * @method _onTouchStart
 * @param {TouchEvent} event
 * @private
 */
TouchInput._onTouchStart = function(event) {
  for (var i = 0; i < event.changedTouches.length; i++) {
    var touch = event.changedTouches[i];
    var x = Graphics.pageToCanvasX(touch.pageX);
    var y = Graphics.pageToCanvasY(touch.pageY);
    if (Graphics.isInsideCanvas(x, y)) {
      this._screenPressed = true;
      this._pressedTime = 0;
      if (event.touches.length >= 2) {
        this._onCancel(x, y);
      } else {
        this._onTrigger(x, y);
      }
      event.preventDefault();
    }
  }
  if (window.cordova || window.navigator.standalone) {
    event.preventDefault();
  }
};

/**
 * @static
 * @method _onTouchMove
 * @param {TouchEvent} event
 * @private
 */
TouchInput._onTouchMove = function(event) {
  for (var i = 0; i < event.changedTouches.length; i++) {
    var touch = event.changedTouches[i];
    var x = Graphics.pageToCanvasX(touch.pageX);
    var y = Graphics.pageToCanvasY(touch.pageY);
    this._onMove(x, y);
  }
};

/**
 * @static
 * @method _onTouchEnd
 * @param {TouchEvent} event
 * @private
 */
TouchInput._onTouchEnd = function(event) {
  for (var i = 0; i < event.changedTouches.length; i++) {
    var touch = event.changedTouches[i];
    var x = Graphics.pageToCanvasX(touch.pageX);
    var y = Graphics.pageToCanvasY(touch.pageY);
    this._screenPressed = false;
    this._onRelease(x, y);
  }
};

/**
 * @static
 * @method _onTouchCancel
 * @param {TouchEvent} event
 * @private
 */
TouchInput._onTouchCancel = function(event) {
  this._screenPressed = false;
};

/**
 * @static
 * @method _onPointerDown
 * @param {PointerEvent} event
 * @private
 */
TouchInput._onPointerDown = function(event) {
  if (event.pointerType === 'touch' && !event.isPrimary) {
    var x = Graphics.pageToCanvasX(event.pageX);
    var y = Graphics.pageToCanvasY(event.pageY);
    if (Graphics.isInsideCanvas(x, y)) {
      // For Microsoft Edge
      this._onCancel(x, y);
      event.preventDefault();
    }
  }
};

/**
 * @static
 * @method _onTrigger
 * @param {Number} x
 * @param {Number} y
 * @private
 */
TouchInput._onTrigger = function(x, y, button) {
  this._events.triggered = true;
  this._x = x;
  this._y = y;
  this._mouseX = x;
  this._mouseY = y;
  this._date = Date.now();
  this._button = button;
};

TouchInput._onRightTrigger = function(x, y, button) {
  this._events.rightTriggered = true;
  this._rightX = x;
  this._rightY = y;
  this._rightDate = Date.now();
  this._button = button;
};

/**
 * @static
 * @method _onCancel
 * @param {Number} x
 * @param {Number} y
 * @private
 */
TouchInput._onCancel = function(x, y) {
  this._events.cancelled = true;
  this._x = x;
  this._y = y;
};

/**
 * @static
 * @method _onMove
 * @param {Number} x
 * @param {Number} y
 * @private
 */
TouchInput._onMove = function(x, y) {
  this._events.moved = true;
  this._x = x;
  this._y = y;
};

/**
 * @static
 * @method _onRelease
 * @param {Number} x
 * @param {Number} y
 * @private
 */
TouchInput._onRelease = function(x, y) {
  this._events.released = true;
  this._x = x;
  this._y = y;
};

TouchInput._onRightRelease = function(x, y) {
  this._events.rightReleased = true;
  this._rightX = x;
  this._rightY = y;
};