const Tilemap = require('./Tilemap');

//-----------------------------------------------------------------------------
/**
 * The tilemap which displays 2D tile-based game map using shaders
 *
 * @class Tilemap
 * @constructor
 */

class ShaderTilemap extends Tilemap {
  constructor() {
    super();
    this.roundPixels = false;
  }

  /**
   * Uploads animation state in renderer
   *
   * @method _hackRenderer
   * @private
   */
  _hackRenderer(renderer) {
    var af = this.animationFrame % 3;
    renderer.plugins.tilemap.tileAnim[0] = af * this._tileWidth;
    renderer.plugins.tilemap.tileAnim[1] = (this.animationFrame % 3) * this._tileHeight;
    return renderer;
  }

  /**
   * PIXI render method
   *
   * @method renderCanvas
   * @param {Object} pixi renderer
   */
  renderCanvas(renderer) {
    this._hackRenderer(renderer);
    PIXI.Container.prototype.renderCanvas.call(this, renderer);
  }


  /**
   * PIXI render method
   *
   * @method render
   * @param {Object} pixi renderer
   */
  render(renderer) {
    this._hackRenderer(renderer);
    PIXI.Container.prototype.render.call(this, renderer);
  }

  /**
   * Forces to repaint the entire tilemap AND update bitmaps list if needed
   *
   * @method refresh
   */
  refresh() {
    if (this._lastBitmapLength !== this.bitmaps.length) {
      this._lastBitmapLength = this.bitmaps.length;
      this.refreshTileset();
    }
    this._needsRepaint = true;
  }

  /**
   * Call after you update tileset
   *
   * @method updateBitmaps
   */
  refreshTileset() {
    var bitmaps = this.bitmaps.map(function(x) { return x._baseTexture ? new PIXI.Texture(x._baseTexture) : x; } );
    this.lowerLayer.setBitmaps(bitmaps);
    this.upperLayer.setBitmaps(bitmaps);
  }

  /**
   * @method updateTransform
   * @private
   */
  updateTransform() {
    var ox;
    var oy;

    if (this.roundPixels) {
      ox = Math.floor(this.origin.x);
      oy = Math.floor(this.origin.y);
    } else {
      ox = this.origin.x;
      oy = this.origin.y;
    }
    var startX = Math.floor((ox - this._margin) / this._tileWidth);
    var startY = Math.floor((oy - this._margin) / this._tileHeight);
    this._updateLayerPositions(startX, startY);

    if (this._needsRepaint || this._lastStartX !== startX || this._lastStartY !== startY) {
      this._lastStartX = startX;
      this._lastStartY = startY;
      this._paintAllTiles(startX, startY);
      this._needsRepaint = false;
    }
    this._sortChildren();
    PIXI.Container.prototype.updateTransform.call(this);
  }

  /**
   * @method _createLayers
   * @private
   */
  _createLayers() {
    this._needsRepaint = true;

    if (!this.lowerZLayer) {
      //@hackerham: create layers only in initialization. Doesn't depend on width/height
      this.addChild(this.lowerZLayer = new PIXI.tilemap.ZLayer(this, 0));
      this.addChild(this.upperZLayer = new PIXI.tilemap.ZLayer(this, 4));

      var useSquareShader = false;
      this.lowerZLayer.zIndex = 0;
      this.lowerZLayer.zOrder = 0;
      this.upperZLayer.zIndex = 4;
      this.upperZLayer.zOrder = 0;

      this.lowerZLayer.addChild(this.lowerLayer = new PIXI.tilemap.CompositeRectTileLayer(0, [], useSquareShader));
      this.lowerLayer.shadowColor = new Float32Array([0.0, 0.0, 0.0, 0.5]);
      this.upperZLayer.addChild(this.upperLayer = new PIXI.tilemap.CompositeRectTileLayer(4, [], useSquareShader));
    }
  }

  /**
   * @method _updateLayerPositions
   * @param {Number} startX
   * @param {Number} startY
   * @private
   */
  _updateLayerPositions(startX, startY) {
    var ox;
    var oy;

    if (this.roundPixels) {
      ox = Math.floor(this.origin.x);
      oy = Math.floor(this.origin.y);
    } else {
      ox = this.origin.x;
      oy = this.origin.y;
    }
    this.lowerZLayer.position.x = startX * this._tileWidth - ox;
    this.lowerZLayer.position.y = startY * this._tileHeight - oy;
    this.upperZLayer.position.x = startX * this._tileWidth - ox;
    this.upperZLayer.position.y = startY * this._tileHeight - oy;
  }

  /**
   * @method _paintAllTiles
   * @param {Number} startX
   * @param {Number} startY
   * @private
   */
  _paintAllTiles(startX, startY) {
    this.lowerZLayer.clear();
    this.upperZLayer.clear();
    var tileCols = Math.ceil(this._width / this._tileWidth) + 1;
    var tileRows = Math.ceil(this._height / this._tileHeight) + 1;
    for (var y = 0; y < tileRows; y++) {
      for (var x = 0; x < tileCols; x++) {
        this._paintTiles(startX, startY, x, y);
      }
    }
  }

  /**
   * @method _paintTiles
   * @param {Number} startX
   * @param {Number} startY
   * @param {Number} x
   * @param {Number} y
   * @private
   */
  _paintTiles(startX, startY, x, y) {
    const mx = startX + x;
    const my = startY + y;
    const dx = x * this._tileWidth, dy = y * this._tileHeight;
    const tileId0 = this._readMapData(mx, my, 0);
    const tileId1 = this._readMapData(mx, my, 1);
    const tileId2 = this._readMapData(mx, my, 2);
    const tileId3 = this._readMapData(mx, my, 3);

    const lowerLayer = this.lowerLayer.children[0];
    const upperLayer = this.upperLayer.children[0];

    if (!lowerLayer || !upperLayer) {
      return;
    }

    this._paintTile(tileId0, dx, dy, lowerLayer, upperLayer);
    this._paintTile(tileId1, dx, dy, lowerLayer, upperLayer);

    if (this._isOverpassPosition(mx, my)) {
      this._drawTile(upperLayer, tileId2, dx, dy);
      this._drawTile(upperLayer, tileId3, dx, dy);
    } else {
      this._paintTile(tileId2, dx, dy, lowerLayer, upperLayer);
      this._paintTile(tileId3, dx, dy, lowerLayer, upperLayer);
    }
  }

  _paintTile(tileId, x, y, lowerLayer, upperLayer) {
    const layer = this._isHigherTile(tileId) ? upperLayer : lowerLayer;
    this._drawTile(layer, tileId, x, y);
  }

  /**
   * @method _drawTile
   * @param {Array} layers
   * @param {Number} tileId
   * @param {Number} dx
   * @param {Number} dy
   * @private
   */
  _drawTile(layer, tileId, dx, dy) {
    if (Tilemap.isVisibleTile(tileId)) {
      if (Tilemap.isAutotile(tileId)) {
        this._drawAutotile(layer, tileId, dx, dy);
      } else {
        this._drawNormalTile(layer, tileId, dx, dy);
      }
    }
  }

  /**
   * @method _drawNormalTile
   * @param {Array} layers
   * @param {Number} tileId
   * @param {Number} dx
   * @param {Number} dy
   * @private
   */
  _drawNormalTile(layer, tileId, dx, dy) {
    let setNumber = 0;

    if (Tilemap.isTileA5(tileId)) {
      setNumber = 4;
    } else {
      setNumber = 5 + Math.floor(tileId / 256);
    }

    const w = this._tileWidth;
    const h = this._tileHeight;
    const sx = (Math.floor(tileId / 128) % 2 * 8 + tileId % 8) * w;
    const sy = (Math.floor(tileId % 256 / 8) % 16) * h;

    layer.addRect(setNumber, sx, sy, dx, dy, w, h);
  }

  /**
   * @method _drawAutotile
   * @param {Array} layers
   * @param {Number} tileId
   * @param {Number} dx
   * @param {Number} dy
   * @private
   */
  _drawAutotile(layer, tileId, dx, dy) {
    const kind = Tilemap.getAutotileKind(tileId);
    const shape = Tilemap.getAutotileShape(tileId);
    const tx = kind % 8;
    const ty = Math.floor(kind / 8);
    let bx = 0;
    let by = 0;
    let setNumber = 0;
    let animX = 0;
    let animY = 0;

    if (Tilemap.isTileA1(tileId)) {
      setNumber = 0;
      if (kind === 0) {
        animX = 2;
        by = 0;
      } else if (kind === 1) {
        animX = 2;
        by = 3;
      } else if (kind === 2) {
        bx = 6;
        by = 0;
      } else if (kind === 3) {
        bx = 6;
        by = 3;
      } else {
        if (kind % 2 !== 0) {
          return;
        }

        bx = Math.floor(tx / 4) * 8;
        by = ty * 6 + Math.floor(tx / 2) % 2 * 3;

        animX = 2;
      }
    } else if (Tilemap.isTileA2(tileId)) {
      setNumber = 1;
      bx = tx * 2;
      by = (ty - 2) * 3;
    } else if (Tilemap.isTileA3(tileId) || Tilemap.isTileA4(tileId)) {
      return;
    }

    const table = Tilemap.FLOOR_AUTOTILE_TABLE[shape];
    const w1 = this._tileWidth / 2;
    const h1 = this._tileHeight / 2;

    for (let i = 0; i < 4; i++) {
      const qsx = table[i][0];
      const qsy = table[i][1];
      const sx1 = (bx * 2 + qsx) * w1;
      const sy1 = (by * 2 + qsy) * h1;
      const dx1 = dx + (i % 2) * w1;
      const dy1 = dy + Math.floor(i / 2) * h1;

      layer.addRect(setNumber, sx1, sy1, dx1, dy1, w1, h1, animX, animY);
    }
  }

  /**
   * @method _drawTableEdge
   * @param {Array} layers
   * @param {Number} tileId
   * @param {Number} dx
   * @param {Number} dy
   * @private
   */
  _drawTableEdge(layer, tileId, dx, dy) {
  }

  /**
   * @method _drawShadow
   * @param {Number} shadowBits
   * @param {Number} dx
   * @param {Number} dy
   * @private
   */
  _drawShadow(layer, shadowBits, dx, dy) {
  }

}

// we need this constant for some platforms (Samsung S4, S5, Tab4, HTC One H8)
// PIXI.glCore.VertexArrayObject.FORCE_NATIVE = true;
PIXI.settings.GC_MODE = PIXI.GC_MODES.AUTO;
PIXI.tilemap.TileRenderer.SCALE_MODE = PIXI.SCALE_MODES.NEAREST;
PIXI.tilemap.TileRenderer.DO_CLEAR = true;

window.ShaderTilemap = ShaderTilemap;
module.exports = ShaderTilemap;