/**
 * MVCommons is the namespace of this module, it holds all the common functions
 * MVC is simple a shorter alias name for MVCommons - for ease of use
 */
var MVCommons = { };
var MVC = MVCommons;

window.MVC = MVC;
window.MVCommons = MVCommons;

/**
 * MVCommons module
 */
(function($){
  /**
   * Use strict mode for better code smell
   */
  "use strict";

  //============================================================================
  // Private functions
  //============================================================================

  /**
   * [ private ] defaultGetter(name)
   * Generates a getter based on a name
   * See reader() for information about the default getter
   * @param name The name of the property.
   * @return a function than can be used as a getter.
   * @private
   */
  function defaultGetter(name) {
    return function () {
      return this['_' + name];
    };
  }

  /**
   * [ private ] defaultSetter(name)
   * Generates a setter based on a name
   * See writer() for information about the default setter
   * @param name The name of the property.
   * @return a function than can be used as a setter.
   * @private
   */
  function defaultSetter(name) {
    return function (value) {
      var prop = '_' + name;
      if ((!this[prop]) || this[prop] !== value) {
        this[prop] = value;
        if (this._refresh) {
          this._refresh();
        }
      }
    };
  }

  /**
   * MVC.reader(obj, name, getter)
   * @param obj The object to add the reader property onto
   * @param name The name of the reader porperty
   * @param getter [optional] The Getter function
   *
   * Makes an easy way to make a reader (a variable you can read)
   * By default it gets the value of this['_' + name]
   */
  function reader(obj, name /*, getter */) {
    Object.defineProperty(obj, name, {
      get: arguments.length > 2 ? arguments[2] : defaultGetter(name),
      configurable: true
    });
  }

  /**
   * MVC.writer(obj, name, setter)
   * @param obj The object to add the property
   * @param name The property name
   * @param setter [optional] The setter function
   *
   * Makes an easy way to define a writer (a setter to a variable)
   * By default it sets the function of the property this['_' + name] = value
   * It also calls a method this._refresh() if that method exists
   */
  function writer(obj, name /*, setter*/) {
    Object.defineProperty(obj, name, {
      set: arguments.length > 2 ? arguments[2] : defaultSetter(name),
      configurable: true
    });
  }

  /**
   * MVC.accessor(obj, name, setter, getter)
   * @param obj The object to add the accessor
   * @param name The property name
   * @param setter [optional] The setter function
   * @param getter [optional] The getter function
   * @see reader
   * @see writer
   *
   * Makes an accessor (both getter and setter) of an object easily
   * See writer() and reader() for information about default values.
   */
  function accessor(value, name /* , setter, getter */) {
    Object.defineProperty(value, name, {
      get: arguments.length > 3 ? arguments[3] : defaultGetter(name),
      set: arguments.length > 2 ? arguments[2] : defaultSetter(name),
      configurable: true
    });
  }

  /**
   * MVC.deepClone(obj)
   * @param obj The object to clone
   * @return The cloned object
   *
   * Deep clones an object and its properties.
   * This function will crash when used on recursive objects.
   * Numbers, Strings and Functions are not deep cloned, thus, if they
   * are used for the obj param, they will be returned without change.
   */
  function deepClone(obj) {
    var result;
    if (obj instanceof Array) {
      return obj.map(function (i) { return $.deepClone(i); });
    } else if (obj && !obj.prototype && (typeof obj == 'object' || obj instanceof Object)) {
      result = {};
      for (var p in obj) {
        result[p] = $.deepClone(obj[p]);
      }
      return result;
    }
    return obj;
  }

  /**
   * MVC.shallowClone(obj)
   * @param obj The object to clone
   * @return The cloned object
   *
   * Clones an object with the same properties as the original.
   * This function does not clone properties.
   * Numbers, Strings and Functions are not shallow cloned, thus, if they
   * are used for the obj param, they will be returned without change.
   */
  function shallowClone(obj) {
    var result;
    if (obj instanceof Array) {
      return obj.slice(0);
    } else if (obj && !obj.prototype && (typeof obj == 'object' || obj instanceof Object)) {
      result = {};
      for (var p in obj) {
        result[p] = obj[p];
      }
      return result;
    }
    return obj;
  }

  /**
   * MVC.safeEval(text)
   * @param text The text to evaluate.
   * @return The result of the expression or null if something fails.
   *
   * Evaluates a context with some safety measure to stop it from breaking.
   */
  function safeEval(text) {
    try {
      return eval(text);
    } catch(e) {
      console.error(e);
      console.log(...log(text));
      return null;
    }
  }

  /**
   * MVC.degToRad(deg)
   * @param deg Degrees to convert
   * @return Radiants equivalent to those degrees.
   * Converts degrees into radians.
   */
  function degToRad(deg) {
    return deg * Math.PI / 180;
  }

  /**
   * MVC.radToDeg(rad)
   * @param rad Radians to convert
   * @return Degrees equivalent to those radians.
   * Converts radians into degrees.
   */
  function radToDeg(rad) {
    return rad * 180 / Math.PI;
  }

  /**
   * MVC.extractEventMeta(event)
   * @param The event to extract the meta on
   * Extracts notetags from all comments on all pages on the event
   * and creates a list of tags on each page.
   */
  function extractEventMeta(event) {
    var the_event = event;
    if (the_event instanceof Objects.Event) {
      the_event = event.event();
    }

    var pages = the_event.pages;
    if (pages === undefined) return;

    var re = /<([^<>:]+)(:?)([^>]*)>/g;
    for (var i = 0; i < pages.length; i++) {
      var page = pages[i];
      page.meta = page.meta || {};

      for (var j = 0; j < page.list.length; j++) {
        var command = page.list[j];
        if (command.code !== 108 && command.code !== 408) continue;

        for (;;) {
          var match = re.exec(command.parameters[0]);
          if (match) {
            if (match[2] === ':') {
              page.meta[match[1]] = match[3];
            } else {
              page.meta[match[1]] = true;
            }
          }
          else {
            break;
          }
        }
      }
    }
  }

  function getNoteValue(notes, tagName) {
    var re = /<([^<>:]+)(:?)([^>]*)>/g;
    while (true) {
      var match = re.exec(notes);
      if (match) {
        if (match[1].toLowerCase() == tagName.toLowerCase()) {
          if (match[2] === ':') {
            return match[3];
          } else {
            return true;
          }
        }
      } else {
        return false;
      }
    }
  }

  /**
   * MVC.getProp(meta, propName)
   * Gets a property value in a case insensitive way
   * @param meta the list of loaded tags from an object
   * @param propName the name of the property
   * @return the value of the property
   */
  function getProp(meta, propName){
    if (meta === undefined) return undefined;
    if (meta[propName] !== undefined) return meta[propName];
    for (var key in meta) {
      if (key.toLowerCase() == propName.toLowerCase()) {
        return meta[key];
      }
    }
    return undefined;
  }  

  //============================================================================
  // Export section
  //============================================================================

  $.extractEventMeta = extractEventMeta;
  $.getProp = getProp;
  $.getNoteValue = getNoteValue;


  /**
   * Some math
   */
  $.degToRad = degToRad;
  $.radToDeg = radToDeg;

  /**
   * Class easily manipulation
   */
  $.reader           = reader;
  $.writer           = writer;
  $.accessor         = accessor;

  /**
   * Object manipulation
   */
  $.shallowClone     = shallowClone;
  $.deepClone        = deepClone;

  /**
   * Evaling things
   */
  $.safeEval = safeEval;


  /**
   * End MVCommons main module
   */
})(MVCommons);

module.exports = {
  MVCommons
};