/**
 * Returns a number whose value is limited to the given range.
 *
 * @method Number.prototype.clamp
 * @param {Number} min The lower boundary
 * @param {Number} max The upper boundary
 * @return {Number} A number in the range (min, max)
 */
Number.prototype.clamp = function(min, max) {
  return Math.min(Math.max(this, min), max);
};

/**
 * Returns a modulo value which is always positive.
 *
 * @method Number.prototype.mod
 * @param {Number} n The divisor
 * @return {Number} A modulo value
 */
Number.prototype.mod = function(n) {
  return ((this % n) + n) % n;
};

/**
 * Replaces %1, %2 and so on in the string to the arguments.
 *
 * @method String.prototype.format
 * @param {Any} ...args The objects to format
 * @return {String} A formatted string
 */
String.prototype.format = function() {
  var args = arguments;
  return this.replace(/%([0-9]+)/g, function(s, n) {
    return args[Number(n) - 1];
  });
};

/**
 * Replaces {0}, {1} and so on in the string to the arguments.
 *
 * @method String.prototype.format
 * @param {Array} args The objects to format
 * @return {String} A formatted string
 */
String.prototype.formatList = function(args) {
  var regex = /\{\d+\}/;
  var regexResult = regex.exec(this);
  var result = this;

  var replaceFn = function(match, number) {
    return typeof args[number] != 'undefined' ? args[number] : match;
  };

  while (regexResult) {
    result = result.replace(/{(\d+)}/g, replaceFn);
    
    var idxParam = parseInt(regexResult[1], 10);
    //If there was a param with number bigger than the number of arguments received, break out of the loop after running it once
    if (idxParam >= args.length) {
      break;
    }

    regexResult = regex.exec(result);
  }

  return result;
};

/**
 * Makes a number string with leading zeros.
 *
 * @method String.prototype.padZero
 * @param {Number} length The length of the output string
 * @return {String} A string with leading zeros
 */
String.prototype.padZero = function(length){
  var s = this;
  while (s.length < length) {
    s = '0' + s;
  }
  return s;
};

/**
 * Makes a number string with leading zeros.
 *
 * @method Number.prototype.padZero
 * @param {Number} length The length of the output string
 * @return {String} A string with leading zeros
 */
Number.prototype.padZero = function(length){
  return String(this).padZero(length);
};

Object.defineProperties(Array.prototype, {
  /**
   * Checks whether the two arrays are same.
   *
   * @method Array.prototype.equals
   * @param {Array} array The array to compare to
   * @return {Boolean} True if the two arrays are same
   */
  equals: {
    enumerable: false,
    value: function(array) {
      if (!array || this.length !== array.length) {
        return false;
      }
      for (var i = 0; i < this.length; i++) {
        if (this[i] instanceof Array && array[i] instanceof Array) {
          if (!this[i].equals(array[i])) {
            return false;
          }
        } else if (this[i] !== array[i]) {
          return false;
        }
      }
      return true;
    }
  },
  /**
   * Makes a shallow copy of the array.
   *
   * @method Array.prototype.clone
   * @return {Array} A shallow copy of the array
   */
  clone: {
    enumerable: false,
    value: function() {
      return this.slice(0);
    }
  },
  /**
   * Checks whether the array contains a given element.
   *
   * @method Array.prototype.contains
   * @param {Any} element The element to search for
   * @return {Boolean} True if the array contains a given element
   */
  contains : {
    enumerable: false,
    value: function(element) {
      return this.indexOf(element) >= 0;
    }
  }
});

/**
 * Checks whether the string contains a given string.
 *
 * @method String.prototype.contains
 * @param {String} string The string to search for
 * @return {Boolean} True if the string contains a given string
 */
String.prototype.contains = function(string) {
  return this.indexOf(string) >= 0;
};

/**
 * Generates a random integer in the range (0, max-1).
 *
 * @static
 * @method Math.randomInt
 * @param {Number} max The upper boundary (excluded)
 * @return {Number} A random integer
 */
Math.randomInt = function(max) {
  return Math.floor(max * Math.random());
};

/**
 * Generates a random integer in the range (min, max-1).
 *
 * @static
 * @method Math.randomRange
 * @param {Number} min The upper boundary (included)
 * @param {Number} max The upper boundary (included)
 * @return {Number} A random integer
 */
Math.randomRange = function(min, max) {
  return Math.randomInt(max - min + 1) + min;
};


/**
 * Drops anything after the 12th decimal place of the number, thus,
 * fixing javascript's issue with decimal operations.
 * Examples:
 *   Math.ceil((0.2 + 0.1) * 10) == 4
 *   Math.ceil(Math.fix(0.2 + 0.1) * 10) == 3
 * @return the fixed number.
 */
Math.fix = function(value, precision){
  var multiplier = Math.pow(10, precision || 12);
  return Math.floor(value * multiplier) / multiplier;
};

/**
 * @return the largest integer less than or equal to the number.
 */
Number.prototype.floor = function(){
  return Math.floor(Math.fix(this));
};

/**
 * @return the smallest integer greater than or equal to the number.
 */
Number.prototype.ceil = function(){
  return Math.ceil(Math.fix(this));
};

/**
* @returns .ceil when positive and .floor when negative
*/
Number.prototype.absCeil = function(){
  if (this === 0) return 0;

  if (this > 0) {
    return this.ceil();
  }

  return this.floor();
};

/**
* @returns .floor when positive and .ceil when negative
*/
Number.prototype.absFloor = function() {
  if (this > 0) {
    return this.floor();
  }

  return this.ceil();
};

/**
 * @return the value of the number rounded to the nearest integer.
 */
Number.prototype.round = function(){
  return Math.round(Math.fix(this));
};

/**
 * @return the absolute value
 */
Number.prototype.abs = function(){
  return Math.abs(this);
};