require('engine/core/MVCommons.js');
const { Items } = require('engine/managers/Items');

class ItemContainer {
  constructor() {
    this.clear();
    this._maxItemTypes = 0;
    this._stackable = true;
  }

  clear() {
    this._items = [];
    this._maxItemTypes = 0;
  }

  get items() {
    return this._items;
  }

  countItems() {
    let count = 0;
    for (let i = 0; i < this._items.length; i++) {
      if (this._items[i]) {
        count++;
      }
    }

    return count;
  }

  getData() {
    const items = [];
    this._items.forEach(function(item){
      if (!item) {
        items.push(null);
        return;
      }

      items.push({
        id : item.id,
        amount : item.amount || 1
      });
    });

    return items;
  }

  setData(newItems) {
    const items = [];
    if (newItems) {
      newItems.forEach(function(item){
        if (!item) {
          items.push(null);
          return;
        }

        const itemData = Items.findItem(item.id);
        if (!itemData) {
          items.push(null);
          return;
        }

        const newItem = MVC.shallowClone(itemData);
        newItem.amount = item.amount || 1;
        items.push(newItem);
      });
    }

    this._items = items;
  }

  isStackable() {
    return this._stackable !== false;
  }

  disableStacks() {
    this._stackable = false;
  }

  enableStacks() {
    this._stackable = true;
  }

  addItem(itemData) {
    // 
  }

  setMaxItemTypes(newMax) {
    this._maxItemTypes = newMax;
  }

  clearList() {
    this._items = this._items.filter(function(n){ return n !== undefined && n !== null; });
  }

  numItems(item) {
    if (!item) return 0;

    let number = 0;
    for (let i = 0; i < this._items.length; i++) {
      if (!this._items[i]) {
        continue;
      }

      if (this._items[i].id === item.id) {
        if (Items.itemTypeIsStackable(item.type)) {
          return this._items[i].amount || 1;
        }

        number++;
      }
    }

    return number;
  }

  hasItem(item) {
    return this.numItems(item) > 0;
  }

  numItemsById(itemId) {
    return this.numItems(Items.getItemData(itemId));
  }

  hasItemById(itemId) {
    return this.hasItem(Items.getItemData(itemId));
  }

  maxItemTypes() {
    if (this._maxItemTypes > 0) {
      return this._maxItemTypes;
    }

    return 9999;
  }

  spaceForItem(item) {
    const amount = this.numItems(item);
    if (amount > 0) {
      return Items.maxItems(item) - amount;
    }

    if (this._items.length >= this.maxItemTypes()) {
      return 0;
    }

    return Items.maxItems(item);
  }

  hasFreeSlots() {
    if (this._items.length < this.maxItemTypes()) return true;

    const maxItems = Math.min(this._items.length, this.maxItemTypes());

    for (let i = 0; i < maxItems; i++) {
      if (this._items[i] === undefined || this._items[i] === null) {
        return true;
      }
    }

    return false;
  }

  itemCount() {
    return this._items.length;
  }

  canGainItem(item) {
    if (item.type == 'seed-box') {
      const seedId = item.id.replace('seed-box', 'seeds');
      return this.canGainItemId(seedId);
    }

    if (!Items.itemTypeCanBeStored(item.type)) {
      return false;
    }

    if (!Items.itemIdCanBeStored(item.id)) {
      return false;
    }

    if (this.hasFreeSlots())  {
      return true;
    }

    if (this.isStackable()) {
      if (Items.itemTypeIsStackable(item.type)) {
        if (this.numItems(item) > 0) {
          return true;
        }
      }
    }

    return false;
  }

  canGainItemId(itemId) {
    return this.canGainItem(Items.getItemData(itemId));
  }

  gainItem(item, amount, reverse) {
    if (!item) return;

    amount = Items.selectAmount(item, amount);
    if (reverse === undefined) reverse = false;

    var i;

    if (this.isStackable() && Items.itemTypeIsStackable(item.type)) {
      var added = false;
      var alreadyHadSome = this.numItems(item) > 0;

      for (i = 0; i < this._items.length; i++) {
        if (!this._items[i]) {
          if (alreadyHadSome) continue;

          let newItem = MVC.shallowClone(item);
          newItem.amount = amount;

          this._items[i] = newItem;
          added = true;
          break;
        }

        if (this._items[i].id == item.id) {
          this._items[i].amount = this._items[i].amount || 0;
          this._items[i].amount += amount;
          added = true;
          break;
        }
      }

      if (!added) {
        var stackItemToAdd = MVC.shallowClone(item);
        stackItemToAdd.amount = amount;

        if (reverse) {
          this._items.splice(0, 0, stackItemToAdd);
        } else {
          this._items.push(stackItemToAdd);
        }
      }
    } else {

      for (i = 0; i < this._items.length && amount > 0; i++) {
        if (!this._items[i]) {
          let newItem = MVC.shallowClone(item);
          newItem.amount = 1;

          this._items[i] = newItem;
          amount--;
        }
      }

      for (i = 0; i < amount; i++) {
        var itemToAdd = MVC.shallowClone(item);
        if (reverse) {
          this._items.splice(0, 0, itemToAdd);
        } else {
          this._items.push(itemToAdd);
        }
      }
    }

    Items.onInventoryChange();
  }

  gainItemId(itemId, amount) {
    return this.gainItem(Items.getItemData(itemId), amount);
  }

  loseItemId(itemId, amount) {
    return this.loseItem(Items.getItemData(itemId), amount);
  }

  loseItem(item, amount) {
    if (!item) return 0;
    if (amount === undefined) amount = 1;
    var amountRemoved = 0;

    var j = 0;
    while (j < this._items.length) {
      if (amountRemoved >= amount) {
        break;
      }

      if (!this._items[j]) {
        j++;
        continue;
      }

      if (this._items[j].id == item.id) {
        if (Items.itemTypeIsStackable(item.type)) {
          this._items[j].amount = this._items[j].amount || 0;

          while (this._items[j].amount > 0 && amountRemoved < amount) {
            this._items[j].amount -= 1;
            amountRemoved++;
          }

          if (this._items[j].amount <= 0) {
            delete this._items[j];
            continue;
          }
        } else {
          delete this._items[j];
          amountRemoved++;
          continue;
        }
      }

      j++;
    }

    Items.onInventoryChange();
    return amountRemoved;
  }

  moveItemId(itemId, newContainer, amount) {
    const item = Items.getItemData(itemId);

    return this.moveItem(item, newContainer, amount);
  }

  moveItem(item, newContainer, amount) {
    if (!newContainer.canGainItem(item)) return false;

    const amountToAdd = this.loseItem(item, amount);

    if (amountToAdd > 0) {
      newContainer.gainItem(item, amountToAdd);
      return true;
    }

    return false;
  }

  moveAllItems(newContainer) {
    for (const item of this._items) {
      if (!item) continue;

      newContainer.gainItem(item);
    }

    this.clear();
  }

  swapIndexes(oldIndex, newIndex) {
    const swap = this._items[newIndex];
    this._items[newIndex] = this._items[oldIndex];
    this._items[oldIndex] = swap;
  }

  sort(fn) {
    this._items.sort(fn);
  }
}

module.exports = {
  ItemContainer,
};