const { Utils } = require('engine/core/Utils');
const { MVCommons } = require('engine/core/MVCommons');

// The wait time of the key repeat in frames.
let keyRepeatWait = 24;
// Has a gamepad been activated before.
let gamepadActivated = false;
// Delay before the gamepad state is read
let gamepadDelay = 0;
// The interval of the key repeat in frames.
let keyRepeatInterval = 6;
let buttonSequenceLength = 12;

class Input {
  static get keyRepeatWait() {
    return keyRepeatWait;
  }
  static set keyRepeatWait(value) {
    keyRepeatWait = value;
  }
  static get gamepadActivated() {
    return gamepadActivated;
  }
  static set gamepadActivated(value) {
    gamepadActivated = value;
  }
  static get gamepadDelay() {
    return gamepadDelay;
  }
  static set gamepadDelay(value) {
    gamepadDelay = value;
  }

  static get keyRepeatInterval() {
    return keyRepeatInterval;
  }
  static set keyRepeatInterval(value) {
    keyRepeatInterval = value;
  }

  static get buttonSequenceLength() {
    return buttonSequenceLength;
  }
  static set buttonSequenceLength(value) {
    buttonSequenceLength = value;
  }

  // The four direction value as a number of the numpad, or 0 for neutral.
  static get dir4() {
    return this._dir4;
  }

  // The eight direction value as a number of the numpad, or 0 for neutral.
  static get dir8() {
    return this._dir8;
  }

  // The time of the last input in milliseconds.
  static get date() {
    return this._date;
  }

  static get keyMapper() {
    return this._keyMapper;
  }

  static set keyMapper(value) {
    this._keyMapper = value;
  }

  static get priorityKeys() {
    return {
      'up' : 87,
      'down' : 83,
      'left' : 65,
      'right' : 68,
      'pageup' : 82,
      'pagedown' : 70,
      'ok' : 13,
      'extra' : 84,
      'use' : 32
    };
  }

  static get keyDescriptions() {
    return {
      8 : 'Backspace',
      9 : 'Tab',
      13 : 'Enter',
      16 : 'Shift',
      17 : 'Control',
      18 : 'Alt',
      19 : 'Pause',
      20 : 'Caps Lock',
      27 : 'Esc',
      32 : 'Space',
      33 : 'Pg Up',
      34 : 'Pg Down',
      35 : 'End',
      36 : 'Home',
      37 : '←',
      38 : '↑',
      39 : '→',
      40 : '↓',
      44 : 'Print Screen',
      45 : 'Insert',
      46 : 'Delete',
      48 : '0',
      49 : '1',
      50 : '2',
      51 : '3',
      52 : '4',
      53 : '5',
      54 : '6',
      55 : '7',
      56 : '8',
      57 : '9',
      65 : 'A',
      66 : 'B',
      67 : 'C',
      68 : 'D',
      69 : 'E',
      70 : 'F',
      71 : 'G',
      72 : 'H',
      73 : 'I',
      74 : 'J',
      75 : 'K',
      76 : 'L',
      77 : 'M',
      78 : 'N',
      79 : 'O',
      80 : 'P',
      81 : 'Q',
      82 : 'R',
      83 : 'S',
      84 : 'T',
      85 : 'U',
      86 : 'V',
      87 : 'W',
      88 : 'X',
      89 : 'Y',
      90 : 'Z',
      96 : 'Num 0',
      97 : 'Num 1',
      98 : 'Num 2',
      99 : 'Num 3',
      100 : 'Num 4',
      101 : 'Num 5',
      102 : 'Num 6',
      103 : 'Num 7',
      104 : 'Num 8',
      105 : 'Num 9',
      106 : 'Num *',
      107 : 'Num +',
      109 : 'Num -',
      111 : 'Num /',
      112 : 'F1',
      113 : 'F2',
      114 : 'F3',
      115 : 'F4',
      116 : 'F5',
      117 : 'F6',
      118 : 'F7',
      119 : 'F8',
      120 : 'F9',
      121 : 'F10',
      122 : 'F11',
      123 : 'F12',
      186 : 'Key 186 (Ç;Ñ)',
      187 : '+',
      188 : ',',
      189 : '-',
      190 : '.',
      191 : 'Key 191 (>;Ç)'      
    };
  }

  static get defaultGamepadProfiles() {
    return $dataProfiles;
  }

  static get defaultKeyMapper() {
    return {
      9: 'map', // tab
      13: 'ok', // enter
      16: 'shift', // shift
      17: 'control', // control
      18: 'control', // alt
      27: 'escape', // escape
      32: 'use', // space


      48 : '0', // 0
      49 : '1', // 1
      50 : '2', // 2
      51 : '3', // 3
      52 : '4', // 4
      53 : '5', // 5
      54 : '6', // 6
      55 : '7', // 7
      56 : '8', // 8
      57 : '9', // 9
      96 : '0', // numpad 0
      97 : '1', // numpad 1
      98 : '2', // numpad 2
      99 : '3', // numpad 3
      100 : '4', // numpad 4
      101 : '5', // numpad 5
      102 : '6', // numpad 6
      103 : '7', // numpad 7
      104 : '8', // numpad 8
      105 : '9', // numpad 9
      109 : '10', // numpad -

      65 : 'left', // A
      66 : 'inventory', // B
      83 : 'down', // S
      68 : 'right', // D
      87 : 'up', // W
      81: 'use', // Q
      84: 'extra', // T
      69: 'ok', // E
      82: 'pageup', // R
      70: 'pagedown', //F
      71: 'change_backpack', //G
      73: 'inventory', // I
      67: 'escape', //C
      77: 'map', // M

      // 81: 'pageup', // Q
      // 69: 'pagedown', // E
      // 88: 'escape', // X
      // 90: 'ok', //Z

      33: 'pageup', // pageup
      34: 'pagedown', // pagedown
      37: 'left', // left arrow
      38: 'up', // up arrow
      39: 'right', // right arrow
      40: 'down', // down arrow

      114 : 'feedback', // F3
      115 : 'fullscreen', // F4
      116 : 'stretchToFit', // F5
      117 : 'quicksave' // F6
    };
  }


  static get protectedKeyCodes() {
    return {
      13 : 'ok',
      27 : 'escape',
      37 : 'left',
      38 : 'up',
      39 : 'right',
      40 : 'down',
      44 : 'print-screen',
      114 : 'feedback',
      115 : 'fullscreen',
      116 : 'stretchToFit',
    };
  }

  static get gamepadProfiles() {
    return this._gamepadProfiles;
  }
  static set gamepadProfiles(value) {
    this._gamepadProfiles = value;
  }

  static initialize() {
    this.clear();
    this._wrapNwjsAlert();
    this._setupEventHandlers();
  }

  static clear() {
    this._currentState = {};
    this._currentCodeState = {};
    this._previousState = {};
    this._gamepadStates = [];
    this._latestButton = null;
    this._latestGamePadIndex = null;
    this._lastKeyDown = null;
    this._lastButton = null;
    this._pressedTime = 0;
    this._dir4 = 0;
    this._dir8 = 0;
    this._preferredAxis = '';
    this._date = 0;
    this._buttonSequence = [];
    this._pressedKeys = [];
  }

  static resetKeyMap() {
    this.keyMapper = this.defaultKeyMapper;
  }

  static _wrapNwjsAlert() {
    if (!Utils.isNwjs()) {
      return;
    }

    const _alert = window.alert;
    window.alert = () => {
      const gui = require('nw.gui');
      const win = gui.Window.get();
      _alert.apply(this, arguments);
      win.focus();
      this.clear();
    };
  }

  static _setupEventHandlers() {
    document.addEventListener('keydown', this._onKeyDown.bind(this));
    document.addEventListener('keyup', this._onKeyUp.bind(this));
    window.addEventListener('blur', this._onLostFocus.bind(this));
  }

  static _onKeyDown(event) {
    if (this._shouldPreventDefault(event.keyCode)) {
      event.preventDefault();
    }
    if (event.keyCode === 144) { // Numlock
      this.clear();
    }
    if (event.altKey && event.keyCode === 13) {
      return;
    }

    this._lastKeyDown = event.keyCode;
    this._pressedKeys.push(event.keyCode);

    var buttonName = this.keyMapper[event.keyCode];
    this._currentCodeState[event.keyCode] = true;

    if (ResourceHandler.exists() && buttonName == "ok") {
      ResourceHandler.retry();
    } else if (buttonName) {
      this._currentState[buttonName] = true;
    } else {
      if (event.keyCode == 16 || event.keyCode == 17) {
        this._currentState.control = true;
      }
    }

    if (event.keyCode == 8) {
      this._currentState.backspace = true;
    }
    this._currentState.altKey = event.altKey;
    this._currentState.ctrlKey = event.ctrlKey;

    this.checkProfiler(event.keyCode);
  }

  static _onKeyUp(event) {
    // if (event.keyCode != 13 && event.keyCode != 27) {
    //   if (this.isTyping()) {
    //     return;
    //   }
    // }

    this._currentCodeState[event.keyCode] = false;

    var buttonName = this.keyMapper[event.keyCode];
    if (buttonName) {
      this._currentState[buttonName] = false;
    } else {
      if (event.keyCode == 16 || event.keyCode == 17) {
        this._currentState.control = false;
      }
    }

    if (event.keyCode == 8) {
      this._currentState.backspace = false;
    }

    if (event.keyCode === 0) { // For QtWebEngine on OS X
      this.clear();
    }
  }

  static _onLostFocus() {
    this.clear();
  }

  static update() {
    this._pollGamepads();
    this._lastKeys = this._pressedKeys;
    this._pressedKeys = [];

    if (this._currentState[this._latestButton]) {
      this._pressedTime++;
    } else {
      this._latestButton = null;
    }
    for (var name in this._currentState) {
      if (this._currentState[name] && !this._previousState[name]) {
        this._latestButton = name;
        this._pressedTime = 0;
        this._date = Date.now();

        this._buttonSequence.push(name);
      }
      this._previousState[name] = this._currentState[name];
    }

    if (this._buttonSequence.length > this.buttonSequenceLength) {
      this._buttonSequence.splice(0, this._buttonSequence.length - this.buttonSequenceLength);
    }

    this._updateDirection();
  }

  static isSequencePressed(sequence) {
    if (!sequence.length) return false;
    if (this._buttonSequence.length < sequence.length) return false;

    const pressedSequence = sequence.length === this._buttonSequence ? this._buttonSequence : this._buttonSequence.slice(0 - sequence.length);

    for (let i = 0; i < sequence.length; i++) {
      if (sequence[i] != pressedSequence[i]) {
        return false;
      }
    }

    return true;
  }

  static clearSequence() {
    this._buttonSequence.splice(0, this._buttonSequence.length);
  }

  static isPressed(keyName) {
    if (this._isEscapeCompatible(keyName) && (this.isPressed('escape') || this.isPressed('start'))) {
      return true;
    } else {
      return !!this._currentState[keyName];
    }
  }

  static isCodePressed(keyCode) {
    return !!this._currentCodeState[keyCode];
  }

  static isTriggered(keyName) {
    if (this._isEscapeCompatible(keyName) && (this.isTriggered('escape') || this.isTriggered('start'))) {
      return true;
    }

    return this._latestButton === keyName && this._pressedTime === 0;
  }

  static isAnyTriggered() {
    if (this.isTriggered('ok') || this.isTriggered('escape') || this.isTriggered('use') || this.isTriggered('map')) {
      return true;
    }

    if (this.isTriggered('extra') || this.isTriggered('control') || this.isTriggered('shift') || this.isTriggered('change_backpack')) {
      return true;
    }

    if (this.isTriggered('pageup') || this.isTriggered('pagedown')) {
      return true;
    }

    if (this.isTriggered('left') || this.isTriggered('right') || this.isTriggered('up') || this.isTriggered('down')) {
      return true;
    }

    if (this.isTriggered('alternate') || this.isTriggered('left_backbutton') || this.isTriggered('right_backbutton')) {
      return true;
    }

    if (this.isTriggered('left_trigger') || this.isTriggered('right_trigger') || this.isTriggered('start')) {
      return true;
    }

    return false;
  }

  static isRepeated(keyName) {
    if (this._isEscapeCompatible(keyName) && (this.isRepeated('escape') || this.isRepeated('start'))) {
      return true;
    }

    return (this._latestButton === keyName && (this._pressedTime === 0 || (this._pressedTime >= this.keyRepeatWait && this._pressedTime % this.keyRepeatInterval === 0)));
  }

  static isLongPressed(keyName) {
    if (this._isEscapeCompatible(keyName) && (this.isLongPressed('escape') || this.isLongPressed('start'))) {
      return true;
    } else {
      return (this._latestButton === keyName && this._pressedTime >= this.keyRepeatWait);
    }
  }

  static isTyping() {
    return document.activeElement && document.activeElement.id == 'inputField';
  }

  static getTextSelection() {
    const el = document.getElementById('inputField');
    if (document.activeElement != el) return false;

    let start = 0;
    let end = 0;

    if (typeof el.selectionStart == "number" && typeof el.selectionEnd == "number") {
      start = el.selectionStart;
      end = el.selectionEnd;
    } else {
      const range = document.selection.createRange();

      if (range && range.parentElement() == el) {
        const len = el.value.length;
        const normalizedValue = el.value.replace(/\r\n/g, "\n");

        // Create a working TextRange that lives only in the input
        const textInputRange = el.createTextRange();
        textInputRange.moveToBookmark(range.getBookmark());

        // Check if the start and end of the selection are at the very end
        // of the input, since moveStart/moveEnd doesn't return what we want
        // in those cases
        const endRange = el.createTextRange();
        endRange.collapse(false);

        if (textInputRange.compareEndPoints("StartToEnd", endRange) > -1) {
          start = end = len;
        } else {
          start = -textInputRange.moveStart("character", -len);
          start += normalizedValue.slice(0, start).split("\n").length - 1;

          if (textInputRange.compareEndPoints("EndToEnd", endRange) > -1) {
            end = len;
          } else {
            end = -textInputRange.moveEnd("character", -len);
            end += normalizedValue.slice(0, end).split("\n").length - 1;
          }
        }
      }
    }

    return {
      start,
      end,
    };
  }

  static _shouldPreventDefault(keyCode) {
    if (this.isTyping()) return false;

    switch (keyCode) {
      case 8: // backspace
      case 9: // tab
      case 33: // pageup
      case 34: // pagedown
      case 37: // left arrow
      case 38: // up arrow
      case 39: // right arrow
      case 40: // down arrow
        return true;
      default:
        return false;
    }
  }

  static checkProfiler(keyCode) {
    if (keyCode == 118 && this.isPressed('altKey')) {
      $gameTemp.requestProfile();
    }
  }

  static _pollGamepads() {
    if (this.gamepadDelay > 0) {
      this.gamepadDelay--;
      return;
    }

    if (!navigator.getGamepads) {
      return;
    }

    const gamepads = navigator.getGamepads();
    if (!gamepads) {
      return;
    }

    // if (!this.gamepadProfiles) {
    //   this.gamepadProfiles = this.defaultGamepadProfiles;
    // }

    for (let i = 0; i < gamepads.length; i++) {
      const gamepad = gamepads[i];
      if (!gamepad || !gamepad.connected) {
        continue;
      }

      if (!this.gamepadActivated) {
        this.gamepadActivated = true;
        this.gamepadDelay = Utils.getFrameCount(30);
        return;
      }
      this._updateGamepadState(gamepad);
    }
  }

  static getTypeFromGamepadId(id) {
    if (id.match(/\w*Xbox 360 Controller\w*/i)) {
      return 'X360';
    }

    if (id.toLowerCase() == 'xinput') {
      return 'XInput';
    }

    // if Microsoft
    if (id.match(/\w*045e\w*/i)) {
      // XBox One controller (wired or not)
      if (id.match(/\w*02ea\w*/i)) {
        return 'XONE';
      }

      // XBox 360 Controller
      if (id.match(/\w*028e\w*/i)) {
        return 'X360';
      }

      // XBOX Wireless Controller 
      if (id.match(/\w*02fd\w*/i)) {
        return 'XBOX';
      }

      return 'MS';
    }

    // if Sony
    if (id.match(/\w*054c\w*/i)) {
      // 
      if (id.match(/\w*09cc\w*/i) || id.match(/\w*05c4\w*/i)) {
        return 'PS4';
      }

      // PLAYSTATION(R)3 Controller
      if (id.match(/\w*0268\w*/i)) {
        return 'PS3';
      }

      return 'SONY';
    }

    // if Nintendo
    if (id.match(/\w*057e\w*/i)) {
      // Left JoyCon
      if (id.match(/\w*2006\w*/i)) {
        return 'LJOY';
      }

      // Right JoyCon
      if (id.match(/\w*2007\w*/i)) {
        return 'RJOY';
      }

      // Pro Controller
      if (id.match(/\w*2009\w*/i)) {
        // #ToDo: get a PRO controller to figure out it's schema.
        return 'Unknown';
        // return 'PRO';
      }
    }

    // Generic, PS3 compatible Gamepad
    if (id.match(/\w*2536\w*/i)) {
      if (id.match(/\w*0575\w*/i)) {
        return 'PS3';
      }

      if (id.match(/\w*0523\w*/i)) {
        return 'PS3';
      }
    }

    // 54c-268-PLAYSTATION(R)3 Controller
    if (id.match(/\w*54c-268\w*/i)) {
      return 'PS3';
    }

    // PS2 controller with adapter
    if (id.includes('Vendor: 0810 Product: 0001')) {
      return 'PS2';
    }

    return 'Unknown';    
  }

  static getAllGamepadTypes() {
    if (!this.gamepadActivated) {
      return [];
    }

    if (!navigator.getGamepads) {
      return [];
    }

    const gamepads = navigator.getGamepads();
    if (!gamepads) {
      return [];
    }

    const list = [];

    for (const gamepad of gamepads) {
      if (!gamepad || !gamepad.connected) {
        continue;
      }

      const type = this.getTypeFromGamepadId(gamepad.id);
      if (type) {
        list.push(type);
      }
    }

    return list;
  }

  static getGamepadType() {
    const allTypes = this.getAllGamepadTypes();

    if (!allTypes || !allTypes.length) {
      return false;
    }

    let index = this._latestGamePadIndex || 0;
    if (!allTypes[index]) {
      index = 0;
    }

    if (allTypes[index] == 'LJOY' || allTypes[index] == 'RJOY') {
      if (allTypes.includes('LJOY') && allTypes.includes('RJOY')) {
        return 'JOYCON-PAIR';
      }
    }

    return allTypes[0];
  }

  static getActiveGamepadProfile() {
    const controllerType = this.getGamepadType();
    if (!controllerType) {
      return false;
    }

    const type = controllerType.toLowerCase();
    return this.gamepadProfiles[type] || this.gamepadProfiles.unknown;
  }

  static checkAnalogSticksJoyCons(gamepad, newState) {
    const axes = gamepad.axes;
    const value = axes[9];

    if (value >= 1.15) {
      return;
    }

    const usingPair = this.isUsingJoyconPair();
    // ignore right joy con analog stick when using both
    if (usingPair && this.getTypeFromGamepadId(gamepad.id) == 'RJOY') {
      return;
    }

    const left = usingPair ? 22 : 24;
    const right = usingPair ? 23 : 25;
    const up = usingPair ? 25 : 22;
    const down = usingPair ? 24 : 23;

    if (value >= 0) {
      if (value >= 0.25) {
        newState[left] = true;
      }

      if (value >= 0.85) {
        newState[up] = true;
      }

      if (value < 0.55) {
        newState[down] = true;
      }

      return;
    }

    const rightValue = Math.abs(value);

    if (rightValue >= 0.7) {
      newState[up] = true;
    }

    if (rightValue >= 0 && rightValue < 0.85) {
      newState[right] = true;
    }

    if (rightValue < 0.3) {
      newState[down] = true;
    }
  }

  static checkAnalogSticksDefaultController(gamepad, newState) {
    const threshold = 0.5;
    const axes = gamepad.axes;

    if (axes[1] < -threshold) {
      newState[12] = true; // up
    } else if (axes[1] > threshold) {
      newState[13] = true; // down
    }
    if (axes[0] < -threshold) {
      newState[14] = true; // left
    } else if (axes[0] > threshold) {
      newState[15] = true; // right
    }
  }

  static typeIsJoyCon(controllerType) {
    return controllerType == 'LJOY' || controllerType == 'RJOY';
  }

  static checkAnalogSticks(controllerType, gamepad, newState) {
    if (this.typeIsJoyCon(controllerType)) {
      this.checkAnalogSticksJoyCons(gamepad, newState);
    } else {
      this.checkAnalogSticksDefaultController(gamepad, newState);
    }
  }

  static isUsingJoyconPair() {
    const allControllerTypes = this.getAllGamepadTypes();
    return allControllerTypes.includes('LJOY') && allControllerTypes.includes('RJOY');
  }

  static manipulateControllerPressedButtons(controllerType, gamepad) {
    if (!this.typeIsJoyCon(controllerType)) {
      return gamepad.buttons;
    }

    if (!this.isUsingJoyconPair()) {
      return gamepad.buttons;
    }

    // When playing with a pair of joy cons, we give each button code a different action based on which joycon it came from
    const baseButtons = gamepad.buttons;
    const buttons = [];

    for (let i = 0; i <= 15; i++) {
      buttons[i] = null;
    }

    if (controllerType == 'LJOY') {
      buttons[8] = baseButtons[8]; // minus
      buttons[10] = baseButtons[10]; // left analog stick
      buttons[13] = baseButtons[13]; // screenshot

      buttons[22] = baseButtons[2]; // up
      buttons[23] = baseButtons[1]; // down
      buttons[24] = baseButtons[0]; // left
      buttons[25] = baseButtons[3]; // right

      buttons[4] = baseButtons[14]; // L
      buttons[6] = baseButtons[15]; // ZL
    } else if (controllerType == 'RJOY') {
      buttons[9] = baseButtons[9]; // plus
      buttons[11] = baseButtons[11]; // right analog stick
      buttons[12] = baseButtons[12]; // home

      buttons[0] = baseButtons[2]; // B
      buttons[1] = baseButtons[0]; // A
      buttons[2] = baseButtons[3]; // Y
      buttons[3] = baseButtons[1]; // X

      buttons[5] = baseButtons[14]; // R
      buttons[7] = baseButtons[15]; // ZR
    }

    return buttons;
  }

  static _updateGamepadState(gamepad) {
    const controllerType = this.getTypeFromGamepadId(gamepad.id);
    const profiles = this.gamepadProfiles;
    if (!profiles) {
      return;
    }

    let profile = profiles[controllerType.toLowerCase()] || this.gamepadProfiles.unknown;

    if (this.typeIsJoyCon(controllerType) && this.isUsingJoyconPair()) {
      profile = this.gamepadProfiles['joycon-pair'] || profile;
    }

    const lastState = this._gamepadStates[gamepad.index] || [];
    const newState = [];
    const buttons = this.manipulateControllerPressedButtons(controllerType, gamepad);

    if (this.typeIsJoyCon(controllerType)) {
      newState[22] = false;
      newState[23] = false;
      newState[24] = false;
      newState[25] = false;
    } else {
      newState[12] = false;
      newState[13] = false;
      newState[14] = false;
      newState[15] = false;
    }

    for (let i = 0; i < buttons.length; i++) {
      if (buttons[i]) {
        newState[i] = buttons[i].pressed;
      }
    }

    this.checkAnalogSticks(controllerType, gamepad, newState);

    for (let j = 0; j < newState.length; j++) {
      if (newState[j] === null) {
        continue;
      }

      if (newState[j] !== lastState[j]) {
        const buttonName = profile[j];
        if (buttonName) {
          this._currentState[buttonName] = newState[j];
        }

        if (newState[j]) {
          this._lastButton = j;
        }
      }
    }

    this._gamepadStates[gamepad.index] = newState;

    if (newState.includes(true)) {
      this._latestGamePadIndex = gamepad.index;
    }
  }

  static _updateDirection() {
    let x = this._signX();
    let y = this._signY();

    this._dir8 = this._makeNumpadDirection(x, y);

    if (x !== 0 && y !== 0) {
      if (this._preferredAxis === 'x') {
        y = 0;
      } else {
        x = 0;
      }
    } else if (x !== 0) {
      this._preferredAxis = 'y';
    } else if (y !== 0) {
      this._preferredAxis = 'x';
    }

    this._dir4 = this._makeNumpadDirection(x, y);
  }

  static _signX() {
    let x = 0;

    if (this.isPressed('left')) {
      x--;
    }
    if (this.isPressed('right')) {
      x++;
    }
    return x;
  }

  static _signY() {
    let y = 0;

    if (this.isPressed('up')) {
      y--;
    }
    if (this.isPressed('down')) {
      y++;
    }
    return y;
  }

  static _makeNumpadDirection(x, y) {
    if (x !== 0 || y !== 0) {
      return 5 - y * 3 + x;
    }
    return 0;
  }

  static _isEscapeCompatible(keyName) {
    return keyName === 'cancel' || keyName === 'menu';
  }

  static getGamepadProfileData() {
    const data = MVCommons.deepClone(this.gamepadProfiles);
    const dataToSave = {};

    for (const key in data) {
      const profile = data[key];
      const defaultProfile = this.defaultGamepadProfiles[key];

      if (!defaultProfile) {
        dataToSave[key] = profile;
        continue;
      }

      let profileNeedsSaving = false;

      delete profile.icons;
      delete profile.labels;
      delete profile.type;

      for (const buttonCode in profile) {
        if (profile[buttonCode] == defaultProfile[buttonCode]) {
          delete profile[buttonCode];
        } else {
          profileNeedsSaving = true;
        }
      }

      if (profileNeedsSaving) {
        dataToSave[key] = profile;
      }
    }

    return dataToSave;
  }

  static loadKeyMapper(keys) {
    this.resetKeyMap();

    const validNames = ['left', 'right', 'up', 'down', 'ok', 'escape',
      'use', 'extra', 'pageup', 'pagedown', 'quicksave',
      'map', 'shift', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '10',
      'mapshot', 'left_backbutton', 'left_trigger', 'change_backpack',
      'inventory'];

    for (const keyCode in keys) {
      if (!keys.hasOwnProperty(keyCode)) continue;
      const keyName = keys[keyCode];

      if (!validNames.includes(keyName)) continue;

      this.setKey(keyCode, keyName);
    }

    for (const protectedKeyCode in this.protectedKeyCodes) {
      if (!this.protectedKeyCodes.hasOwnProperty(protectedKeyCode)) continue;

      this.setKey(protectedKeyCode, this.protectedKeyCodes[protectedKeyCode]);
    }
  }

  static setKey(keyCode, keyName) {
    if (this.protectedKeyCodes[keyCode] !== undefined) {
      return;
    }

    this.keyMapper[keyCode] = keyName;
  }

  static removeKey(key) {
    if (this.protectedKeyCodes[key] !== undefined) {
      return;
    }

    delete this.keyMapper[key];
  }

  static onLoadProfiles() {
    this.gamepadProfiles = this.defaultGamepadProfiles;
  }
}

Input.keyMapper = {};
Input.gamepadProfiles = {};

module.exports = {
  Input,
};