//-----------------------------------------------------------------------------
/**
 * The resource class. Allows to be collected as a garbage if not use for some time or ticks
 *
 * @class CacheEntry
 * @constructor
 * @param {ResourceManager} resource manager
 * @param {string} key, url of the resource
 * @param {string} item - Bitmap, HTML5Audio, WebAudio - whatever you want to store in the cache
 */
Core.CacheEntry = class CacheEntry {
  constructor(cache, key, item) {
    this.cache = cache;
    this.key = key;
    this.item = item;
    this.cached = false;
    this.touchTicks = 0;
    this.touchSeconds = 0;
    this.ttlTicks = 0;
    this.ttlSeconds = 0;
    this.freedByTTL = false;

    if (item) {
      item.cacheEntry = this;
    }
  }

  /**
   * frees the resource
   */
  free(byTTL) {
    this.freedByTTL = byTTL || false;
    if (this.cached) {
      this.cached = false;
      delete this.cache._inner[this.key];
    }
  }

  /**
   * Allocates the resource
   * @returns {CacheEntry}
   */
  allocate() {
    if (!this.cached) {
      this.cache._inner[this.key] = this;
      this.cached = true;
    }
    this.touch();
    return this;
  }

  /**
   * Sets the time to live
   * @param {number} ticks TTL in ticks, 0 if not set
   * @param {number} time TTL in seconds, 0 if not set
   * @returns {CacheEntry}
   */
  setTimeToLive(ticks, seconds) {
    this.ttlTicks = ticks || 0;
    this.ttlSeconds = seconds || 0;
    return this;
  }

  isStillAlive() {
    const cache = this.cache;

    let ticksExpired = false;
    let secondsExpired = false;

    if (this.ttlTicks !== 0) {
      ticksExpired = cache.updateTicks < (this.touchTicks + this.ttlTicks);
    }

    if (this.ttlSeconds !== 0) {
      secondsExpired = cache.updateSeconds > (this.touchSeconds + this.ttlSeconds);
    }

    return (!ticksExpired && !secondsExpired);
  }

  /**
   * makes sure that resource wont freed by Time To Live
   * if resource was already freed by TTL, put it in cache again
   */
  touch() {
    const cache = this.cache;
    if (this.cached) {
      this.touchTicks = cache.updateTicks;
      this.touchSeconds = cache.updateSeconds;
    } else if (this.freedByTTL) {
      this.freedByTTL = false;
      //TODO: shall we log this event? its not normal
      if (!cache._inner[this.key]) {
        cache._inner[this.key] = this;
      }
    }
  }
};