const Window = require('./Window');
const { Inventory } = require('engine/managers/Inventory');

//-----------------------------------------------------------------------------
// BasicWindow
//
// The superclass of Window_Base and FreeContent

class BasicWindow extends Window {
  initialize(x, y, width, height) {
    super.initialize();
    this.loadWindowskin();
    this.move(x, y, width, height);
    this.updatePadding();
    this.createContents();
    this._heightCache = [];
    this._widthCache = [];
  }

  lineHeight() {
    return 16 + (8 * Graphics.windowZoomLevel);
  }

  standardFontFace() {
    // If the $gameSystem was not yet loaded
    if (!window.$gameSystem) {
      return 'MenuFont';
    }

    if ($gameSystem.isChinese()) {
      return 'SimHei, Heiti TC, sans-serif';
    } else if ($gameSystem.isKorean()) {
      return 'Dotum, AppleGothic, sans-serif';
    } else {
      return 'MenuFont';
    }
  }

  standardFontSize() {
    return Utils.convertFontSize(32, Graphics.windowZoomLevel);
  }

  loadWindowskin() {
    this.windowskin = Managers.Images.loadSystem('DiarySkin');
  }

  textPadding() {
    return 6;
  }

  updatePadding() {
    this._padding = this.standardPadding();
    this._paddingV = this.verticalPadding();
    this._paddingH = this.horizontalPadding();
    this._refreshAllParts();
  }

  contentsWidth() {
    return this.width - this.horizontalPadding() * 2;
  }

  contentsHeight() {
    return this.height - this.verticalPadding() * 2;
  }

  fittingHeight(numLines) {
    return numLines * this.lineHeight() + this.verticalPadding() * 2;
  }

  createContents() {
    const w = this.contentsWidth();
    const h = this.contentsHeight();

    if (!this.contents) {
      this.contents = new Bitmap(w, h);
    } else if (this.contents.width != w || this.contents.height != h) {
      this.contents.resize(w, h);
      this.contents.clear();
    } else {
      this.contents.clear();
    }

    this.resetFontSettings();
  }

  standardOutlineColor() {
    return 'rgba(0, 0, 0, 0.5)';
  }

  standardOutlineWidth() {
    return 4;
  }

  resetFontSettings() {
    this.contents.fontFace = this.standardFontFace();
    this.contents.fontSize = this.standardFontSize();
    this.resetTextColor();
    this.contents.fontItalic = false;
    this.contents.outlineColor = this.standardOutlineColor();
    this.contents.outlineWidth = this.standardOutlineWidth();
  }

  resetCustomFontSettings() {
    this._quietMode = false;
  }

  resetTextColor() {
    this.changeTextColor(this.normalColor());
  }

  show() {
    this.visible = true;
  }

  hide() {
    this.visible = false;
  }

  textColor(n) {
    if (n >= 100) {
      switch(n) {
        case 100:
          return this.normalColor();
        case 101:
          return this.importantColor();
        case 102:
          return this.positiveColor();
        case 103:
          return this.negativeColor();
        case 104:
          return this.blackColor();
        case 105:
          return this.baseColor();
        case 106:
          return this.darkBaseColor();
        default:
          return this.normalColor();
      }
    }

    const colors = [
      '#FFFFFF', // branco
      '#FB3153', // vermelho
      '#328FFA', // azul
      '#FB3153', // vermelho
      '#3D2B25', // preto
      '#F4DEA8', // base
      '#D7A678'  // base escura
    ];
    
    return colors[n % colors.length];
  }

  normalColor() {
    return this.textColor(0);
  }

  importantColor() {
    return this.textColor(1);
  }

  positiveColor() {
    return this.textColor(2);
  }

  blueColor() {
    return this.textColor(2);
  }

  negativeColor() {
    return this.textColor(3);
  }

  redColor() {
    return this.textColor(3);
  }

  blackColor() {
    return this.textColor(4);
  }

  baseColor() {
    return this.textColor(5);
  }

  darkBaseColor() {
    return this.textColor(6);
  }

  translucentOpacity() {
    return 100;
  }

  changeTextColor(color) {
    this.contents.textColor = color;
  }

  changePaintOpacity(enabled) {
    this.contents.paintOpacity = enabled ? 255 : this.translucentOpacity();
  }

  calcTextY(desiredY) {
    const lineHeight = this.lineHeight();
    const diff = lineHeight - (lineHeight - this.contents.fontSize * 0.7) / 2;

    return desiredY - diff;
  }

  drawText(text, x, y, maxWidth, align) {
    this.contents.drawText(text, x, y, maxWidth, this.lineHeight(), align);
  }

  textWidth(text) {
    return this.contents.measureTextWidth(text);
  }

  drawTextEx(text, x, y, targetWidth = undefined) {
    return this.drawTextExGetData(text, x, y, targetWidth).width;
  }

  drawTextExGetData(text, x, y, targetWidth = undefined) {
    const result = {
      maxWidth : 0,
      width : 0,
      height : 0,
      lines : 1
    };

    if (text) {
      let lines = 0;
      const textState = { index: 0, x, y, left: x, targetWidth };
      textState.text = this.convertEscapeCharacters(text);
      textState.height = this.calcTextHeight(textState, false);
      this.resetFontSettings();

      let lastX = Graphics.width * 3;

      while (textState.index < textState.text.length) {
        this.processCharacter(textState);

        const newWidth = textState.x - x;

        if (textState.x < lastX) {
          lines++;
        }

        lastX = textState.x;
        result.maxWidth = Math.max(result.maxWidth, newWidth);
      }

      result.height = textState.height * lines;
      result.lines = lines;
      result.width = textState.x - x;
    }

    return result;
  }

  setWordWrap(text) {
    this._wordWrap = false;
    if (text.match(/<(?:WordWrap)>/i)) {
      this._wordWrap = true;
      text = text.replace(/<(?:WordWrap)>/gi, '');
    }
    if (this._wordWrap) {
      const replace = '';
      text = text.replace(/[\n\r]+/g, replace);
    }
    text = text.replace(/<(?:BR|line break)>/gi, '\n');
    return text;
  }

  convertEscapeCharacters(text) {
    text = this.setWordWrap(text);
    text = text.replace(/\\/g, '\x1b');
    text = text.replace(/\x1b\x1b/g, '\\');
    text = text.replace(/\x1bV\[(\d+)\]/gi, function() {
      return $gameVariables.value(parseInt(arguments[1]));
    }.bind(this));
    text = text.replace(/\x1bV\[(\d+)\]/gi, function() {
      return $gameVariables.value(parseInt(arguments[1]));
    }.bind(this));
    text = text.replace(/\x1bN\[(\d+)\]/gi, function() {
      return this.actorName(parseInt(arguments[1]));
    }.bind(this));
    text = text.replace(/\x1bP\[(\d+)\]/gi, function() {
      return this.partyMemberName(parseInt(arguments[1]));
    }.bind(this));
    text = text.replace(/\x1bG/gi, Managers.Text.currencyUnit);

    // \NI[n]
    text = text.replace(/\x1bNI\[(\d+)\]/gi, function() {
      return $dataItems[parseInt(arguments[1])].name;
    }.bind(this));
    // \II[n]
    text = text.replace(/\x1bII\[(\d+)\]/gi, function() {
      return this.escapeIconItem(arguments[1], $dataItems);
    }.bind(this));

    return text;
  }

  escapeIconItem(n, database) {
    return `\x1bI[${database[n].iconIndex}]${database[n].name}`;
  }

  actorName(n) {
    return $gameActors.actorName(n);
  }

  partyMemberName(n) {
    const actor = n >= 1 ? $gameParty.members()[n - 1] : null;
    return actor ? actor.name() : '';
  }

  processCharacter(textState) {
    switch (textState.text[textState.index]) {
      case '\n':
        this.processNewLine(textState);
        break;
      case '\f':
        this.processNewPage(textState);
        break;
      case '\x1b':
        this.processEscapeCharacter(this.obtainEscapeCode(textState), textState);
        break;
      default:
        this.processNormalCharacter(textState);
        break;
    }
  }

  textMargin() {
    return 0;
  }

  checkWordWrap(textState) {
    if (!textState) return false;
    if (!this._wordWrap) return false;
    let size = 0;

    //If the current character is a space, check if there' enough space for the next word in this line.
    if (textState.text[textState.index] === ' ') {
      let nextSpace = textState.text.indexOf(' ', textState.index + 1);
      const nextBreak = textState.text.indexOf('\n', textState.index + 1);
      if (nextSpace < 0) nextSpace = textState.text.length + 1;
      if (nextBreak > 0) nextSpace = Math.min(nextSpace, nextBreak);
      const word = textState.text.substring(textState.index, nextSpace);
      size = this.textWidthExCheck(word);
    }

    let lineWidth;
    if (textState.targetWidth) {
      lineWidth = textState.left + textState.targetWidth;
    } else {
      lineWidth = this.contents.width - this.textMargin() * 2;
    }

    //If the current character wasn't a space, then there's no next word here, so it only checks if the line is already past the the end.
    return (size + textState.x > lineWidth);
  }

  saveCurrentWindowSettings() {
    this._saveFontFace = this.contents.fontFace;
    this._saveFontSize = this.contents.fontSize;
    this._savetextColor = this.contents.textColor;
    this._saveFontItalic = this.contents.fontItalic;
    this._saveOutlineColor = this.contents.outlineColor;
    this._saveOutlineWidth = this.contents.outlineWidth;
  }

  restoreCurrentWindowSettings() {
    this.contents.fontFace = this._saveFontFace;
    this.contents.fontSize = this._saveFontSize;
    this.contents.textColor = this._savetextColor;
    this.contents.fontItalic = this._saveFontItalic;
    this.contents.outlineColor = this._saveOutlineColor;
    this.contents.outlineWidth = this._saveOutlineWidth;
  }

  clearCurrentWindowSettings() {
    this._saveFontFace = undefined;
    this._saveFontSize = undefined;
    this._savetextColor = undefined;
    this._saveFontItalic = undefined;
    this._saveOutlineColor = undefined;
    this._saveOutlineWidth = undefined;
  }

  textWidthExCheck(text) {
    const setting = this._wordWrap;
    this._wordWrap = false;
    this.saveCurrentWindowSettings();
    this._testMode = true;
    const value = this.drawTextEx(text, 0, this.contents.height);
    this._testMode = false;
    this.restoreCurrentWindowSettings();
    this.clearCurrentWindowSettings();
    this._wordWrap = setting;
    return value;
  }

  textWidthEx(text) {
    return this.drawTextExGetData(text, 0, this.contents.height).maxWidth;
  }

  textSize(text) {
    return this.drawTextExGetData(text, 0, this.contents.height);
  }

  processNormalCharacter(textState) {
    if (this.checkWordWrap(textState)) {
      return this.processNewLine(textState);
    }

    if (this._quietMode) {
      this.makeFontSlightlySmaller();
    }

    const c = textState.text[textState.index++];
    const w = this.textWidth(c);
    this.contents.drawText(c, textState.x, textState.y, w * 2, textState.height);
    textState.x += w;
  }

  processNewLine(textState) {
    textState.x = textState.left;
    textState.y += textState.height;
    // textState.height = this.calcTextHeight(textState, false);
    textState.index++;
  }

  processNewPage(textState) {
    textState.index++;
  }

  obtainEscapeCode(textState) {
    textState.index++;
    const regExp = /^[$.|^!><{}\\]|^[A-Z]+/i;
    const arr = regExp.exec(textState.text.slice(textState.index));
    if (arr) {
      textState.index += arr[0].length;
      return arr[0].toUpperCase();
    } else {
      return '';
    }
  }

  obtainEscapeParam(textState) {
    const arr = /^\[\d+\]/.exec(textState.text.slice(textState.index));
    if (arr) {
      textState.index += arr[0].length;
      return parseInt(arr[0].slice(1));
    } else {
      return '';
    }
  }

  obtainEscapeString(textState) {
    const arr = /^<(.*?)>/.exec(textState.text.slice(textState.index));
    if (arr) {
      textState.index += arr[0].length;
      return String(arr[0].slice(1, arr[0].length - 1));
    } else {
      return '';
    }
  }

  obtainColorString(textState) {
    const arr = /^\[(.*?)\]/.exec(textState.text.slice(textState.index));
    if (arr) {
      textState.index += arr[0].length;
      return `#${String(arr[0].slice(1, arr[0].length - 1))}`;
    } else {
      return '#ffffff';
    }
  }

  processEscapeCharacter(code, textState) {
    let id;

    switch (code) {
      case 'FS':
        this.contents.fontSize = this.obtainEscapeParam(textState);
        break;
      case 'FN':
        this.contents.fontFace = this.obtainEscapeString(textState);
        break;
      case 'OC':
        id = this.obtainEscapeParam(textState);
        this.contents.outlineColor = this.textColor(id);
        break;
      case 'OW':
        this.contents.outlineWidth = this.obtainEscapeParam(textState);
        break;
      case 'PX':
        textState.x = this.obtainEscapeParam(textState);
        break;
      case 'PY':
        textState.y = this.obtainEscapeParam(textState);
        break;
      case 'C':
        this.changeTextColor(this.textColor(this.obtainEscapeParam(textState)));
        break;
      case 'I':
        this.processDrawIcon(this.obtainEscapeParam(textState), textState);
        break;
      case 'SI':
        this.processDrawSmallIcon(this.obtainEscapeParam(textState), textState);
        break;
      case 'HI':
        this.processDrawHalfIcon(this.obtainEscapeParam(textState), textState);
        break;
      case '{':
        this.makeFontBigger();
        break;
      case '}':
        this.makeFontSmaller();
        break;
      case 'QUIET':
        this._quietMode = true;
        break;
      case 'RESET' :
        this.resetCustomFontSettings();
        break;
      case 'HC':
        this.changeTextColor(this.obtainColorString(textState).toLowerCase());
        break;
    }
  }

  processDrawIcon(iconIndex, textState) {
    this.drawIcon(iconIndex, textState.x + 2, textState.y + 2);
    textState.x += Core.BasicWindow._iconWidth + 2;
  }

  processDrawSmallIcon(iconIndex, textState) {
    this.drawSmallIcon(iconIndex, textState.x + 2, textState.y + 6);
    textState.x += (Core.BasicWindow._iconWidth / 4) + 6;
  }

  processDrawHalfIcon(iconIndex, textState) {
    const z = Graphics.windowZoomLevel;
    this.drawIcon(iconIndex, textState.x, textState.y + 2 * z, z / 2);
    textState.x += (Core.BasicWindow._iconWidth * z / 2) + 2 * z;
  }

  makeFontBigger() {
    if (this.contents.fontSize <= 96) {
      this.contents.fontSize += Utils.convertFontSize(4, Graphics.windowZoomLevel);
    }
  }

  makeFontSmaller() {
    if (this.contents.fontSize >= 18) {
      this.contents.fontSize -= Utils.convertFontSize(4, Graphics.windowZoomLevel);
    }
  }

  makeFontSlightlySmaller() {
    if (this.contents.fontSize >= 2) {
      this.contents.fontSize -= Utils.convertFontSize(0.5, Graphics.windowZoomLevel);
    }
  }

  getHeightFromCache(text, all, fontSize) {
    const len = this._heightCache.length;
    for (let i = 0; i < len; i++) {
      const itemData = this._heightCache[i];

      if (!itemData) continue;

      if (itemData.all != all) continue;
      if (itemData.fontSize != fontSize) continue;
      if (itemData.text != text) continue;

      return itemData.textHeight;
    }

    return null;
  }

  storeTextHeightOnCache(text, all, fontSize, textHeight) {
    const itemData = {
      all,
      fontSize,
      text,
      textHeight
    };

    this._heightCache.push(itemData);
  }

  extraLineHeight() {
    return 1 * Graphics.windowZoomLevel;
  }

  calcTextHeight(textState, all) {
    const text = textState.text;

    const heightFromCache = this.getHeightFromCache(text, all, this.contents.fontSize);
    if (heightFromCache) return heightFromCache;

    const lastFontSize = this.contents.fontSize;
    let textHeight = 0;
    const lines = text.slice(textState.index).split('\n');
    const maxLines = all ? lines.length : 1;

    for (let i = 0; i < maxLines; i++) {
      let maxFontSize = this.contents.fontSize;
      const regExp = /\x1b[{}]/g;
      for (;;) {
        const array = regExp.exec(lines[i]);
        if (array) {
          if (array[0] === '\x1b{') {
            this.makeFontBigger();
          }
          if (array[0] === '\x1b}') {
            this.makeFontSmaller();
          }
          if (maxFontSize < this.contents.fontSize) {
            maxFontSize = this.contents.fontSize;
          }
        } else {
          break;
        }
      }

      textHeight += maxFontSize;
      textHeight += this.extraLineHeight();
    }

    this.contents.fontSize = lastFontSize;
    this.storeTextHeightOnCache(text, all, lastFontSize, textHeight);
    return textHeight;
  }

  drawIcon(iconIndex, x, y, zoom) {
    zoom = zoom || Graphics.windowZoomLevel;

    const gw = Core.BasicWindow._iconWidth;
    const gh = Core.BasicWindow._iconHeight;

    const bitmap = Managers.Images.loadIcon(iconIndex);
    this.contents.blt(bitmap, 0, 0, gw, gh, x, y, gw * zoom, gh * zoom);
  }

  _drawBitmap(bitmap, x, y, zoomLevel, preferRefresh) {
    if (x === undefined) x = 0;
    if (y === undefined) y = 0;
    if (zoomLevel === undefined) zoomLevel = 1;

    const drawBitmap = () => {
      let drawWidth = bitmap._canvas.width;
      let drawHeight = bitmap._canvas.height;

      drawWidth *= zoomLevel;
      drawHeight *= zoomLevel;

      this.contents.bltBitmap(bitmap, 0, 0, bitmap._canvas.width, bitmap._canvas.height, x, y, drawWidth, drawHeight);
    };

    if (bitmap.isReady()) {
      drawBitmap();
    } else {
      bitmap.addLoadListener(() => {
        if (preferRefresh) {
          this.refresh();
        } else {
          drawBitmap();
        }
      });
    }
  }

  drawPicture(filename, x, y, zoomLevel, preferRefresh) {
    const bitmap = Managers.Images.loadPicture(filename);
    this._drawBitmap(bitmap, x, y, zoomLevel, preferRefresh);
  }

  drawItemBadge(iconIndex, x, y, w, h) {
    const bitmap = Managers.Images.loadItemBadge(iconIndex);

    const pw = Core.BasicWindow._iconWidth;
    const ph = Core.BasicWindow._iconHeight;

    if (w === undefined) w = pw;
    if (h === undefined) h = ph;

    this.contents.blt(bitmap, 0, 0, pw, ph, x, y, w, h);
  }

  drawIconResized(iconIndex, x, y, w, h) {
    const bitmap = Managers.Images.loadResizedIcon(iconIndex, w, h);
    this.contents.blt(bitmap, 0, 0, w, h, x, y, w, h);

    // var bitmap = Managers.Images.loadSystem('ItemIcons');
    // var pw = Core.BasicWindow._iconWidth;
    // var ph = Core.BasicWindow._iconHeight;
    // var sx = iconIndex % 16 * pw;
    // var sy = Math.floor(iconIndex / 16) * ph;
    // var gw = Core.BasicWindow._iconWidth;
    // var gh = Core.BasicWindow._iconHeight;

    // var diffW = pw - gw;
    // var diffH = ph - gh;
    // var diffX = diffW / 2;
    // var diffY = diffH / 2;

    // var ratioW = pw / gw;
    // var ratioH = ph / gw;

    // this.contents.bltBitmap(bitmap, sx, sy, pw, ph, x - diffX, y - diffY, w * ratioW, h * ratioH);
  }

  drawFace(faceName, faceIndex, x, y, width=Core.BasicWindow._faceWidth, height=Core.BasicWindow._faceHeight) {
    const bitmap = Managers.Images.loadFace(faceName);
    const pw = Core.BasicWindow._faceWidth;
    const ph = Core.BasicWindow._faceHeight;
    const sw = Math.min(width, pw);
    const sh = Math.min(height, ph);
    const dx = Math.floor(x + Math.max(width - pw, 0) / 2);
    const dy = Math.floor(y + Math.max(height - ph, 0) / 2);
    const sx = faceIndex % 4 * pw + (pw - sw) / 2;
    const sy = Math.floor(faceIndex / 4) * ph + (ph - sh) / 2;
    this.contents.bltBitmap(bitmap, sx, sy, sw, sh, dx, dy);
  }

  drawCharacter(characterName, characterIndex, x, y, zoom) {
    if (zoom === undefined) zoom = 1;
    if (Managers.Images.characterIsInvalid(characterName)) return;

    const match = characterName.match(/\[(\d*)\]/);
    const frameCount = (match && match.length >= 2) ? parseInt(match[1]) : 3;

    const bitmap = Managers.Images.loadCharacter(characterName);
    const big = Managers.Images.isBigCharacter(characterName);
    const isRow = Managers.Images.isCharacterRow(characterName);

    const pw = bitmap.width / (big ? frameCount : (frameCount * 4));
    const ph = isRow ? bitmap.height : (bitmap.height / (big ? 4 : 8));
    const n = characterIndex;
    const sx = (n % 4 * frameCount + 1) * pw;
    const sy = (Math.floor(n / 4) * 4) * ph;

    const dw = pw * zoom;
    const dh = ph * zoom;
    const dx = x - dw / 2;
    const dy = y - dh;

    this.contents.bltBitmap(bitmap, sx, sy, pw, ph, dx, dy, dw, dh);
  }

  drawCharacterIcon(characterName, characterIndex, x, y, zoom) {
    const bitmap = Managers.Images.loadCharacterIcon(characterName, characterIndex, zoom);

    if (!bitmap.isReady()) return false;

    this.contents.blt(bitmap, 0, 0, bitmap._canvas.width, bitmap._canvas.height, x, y);
    return true;
  }

  drawCharacterIconCentered(characterName, characterIndex, x, y, w, h, zoom, direction) {
    const bitmap = Managers.Images.loadCharacterIcon(characterName, characterIndex, zoom, direction);

    if (!bitmap.isReady()) return false;

    const diffX = w - bitmap._canvas.width;
    const diffY = h - bitmap._canvas.height;
    const newX = x += diffX / 2;
    const newY = y += diffY / 2;

    this.contents.blt(bitmap, 0, 0, bitmap._canvas.width, bitmap._canvas.height, newX, newY);
    return true;
  }

  drawActorCharacter(actor, x, y) {
    this.drawCharacter(actor.characterName(), actor.characterIndex(), x, y);
  }

  drawActorFace(actor, x, y, width, height) {
    this.drawFace(actor.faceName(), actor.faceIndex(), x, y, width, height);
  }

  drawActorName(actor, x, y, width=168) {
    this.changeTextColor(this.normalColor());
    this.drawText(actor.name(), x, y, width);
  }

  drawActorNickname(actor, x, y, width=270) {
    this.resetTextColor();
    this.drawText(actor.nickname(), x, y, width);
  }

  drawCurrentAndMax(current, max, x, y, width, color1, color2) {
    const labelWidth = this.textWidth('HP');
    const valueWidth = this.textWidth('0000');
    const slashWidth = this.textWidth('/');
    const x1 = x + width - valueWidth;
    const x2 = x1 - slashWidth;
    const x3 = x2 - valueWidth;
    if (x3 >= x + labelWidth) {
      this.changeTextColor(color1);
      this.drawText(current, x3, y, valueWidth, 'right');
      this.changeTextColor(color2);
      this.drawText('/', x2, y, slashWidth, 'right');
      this.drawText(max, x1, y, valueWidth, 'right');
    } else {
      this.changeTextColor(color1);
      this.drawText(current, x1, y, valueWidth, 'right');
    }
  }

  drawItemName(item, x, y, width, drawNumber, drawIcon, align) {
    if (!item) {
      return;
    }

    if (drawNumber === undefined) drawNumber = true;
    if (drawIcon === undefined) drawIcon = true;

    width = width || 312;

    let iconBoxWidth = 0;
    this.resetTextColor();

    if (drawIcon) {
      iconBoxWidth = Core.BasicWindow._iconWidth + 4;
      this.drawIcon(item.iconIndex, x + 2, y + 2);
    }

    if (drawNumber && Inventory.itemTypeIsStackable(item.type)) {
      const number = Inventory.numItems(item);
      const numberWidth = this.textWidth(number);

      this.drawText(number, x + iconBoxWidth, y, numberWidth);
      this.drawText('x', x + iconBoxWidth + numberWidth, y, 20);
      iconBoxWidth += numberWidth + 20;
    }

    this.drawText(Managers.Text.item(item.id), x + iconBoxWidth, y, width - iconBoxWidth, align);
  }

  drawCurrencyValue(value, unit, x, y, width) {
    const unitWidth = Math.min(80, this.textWidth(unit));
    this.resetTextColor();
    this.drawText(value, x, y, width - unitWidth - 6, 'right');
    this.drawText(unit, x + width - unitWidth, y, unitWidth, 'right');
  }

  canvasToLocalX(x) {
    let node = this;
    while (node) {
      x -= node.x;
      node = node.parent;
    }
    return x;
  }

  canvasToLocalY(y) {
    let node = this;
    while (node) {
      y -= node.y;
      node = node.parent;
    }
    return y;
  }

  hit(x, y) {
    if (x >= this.x && x <= this.x + this._width) {
      if (y >= this.y && y <= this.y + this._height) {
        return true;
      }
    }

    return false;
  }

  isOnCurrentScene() {
    const layer = Managers.Scenes._scene._windowLayer;
    if (!layer) return false;

    return this.parent == layer;
  }

  drawSystem(filename, x, y) {
    if (x === undefined) x = 0;
    if (y === undefined) y = 0;

    const bitmap = Managers.Images.loadSystem(filename);
    const me = this;

    let drawBitmap;

    drawBitmap = () => {
      const drawWidth = bitmap._canvas.width;
      const drawHeight = bitmap._canvas.height;

      me.contents.bltBitmap(bitmap, 0, 0, bitmap._canvas.width, bitmap._canvas.height, x, y, drawWidth, drawHeight);
    };

    drawBitmap.bind(this);
    bitmap.addLoadListener(drawBitmap);
  }

  getPositionLabel(/*x, y*/) {
    return '';
  }
}

Core.BasicWindow = BasicWindow;
module.exports = BasicWindow;

Core.BasicWindow._iconWidth  = 24;
Core.BasicWindow._iconHeight = 24;

Core.BasicWindow._faceWidth  = 72;
Core.BasicWindow._faceHeight = 72;