--[[----------------------------------------------------------------------------
JPEGminiPluginManager.lua
Copyright 2009-2013 ICVT Ltd. All rights reserved
------------------------------------------------------------------------------]]

local LrFileUtils = import "LrFileUtils"
local LrView = import "LrView"
local LrHttp = import "LrHttp"
local LrBinding = import "LrBinding"
local app = import 'LrApplication'
local LrPathUtils = import 'LrPathUtils'
local LrTasks = import "LrTasks"
local LrDialogs = import 'LrDialogs'
local LrRecursionGuard = import 'LrRecursionGuard'
local LrFunctionContext = import 'LrFunctionContext'
local LrLogger = import 'LrLogger'
local myLogger = LrLogger( 'libraryLogger' )

local paneWidth = 440

myLogger:enable( "print" ) -- or "logfile"

PluginManager = {}

errorTable = {
	[0]="No error",
	[1]="No license",
	[2]="Bad Parameters",
	[3]="Library initialization failed",
	[6]="Incompatible .Net version",
	[7]="Can't run dot net subprocess",
	[100]="Bad license key",
	[101]="No connection",
	[102]="Server error",
	[103]="Invalid license",
	[104]="License expired",
	[105]="License update period expired",
	[106]="The license is already activated on a different computer",
	[107]="The license doesn't apply to this product",
	[108]="General Error",
	[1001]="Failed to load image data",
	[1002]="Failed to init image",
	[1003]="Failed to get image info",
	[1004]="Failed to optimize image",
	[1005]="No further optimization is possible",
	[1006]="Image already optimized",
	[1007]="Image exceed size (MP) limit",
	[1008]="Image file is locked",
	[1009]="Failed to create destination directory",
	[1010]="Image file write failed",
	[1011]="Out of memory",
	[1012]="File is hard-link",
	[1013]="Image is hightly compressed",
	[1014]="Image format not supported",
	[1015]="Image format not supported for in-place recompression",
	[1016]="Failed to initialize HEIC extract library",
	[1017]="Failed to initialize HEIC image",
	[1018]="Failed to extract HEIC image",
	[1019]="Failed to initialize raw image",
	[1020]="Failed to set metadata",
	[1021]="Failed to delete original",
	[1022]="Failed to initialize JPEGmini library",
	[1023]="Failed to get HEIC image info",
	[3009]="License not bound",
	[3011]="License payload invalid",
	[3012]="License binding failed",
	[3013]="Post modification re-fetch failed",
	[3014]="Unbinding failed",
	[3015]="Invalid Machine ID",
	[3016]="Yarls internal error"
}

function stringFromToolResult(r)
	errorString = errorTable[r]
	if errorString == nil then
		errorString = "Unknown error: " .. tostring(r)
	end
	return errorString
end

function switch_to_status(p, prompt)
	p.status_pane_caption = prompt
	p.current_pane = "status_pane"
end

function switch_to_code(p)
	p.current_pane = "enter_code_pane"
end

function switch_to_rebind(p)
	p.current_pane = "rebind_pane"
end

function switch_to_limbo(p, error_code)
	p.current_pane = "limbo_pane"
	if error_code ~= 0 then
		p.limbo_message = "Your license cannot be used to activate JPEGmini because of the following error: " .. stringFromToolResult(error_code) .. "."
	end
end

function setup_activation_ok(p)
	switch_to_status(p, "Activated")
	_G.activationStatus = 1;
end

function rebind_action(p)
	switch_to_status(p, "Activating...")
	LrFunctionContext.postAsyncTaskWithContext("",function(context)
		LrDialogs.attachErrorDialogToFunctionContext(context)
		if p.activation_code:len() > 0 then
			params = {action="rebind", code=p.activation_code}
		else
			params = {action="rebind"}
		end
		rebindResult = PluginManager.executeCommand(params,false)
		if rebindResult == 0 then
			setup_activation_ok(p)
		else
			LrDialogs.message("Activation failed (" .. stringFromToolResult(rebindResult) .. ")")
			if p.activation_code:len() > 0 then
				switch_to_code(p)
			else
				switch_to_rebind(p)
			end
		end
	end)
end

function refetch_action(p)
	switch_to_status(p, "Checking License...")

	LrFunctionContext.postAsyncTaskWithContext("",function(context)
		LrDialogs.attachErrorDialogToFunctionContext(context)
		
		refetchResult = PluginManager.executeCommand({action="refetch"},false)
		if refetchResult == 0 then
			setup_activation_ok(p)
		else
			LrDialogs.message("Check failed (" .. stringFromToolResult(refetchResult) .. ")")
			switch_to_limbo(p);
		end
	end)
end

function activate_action(p)
	if p.activation_code:len() > 0 then
		switch_to_status(p, "Activating...")
		LrFunctionContext.postAsyncTaskWithContext("",function(context)
			LrDialogs.attachErrorDialogToFunctionContext(context)
			activationResult = PluginManager.executeCommand({action="activate", code=p.activation_code},false)

			if activationResult == 0 then
				setup_activation_ok(p)
			elseif activationResult == 106 then
				if LrDialogs.confirm("Number of activations exceeded", "Your license has been activated on the maximum allowed number of computers. Click \"Deactivate other license\" to deactivate JPEGmini Pro on one of the other computers and activate it on this computer.", "Deactivate other license", "Cancel") == "ok" then
					rebind_action(p)
				else
					switch_to_code(p)
				end
			else
				LrDialogs.message("Activation failed (" .. stringFromToolResult(activationResult) .. ")")
				switch_to_code(p)
			end
		end)
	else
		LrDialogs.message("Please enter a valid activation code")
	end
end


function make_status_pane(f,p)
	return f:view {
		bind_to_object = p,
		visible = LrBinding.keyEquals("current_pane","status_pane"),
		f:row {
			height = 15,
		},
		f:row {
			spacing = f:label_spacing(),
			f:static_text {
				width = paneWidth,
				height = 100,
				size = 24,
				font = "<system/bold>",
				title = LrView.bind("status_pane_caption"),
			},
		},
	}
end

function make_limbo_pane(f,p)
	return f:view {
		bind_to_object = p,
		visible = LrBinding.keyEquals("current_pane","limbo_pane"),
		f:row {
			height = 15,
		},
		f:row {
			f:static_text {
				width = paneWidth,
				title = "License Invalid",
				font = "<system/bold>"
			},
		},
		f:row {
			height = 5,
		},
		f:row {
			f:static_text {
				width = paneWidth,
				height = 75,
				title = LrView.bind('limbo_message',p),
			},
		},
		f:row {
			height = 10,
		},
		f:row {
			height = 20,
			f:static_text {
				width = 285,
				title = ""
			},
			f:push_button {
				width = 150,
				title = 'Try Again',
				action = function ()
					refetch_action(p)
				end
			},
		},		
	}
end


function make_rebind_pane(f,p)
	return f:view {
		bind_to_object = p,
		visible = LrBinding.keyEquals("current_pane","rebind_pane"),
		f:row {
			height = 15,
		},

		f:row {
			f:static_text {
				width = paneWidth,
				title = "Number of activations exceeded",
				font = "<system/bold>"
			},
		},
		f:row {
			height = 5,
		},
		f:row {
			f:static_text {
				width = paneWidth,
				height = 75,
				title = "Your license has been activated on the maximum allowed number of computers. Click \"Deactivate other license\" to deactivate JPEGmini Pro on one of the other computers and activate it on this computer."
			},
		},
		f:row {
			height = 10,
		},
		f:row {
			height = 20,
			f:static_text {
				width = 285,
				title = ""
			},
			f:push_button {
				width = 150,
				title = 'Deactivate other license',
				action = function ()
					rebind_action(p)
				end
			},
		},		
	}
end

function make_enter_code_pane(f,p)
	return f:view {
		bind_to_object = p,
		visible = LrBinding.keyEquals("current_pane","enter_code_pane"),
		f:row {
			height = 15,
		},
		f:row {
			f:static_text {
				width = paneWidth,
				title = "Please Enter your Activation Code:"
			}
		},
		f:row {
			height = 10,
				},
				f:row {
					spacing = f:control_spacing(),
					f:edit_field {
						width = 275,
				value = LrView.bind('activation_code',p),
					},
					f:push_button {
						width = 150,
						title = 'Activate',
				action = function ()
					activate_action(p)	
				end
			},
		},
	}
end


function PluginManager.sectionsForTopOfDialog( f, p )
	_G.activationStatus = 0;
										p.activation_code = ""
	switch_to_status(p, "Checking License...")
	LrFunctionContext.postAsyncTaskWithContext("check license",function(context)
		LrDialogs.attachErrorDialogToFunctionContext(context)
		checkResult = PluginManager.executeCommand({action="check"},false)
			if checkResult == 0 then
				setup_activation_ok(p)
			elseif checkResult == 106 then
				switch_to_rebind(p)
			elseif checkResult == 1 then
				switch_to_code(p)
			else
				switch_to_limbo(p, checkResult)
			end
								end)

		return {
			{
				bind_to_object = p,
				title = "JPEGmini",
				f:row {
					f:column {
					f:picture {
						width = 128,
						value = _PLUGIN.path .. "/can_big.png",
						},
					},
				f:column {
					place = "overlapping",
					make_status_pane(f,p),
					make_enter_code_pane(f,p),
					make_limbo_pane(f,p),
					make_rebind_pane(f,p)
				},
				},
			},
		}
	end

function unescape_string(escaped_string)
	local length = string.len(escaped_string)
	local result = ""
	local status = 1
	for i=1,length do
		local c = string.sub(escaped_string,i,i)
		if status == 1 then
			if c == "\\" then
				status = 2
			else
				result = result .. c
			end
		elseif status == 2 then
			if c == '\\' then
				result = result .. '\\'
				status = 1
			elseif c == '3' then
				status = 3
			else
				status = 1
			end
		elseif status == 3 then
			if c == 'A' then
				result = result .. ':'
			elseif c == 'B' then
				result = result .. ';'
			end
			status = 1
		else
			status = 1
		end
	end
	return result
end


function result_dictionary(string)
	local length = string.len(string)
	local result = {}
	local status = 1
	local key, value
	key = ""
	value = ""
	for i=1,length do
		local c = string.sub(string,i,i)
		if status == 1 then
			if c == ":" then
				status = 2
			else
				key = key .. c
			end
		elseif status == 2 then
			if c == ";" then
				result[key] = unescape_string(value)
				key = ""
				value = ""
				status = 1
			else
				value = value .. c
			end
		end
	end
	return result
end

function describe_dictionary(dictionary)
	if dictionary ~= nil then
		local result = ""
		for key, value in pairs(dictionary) do
			local v = value
			if type(v) ~= "string" then
				v = tostring(value)
			end
			result = result .. "\"" .. key .. "\":\"" .. v .. "\",\n"
		end
		return result;
	else
		return "(nil)"
	end
end

function describe_dictionary_types(dictionary)
	local result = ""
	for key, value in pairs(dictionary) do
		result = result .. "\"" .. key .. "\":\"" .. type(value) .. "\",\n"
	end
	return result;
end



function PluginManager.executeCommand(params, full)
	local command;
	
	if MAC_ENV == true then
		command = LrPathUtils.child( _PLUGIN.path, "mac" )
	else
		command = LrPathUtils.child( _PLUGIN.path, "win" )
	end
	command = '"' .. LrPathUtils.child(command, "JPEGminiTool" ) .. '"'

	for k,v in pairs(params) do
		command = command .. " -" .. k .. ' "' .. v .. '"'
	end
	
	local tempfile = os.tmpname ()
	command = command .. " -result " .. ' "' .. tempfile .. '"' .. " -full-info YES"

	if WIN_ENV == true then
		command = '"' .. command .. '"'
	end
	local resultContent = LrTasks.execute(command)
	if LrFileUtils.exists(tempfile) then
		    local f = io.open(tempfile, "r")
    		resultContent = f:read("*all")
   		 	f:close()
			LrFileUtils.delete( tempfile )
			local rd = result_dictionary(resultContent)
			if full then
				return rd
			else
				return tonumber(rd["result"])
			end
	end
	if full then
		return nil
	else
		return 9999
	end
end
