--[[----------------------------------------------------------------------------
JPEGminiFilterProvider.lua
Copyright 2009-2013 ICVT Ltd. All rights reserved
------------------------------------------------------------------------------]]

-- Lightroom SDK

local LrView = import 'LrView'
local bind = LrView.bind
local LrPathUtils = import 'LrPathUtils'
local LrTasks = import "LrTasks"
local LrLogger = import 'LrLogger'
local myLogger = LrLogger( 'libraryLogger' )
local LrDialogs = import 'LrDialogs'
local LrColor = import 'LrColor'
local LrBinding = import "LrBinding"

require 'JPEGminiPluginManager'


myLogger:enable( "print" ) -- or "logfile"

function magnitudeSuffixWithMagnitude(magnitude)
	local ss = {"B","KB","MB","GB","TB","PB","EB","ZB"}
	local r = ss[magnitude]
	if r == nil then
		return "??"
	end
	return r
end	
	

function qualifiedSizeStringWithBytes(size)
	local m
		
	m = 1
	while size >= 1000 do
		size = size / 1000
		m = m + 1
	end
	return string.format("%.1f ",size) .. magnitudeSuffixWithMagnitude(m)
end

function updateStats(infoDict, propertyTable)
	local ts	-- total savings
	local ss	-- session savings
	local sp	-- session percent
	
	if tonumber(infoDict["total-output"]) > 0 then
		local so = tonumber(infoDict["session-output"])
		local si = tonumber(infoDict["session-input"]);
		ts = tonumber(infoDict["total-input"]) - tonumber(infoDict["total-output"])
		ss = si - so
		if (si > 0) then
			sp = (ss / si)*100
		else
			sp = 0;
		end
				
		ts = qualifiedSizeStringWithBytes(ts)
		if si > 0 then
			ss = qualifiedSizeStringWithBytes(ss)
			sp = string.format("%.1f %%",sp)
		else
			ss = "none"
			sp = "none"
		end
	else
		ts = "0"
		ss = "0"
		sp = "0"
	end
	propertyTable.session_savings = ss
	propertyTable.session_percentage = sp
	propertyTable.total_savings = ts
end
function make_block_pane(f,p)
	return f:view {
		bind_to_object = p,
		visible = LrBinding.keyEquals("current_pane","block_pane"),
		f:picture {
			value = _PLUGIN.path .. "/export_panel_background_inactive.png",
		}
	}
end

function make_export_pane(f, p)
	local leftMarginWidth = 185
	local leftSideWidth = 78
	local rightSideWidth = 130
	local fontName = "<system/bold>"
	local fontSize = 24
	local topSpacingHeight = 39
	local spacingRowHeight = 10
	return f:view {
		bind_to_object = p,
		visible = LrBinding.keyEquals("current_pane","export_pane"),
		f:row {
			place = "overlapping",
			f:column {
				f:row {
					height = topSpacingHeight,
				},
				f:row {
					f:column {
						width = leftMarginWidth
					},
					f:static_text {
						width = leftSideWidth,
						alignment = "center",
						size = "regular",
						font = fontName,
						title = LrView.bind('session_savings',p)
					},
					f:static_text {
						width = rightSideWidth,
						alignment = "center",
						size = "regular",
						font = fontName,
						title = "Last Savings"
					},
				},
				f:row {
					height = spacingRowHeight,
				},
				f:row {
					f:column {
						width = leftMarginWidth
					},
					f:static_text {
						width = leftSideWidth,
						alignment = "center",
						size = "regular",
						font = fontName,
						title = LrView.bind('session_percentage',p)
					},
					f:static_text {
						width = rightSideWidth,
						alignment = "center",
						size = "regular",
						font = fontName,
						title = "Reduction Ratio"
					},
				},
				f:row {
					height = spacingRowHeight,
				},
				f:row {
					f:column {
						width = leftMarginWidth
					},
					f:static_text {
						width = leftSideWidth,
						alignment = "center",
						size = "regular",
						font = fontName,
						title = LrView.bind('total_savings',p)
					},
					f:static_text {
						width = rightSideWidth,
						alignment = "center",
						size = "regular",
						font = fontName,
						title = "Total Savings"
					},
				},
			},
			f:column {
				f:picture {
					value = _PLUGIN.path .. "/export_panel_background.png",
				}
			}
		}
	}
end

--============================================================================--

local JPEGminiFilterProvider = {}

function JPEGminiFilterProvider.startDialog(propertyTable)
	LrTasks.startAsyncTask(function()
		while _G.activationStatus == 0 do
			LrTasks.sleep(0.5)
		end
		if _G.activationStatus ~= 1 then
			LrDialogs.message("JPEGmini Plug-in not activated","To Activate JPEGmini Please Open the plug-in manager, Click \"JPEGmini\" and follow the on screen instructions.")
		end
		if _G.activationStatus == 1 then
			updateStats(PluginManager.executeCommand({action="check"}, true), propertyTable)
		end
	end)
	return {}
end

function JPEGminiFilterProvider.sectionForFilterInDialog(f, p)
	p.current_pane = "block_pane"
	if _G.activationStatus == 1 then
		p.current_pane = "export_pane"
	else
		p.current_pane = "block_pane"
	end
	if _G.activationStatus == 0 then
		LrTasks.startAsyncTask(function()
			while _G.activationStatus == 0 do
				LrTasks.sleep(0.5)
			end
			if _G.activationStatus == 1 then
				p.current_pane = "export_pane"
			else
				p.current_pane = "block_pane"
			end
		end)
	end
	return {
		title = 'JPEGmini',
		f:row {
			place = "overlapping",
			make_export_pane(f, p),
			make_block_pane(f,p)
		}
	}
end

-------------------------------------------------------------------------------

function JPEGminiFilterProvider.postProcessRenderedPhotos( functionContext, filterContext )
	PluginManager.executeCommand({action="reset-session"},false)

	local renditionOptions = {
		plugin = _PLUGIN,
		renditionsToSatisfy = filterContext.renditionsToSatisfy,
		filterSettings = function( renditionToSatisfy, exportSettings )
			--exportSettings.LR_format = 'JPEG'
			--exportSettings.LR_jpeg_quality = 1
			--return os.tmpname()
			-- This hook function gives you the opportunity to change the render
			-- settings for each photo before Lightroom renders it.
			-- For example, if you wanted Lightroom to generate TIFF files,
			-- you can add below statements:
			-- exportSettings.LR_format = TIFF
			-- return os.tmpname()
			-- By doing so, you assume responsibility for creating
			-- the file type that was originally requested and placing it
			-- in the location that was originally requested in your filter loop below.
			
		end,
	}
		
	local command
	local quotedCommand
			
			
	
	for sourceRendition, renditionToSatisfy in filterContext:renditions( renditionOptions ) do
	
		-- Wait for the upstream task to finish its work on this photo.
		
		local success, pathOrMessage = sourceRendition:waitForRender()
		if success then
		
			-- Now that the photo is completed and available to this filter,
			-- you can do your work on the photo here. This sample passes the user's settings,
			-- the title and language to an external application to update the metadata.
			
			local path = pathOrMessage
			local ext = string.lower(string.match(path,".*%.(.*)"))

			if ext == "jpeg" or ext == "jpg" then
				local commandResult = PluginManager.executeCommand({action="process", path=path},false)
				if commandResult == 0 or commandResult == 1005 or commandResult == 1006 then
					renditionToSatisfy:renditionIsDone (true,"")
				else
					renditionToSatisfy:renditionIsDone(false, "Failed to process image: " .. stringFromToolResult(commandResult))
				end
			else
				renditionToSatisfy:renditionIsDone (true,"")
			end

			
		
		end
	
	end

end

-------------------------------------------------------------------------------

return JPEGminiFilterProvider
