/*
 * Decompiled with CFR 0.152.
 */
package net.sourceforge.jocular.math;

import java.util.ArrayList;
import java.util.List;
import net.sourceforge.jocular.math.Polynomial;

public class SturmSolver {
    private final Polynomial m_poly;
    private final Polynomial m_deriv;
    private final Polynomial m_2ndDeriv;
    private final List<Polynomial> m_polys;
    private final List<RangeResult> m_roots = new ArrayList<RangeResult>();
    private static int maxId = 0;

    public SturmSolver(Polynomial p) {
        double g = p.get(p.order());
        this.m_poly = p.multiplyBy(1.0 / g).removeInsignificantCoefficients();
        this.m_deriv = this.m_poly.derivative();
        this.m_2ndDeriv = this.m_deriv.derivative();
        this.m_polys = this.makePolyList();
    }

    public boolean solve(double min, double max) {
        this.m_roots.clear();
        RangeResult rr = new RangeResult(new PointResult(min), new PointResult(max));
        this.solveStep(rr);
        return this.m_roots.size() > 0;
    }

    private void solveStep(RangeResult rr) {
        if (rr != null) {
            if (rr.isAtRoot()) {
                this.addRoot(rr);
            } else if (rr.containsRoot()) {
                PointResult mid = new PointResult(rr.getMid());
                RangeResult rrMax = new RangeResult(mid, rr.pMax);
                rr.pMax = mid;
                this.solveStep(rr);
                this.solveStep(rrMax);
            }
        }
    }

    private synchronized void addRoot(RangeResult r) {
        boolean found = false;
        for (RangeResult rr : this.m_roots) {
            if (rr.pMin == r.pMax) {
                rr.setMin(r);
                found = true;
                continue;
            }
            if (rr.pMax == r.pMin) {
                rr.setMax(r);
                found = true;
                break;
            }
            if (!(Math.abs(rr.getRoot() - r.getRoot()) < 5.0E-14)) continue;
            found = true;
            if (rr.pMax.x < r.pMin.x) {
                rr.setMax(r);
                continue;
            }
            if (!(rr.pMin.x > r.pMax.x)) continue;
            rr.setMin(r);
        }
        if (!found) {
            this.m_roots.add(r);
        }
    }

    public double[] getRoots() {
        double[] result = new double[this.m_roots.size()];
        for (int i = 0; i < this.m_roots.size(); ++i) {
            result[i] = this.m_roots.get((int)i).pMin.x;
        }
        return result;
    }

    public List<Polynomial> getRealRoots() {
        ArrayList<Polynomial> result = new ArrayList<Polynomial>();
        for (RangeResult d : this.m_roots) {
            double v = this.m_poly.evaluate(d.getRoot());
            if (!(Math.abs(v) < 5.0E-14)) continue;
            result.add(Polynomial.makeFromRealRoots(new double[]{d.getRoot()}));
        }
        return result;
    }

    public static void main(String[] args) {
        Polynomial p1 = Polynomial.makeFromRealRoots(new double[]{0.0, 0.5, 0.5, 1.0});
        p1 = Polynomial.parseString("-5.42882554047509E-6*x^6 + -17.8558722976245E-6*x^5 + 148.319840372031E-6*x^4 + 1.65332150903931E-6*x^3 + -445.133303772598E-6*x^2 + 0E0*x + 191.356656138809E-9");
        p1 = Polynomial.parseString("2.29306845777999E-3*x^6 + -8.77347931672343E-3*x^5 + 8.3920236942572E-3*x^4 + 429.88884886378E-6*x^3 + -922.396058695927E-6*x^2 + 0E0*x + 20.1481580009546E-6");
        p1 = p1.dividedBy(Polynomial.makeFromRealRoots(new double[]{0.17999588317834}))[0];
        p1 = p1.dividedBy(Polynomial.makeFromRealRoots(new double[]{-0.16791228742781}))[0];
        p1 = p1.dividedBy(Polynomial.makeFromRealRoots(new double[]{-0.26114381820723}))[0];
        p1 = p1.dividedBy(Polynomial.makeFromRealRoots(new double[]{0.31635989894961}))[0];
        p1 = p1.dividedBy(Polynomial.makeFromRealRoots(new double[]{1.764595866739104}))[0];
        p1 = p1.dividedBy(Polynomial.makeFromRealRoots(new double[]{1.994191413289727}))[0];
        p1 = Polynomial.parseString("23.1878883820835E6*x^6 + -139.127330292502E6*x^5 + 208.690995438755E6*x^4 + 92.751470662322E6*x^3 + -278.254411986972E6*x^2 + 3.10056134599332E-6*x + 92.7513877963854E6");
        List<Polynomial> rs = SturmSolver.factor(p1, 0.0, 1.0);
        rs = p1.factor();
        Polynomial r = null;
        if (rs.size() > 0) {
            r = rs.get(0);
        }
        System.out.println("SturmSolver.main found " + rs.size() + " roots: " + r + " ");
    }

    public static List<Polynomial> factor(Polynomial p, double min, double max) {
        List<Polynomial> result;
        if (p.order() <= 3) {
            List<Polynomial> ps = p.cubicRealRoots();
            result = new ArrayList<Polynomial>();
            for (Polynomial tp : ps) {
                double r;
                if (tp.order() != 1 || !((r = tp.linearRoot()) >= min) || !(r <= max)) continue;
                result.add(tp);
            }
        } else {
            SturmSolver ss = new SturmSolver(p);
            int startMaxId = maxId;
            ss.solve(min, max);
            result = ss.getRealRoots();
            if (maxId - startMaxId > 500) {
                System.out.println("SturmSolver.factor " + maxId + " points calc'ed");
                System.out.println("SturmSolver.factor " + p);
                System.out.println("SturmSolver.factor " + result);
            }
        }
        return result;
    }

    private int signChangeCountAroundVal(double v) {
        int signChangeCount = 0;
        double lastSign = 0.0;
        for (Polynomial p : this.m_polys) {
            double pMin = p.evaluate(v);
            if (pMin > 5.0E-14) {
                if (lastSign < 0.0) {
                    ++signChangeCount;
                }
                lastSign = 1.0;
                continue;
            }
            if (!(pMin < -5.0E-14)) continue;
            if (lastSign > 0.0) {
                ++signChangeCount;
            }
            lastSign = -1.0;
        }
        return signChangeCount;
    }

    boolean isZero(double x) {
        boolean result = false;
        double v = this.m_poly.evaluate(x);
        double d = this.m_deriv.evaluate(x);
        if (Math.abs(v) < 5.0E-14 * Math.abs(d)) {
            result = true;
        }
        return result;
    }

    private List<Polynomial> makePolyList() {
        ArrayList<Polynomial> ps = new ArrayList<Polynomial>();
        boolean keepGoing = true;
        int i = 0;
        while (keepGoing) {
            switch (i) {
                case 0: {
                    ps.add(i, this.m_poly);
                    break;
                }
                case 1: {
                    ps.add(i, this.m_deriv);
                    break;
                }
                default: {
                    ps.add(i, ((Polynomial)ps.get(i - 2)).dividedBy((Polynomial)ps.get(i - 1))[1].multiplyBy(-1.0));
                    if (ps.get(i) != Polynomial.ZEROTH) break;
                    keepGoing = false;
                }
            }
            if (++i <= this.m_poly.order()) continue;
            keepGoing = false;
        }
        return ps;
    }

    private class RangeResult {
        PointResult pMin;
        PointResult pMax;
        PointResult pBest;

        RangeResult(PointResult min, PointResult max) {
            this.pMin = min;
            this.pMax = max;
            this.pBest = this.getBest();
        }

        PointResult getBest() {
            PointResult result = Math.abs(this.pMin.y) < Math.abs(this.pMax.y) ? this.pMin : this.pMax;
            if (this.pBest != null && Math.abs(this.pBest.y) < Math.abs(result.y)) {
                result = this.pBest;
            }
            return result;
        }

        public void setMin(RangeResult rr) {
            this.pMin = rr.pMin;
            this.pBest = this.getBest();
        }

        public void setMax(RangeResult rr) {
            this.pMax = rr.pMax;
            this.pBest = this.getBest();
        }

        double getMid() {
            return (this.pMin.x + this.pMax.x) / 2.0;
        }

        boolean isRootsInRange() {
            return this.pMin.num - this.pMax.num > 0;
        }

        boolean containsRoot() {
            boolean result = this.pMin.isRoot();
            result |= this.pMax.isRoot();
            return result |= this.isRootsInRange();
        }

        boolean isAtRoot() {
            double e2;
            boolean result;
            double e = Math.abs(this.pMax.x - this.pMin.x);
            boolean bl = result = e <= 5.0E-14;
            if (!result && this.pMax.isRoot() && this.pMin.isRoot() && (e2 = Math.abs(this.pMin.x0 - this.pMax.x0)) < 5.0E-9 && e < 5.0E-9) {
                result = true;
            }
            return result;
        }

        public String toString() {
            return "RangeResult " + this.pMin.x + " : " + this.pMax.x + (this.containsRoot() ? "*" : "");
        }

        public double getRoot() {
            double result = this.pBest.x;
            return result;
        }
    }

    private class PointResult {
        final double x;
        final double y;
        final double dy;
        final double ddy;
        final double x0;
        final boolean root;
        final int num;

        PointResult(double x) {
            this.x = x;
            this.y = SturmSolver.this.m_poly.evaluate(x);
            this.dy = SturmSolver.this.m_deriv.evaluate(x);
            this.ddy = SturmSolver.this.m_2ndDeriv.evaluate(x);
            this.num = SturmSolver.this.signChangeCountAroundVal(x);
            this.root = this.calcIsRoot();
            ++maxId;
            this.x0 = this.bestFitRoot();
        }

        boolean isRoot() {
            return this.root;
        }

        boolean calcIsRoot() {
            boolean result = Math.abs(this.y) <= 5.0E-14 * Math.abs(this.dy);
            return result;
        }

        public String toString() {
            return "PointResult " + this.x + (this.root ? " root" : "");
        }

        public double bestFitRoot() {
            double result;
            double a = this.ddy / 2.0;
            double b = this.dy - this.ddy * this.x;
            double c = this.y - (a * this.x + b) * this.x;
            if (this.ddy != 0.0) {
                double dyOverddy = this.dy / this.ddy;
                double sqrt = dyOverddy * dyOverddy - 2.0 * this.y / this.ddy;
                if (sqrt >= 0.0) {
                    double r;
                    sqrt = Math.sqrt(sqrt);
                    double r1 = -dyOverddy + this.x - sqrt;
                    double r2 = -dyOverddy + this.x + sqrt;
                    result = r = Math.abs(this.x - r1) < Math.abs(this.x - r2) ? r1 : r2;
                } else {
                    result = this.x;
                }
            } else {
                result = -c / b;
            }
            return result;
        }
    }
}

