/*
 * Decompiled with CFR 0.152.
 */
package net.sourceforge.jocular.math;

import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.List;
import net.sourceforge.jocular.math.Complex;
import net.sourceforge.jocular.math.JenkinsAndTraubPolySolver;

public class Polynomial {
    public static final Polynomial ZEROTH = new Polynomial(new double[]{0.0}){

        @Override
        public String toString() {
            return "Zero";
        }
    };
    public static final Polynomial UNIT = new Polynomial(new double[]{1.0});
    public static final double EPSILON = 5.0E-14;
    private final double[] m_coeffs;
    private static final int ITERATION_LIMIT = 30;
    private static final int NEWTON_ITERATION_LIMIT = 100;

    private Polynomial(double[] coeffs) {
        this.m_coeffs = coeffs;
    }

    public Complex evaluate(Complex x) {
        Complex result = new Complex(0.0, 0.0);
        for (int i = this.m_coeffs.length - 1; i >= 0; --i) {
            result = result.multiply(x);
            result = result.add(this.m_coeffs[i]);
        }
        return result;
    }

    public double evaluate(double x) {
        double result = 0.0;
        for (int i = this.m_coeffs.length - 1; i >= 0; --i) {
            result *= x;
            result += this.m_coeffs[i];
        }
        return result;
    }

    public int order() {
        return this.m_coeffs.length - 1;
    }

    public static Polynomial makeRandom(int order, double maxCoeff) {
        int nOrder = (int)Math.round(Math.random() * (double)order);
        double[] cs = new double[nOrder + 1];
        for (int i = 0; i < cs.length; ++i) {
            cs[i] = (Math.random() * 2.0 - 1.0) * maxCoeff;
        }
        return Polynomial.makeFromCoefficients(cs);
    }

    public Polynomial multiplyBy(Polynomial p) {
        int i;
        if (p == ZEROTH || this == ZEROTH) {
            return ZEROTH;
        }
        if (this == UNIT) {
            return p;
        }
        if (p == UNIT) {
            return this;
        }
        double[] result = new double[p.order() + this.order() + 1];
        for (i = 0; i < result.length; ++i) {
            result[i] = 0.0;
        }
        for (i = 0; i < p.m_coeffs.length; ++i) {
            for (int j = 0; j < this.m_coeffs.length; ++j) {
                int n = i + j;
                result[n] = result[n] + p.m_coeffs[i] * this.m_coeffs[j];
            }
        }
        return Polynomial.makeFromCoefficients(result);
    }

    public Polynomial multiplyBy(double d) {
        if (d == 0.0 || this == ZEROTH) {
            return ZEROTH;
        }
        double[] result = new double[this.m_coeffs.length];
        for (int i = 0; i < result.length; ++i) {
            result[i] = this.m_coeffs[i] * d;
        }
        return Polynomial.makeFromCoefficients(result);
    }

    public void printPlotData(double min, double max) {
        for (double x = min; x < max; x += (max - min) / 1000.0) {
            System.out.println(x + ", " + this.evaluate(x));
        }
    }

    public boolean isOrderGreaterThanOrEqualTo(Polynomial p) {
        boolean result = true;
        result = this.order() >= p.order() ? true : (this.order() < p.order() ? false : false);
        return result;
    }

    public Polynomial leftShift(int n) {
        double[] result = new double[this.m_coeffs.length + n];
        for (int i = 0; i < result.length; ++i) {
            result[i] = i < n ? 0.0 : this.m_coeffs[i - n];
        }
        return Polynomial.makeFromCoefficients(result);
    }

    public Polynomial rightShift(int n) {
        if (n > this.order()) {
            return ZEROTH;
        }
        double[] result = new double[this.m_coeffs.length - n];
        for (int i = 0; i < result.length; ++i) {
            result[i] = i < n ? 0.0 : this.m_coeffs[i - n];
        }
        return Polynomial.makeFromCoefficients(result);
    }

    public Polynomial[] dividedBy(double d) {
        Polynomial[] result = new Polynomial[2];
        result[1] = ZEROTH;
        result[0] = this.multiplyBy(1.0 / d);
        return result;
    }

    public Polynomial[] dividedBy(Polynomial p) {
        int pOrder = p.order();
        if (pOrder == 0) {
            return this.dividedBy(p.get(0));
        }
        Polynomial[] result = new Polynomial[2];
        if (this == ZEROTH || !this.isOrderGreaterThanOrEqualTo(p)) {
            result[0] = ZEROTH;
            result[1] = this;
            return result;
        }
        int order = this.order();
        int dOrder = order - pOrder;
        if (p == ZEROTH) {
            throw new RuntimeException("Divide by zero.");
        }
        double[] quotient = new double[this.m_coeffs.length];
        double[] remainder = new double[this.m_coeffs.length];
        System.arraycopy(this.m_coeffs, 0, remainder, 0, this.m_coeffs.length);
        for (int i = dOrder; i >= 0; --i) {
            double r = remainder[i + pOrder];
            if (Math.abs(r) > 5.0E-14) {
                quotient[i] = r /= p.get(pOrder);
                for (int j = pOrder; j >= 0; --j) {
                    if (j == pOrder) {
                        remainder[j + i] = 0.0;
                        continue;
                    }
                    int n = j + i;
                    remainder[n] = remainder[n] - r * p.get(j);
                }
                continue;
            }
            quotient[i] = 0.0;
        }
        result[0] = Polynomial.makeFromCoefficients(quotient);
        result[1] = Polynomial.makeFromCoefficients(remainder);
        return result;
    }

    public Polynomial subtract(Polynomial p) {
        int size = this.m_coeffs.length > p.m_coeffs.length ? this.m_coeffs.length : p.m_coeffs.length;
        double[] result = new double[size];
        int zeroCount = 0;
        for (int i = 0; i < size; ++i) {
            result[i] = 0.0;
            if (this.m_coeffs.length > i) {
                int n = i;
                result[n] = result[n] + this.m_coeffs[i];
            }
            if (p.m_coeffs.length > i) {
                int n = i;
                result[n] = result[n] - p.m_coeffs[i];
            }
            if (result[i] != 0.0) continue;
            ++zeroCount;
        }
        if (zeroCount == size) {
            return ZEROTH;
        }
        return Polynomial.makeFromCoefficients(result);
    }

    public Polynomial add(Polynomial p) {
        int size = this.m_coeffs.length > p.m_coeffs.length ? this.m_coeffs.length : p.m_coeffs.length;
        double[] result = new double[size];
        int zeroCount = 0;
        for (int i = 0; i < size; ++i) {
            result[i] = 0.0;
            if (this.m_coeffs.length > i) {
                int n = i;
                result[n] = result[n] + this.m_coeffs[i];
            }
            if (p.m_coeffs.length > i) {
                int n = i;
                result[n] = result[n] + p.m_coeffs[i];
            }
            if (result[i] != 0.0) continue;
            ++zeroCount;
        }
        if (zeroCount == size) {
            return ZEROTH;
        }
        return Polynomial.makeFromCoefficients(result);
    }

    public Polynomial derivative() {
        if (this.order() == 0) {
            return ZEROTH;
        }
        double[] result = new double[this.m_coeffs.length - 1];
        for (int i = 0; i < result.length; ++i) {
            result[i] = this.m_coeffs[i + 1] * ((double)i + 1.0);
        }
        return Polynomial.makeFromCoefficients(result);
    }

    public static Polynomial makeFromCoefficients(double[] coeffs) {
        if (coeffs[coeffs.length - 1] != 0.0) {
            return new Polynomial(coeffs);
        }
        int n = 0;
        for (int i = coeffs.length - 1; i >= 0 && coeffs[i] == 0.0; --i) {
            ++n;
        }
        if (n == coeffs.length) {
            return ZEROTH;
        }
        double[] result = new double[coeffs.length - n];
        for (int i = 0; i < result.length; ++i) {
            result[i] = coeffs[i];
        }
        return new Polynomial(result);
    }

    public static Polynomial makeFromRealRoots(double[] roots) {
        if (roots.length < 1) {
            return ZEROTH;
        }
        Polynomial result = UNIT;
        for (double r : roots) {
            result = result.multiplyBy(Polynomial.makeFromCoefficients(new double[]{-r, 1.0}));
        }
        return result;
    }

    public static Polynomial makeFromComplexPair(double r, double c) {
        return Polynomial.makeFromCoefficients(new double[]{c * c + r * r, -2.0 * r, 1.0});
    }

    public static Polynomial makeFromComplexPair(Complex c1, Complex c2) {
        return Polynomial.makeFromComplexPair(c1, c2, false);
    }

    public static Polynomial makeFromComplexPair(Complex c1, Complex c2, boolean force) {
        Complex p0 = c1.multiply(c2);
        Complex p1 = c1.neg().subtract(c2);
        if (!force && (p0.imag() > 5.0E-14 || p1.imag() > 5.0E-14)) {
            throw new RuntimeException("Specified complex numbers don't combine to yeild real coefficients: " + c1 + ", " + c2);
        }
        return Polynomial.makeFromCoefficients(new double[]{p0.real(), p1.real(), 1.0});
    }

    public Polynomial roundToZero(double epsilon) {
        double[] result = new double[this.m_coeffs.length];
        int n = 0;
        for (int i = 0; i < result.length; ++i) {
            if (Math.abs(this.m_coeffs[i]) <= epsilon) {
                result[i] = 0.0;
                ++n;
                continue;
            }
            result[i] = this.m_coeffs[i];
        }
        if (n == result.length) {
            return ZEROTH;
        }
        return Polynomial.makeFromCoefficients(result);
    }

    public int numRoots(double min, double max) {
        if (Math.abs(this.evaluate(min)) < 5.0E-14) {
            return 1;
        }
        if (Math.abs(this.evaluate(max)) < 5.0E-14) {
            return 1;
        }
        if (max <= min) {
            return 0;
        }
        if (this.order() == 0) {
            return 0;
        }
        Polynomial poly = this.multiplyBy(1.0 / this.get(this.order()));
        ArrayList<Polynomial> ps = new ArrayList<Polynomial>();
        boolean keepGoing = true;
        int i = 0;
        while (keepGoing) {
            switch (i) {
                case 0: {
                    ps.add(i, poly);
                    break;
                }
                case 1: {
                    ps.add(i, poly.derivative());
                    break;
                }
                default: {
                    ps.add(i, ((Polynomial)ps.get(i - 2)).dividedBy((Polynomial)ps.get(i - 1))[1].multiplyBy(-1.0));
                    if (ps.get(i) != ZEROTH) break;
                    keepGoing = false;
                }
            }
            if (++i <= this.order()) continue;
            keepGoing = false;
        }
        int minSign = 0;
        int maxSign = 0;
        int signChangeCount = 0;
        for (Polynomial p : ps) {
            double pMin = p.evaluate(min);
            double pMax = p.evaluate(max);
            if (pMin > 5.0E-14) {
                if (minSign < 0) {
                    ++signChangeCount;
                }
                minSign = 1;
            } else if (pMin < -5.0E-14) {
                if (minSign > 0) {
                    ++signChangeCount;
                }
                minSign = -1;
            }
            if (pMax > 5.0E-14) {
                if (maxSign < 0) {
                    --signChangeCount;
                }
                maxSign = 1;
                continue;
            }
            if (!(pMax < -5.0E-14)) continue;
            if (maxSign > 0) {
                --signChangeCount;
            }
            maxSign = -1;
        }
        return signChangeCount;
    }

    public double getMostSignificantCoefficient() {
        return this.m_coeffs[this.m_coeffs.length - 1];
    }

    public String toString() {
        Object result = "";
        boolean firstTime = true;
        DecimalFormat nf = new DecimalFormat("##0.##############E0");
        for (int i = this.order(); i >= 0; --i) {
            if (!firstTime) {
                result = (String)result + " + ";
            } else {
                firstTime = false;
            }
            result = Double.isNaN(this.m_coeffs[i]) ? (String)result + "NaN" : (String)result + nf.format(this.m_coeffs[i]);
            if (i > 1) {
                result = (String)result + "*x^" + i;
                continue;
            }
            if (i != 1) continue;
            result = (String)result + "*x";
        }
        return result;
    }

    public double get(int n) {
        if (n >= this.m_coeffs.length) {
            return 0.0;
        }
        return this.m_coeffs[n];
    }

    public double maxCoeff() {
        double result = Double.MIN_NORMAL;
        for (double c : this.m_coeffs) {
            if (!(Math.abs(c) > result)) continue;
            result = Math.abs(c);
        }
        return result;
    }

    public Polynomial absorb(Polynomial p) {
        Polynomial result = ZEROTH;
        for (int i = 0; i < this.m_coeffs.length; ++i) {
            double c = this.m_coeffs[i];
            result = result.add(p.power(i).multiplyBy(c));
        }
        return result;
    }

    public Polynomial power(int n) {
        Polynomial result = UNIT;
        for (int i = 0; i < n; ++i) {
            result = result.multiplyBy(this);
        }
        return result;
    }

    @Deprecated
    public List<Polynomial> factor2() {
        ArrayList<Polynomial> result = new ArrayList<Polynomial>();
        Polynomial dividend = this;
        double k = this.get(this.order());
        if (k != 1.0) {
            Polynomial pk = Polynomial.makeFromCoefficients(new double[]{k});
            dividend = dividend.dividedBy(pk)[0];
            result.add(pk);
        }
        while (dividend.order() > 2) {
            Complex c = dividend.laguerreSolve(0.5, 10);
            Polynomial d = Polynomial.makeFromComplex(c);
            if (c.isReal()) {
                Complex newC = this.laguerreSolve(c, 0);
                if (newC.divide(c).abs() > 1.1) {
                    System.out.println("Polynomial.factor2 probably found different root: " + c + ", " + newC);
                }
                d = Polynomial.makeFromComplex(newC);
                result.add(d);
            } else {
                double u = d.get(1) / d.get(2);
                double v = d.get(0) / d.get(2);
                d = this.bairstowSolve(u, v);
                result.addAll(d.quadraticRealRoots());
            }
            dividend = dividend.dividedBy(d)[0];
        }
        if (dividend.order() == 2) {
            result.addAll(dividend.quadraticRealRoots());
        } else {
            result.add(dividend);
        }
        return result;
    }

    public List<Polynomial> factor() {
        JenkinsAndTraubPolySolver js = new JenkinsAndTraubPolySolver();
        return js.factor(this);
    }

    @Deprecated
    public List<Polynomial> factor3() {
        ArrayList<Polynomial> result = new ArrayList<Polynomial>();
        double k = this.get(this.order());
        if (Math.abs(k - 1.0) > 5.0E-14 && this.order() > 0) {
            result.addAll(this.multiplyBy(1.0 / k).factor());
            result.add(Polynomial.makeFromCoefficients(new double[]{k}));
        } else {
            switch (this.order()) {
                case 0: 
                case 1: {
                    result.add(this);
                    break;
                }
                case 2: {
                    result.addAll(this.quadraticRealRoots());
                    break;
                }
                default: {
                    Polynomial r;
                    Complex c = this.laguerreSolve(1.0, 10);
                    double u = 0.0;
                    double v = 0.0;
                    if (this.evaluate(c).abs() < 5.0E-14) {
                        r = Polynomial.makeFromComplex(c);
                    } else if (!c.isReal()) {
                        r = Polynomial.makeFromComplex(c);
                        u = r.get(1) / r.get(2);
                        v = r.get(0) / r.get(2);
                        r = this.bairstowSolve(u, v);
                    } else {
                        r = Polynomial.makeFromComplex(this.laguerreSolve(c.abs(), 10000));
                    }
                    if (r == null) {
                        throw new RuntimeException("Solving failed Bairstow and Newton for polynomial " + this.toArrayString());
                    }
                    if (r.order() > 1) {
                        result.addAll(r.quadraticRealRoots());
                    } else {
                        result.add(r);
                    }
                    result.addAll(this.dividedBy(r)[0].factor());
                }
            }
        }
        return result;
    }

    protected List<Polynomial> cubicRealRoots() {
        if (this.order() > 3) {
            throw new RuntimeException("Order must be less that 3, not " + this.order());
        }
        if (this.order() < 3) {
            return this.quadraticRealRoots();
        }
        ArrayList<Polynomial> result = new ArrayList<Polynomial>();
        double a = this.m_coeffs[3];
        double b = this.m_coeffs[2];
        double c = this.m_coeffs[1];
        double d = this.m_coeffs[0];
        double delta = 18.0 * a * b * c * d - 4.0 * b * b * b * d + b * b * c * c - 4.0 * a * c * c * c - 27.0 * a * a * d * d;
        double delta0 = b * b - 3.0 * a * c;
        double delta1 = 2.0 * b * b * b - 9.0 * a * b * c + 27.0 * a * a * d;
        if (delta == 0.0) {
            if (delta0 == 0.0) {
                double r = -b / 3.0 / a;
                Polynomial p = Polynomial.makeFromRealRoots(new double[]{r});
                result.add(p);
                result.addAll(this.dividedBy(p)[0].quadraticRealRoots());
            } else {
                double r = (9.0 * a * d - b * c) / 2.0 / delta0;
                Polynomial p = Polynomial.makeFromRealRoots(new double[]{r});
                result.add(p);
                result.add(p);
                r = (4.0 * a * b * c - 9.0 * a * a * d - b * b * b) / a / delta0;
                p = Polynomial.makeFromRealRoots(new double[]{r});
                result.add(p);
            }
        } else if (delta0 == 0.0) {
            double capC = Math.pow(delta1, 0.3333333333333333);
            double r = (b + capC + delta0 / capC) / -3.0 / a;
            Polynomial p = Polynomial.makeFromRealRoots(new double[]{r});
            result.add(p);
            Polynomial[] ps = this.dividedBy(p);
            result.addAll(ps[0].quadraticRealRoots());
        } else {
            Complex crii;
            Complex cri;
            Complex cr;
            Complex sqr1 = new Complex(-27.0 * a * a * delta);
            sqr1 = sqr1.sqrt();
            Complex sqr2 = sqr1.add(delta1);
            Complex sqr3 = sqr1.neg().add(delta1);
            if (sqr2.abs() < sqr3.abs()) {
                sqr2 = sqr3;
            }
            Complex u1 = new Complex(1.0);
            Complex u2 = new Complex(-0.5, Math.sqrt(0.75));
            Complex u3 = new Complex(-0.5, -Math.sqrt(0.75));
            Complex cc = sqr2.divide(2.0).rootn(3.0);
            Complex cdelta0 = new Complex(delta0);
            Complex u1c = u1.multiply(cc);
            Complex u2c = u2.multiply(cc);
            Complex u3c = u3.multiply(cc);
            Complex cr1 = u1c.add(b).add(cdelta0.divide(u1c)).divide(-3.0 * a);
            Complex cr2 = u2c.add(b).add(cdelta0.divide(u2c)).divide(-3.0 * a);
            Complex cr3 = u3c.add(b).add(cdelta0.divide(u3c)).divide(-3.0 * a);
            double i1 = Math.abs(cr1.imag());
            double i2 = Math.abs(cr2.imag());
            double i3 = Math.abs(cr3.imag());
            if (i2 < i1 && i2 < i3) {
                cr = cr2;
                cri = cr1;
                crii = cr3;
            } else if (i3 < i1 && i3 < i2) {
                cr = cr3;
                cri = cr1;
                crii = cr2;
            } else {
                cr = cr1;
                cri = cr2;
                crii = cr3;
            }
            double qe = cri.subtract(crii).sqr().real();
            if (cr.isNaN()) {
                throw new RuntimeException("Chosen root is NaN for " + this);
            }
            if (qe > 0.0) {
                // empty if block
            }
            Polynomial p = Polynomial.makeFromRealRoots(new double[]{cr.real()});
            result.add(p);
            if (qe < 0.0) {
                result.add(Polynomial.makeFromComplexPair(cri, crii, true));
            } else {
                double ri = cri.real();
                double rii = crii.real();
                result.add(Polynomial.makeFromRealRoots(new double[]{ri}));
                result.add(Polynomial.makeFromRealRoots(new double[]{rii}));
            }
        }
        return result;
    }

    protected List<Polynomial> quadraticRealRoots() {
        ArrayList<Polynomial> result = new ArrayList<Polynomial>();
        if (this.order() > 2) {
            return null;
        }
        if (this.order() < 2) {
            result.add(this);
            return result;
        }
        double b = this.m_coeffs[1];
        double a = this.m_coeffs[2];
        double b2a = b / 2.0 / a;
        double c = this.m_coeffs[0];
        double sqr = b2a * b2a - c / a;
        if (sqr < 0.0) {
            result.add(this);
        } else {
            sqr = Math.sqrt(sqr);
            double r1 = -b2a + sqr;
            double r2 = -b2a - sqr;
            result.add(Polynomial.makeFromRealRoots(new double[]{r1}));
            result.add(Polynomial.makeFromRealRoots(new double[]{r2}));
        }
        return result;
    }

    protected Complex[] quadraticRoot() {
        if (this.order() == 1) {
            Complex[] result = new Complex[]{new Complex(-this.get(0) / this.get(1))};
            return result;
        }
        if (this.order() != 2) {
            return null;
        }
        Complex[] result = new Complex[2];
        double a = this.m_coeffs[2];
        double b = this.m_coeffs[1];
        double c = this.m_coeffs[0];
        double negb2a = -b / 2.0 / a;
        double sqr = (b * b / 4.0 / a - c) / a;
        if (sqr >= 0.0) {
            result[0] = new Complex(negb2a + Math.sqrt(sqr), 0.0);
            result[1] = new Complex(negb2a - Math.sqrt(sqr), 0.0);
        } else {
            result[0] = new Complex(negb2a).add(new Complex(sqr).sqrt());
            result[1] = new Complex(negb2a).subtract(new Complex(sqr).sqrt());
        }
        return result;
    }

    protected static int countRealRootsFromFactors(List<Polynomial> ps, double min, double max) {
        int result = 0;
        for (Polynomial p : ps) {
            double r;
            if (p.order() != 1 || !((r = -p.get(0) / p.get(1)) >= min) || !(r <= max)) continue;
            ++result;
        }
        return result;
    }

    public Polynomial newtonSolve(double guess) {
        Polynomial dThis = this.derivative();
        double error = Double.MAX_VALUE;
        double result = guess;
        int n = 0;
        double lastError = Double.MAX_VALUE;
        double lastGuess = guess;
        boolean done = false;
        while (!done) {
            ++n;
            error = this.evaluate(result);
            result -= error / dThis.evaluate(result);
            if (Math.abs(error / lastError + 1.0) < 0.3) {
                result = (result + lastGuess) / 2.0;
            }
            if (result == lastGuess) {
                done = true;
            }
            lastGuess = result;
            lastError = error;
            System.out.println("Polynomial.newtonSolve: guess: " + result + ", error: " + error);
            if (!(Math.abs(error) <= 5.0E-14) && n <= 100) continue;
            done = true;
        }
        if (n >= 100) {
            return null;
        }
        return Polynomial.makeFromRealRoots(new double[]{result});
    }

    public Complex laguerreSolve(double guess, int maxIterations) {
        return this.laguerreSolve(new Complex(guess, 0.0), maxIterations);
    }

    public Complex laguerreSolve(Complex guess, int maxIterations) {
        Polynomial d = this.derivative();
        Polynomial dd = d.derivative();
        double n = this.order();
        Complex result = guess;
        boolean done = false;
        Complex lastResult = result;
        int i = 0;
        ArrayList<Complex> iters = new ArrayList<Complex>();
        while (!done) {
            iters.add(result);
            ++i;
            Complex p = this.evaluate(result);
            if (p.abs() == 0.0) {
                done = true;
                continue;
            }
            Complex g = d.evaluate(result).divide(p);
            Complex h = g.multiply(g).subtract(dd.evaluate(result).divide(p));
            Complex den = h.multiply(n).subtract(g.multiply(g)).multiply(n - 1.0);
            den = den.sqrt();
            Complex a = new Complex(n, 0.0);
            Complex a1 = a.divide(g.add(den));
            Complex a2 = a.divide(g.subtract(den));
            a = a1.isNaN() ? a2 : (a2.isNaN() ? a1 : (a1.abs() > a2.abs() ? a2 : a1));
            double resultRatio = Math.abs(Math.abs((result = result.subtract(a)).abs() / lastResult.abs()) - 1.0);
            if (resultRatio < 5.0E-14) {
                done = true;
            }
            if (result.equals(lastResult)) {
                done = true;
            }
            if (maxIterations > 0 && i >= maxIterations) {
                done = true;
            }
            if (p.abs() < 5.0E-14) {
                done = true;
            }
            if (result.isNaN()) {
                throw new RuntimeException("Result is NaN.");
            }
            if (i > 30 && !done) {
                done = true;
                System.out.println("Polynomial.laguerreSolve failed with error " + p.abs() + " and result ratio " + resultRatio + " when solving polynomial " + this.toArrayString());
                result = null;
            }
            lastResult = result;
        }
        if (result == null) {
            return null;
        }
        if (Math.abs(result.imag()) < 5.0E-14) {
            result = result.onlyReal();
        }
        return result;
    }

    public Polynomial bairstowSolve(double uGuess, double vGuess) {
        double u = uGuess;
        double v = vGuess;
        boolean done = false;
        int n = 0;
        while (!done) {
            double error;
            double h;
            ++n;
            Polynomial test = Polynomial.makeFromCoefficients(new double[]{v, u, 1.0});
            Polynomial[] qr = this.dividedBy(test);
            double c = qr[1].get(1);
            double d = qr[1].get(0);
            double g = (qr = qr[0].dividedBy(test))[1].get(1);
            double k = v * g * g + (h = qr[1].get(0)) * h - h * u * g;
            if (k == 0.0) {
                throw new RuntimeException("k is 0");
            }
            double du = (-c * h + g * d) / k;
            double dv = (-g * v * c + g * u * d - h * d) / k;
            u -= du;
            v -= dv;
            if (Math.abs(du) < 5.0E-14 && Math.abs(dv) < 5.0E-14) {
                done = true;
            }
            if ((error = this.evaluate(test.quadraticRoot()[0]).abs()) < 5.0E-14) {
                done = true;
            }
            if (n > 30 && !done) {
                System.out.println("Polynomial.bairstowSolve failed with error: " + (Math.abs(du) + Math.abs(dv)));
                return null;
            }
            if (!Double.isNaN(u) && !Double.isNaN(v)) continue;
            return null;
        }
        return Polynomial.makeFromCoefficients(new double[]{v, u, 1.0});
    }

    protected static Polynomial makeFromComplex(Complex c) {
        Polynomial result;
        if (c == null) {
            return null;
        }
        if (c.isReal()) {
            result = Polynomial.makeFromRealRoots(new double[]{c.real()});
        } else {
            double c1 = -2.0 * c.real();
            double c0 = c.absSq();
            result = Polynomial.makeFromCoefficients(new double[]{c0, c1, 1.0});
        }
        return result;
    }

    public static Polynomial multiplyAll(List<Polynomial> ps) {
        Polynomial result = UNIT;
        for (Polynomial p : ps) {
            result = result.multiplyBy(p);
        }
        return result;
    }

    public String toArrayString() {
        Object result = "{";
        for (int i = 0; i < this.m_coeffs.length; ++i) {
            if (i != 0) {
                result = (String)result + ", ";
            }
            result = (String)result + this.m_coeffs[i];
        }
        result = (String)result + "}";
        return result;
    }

    private static String arrayStringFromRootList(List<Polynomial> ps) {
        Object result = "";
        for (Polynomial p : ps) {
            result = (String)result + "(" + p + ")";
        }
        return result;
    }

    public boolean isNan() {
        boolean result = false;
        for (double d : this.m_coeffs) {
            if (!Double.isNaN(d)) continue;
            result = true;
            break;
        }
        return result;
    }

    public double linearRoot() {
        if (this.order() != 1) {
            throw new RuntimeException("Order must be 1, not " + this.order());
        }
        return -this.m_coeffs[0] / this.m_coeffs[1];
    }

    public double distance(Polynomial p) {
        if (this.order() != p.order()) {
            return Double.MAX_VALUE;
        }
        double result = 0.0;
        for (int i = 0; i < this.m_coeffs.length; ++i) {
            result += Math.pow(this.m_coeffs[i] - p.m_coeffs[i], 2.0);
        }
        return Math.sqrt(result);
    }

    public static void main(String[] args) {
        Polynomial p1 = Polynomial.makeFromRealRoots(new double[]{-0.9, -0.8, -0.5, -4.0}).multiplyBy(Polynomial.makeFromComplexPair(2.0, 3.0));
        Polynomial p2 = Polynomial.makeFromRealRoots(new double[]{3.0, 2.0, 4.0});
        Polynomial p4 = Polynomial.makeFromCoefficients(new double[]{590.129348046111, 774.5739602889239, 920.9974504798464, -849.5819426370368, 634.5223738278904, 65.11549151468743});
        Polynomial p5 = Polynomial.makeFromCoefficients(new double[]{5.0, 4.0, -6.0, 10.0, 11.0, 22.0, 88.0, 40.0, -80.0, -2.0});
        Polynomial p8 = Polynomial.makeFromCoefficients(new double[]{0.8855898648893807, -2.7648988374959007, 2.6424531065208012, 1.5770850501030125, 0.9133455717735042, 2.7771209635420036, 1.0});
        System.out.println("P1 = " + p1);
        System.out.println("P2 = " + p2);
        Polynomial[] qs = p1.dividedBy(p2);
        System.out.println("Quotient: " + qs[0]);
        System.out.println("Remainder: " + qs[1]);
        System.out.println("P1 should be: " + qs[0].multiplyBy(p2).add(qs[1]));
        System.out.println("Greater than: " + p1.isOrderGreaterThanOrEqualTo(p2));
        System.out.println("Zero: " + ZEROTH);
        System.out.println("Sturm: " + p1.numRoots(0.0, 1.0));
        System.out.println("Square: " + Polynomial.makeFromRealRoots(new double[]{3.0, 4.0}).power(2));
        System.out.println("Solve: " + p2.newtonSolve(2.5));
        System.out.println("Quadratic Solve: " + Polynomial.makeFromRealRoots(new double[]{3.0, 4.0}).quadraticRealRoots().get(0));
        System.out.println("Laguerre Solve " + p4 + " --> " + p4.laguerreSolve(0.0, 0));
        int sturmCount = 0;
        int badCount = 0;
        JenkinsAndTraubPolySolver solver = new JenkinsAndTraubPolySolver();
        long startTime = System.currentTimeMillis();
        int NUM_CYCLES = 10000000;
        long totalIterations = 0L;
        for (int i = 0; i < 10000000; ++i) {
            if (i % 1000000 == 0) {
                double iPer = totalIterations;
                double tPer = System.currentTimeMillis() - startTime;
                System.out.print("Polynomial.main done " + i + " so far, with " + (iPer /= (double)i) + " iterations and " + (tPer /= (double)i * 1000.0) + " seconds per poly.\r");
            }
            p5 = Polynomial.makeRandom(6, 1000.0);
            int nSturm = p5.numRoots(0.0, 1.0);
            List<Polynomial> ps = null;
            try {
                ps = solver.factor(p5);
                totalIterations += (long)solver.getIterations();
                int nCount = Polynomial.countRealRootsFromFactors(ps, 0.0, 1.0);
                double r = Double.MIN_NORMAL;
                Polynomial result = UNIT;
                for (Polynomial p : ps) {
                    Complex[] cs = p.quadraticRoot();
                    if (cs != null && cs.length > 0) {
                        for (Complex c : cs) {
                            double tr = p.evaluate(c).abs();
                            if (!(tr > r)) continue;
                            r = tr;
                        }
                    }
                    result = result.multiplyBy(p);
                }
                if (p5.subtract(result).maxCoeff() < 5.0E-14) {
                    r = 0.0;
                }
                Object doubleRoots = "";
                block5: for (int j = 0; j < ps.size(); ++j) {
                    Polynomial f1 = ps.get(j);
                    if (f1.order() != 1) continue;
                    double r1 = -f1.get(0) / f1.get(1);
                    for (int k = j; k < ps.size(); ++k) {
                        Polynomial f2 = ps.get(k);
                        double r2 = -f2.get(0) / f2.get(1);
                        if (!(Math.abs(r2 - r1) < 5.0E-14)) continue;
                        doubleRoots = "double-rooted (" + r2 + ")";
                        continue block5;
                    }
                }
                if (r > 1.0E-12) {
                    System.out.println("Remainder " + r + " too large for Factors of " + p5.subtract(result).toArrayString());
                }
                if (nSturm > 0 || nCount <= 0) continue;
                System.out.println(++sturmCount + " This " + (String)doubleRoots + " has " + nCount + " not " + nSturm + " roots between 0.0 and 1.0. " + Polynomial.arrayStringFromRootList(ps));
                System.out.println("          " + p5 + " should equal " + Polynomial.multiplyAll(ps));
                continue;
            }
            catch (RuntimeException e) {
                ++badCount;
                System.out.println("Polynomial.main bad root: " + p5.toArrayString() + " " + e);
                e.printStackTrace();
            }
        }
        double polyPerSecond = 1.0 / ((double)(System.currentTimeMillis() - startTime) / 1.0E7 / 1000.0);
        System.out.println("JenkinsAndTraubPolySolver.main poly per second: " + polyPerSecond);
        System.out.println("Polynomial.main num bad roots: " + badCount);
        System.out.println("Numroots: " + p8.numRoots(0.0, 1.0));
        System.out.println("PHEW! DONE.");
    }

    public static Polynomial parseString(String s) {
        int i;
        String t = s.trim().replace(" ", "");
        t = t.toLowerCase();
        t = t.replace("e-", "en");
        t = t.replace("+-", "-");
        t = t.replace("+", ",+");
        t = t.replace("-", ",-");
        t = t.replace("*", "");
        t = t.replace("x^", "x");
        t = t.replace("x,", "x1,");
        if ((t = t.replace("n", "-")).startsWith(",")) {
            t = t.substring(1);
        }
        String[] ts = t.split(",");
        double[] ds = new double[ts.length];
        int[] is = new int[ts.length];
        for (int i2 = 0; i2 < ts.length; ++i2) {
            int j = ts[i2].indexOf("x");
            if (j != -1) {
                String c = ts[i2].substring(0, j);
                String x = ts[i2].substring(j + 1);
                ds[i2] = Double.parseDouble(c);
                is[i2] = Integer.parseInt(x);
                continue;
            }
            if (ts[i2].equals("")) continue;
            ds[i2] = Double.parseDouble(ts[i2]);
            is[i2] = 0;
        }
        int maxI = 0;
        for (int i3 : is) {
            if (i3 <= maxI) continue;
            maxI = i3;
        }
        double[] coeffs = new double[maxI + 1];
        for (i = 0; i < coeffs.length; ++i) {
            coeffs[i] = 0.0;
        }
        for (i = 0; i < ds.length; ++i) {
            coeffs[is[i]] = ds[i];
        }
        return Polynomial.makeFromCoefficients(coeffs);
    }

    public Polynomial removeInsignificantCoefficients() {
        int i;
        int n = this.m_coeffs.length;
        double[] cs = new double[n];
        double max = 0.0;
        for (i = 0; i < n; ++i) {
            cs[i] = this.m_coeffs[i];
            if (!(Math.abs(cs[i]) > Math.abs(max))) continue;
            max = cs[i];
        }
        for (i = 0; i < n; ++i) {
            int n2 = i;
            cs[n2] = cs[n2] / max;
            if (!(Math.abs(cs[i]) < 5.0E-14)) continue;
            cs[i] = 0.0;
        }
        return Polynomial.makeFromCoefficients(cs);
    }
}

