/* -*- mode: bison -*- */
%{
/*
 * Grace - GRaphing, Advanced Computation and Exploration of data
 *
 * Home page: http://plasma-gate.weizmann.ac.il/Grace/
 *
 * Copyright (c) 1991-1995 Paul J Turner, Portland, OR
 * Copyright (c) 1996-2003 Grace Development Team
 * and after 2010 P. Vincent
 *
 *
 *
 *                           All Rights Reserved
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 *
 * evaluate expressions, commands, parameter files
 *
 */

#include <config.h>
#include <cmath.h>

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#if defined(HAVE_SYS_PARAM_H)
#  include <sys/param.h>
#endif

/* bison not always handles it well itself */
#if defined(HAVE_ALLOCA_H)
#  include <alloca.h>
#endif

#include "defines.h"
#include "globals.h"
#include "cephes/cephes.h"
#include "device.h"
#include "utils.h"
#include "files.h"
#include "graphs.h"
#include "graphutils.h"
#include "plotone.h"
#include "dlmodule.h"
#include "t1fonts.h"
#include "ssdata.h"
#include "parser.h"
#ifndef HAVE_DRAND48
  extern double drand48(void);
#endif
#include "mathstuff.h"
#ifndef WITH_GTK
#  include "protos.h"
#else     /* WITH_GTK */
#  include "noxprotos.h"
#  include <gtk/gtk.h>
#  include "gg_protos.h"
#endif    /* WITH_GTK */
#  include "ng_objects.h"
#  include "nn_tree.h"
#  include "zc.h"

#include "fourier.h"
#include "wavelet.h"

#define MAX_PARS_STRING_LENGTH  4096
#define MAX_GRARR_TMP 100

#define CAST_DBL_TO_BOOL(x) (fabs(x) < 0.5 ? 0:1)

typedef double (*ParserFnc)();

extern graph *g;

static int file_project_version = 0;
static int curid;
static int curobjtyp;
static int curpmask = MA_NONE;

#include "pars_tree.h"
 extern ParsTreeNode *ptree;

static double  s_result;    /* return value if a scalar expression is scanned*/
static grarr  *v_result;    /* return value if a vector or matrix expression is scanned*/

static int expr_parsed, vexpr_parsed;

static int interr;

static grarr freelist[MAX_GRARR_TMP]; /* temporary vectors */
static int fcnt = 0;		      /* number of the temporary vectors allocated */

static grarr mattmp[MAX_GRARR_TMP];   /* temporary matrices */
static int   matcnt = 0;              /* number of the temporary matrices allocated */

static target trgt_pool[100]; 	/* pool of temporary targets */
static int tgtn = 0;		/* number of the temporary targets used */

int naxis = 0;	/* current axis */
//static int curstring;
/* these guys attempt to avoid reentrancy problems */
static int gotparams = FALSE, gotread = FALSE, gotnlfit = FALSE;
int readxformat;
static int nlfit_gno, nlfit_setno, nlfit_nsteps ,nlfit_trace;
static double *nlfit_warray = NULL;

char batchfile[GR_MAXPATHLEN] = "",
     paramfile[GR_MAXPATHLEN] = "",
     readfile[GR_MAXPATHLEN] = "";

static FILE *fplog = NULL;  /* Log file to record commands */

static char f_string[MAX_PARS_STRING_LENGTH]; /* buffer for string to parse */
static int pos;

/* the graph, set, and its length of the parser's current state */
 static int gn;  		     /* whichgraph in grace-5   */
 static int sn;  		     /* whichset   in grace-5   */
 static int jo;  		     /* synonym for cur_obj_num */

/* the graph and set of the left part of a vector assignment */
static int vasgn_gno;
static int vasgn_setno;

/* to assign  matrix columns */
static int masgn_row1;
static int masgn_row2;
static int masgn_col1;
static int masgn_col2;

static int alias_force = FALSE; /* controls whether aliases can override
                                                       existing keywords */

extern char print_file[];
extern char *close_input;

static int filltype_obs;

 static int string_font    = 0;    //24
 static double string_size = 1.0;  //24

static int index_shift = 0;     /* 0 for C, 1 for F77 index notation */

static void free_grarr (grarr *gar);
static void copy_grarr     (grarr *dest, grarr *src);
static int find_set_bydata (double *data, target *tgt);

static int getcharstr(void);
static void ungetchstr(void);
static int follow(int expect, int ifyes, int ifno);

static int yylex(void);
static int yyparse(void);
static void yyerror(char *s);

static int findf(symtab_entry *keytable, char *s);

/* Total (intrinsic + user-defined) list of functions and keywords */
symtab_entry *key;
static void undefine_all_users_var (void);

/* Functions to shorten C part of a lot of rules */
static int bad_gno  (void);
static int bad_axis (void);
static int div_null (double div);
static grarr *tmp_vec (grarr *src);
static grarr *tmp_mat (grarr *src);

%}

/* --------------  BISON DECLARATIONS -------------- */


%union {
    int     ival;
    double  dval;
    char   *sval;
    double *dptr;
    target *trgt;
    grarr  *vrbl;
    grarr  *matp;
  QDobject *qobj;
}

%token KEY_VAR
%token KEY_VEC
%token KEY_MAT

%token KEY_CONST     
%token KEY_UNIT      
%token KEY_FUNC_I    
%token KEY_FUNC_D    
%token KEY_FUNC_NN   
%token KEY_FUNC_ND   
%token KEY_FUNC_DD   
%token KEY_FUNC_NND  
%token KEY_FUNC_PPD  
%token KEY_FUNC_PPPD 
%token KEY_FUNC_PPPPD 
%token KEY_FUNC_PPPPPD 

%token <ival> INDEX
%token <ival> DATE

%token <dptr> VAR_D	 /* a (pointer to) double variable                          */
%token <vrbl> VEC_D	 /* a (pointer to) double array variable                    */
%token <matp> MAT_D	 /* a (pointer to) 2D array of doubles                      */

%token <ival> CONSTANT	 /* a (double) constant                                     */
%token <ival> UCONSTANT	 /* a (double) unit constant                                */
%token <ival> FUNC_I	 /* a function of 1 int variable                            */
%token <ival> FUNC_D	 /* a function of 1 double variable                         */
%token <ival> FUNC_NN    /* a function of 2 int parameters                          */
%token <ival> FUNC_ND    /* a function of 1 int parameter and 1 double variable     */
%token <ival> FUNC_DD    /* a function of 2 double variables                        */
%token <ival> FUNC_NND   /* a function of 2 int parameters and 1 double variable    */
%token <ival> FUNC_PPD   /* a function of 2 double parameters and 1 double variable */
%token <ival> FUNC_PPPD  /* a function of 3 double parameters and 1 double variable */
%token <ival> FUNC_PPPPD /* a function of 4 double parameters and 1 double variable */
%token <ival> FUNC_PPPPPD/* a function of 5 double parameters and 1 double variable */

%token <ival> ABOVE
%token <ival> ABSOLUTE
%token <ival> ALIAS
%token <ival> ALLVARS
%token <ival> ALT
%token <ival> ALTXAXIS
%token <ival> ALTYAXIS
%token <ival> ANCHOR
%token <ival> ANGLE
%token <ival> ANGLES
%token <ival> ANTIALIASING
%token <ival> APPEND
%token <ival> ARC
%token <ival> ARRANGE
%token <ival> ARROW
%token <ival> ASCENDING
%token <ival> ASPLINE
%token <ival> AUTO
%token <ival> AUTOSCALE
%token <ival> AUTOTICKS
%token <ival> AVALUE
%token <ival> AVG
%token <ival> BACKGROUND
%token <ival> BAR
%token <ival> BARDY
%token <ival> BARDYDY
%token <ival> BASELINE
%token <ival> BATCH
%token <ival> BEGIN
%token <ival> BELOW
%token <ival> BETWEEN
%token <ival> BLACKMAN
%token <ival> BLOCK
%token <ival> BOTH
%token <ival> BOTTOM
%token <ival> BOX
%token <ival> BREAK
%token <ival> BSPLINE
%token <ival> BSPLINE_CENTERED
%token <ival> CD
%token <ival> CENTER
%token <ival> CHAR
%token <ival> CHART
%token <sval> CHRSTR
%token <ival> CLEAR
%token <ival> CLICK
%token <ival> CLIP
%token <ival> CLOSE
%token <ival> CMAP
%token <ival> COEFFICIENTS
%token <ival> COLOR
%token <ival> COLORBAR
%token <ival> COMMENT
%token <ival> COMPLEX
%token <ival> COMPOUND
%token <ival> COMPUTING
%token <ival> CONSTRAINTS
%token <ival> CONTOUR
%token <ival> COPY
%token <ival> CYCLE
%token <ival> DAUBECHIES
%token <ival> DAUBECHIES_CENTERED
%token <ival> DAYMONTH
%token <ival> DAYOFWEEKL
%token <ival> DAYOFWEEKS
%token <ival> DAYOFYEAR
%token <ival> DDMMYY
%token <ival> DECIMAL
%token <ival> DEF
%token <ival> DEFAULT
%token <ival> DEFINE
%token <ival> DEGREES
%token <ival> DEGREESLAT
%token <ival> DEGREESLON
%token <ival> DEGREESMMLAT
%token <ival> DEGREESMMLON
%token <ival> DEGREESMMSSLAT
%token <ival> DEGREESMMSSLON
%token <ival> DESCENDING
%token <ival> DESCRIPTION
%token <ival> DEVICE
%token <ival> DFT
%token <ival> DIFFERENCE
%token <ival> DISK
%token <ival> DOWN
%token <ival> DPI
%token <ival> DROP
%token <ival> DROPLINE
%token <ival> ECHO
%token <ival> ELLIPSE
%token <ival> ENGINEERING
%token <ival> ERRORBAR
%token <ival> EXIT
%token <ival> EXPONENTIAL
%token <ival> FFT
%token <ival> FILEP
%token <ival> FILL
%token <ival> FIT
%token <ival> FIXED
%token <ival> FIXEDPOINT
%token <ival> FLUSH
%token <ival> FOCUS
%token <ival> FOLLOWS
%token <ival> FONTP
%token <ival> FORCE
%token <ival> FORMAT
%token <ival> FORMULA
%token <ival> FRAMEP
%token <ival> FREE
%token <ival> FREQUENCY
%token <ival> FROM
%token <ival> GENERAL
%token <ival> GETP
%token <ival> GRAPH
%token <ival> GRAPHNO
%token <ival> GRID
%token <ival> HAAR
%token <ival> HAAR_CENTERED
%token <ival> HALF
%token <ival> HAMMING
%token <ival> HANNING
%token <ival> HARDCOPY
%token <ival> HBAR
%token <ival> HELP
%token <ival> HGAP
%token <ival> HIDDEN
%token <ival> HISTOGRAM
%token <ival> HMS
%token <ival> HORIZI
%token <ival> HORIZONTAL
%token <ival> HORIZO
%token <ival> ID
%token <ival> IF
%token <ival> IFILTER
%token <ival> IMAX
%token <ival> IMIN
%token <ival> IN
%token <ival> INCREMENT
%token <ival> INOUT
%token <ival> INSEGX
%token <ival> INT
%token <ival> INTEGRATE
%token <ival> INTERPOLATE
%token <ival> INVDFT
%token <ival> INVERT
%token <ival> INVFFT
%token <ival> JUST
%token <ival> KILL
%token <ival> LABEL
%token <ival> LANDSCAPE
%token <ival> LAYER
%token <ival> LAYOUT
%token <ival> LEFT
%token <ival> LEGEND
%token <ival> LENGTH
%token <ival> LINCONV
%token <ival> LINE
%token <ival> LINEAR
%token <ival> LINESTYLE
%token <ival> LINEWIDTH
%token <ival> LINK
%token <ival> LOAD
%token <ival> LOCTYPE
%token <ival> LOG
%token <ival> LOGARITHMIC
%token <ival> LOGIT
%token <ival> LOGX
%token <ival> LOGXY
%token <ival> LOGY
%token <ival> MAGIC
%token <ival> MAGNITUDE
%token <ival> MAJOR
%token <ival> MAP
%token <ival> MAXP
%token <ival> MESH
%token <ival> MESH2D
%token <ival> MINP
%token <ival> MINOR
%token <ival> MMDD
%token <ival> MMDDHMS
%token <ival> MMDDYY
%token <ival> MMDDYYHMS
%token <ival> MMSSLAT
%token <ival> MMSSLON
%token <ival> MMYY
%token <ival> MONTHDAY
%token <ival> MONTHL
%token <ival> MONTHS
%token <ival> MONTHSY
%token <ival> MOVE
%token <ival> NEGATE
%token <ival> NEW
%token <ival> NONE
%token <ival> NONLFIT
%token <ival> NORMAL
%token <ival> NTICKS
%token <ival> NXY
%token <ival> OBJ
%token <ival> OFF
%token <ival> OFFSET
%token <ival> OFFSETX
%token <ival> OFFSETY
%token <ival> OFILTER
%token <ival> ON
%token <ival> ONREAD
%token <ival> OP
%token <ival> OPEN
%token <ival> OPPOSITE
%token <ival> OUT
%token <ival> PAGE
%token <ival> PARA
%token <ival> PARAMETERS
%token <ival> PARZEN
%token <ival> PATTERN
%token <ival> PERIOD
%token <ival> PERP
%token <ival> PHASE
%token <ival> PIE
%token <ival> PIPE
%token <ival> PLACE
%token <ival> POINT
%token <ival> POLAR
%token <ival> POLYI
%token <ival> POLYLINE
%token <ival> POLYO
%token <ival> POP
%token <ival> PORTRAIT
%token <ival> POSITIVE
%token <ival> POWER
%token <ival> PREC
%token <ival> PREPEND
%token <ival> PRINT
%token <ival> PS
%token <ival> PUSH
%token <ival> PUTP
%token <ival> RAND
%token <ival> READ
%token <ival> REAL
%token <ival> RECIPROCAL
%token <ival> REDRAW
%token <ival> REFERENCE
%token <ival> REGNUM
%token <ival> REGRESS
%token <ival> RESIZE
%token <ival> RESTRICT
%token <ival> REVERSE
%token <ival> RIGHT
%token <ival> RISER
%token <ival> ROT
%token <ival> ROUNDED
%token <ival> RSUM
%token <ival> RULE
%token <ival> RUNAVG
%token <ival> RUNMAX
%token <ival> RUNMED
%token <ival> RUNMIN
%token <ival> RUNSTD
%token <ival> SAVEALL
%token <ival> SCALE
%token <pset> SCIENTIFIC
%token <ival> SELXY
%token <ival> SCROLL
%token <ival> SD
%token <ival> SET
%token <ival> SETNUM
%token <ival> SFORMAT
%token <ival> SIGN
%token <ival> SIZE
%token <ival> SKIP
%token <ival> SLEEP
%token <ival> SMITH
%token <ival> SORT
%token <ival> SOURCE
%token <ival> SPEC
%token <ival> SPLINE
%token <ival> SPLIT
%token <ival> STACK
%token <ival> STACKED
%token <ival> STACKEDBAR
%token <ival> STACKEDHBAR
%token <ival> STAGGER
%token <ival> START
%token <ival> STOP
%token <ival> STRING
%token <ival> SUM
%token <ival> SUBTITLE
%token <ival> SWAP
%token <ival> SYMBOL
%token <ival> TARGET
%token <ival> TICKLABEL
%token <ival> TICKP
%token <ival> TICKSP
%token <ival> TIMER
%token <ival> TIMESTAMP
%token <ival> TITLE
%token <ival> TO
%token <ival> TOP
%token <ival> TRIANGULAR
%token <ival> TYPE
%token <ival> UP
%token <ival> UPDATEALL
%token <ival> USE
%token <ival> VERSION
%token <ival> VERTI
%token <ival> VERTICAL
%token <ival> VERTO
%token <ival> VGAP
%token <ival> VIEW
%token <ival> VMAP
%token <ival> VX1
%token <ival> VX2
%token <ival> VXMAX
%token <ival> VY1
%token <ival> VY2
%token <ival> VYMAX
%token <ival> WAVELET
%token <ival> WELCH
%token <ival> WITH
%token <ival> WORLD
%token <ival> WRAP
%token <ival> WRITE
%token <ival> WX1
%token <ival> WX2
%token <ival> WY1
%token <ival> WY2
%token <ival> X_TOK
%token <ival> X0
%token <ival> X1
%token <ival> XAXES
%token <ival> XAXIS
%token <ival> XCOR
%token <ival> XMAX
%token <ival> XMIN
%token <ival> XSPLINE
%token <ival> XY
%token <ival> XYAXES
%token <ival> XYBOXPLOT
%token <ival> XYCOLOR
%token <ival> XYCOLPAT
%token <ival> XYDX
%token <ival> XYDXDX
%token <ival> XYDXDXDYDY
%token <ival> XYDXDY
%token <ival> XYDY
%token <ival> XYDYDY
%token <ival> XYHILO
%token <ival> XYR
%token <ival> XYSIZE
%token <ival> XYSPLINE
%token <ival> XYSTRING
%token <ival> XYCMAP
%token <ival> XYVMAP
%token <ival> XYZ
%token <ival> Y_TOK
%token <ival> Y0
%token <ival> Y1
%token <ival> Y2
%token <ival> Y3
%token <ival> Y4
%token <ival> YAXES
%token <ival> YAXIS
%token <ival> YEAR
%token <ival> YMAX
%token <ival> YMIN
%token <ival> YYMMDD
%token <ival> YYMMDDHMS
%token <ival> ZERO
%token <ival> ZNORM
%token <ival> ZSCALE

%token <ival> FITPARM
%token <ival> FITPMAX
%token <ival> FITPMIN
%token <dval> NUMBER

%token <sval> NEW_TOKEN

%type <ival> onoff

%type <ival> selectgraph
%type <trgt> selectset

%type <qobj> selobj;
%type <qobj> thiscompound;

%type <ival> pagelayout
%type <ival> pageorient

%type <ival> regiontype

%type <ival> color_select
%type <ival> pattern_select
%type <ival> font_select

%type <ival> lines_select
%type <dval> linew_select

%type <ival> graphtype
%type <ival> xytype

%type <ival> scaletype
%type <ival> signchoice

%type <ival> colpat_obs
%type <ival> direction

%type <ival> formatchoice
%type <ival> inoutchoice
%type <ival> justchoice

%type <ival> opchoice
%type <ival> opchoice_sel
%type <ival> opchoice_obs
%type <ival> opchoice_sel_obs

%type <ival> worldview

%type <ival> filtermethod
%type <ival> filtertype

%type <ival> tickspectype

%type <ival> sourcetype

%type <ival> interpmethod
%type <ival> stattype

%type <ival> datacolumn

%type <ival> runtype

%type <ival> ffttype
%type <ival> fourierdata
%type <ival> fourierloadx
%type <ival> fourierloady
%type <ival> fourierinseg
%type <ival> fourieroutsg
%type <ival> waveletype
%type <ival> vectormap
%type <ival> colorbar
%type <ival> windowtype

%type <ival> nonlfitopts

%type <ival> sortdir
%type <ival> sorton

%type <ival> proctype

%type <ival> objtyp

%type <ival> indx
%type <ival> iexpr
%type <ival> nexpr
%type <sval> sexpr
%type <dval> jdate
%type <dval> jrawdate
%type <dval> expr

%type <vrbl> array
%type <vrbl> lside_array
%type <vrbl> vexpr

%type <matp> matrix
%type <matp> lside_matrix
%type <matp> mexpr

/* Precedence */
%nonassoc '?' ':'
%left OR
%left AND
%nonassoc GT LT LE GE EQ NE
%right UCONSTANT
%left '+' '-'
%left '*' '/' '%'
%nonassoc UMINUS NOT	/* negation--unary minus */
%right '^'		/* exponentiation        */
%left '.'


%%


/* -----------------------  GRAMMAR RULES ----------------------- */

full_list:
        multi_list
        | expr {
            expr_parsed = TRUE;
            s_result = $1;
        }
        | vexpr {
            vexpr_parsed = TRUE;
            v_result = $1;
        }
        ;

multi_list:
        list
        | multi_list ';' list
        ;

list:
	/* empty */
	| parmset {}
	| parmset_obs {}
	| setpobj {}
	| objects {}
	| regionset {}
	| setaxis {}
	| set_setprop {}
	| actions {}
	| options {}
	| asgn {}
	| vasgn {}
	| masgn {}
        | flowcontrol {}
	| defines {}
	| error {
	    return 1;
	}
	;



expr:	NUMBER { $$ = $1; }
	|  VAR_D { $$ = *($1); }
	|  FITPARM { $$ = nonl_parms[$1].value;	}
	|  FITPMAX { $$ = nonl_parms[$1].max; }
	|  FITPMIN { $$ = nonl_parms[$1].min; }
	|  array indx {
            if ($2 >= $1->length) {
                errmsg("Access beyond array bounds");
                return 1;
            }
            $$ = $1->data[$2];
	}
	| stattype '(' vexpr ')' {
	    double dummy, dummy2;
            int idummy, ind, length = $3->length;
	    if ($3->data == NULL) {
		yyerror("NULL variable, check set type");
		return 1;
	    }
	    switch ($1) {
	    case MINP: $$ = vmin($3->data, length);		break;
	    case MAXP: $$ = vmax($3->data, length);		break;
            case AVG:  stasum($3->data, length, &$$, &dummy);   break;
            case SD:   stasum($3->data, length, &dummy, &$$);   break;
            case SUM:  stasum($3->data, length, &$$, &dummy); $$ *= length; break;
            case IMIN: minmax($3->data, length, &dummy, &dummy2, &ind, &idummy); $$ = (double) ind; break;
            case IMAX: minmax($3->data, length, &dummy, &dummy2, &idummy, &ind); $$ = (double) ind; break;
	    }
	}
	| INT '(' vexpr ',' vexpr ')' {
	    if ($3->length != $5->length) {
		yyerror("X and Y are of different length");
		return 1;
            } else {
                $$ = trapint($3->data, $5->data, NULL, NULL, $3->length);
            }
        }
        | array '.' LENGTH 		  { $$ = $1->length; }
        | selectset '.' LENGTH 		  { $$ = getsetlength($1->gno, $1->setno); }
        | selectset '.' ID 		  { $$ = $1->setno; }
        | selectgraph '.' ID 		  { $$ = $1; }
        | CONSTANT      		  { $$ = ((ParserFnc) (key[$1].data)) (); }
        | expr UCONSTANT        	  { $$ = $1 * ((ParserFnc) (key[$2].data)) (); }
        | RAND  		 	  { $$ = drand48(); }
        | FUNC_I '(' iexpr ')'  	  { $$ = ((ParserFnc) (key[$1].data)) ($3);     }
        | FUNC_D '(' expr ')'   	  { $$ = ((ParserFnc) (key[$1].data)) ($3);     }
        | FUNC_ND '(' iexpr ',' expr ')'  { $$ = ((ParserFnc) (key[$1].data)) ($3, $5); }
        | FUNC_NN '(' iexpr ',' iexpr ')' { $$ = ((ParserFnc) (key[$1].data)) ($3, $5);	}
	| FUNC_DD '(' expr ',' expr ')'	  { $$ = ((ParserFnc) (key[$1].data)) ($3, $5);	}
	| FUNC_NND '(' iexpr ',' iexpr ',' expr ')'
	{
	    $$ = ((ParserFnc) (key[$1].data)) ($3, $5, $7);
	}
	| FUNC_PPD '(' expr ',' expr ',' expr ')'
	{
	    $$ = ((ParserFnc) (key[$1].data)) ($3, $5, $7);
	}
	| FUNC_PPPD '(' expr ',' expr ',' expr ',' expr ')'
	{
	    $$ = ((ParserFnc) (key[$1].data)) ($3, $5, $7, $9);
	}
	| FUNC_PPPPD '(' expr ',' expr ',' expr ',' expr ',' expr ')'
	{
	    $$ = ((ParserFnc) (key[$1].data)) ($3, $5, $7, $9, $11);
	}
	| FUNC_PPPPPD '(' expr ',' expr ',' expr ',' expr ',' expr ',' expr ')'
	{
	    $$ = ((ParserFnc) (key[$1].data)) ($3, $5, $7, $9, $11, $13);
	}
        | selectgraph '.' VX1 { $$ = g[$1].v.xv1; }
        | selectgraph '.' VX2 { $$ = g[$1].v.xv2; }
        | selectgraph '.' VY1 { $$ = g[$1].v.yv1; }
        | selectgraph '.' VY2 { $$ = g[$1].v.yv2; }
        | selectgraph '.' WX1 { $$ = g[$1].w.xg1; }
        | selectgraph '.' WX2 { $$ = g[$1].w.xg2; }
        | selectgraph '.' WY1 { $$ = g[$1].w.yg1; }
        | selectgraph '.' WY2 { $$ = g[$1].w.yg2; }
        | DATE '(' jdate ')' {            $$ = $3;      }
	| DATE '(' iexpr ',' nexpr ',' nexpr ')' { /* yr, mo, day */
	    $$ = cal_and_time_to_jul($3, $5, $7, 12, 0, 0.0);
	}
	| DATE '(' iexpr ',' nexpr ',' nexpr ',' nexpr ',' nexpr ',' expr ')'
	{ /* yr, mo, day, hr, min, sec */
	    $$ = cal_and_time_to_jul($3, $5, $7, $9, $11, $13);
	}
	| VX1 { if (bad_gno ()) return 1; $$ = g[gn].v.xv1; }
	| VX2 { if (bad_gno ()) return 1; $$ = g[gn].v.xv2; }
	| VY1 { if (bad_gno ()) return 1; $$ = g[gn].v.yv1; }
	| VY2 { if (bad_gno ()) return 1; $$ = g[gn].v.yv2; }
	| WX1 { if (bad_gno ()) return 1; $$ = g[gn].w.xg1; }
	| WX2 { if (bad_gno ()) return 1; $$ = g[gn].w.xg2; }
	| WY1 { if (bad_gno ()) return 1; $$ = g[gn].w.yg1; }
	| WY2 { if (bad_gno ()) return 1; $$ = g[gn].w.yg2; }
	| VXMAX {
	    double vx, vy;
            get_page_viewport(&vx, &vy);
            $$ = vx;
	}
	| VYMAX {
	    double vx, vy;
            get_page_viewport(&vx, &vy);
            $$ = vy;
	}
        | '(' expr ')'  	{ $$ = $2; }
        | expr '+' expr 	{ $$ = $1 + $3; }
        | expr '-' expr 	{ $$ = $1 - $3; }
        | '-' expr %prec UMINUS { $$ = -$2; }
        | '+' expr %prec UMINUS { $$ = $2; }
        | expr '*' expr 	{ $$ = $1 * $3; }
	| expr '/' expr	        { if (div_null ($3)) return 1; $$ = $1 / $3; }
	| expr '%' expr	        { if (div_null ($3)) return 1; $$ = fmod($1, $3); }
	| expr '^' expr {
	    if ($1 < 0 && rint($3) != $3) {
		yyerror("Negative value raised to non-integer power");
		return 1;
            } else if ($1 == 0.0 && $3 <= 0.0) {
		yyerror("Zero raised to non-positive power");
		return 1;
            } else {
                $$ = pow($1, $3);
            }
	}
        | expr '?' expr ':' expr { $$ = $1 ? $3 : $5;  }
        | expr GT expr 	{ $$ = ($1 > $3); }
        | expr LT expr 	{ $$ = ($1 < $3); }
        | expr LE expr 	{ $$ = ($1 <= $3); }
        | expr GE expr 	{ $$ = ($1 >= $3); }
        | expr EQ expr 	{ $$ = ($1 == $3); }
        | expr NE expr 	{ $$ = ($1 != $3); }
        | expr AND expr { $$ = $1 && $3;   }
        | expr OR expr  { $$ = $1 || $3;   }
        | NOT expr      { $$ = !($2);      }
	;

sexpr:	CHRSTR {
            $$ = $1;
	}
        | sexpr '.' sexpr {
            $$ = concat_strings($1, $3);
            xfree($3);
        }
        | sexpr '.' expr {
            char buf[32];
            set_locale_num(TRUE);
            sprintf(buf, "%g", $3);
            set_locale_num(FALSE);
            $$ = concat_strings($1, buf);
        }
        ;

iexpr:  expr {
	    int itmp = rint($1);
            if (fabs(itmp - $1) > 1.e-6) {
		yyerror("Non-integer value supplied for integer");
		return 1;
            }
            $$ = itmp;
        }
        ;

nexpr:	iexpr {
            if ($1 < 0) {
		yyerror("Negative value supplied for non-negative");
		return 1;
            }
            $$ = $1;
	}
        ;

indx:	'[' iexpr ']' {
	    int itmp = $2 - index_shift;
            if (itmp < 0) {
		yyerror("Negative index");
		return 1;
            }
            $$ = itmp;
	}
        ;

jdate:  expr {
            $$ = $1;
        }
        | sexpr {
            double jul;
            Dates_format dummy;
            if (parse_date($1, get_date_hint(), FALSE, &jul, &dummy)
                == RETURN_SUCCESS) {
                xfree($1);
                $$ = jul;
            } else {
                xfree($1);
		yyerror("Invalid date");
		return 1;
            }
        }
        ;

jrawdate:  expr {
            $$ = $1;
        }
        | sexpr {
            double jul;
            Dates_format dummy;
            if (parse_date($1, get_date_hint(), TRUE, &jul, &dummy)
                == RETURN_SUCCESS) {
                xfree($1);
                $$ = jul;
            } else {
                xfree($1);
		yyerror("Invalid date");
		return 1;
            }
        }
        ;

array:	VEC_D { $$ = $1; }
        | datacolumn
	{
	    double *ptr = getcol(vasgn_gno, vasgn_setno, $1);
            $$ = &freelist[fcnt++];
            $$->type = GRARR_SET;
            $$->data = ptr;
            if (ptr == NULL) {
                errmsg("NULL variable - check set type");
                return 1;
            } else {
                $$->length = getsetlength(vasgn_gno, vasgn_setno);
            }
	}
	| selectset '.' datacolumn
	{
	    double *ptr = getcol($1->gno, $1->setno, $3);
            $$ = &freelist[fcnt++];
            $$->type = GRARR_SET;
            $$->data = ptr;
            if (ptr == NULL) {
                errmsg("NULL variable - check set type");
                return 1;
            } else {
                $$->length = getsetlength($1->gno, $1->setno);
            }
	}
        ;

vexpr:
	array
	{
            $$ = $1;
	}
	| array '[' iexpr ':' iexpr ']'
	{
            int start = $3 - index_shift, stop = $5 - index_shift;
            if (start < 0 || stop < start || stop >= $1->length) {
		yyerror("Invalid index range");
            } else {
                int len = stop - start + 1;
	        double *ptr = xmalloc(len*SIZEOF_DOUBLE);
                if ($$->data == NULL) {
                    yyerror("Not enough memory");
                } else {
                    int i;
                    $$ = &freelist[fcnt++];
	            $$->data = ptr;
                    $$->length = len;
                    $$->type = GRARR_TMP;
                    for (i = 0; i < len; i++) {
                        $$->data[i] = $1->data[i + $3];
                    }
                }
            }
	}
	| MESH '(' nexpr ')'
	{
            int len = $3;
            if (len < 1) {
                yyerror("npoints must be > 0");
            } else {
                double *ptr = allocate_index_data(len);
                if (ptr == NULL) {
                    errmsg("Malloc failed");
                    return 1;
                } else {
                    $$ = &freelist[fcnt++];
                    $$->type = GRARR_TMP;
                    $$->data = ptr;
                    $$->length = len;
                }
            }
	}
	| MESH '(' expr ',' expr ',' nexpr ')'
	{
            int len = $7;
            if (len < 2) {
                yyerror("npoints must be > 1");
            } else {
                double *ptr = allocate_mesh($3, $5, len);
                if (ptr == NULL) {
                    errmsg("Malloc failed");
                    return 1;
                } else {
                    $$ = &freelist[fcnt++];
                    $$->type = GRARR_TMP;
                    $$->data = ptr;
                    $$->length = len;
                }
            }
	}
	| RAND '(' nexpr ')'
	{
	    int i;
            int len = $3;
            $$ = &freelist[fcnt++];
	    $$->data = xmalloc($3*SIZEOF_DOUBLE);
            if ($$->data == NULL) {
                errmsg("Not enough memory");
                return 1;
            } else {
                $$->length = len;
                $$->type = GRARR_TMP;
            }
            for (i = 0; i < $$->length; i++) {
		$$->data[i] = drand48();
	    }
	}
	| REGNUM '(' selectset ')'
	{
	    int rtype, i, len;
            char *rarray;

            rtype = RESTRICT_REG0 + $1;

	    if (get_restriction_array($3->gno, $3->setno,
                rtype, FALSE, &rarray) != RETURN_SUCCESS) {
                errmsg("Error in region evaluation");
                return 1;
	    }

            len = getsetlength($3->gno, $3->setno);
            $$ = &freelist[fcnt++];
	    $$->data = xmalloc(len*SIZEOF_DOUBLE);
            if ($$->data == NULL) {
                errmsg("Not enough memory");
                return 1;
            } else {
                $$->length = len;
                $$->type = GRARR_TMP;
            }
            for (i = 0; i < $$->length; i++) {
		$$->data[i] = rarray[i];
	    }

            xfree(rarray);
	}
	| RSUM '(' vexpr ')' {
            int i;
            $$ = tmp_vec ($3);
            for (i = 1; i < $$->length; i++) {
                $$->data[i] += $$->data[i - 1];
            }
	}
	| FUNC_I '(' vexpr ')' {
	    int i;
            $$ = tmp_vec ($3);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = ((ParserFnc) (key[$1].data)) ((int) ($3->data[i]));
	    }
	}
	| FUNC_D '(' vexpr ')'
	{
	    int i;
            $$ = tmp_vec ($3);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = ((ParserFnc) (key[$1].data)) (($3->data[i]));
	    }
	}
	| FUNC_DD '(' vexpr ',' vexpr ')' {
	    int i;
	    if ($3->length != $5->length) {
                errmsg("Can't operate on vectors of different lengths");
                return 1;
            }
            $$ = tmp_vec ($3);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = ((ParserFnc) (key[$1].data)) ($3->data[i], $5->data[i]);
	    }
	}
	| FUNC_DD '(' expr ',' vexpr ')' {
	    int i;
            $$ = tmp_vec ($5);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = ((ParserFnc) (key[$1].data)) ($3, $5->data[i]);
	    }
	}
	| FUNC_DD '(' vexpr ',' expr ')' {
	    int i;
            $$ = tmp_vec ($3);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = ((ParserFnc) (key[$1].data)) ($3->data[i], $5);
	    }
	}
	| FUNC_ND '(' iexpr ',' vexpr ')' {
	    int i;
            $$ = tmp_vec ($5);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = ((ParserFnc) (key[$1].data)) ($3, $5->data[i]);
	    }
	}
	| FUNC_NND '(' iexpr ',' iexpr ',' vexpr ')' {
	    int i;
            $$ = tmp_vec ($7);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = ((ParserFnc) (key[$1].data)) ($3, $5, $7->data[i]);
	    }
	}
	| FUNC_PPD '(' expr ',' expr ',' vexpr ')' {
	    int i;
            $$ = tmp_vec ($7);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = ((ParserFnc) (key[$1].data)) ($3, $5, $7->data[i]);
	    }
	}
	| FUNC_PPPD '(' expr ',' expr ',' expr ',' vexpr ')' {
	    int i;
            $$ = tmp_vec ($9);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = ((ParserFnc) (key[$1].data)) ($3, $5, $7, $9->data[i]);
	    }
	}
	| FUNC_PPPPD '(' expr ',' expr ',' expr ',' expr ',' vexpr ')' {
	    int i;
            $$ = tmp_vec ($11);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = ((ParserFnc) (key[$1].data)) ($3, $5, $7, $9, $11->data[i]);
	    }
	}
	| FUNC_PPPPPD '(' expr ',' expr ',' expr ',' expr ',' expr ',' vexpr ')' {
	    int i;
            $$ = tmp_vec ($13);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = ((ParserFnc) (key[$1].data)) ($3, $5, $7, $9, $11, $13->data[i]);
	    }
	}
	| vexpr '+' vexpr {
	    int i;
	    if ($1->length != $3->length) {
                errmsg("Can't operate on vectors of different lengths");
                return 1;
            }
            $$ = tmp_vec ($1);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = $1->data[i] + $3->data[i];
	    }
	}
	| vexpr '+' expr {
	    int i;
            $$ = tmp_vec ($1);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = $1->data[i] + $3;
	    }
	}
	| expr '+' vexpr {
	    int i;
            $$ = tmp_vec ($3);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = $1 + $3->data[i];
	    }
	}
	| vexpr '-' vexpr {
	    int i;
	    if ($1->length != $3->length) {
                errmsg("Can't operate on vectors of different lengths");
                return 1;
            }
            $$ = tmp_vec ($1);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = $1->data[i] - $3->data[i];
	    }
	}
	| vexpr '-' expr {
	    int i;
            $$ = tmp_vec ($1);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = $1->data[i] - $3;
	    }
	}
	| expr '-' vexpr {
	    int i;
            $$ = tmp_vec ($3);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = $1 - $3->data[i];
	    }
	}
	| vexpr '*' vexpr {
	    int i;
	    if ($1->length != $3->length) {
                errmsg("Can't operate on vectors of different lengths");
                return 1;
            }
            $$ = tmp_vec ($1);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = $1->data[i] * $3->data[i];
	    }
	}
	| vexpr '*' expr {
	    int i;
            $$ = tmp_vec ($1);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = $1->data[i] * $3;
	    }
	}
	| expr '*' vexpr
	{
	    int i;
            $$ = &freelist[fcnt++];
	    copy_grarr ($$, $3);
            $$->type = GRARR_TMP;

	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = $1 * $3->data[i];
	    }
	}
	| vexpr '/' vexpr {
	    int i;
	    if ($1->length != $3->length) {
                errmsg("Can't operate on vectors of different lengths");
                return 1;
            }
            $$ = tmp_vec ($1);
	    for (i = 0; i < $$->length; i++) {
	      if (div_null ($3->data[i])) return 1;
              $$->data[i] = $1->data[i] / $3->data[i];
	    }
	}
	| vexpr '/' expr {
	    int i;
	    if (div_null ($3)) return 1;
            $$ = tmp_vec ($1);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = $1->data[i] / $3;
	    }
	}
	| expr '/' vexpr {
	    int i;
            $$ = tmp_vec ($3);
	    for (i = 0; i < $$->length; i++) {
	      if (div_null ($3->data[i])) return 1;
	      $$->data[i] = $1 / $3->data[i];
	    }
	}
	| vexpr '%' vexpr {
	    int i;
	    if ($1->length != $3->length) {
                errmsg("Can't operate on vectors of different lengths");
                return 1;
            }
            $$ = tmp_vec ($1);
	    for (i = 0; i < $$->length; i++) {
	      if (div_null ($3->data[i])) return 1;
	      $$->data[i] = fmod($1->data[i], $3->data[i]);
	    }
	}
	| vexpr '%' expr {
	    int i;
	    if (div_null ($3)) return 1;
            $$ = tmp_vec ($1);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = fmod($1->data[i], $3);
	    }
	}
	| expr '%' vexpr {
	    int i;
            $$ = tmp_vec ($3);
	    for (i = 0; i < $$->length; i++) {
	      if (div_null ($3->data[i])) return 1;
	      $$->data[i] = fmod($1, $3->data[i]);
	    }
	}
	| vexpr '^' vexpr {
	    int i;
	    if ($1->length != $3->length) {
                errmsg("Can't operate on vectors of different lengths");
                return 1;
            }
            $$ = tmp_vec ($1);
	    for (i = 0; i < $$->length; i++) {
	        if ($1->data[i] < 0 && rint($3->data[i]) != $3->data[i]) {
	            yyerror("Negative value raised to non-integer power");
	            return 1;
                } else if ($1->data[i] == 0.0 && $3->data[i] <= 0.0) {
	            yyerror("Zero raised to non-positive power");
	            return 1;
                } else {
                    $$->data[i] = pow($1->data[i], $3->data[i]);
                }
	    }
	}
	| vexpr '^' expr {
	    int i;
            $$ = tmp_vec ($1);
	    for (i = 0; i < $$->length; i++) {
	        if ($1->data[i] < 0 && rint($3) != $3) {
	            yyerror("Negative value raised to non-integer power");
	            return 1;
                } else if ($1->data[i] == 0.0 && $3 <= 0.0) {
	            yyerror("Zero raised to non-positive power");
	            return 1;
                } else {
                    $$->data[i] = pow($1->data[i], $3);
                }
	    }
	}
	| expr '^' vexpr {
	    int i;
            $$ = tmp_vec ($3);
	    for (i = 0; i < $$->length; i++) {
	        if ($1 < 0 && rint($3->data[i]) != $3->data[i]) {
	            yyerror("Negative value raised to non-integer power");
	            return 1;
                } else if ($1 == 0.0 && $3->data[i] <= 0.0) {
	            yyerror("Zero raised to non-positive power");
	            return 1;
                } else {
                    $$->data[i] = pow($1, $3->data[i]);
                }
	    }
	}
	| vexpr UCONSTANT {
	    int i;
            $$ = tmp_vec ($1);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = $1->data[i] * ((ParserFnc) (key[$2].data)) ();
	    }
	}
	| vexpr '?' expr ':' expr {
            int i;
            $$ = &freelist[fcnt++];
	    copy_grarr ($$, $1);
            $$->type = GRARR_TMP;
            for (i = 0; i < $$->length; i++) {
                $$->data[i] = CAST_DBL_TO_BOOL($1->data[i]) ? $3 : $5;
            }
	}
	| vexpr '?' expr ':' vexpr {
            int i;
	    if ($1->length != $5->length) {
                errmsg("Can't operate on vectors of different lengths");
                return 1;
            }
            $$ = &freelist[fcnt++];
	    copy_grarr ($$, $1);
            $$->type = GRARR_TMP;
            for (i = 0; i < $$->length; i++) {
                $$->data[i] = CAST_DBL_TO_BOOL($1->data[i]) ? $3 : $5->data[i];
            }
	}
	| vexpr '?' vexpr ':' expr {
            int i;
	    if ($1->length != $3->length) {
                errmsg("Can't operate on vectors of different lengths");
                return 1;
            }
            $$ = &freelist[fcnt++];
	    copy_grarr ($$, $1);
            $$->type = GRARR_TMP;
            for (i = 0; i < $$->length; i++) {
                $$->data[i] = CAST_DBL_TO_BOOL($1->data[i]) ? $3->data[i] : $5;
            }
	}
	| vexpr '?' vexpr ':' vexpr {
            int i;
	    if ($1->length != $5->length || $1->length != $3->length) {
                errmsg("Can't operate on vectors of different lengths");
                return 1;
            }
            $$ = &freelist[fcnt++];
	    copy_grarr ($$, $1);
            $$->type = GRARR_TMP;
            for (i = 0; i < $$->length; i++) {
                $$->data[i] = CAST_DBL_TO_BOOL($1->data[i]) ? $3->data[i] : $5->data[i];
            }
	}
	| vexpr OR vexpr {
	    int i;
	    if ($1->length != $3->length) {
                errmsg("Can't operate on vectors of different lengths");
                return 1;
            }
            $$ = tmp_vec ($1);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = $1->data[i] || $3->data[i];
	    }
	}
	| vexpr OR expr	{
	    int i;
            $$ = tmp_vec ($1);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = $1->data[i] || $3;
	    }
	}
	| expr OR vexpr	{
	    int i;
            $$ = tmp_vec ($3);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = $1 || $3->data[i];
	    }
	}
	| vexpr AND vexpr {
	    int i;
	    if ($1->length != $3->length) {
                errmsg("Can't operate on vectors of different lengths");
                return 1;
            }
	    copy_grarr ($$, $1);
            $$->type = GRARR_TMP;

	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = $1->data[i] && $3->data[i];
	    }
	}
	| vexpr AND expr {
	    int i;
            $$ = tmp_vec ($1);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = $1->data[i] && $3;
	    }
	}
	| expr AND vexpr {
	    int i;
            $$ = tmp_vec ($3);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = $1 && $3->data[i];
	    }
	}
	| vexpr GT vexpr {
	    int i;
	    if ($1->length != $3->length) {
                errmsg("Can't operate on vectors of different lengths");
                return 1;
            }
            $$ = tmp_vec ($1);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = ($1->data[i] > $3->data[i]);
	    }
	}
	| vexpr GT expr	{
	    int i;
            $$ = tmp_vec ($1);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = ($1->data[i] > $3);
	    }
	}
	| expr GT vexpr	{
	    int i;
            $$ = tmp_vec ($3);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = ($1 > $3->data[i]);
	    }
	}
	| vexpr LT vexpr {
	    int i;
	    if ($1->length != $3->length) {
                errmsg("Can't operate on vectors of different lengths");
                return 1;
            }
            $$ = tmp_vec ($1);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = ($1->data[i] < $3->data[i]);
	    }
	}
	| vexpr LT expr	{
	    int i;
            $$ = tmp_vec ($1);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = ($1->data[i] < $3);
	    }
	}
	| expr LT vexpr	{
	    int i;
            $$ = tmp_vec ($3);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = ($1 < $3->data[i]);
	    }
	}
	| vexpr GE vexpr {
	    int i;
	    if ($1->length != $3->length) {
                errmsg("Can't operate on vectors of different lengths");
                return 1;
            }
            $$ = tmp_vec ($1);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = ($1->data[i] >= $3->data[i]);
	    }
	}
	| vexpr GE expr	{
	    int i;
            $$ = tmp_vec ($1);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = ($1->data[i] >= $3);
	    }
	}
	| expr GE vexpr	{
	    int i;
            $$ = tmp_vec ($3);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = ($1 >= $3->data[i]);
	    }
	}
	| vexpr LE vexpr {
	    int i;
	    if ($1->length != $3->length) {
                errmsg("Can't operate on vectors of different lengths");
                return 1;
            }
            $$ = tmp_vec ($1);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = ($1->data[i] <= $3->data[i]);
	    }
	}
	| vexpr LE expr	{
	    int i;
            $$ = tmp_vec ($1);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = ($1->data[i] <= $3);
	    }
	}
	| expr LE vexpr	{
	    int i;
            $$ = tmp_vec ($3);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = ($1 <= $3->data[i]);
	    }
	}
	| vexpr EQ vexpr {
	    int i;
	    if ($1->length != $3->length) {
                errmsg("Can't operate on vectors of different lengths");
                return 1;
            }
            $$ = tmp_vec ($1);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = ($1->data[i] == $3->data[i]);
	    }
	}
	| vexpr EQ expr	{
	    int i;
            $$ = tmp_vec ($1);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = ($1->data[i] == $3);
	    }
	}
	| expr EQ vexpr	{
	    int i;
            $$ = tmp_vec ($3);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = ($1 == $3->data[i]);
	    }
	}
	| vexpr NE vexpr {
	    int i;
	    if ($1->length != $3->length) {
                errmsg("Can't operate on vectors of different lengths");
                return 1;
            }
            $$ = tmp_vec ($1);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = ($1->data[i] != $3->data[i]);
	    }
	}
	| vexpr NE expr	{
	    int i;
            $$ = tmp_vec ($1);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = ($1->data[i] != $3);
	    }
	}
	| expr NE vexpr	{
	    int i;
            $$ = tmp_vec ($3);
	    for (i = 0; i < $$->length; i++) {
		$$->data[i] = ($1 != $3->data[i]);
	    }
	}
	| NOT vexpr {
	    int i;
            $$ = tmp_vec ($2);
            for (i = 0; i < $$->length; i++) {
                $$->data[i] = !$2->data[i];
            }
	}
	| '(' vexpr ')'	{
	    int i;
            $$ = tmp_vec ($2);
            for (i = 0; i < $$->length; i++) {
                $$->data[i] = $2->data[i];
            }
	}
	| '-' vexpr %prec UMINUS {
	    int i;
            $$ = tmp_vec ($2);
            for (i = 0; i < $$->length; i++) {
                $$->data[i] = - $2->data[i];
            }
	}
//	| mexpr '*' vexpr {
//	    int i ,j ,k;
//            $$ = tmp_vec ($3);
//            for (i = 0; i < $$->length; i++) {
//                $$->data[i] = - $2->data[i];
//            }
//	}
	;

matrix: MAT_D { $$ = $1; }
;

mexpr: matrix { $$ = $1; }
       | mexpr '+' mexpr {
	   int i ,j;
	   if ($1->nrows != $3->nrows || $1->ncols != $3->ncols) {
	     errmsg ("mexpr + mexpr: Left and right matrices are of different size");
	   }
	   $$ = tmp_mat ($1);
	   for (i = 0; i < $1->nrows; i++) {
	     for (j = 0; j < $1->ncols; j++) {
	       $$->matrix[i][j] = $1->matrix[i][j] + $3->matrix[i][j];
	     }
	   }
	}
;

lside_matrix: matrix {
  	       switch ($1->type) {
  	       case GRARR_MAT:
  	         vasgn_gno   = -1;
  	         vasgn_setno = -1;
		 masgn_row1  = -1;
		 masgn_row2  = -1;
		 masgn_col1  = -1;
		 masgn_col2  = -1;
  	         break;
  	         // TODO add assignation to a set */
  	       default:
  	         errmsg("lside_matrix: Internal error");
  	         return 1;
  	       }
  	       $$ = $1;
              }
         | matrix '[' ']' '[' nexpr ']' {
  	       switch ($1->type) {
  	       case GRARR_MAT:
  	         vasgn_gno   = -1;
  	         vasgn_setno = -1;
		 masgn_row1  = -1;
		 masgn_row2  = -1;
		 masgn_col1 = $5;
		 masgn_col2 = $5;
  	         break;
  	         // TODO add assignation to a set */
  	       default:
  	         errmsg("lside_matrix: Internal error");
  	         return 1;
  	       }
  	       $$ = $1;
          }
         | matrix '[' nexpr ']' '[' ']' {
  	       switch ($1->type) {
  	       case GRARR_MAT:
  	         vasgn_gno   = -1;
  	         vasgn_setno = -1;
		 masgn_row1  = $3;
		 masgn_row2  = $3;
		 masgn_col1  = -1;
		 masgn_col2  = -1;
  	         break;
  	         // TODO add assignation to a set */
  	       default:
  	         errmsg("lside_matrix: Internal error");
  	         return 1;
  	       }
  	       $$ = $1;
          }
         | matrix '[' nexpr ':' nexpr  ']' '[' ']' {
  	       switch ($1->type) {
  	       case GRARR_MAT:
  	         vasgn_gno   = -1;
  	         vasgn_setno = -1;
		 masgn_row1  = $3;
		 masgn_row2  = $5;
		 masgn_col1  = -1;
		 masgn_col2  = -1;
  	         break;
  	         // TODO add assignation to a set */
  	       default:
  	         errmsg("lside_matrix: Internal error");
  	         return 1;
  	       }
  	       $$ = $1;
          }
         | matrix '[' ']' '[' nexpr ':' nexpr  ']' {
  	       switch ($1->type) {
  	       case GRARR_MAT:
  	         vasgn_gno   = -1;
  	         vasgn_setno = -1;
		 masgn_row1  = -1;
		 masgn_row2  = -1;
		 masgn_col1  = $5;
		 masgn_col2  = $7;
  	         break;
  	         // TODO add assignation to a set */
  	       default:
  	         errmsg("lside_matrix: Internal error");
  	         return 1;
  	       }
  	       $$ = $1;
          }
;

masgn: lside_matrix '=' mexpr {                   /* matrix = matrix */
	   int i ,j;
	   if ($1->nrows != $3->nrows || $1->ncols != $3->ncols) {
	     errmsg ("Left and right matrices are of different size");
	   }
	   for (i = 0; i < $1->nrows; i++) {
	     for (j = 0; j < $1->ncols; j++) {
	       $1->matrix[i][j] = $3->matrix[i][j];
	     }
	   }
	 }
	 | lside_matrix '=' expr {                /* matrix = scalar */
	   int i ,j;
	   for (i = 0; i < $1->nrows; i++) {
	     for (j = 0; j < $1->ncols; j++) {
	       $1->matrix[i][j] = $3;
	     }
	   }
	 }
         | lside_matrix '=' vexpr {
	   int i ,j ,k;
	   if (masgn_row1 < 0 && masgn_col1 < 0) {
	     /* matrix = vector : casting of the vector into a matrix */
	     if ($1->nrows * $1->ncols != $3->length) {
	       errmsg ("Left matrice size incompatible with right vector length ");
	     }
	     k = 0;
	     for (i = 0; i < $1->nrows; i++) {
	       for (j = 0; j < $1->ncols; j++) {
		 $1->matrix[i][j] = $3->data[k];
		 k++;
	       }
	     }
	   } else if (masgn_row1 < 0 && masgn_col1 >= 0) {
	     /* matrix[][n]     = vector :  vector is copied into column n of the matrix */
	     /* matrix[][n1:n2] = vector :  vector is duplicated into columns n1 to n2 of the matrix */
	     if ($1->nrows != $3->length) {
	       errmsg ("Left matrice number of rows not equal to vector length ");
	     }
	     for (j = masgn_col1; j <= masgn_col2; j++) {
	       for (i = 0; i < $1->nrows; i++) {
		 $1->matrix[i][j] = $3->data[i];
	       }
	     }
	   } else if (masgn_row1 >= 0  && masgn_col1 < 0) {
	     /* matrix[n1:n2][] = vector : vector is duplicated into rows of the matrix */
	     if ($1->ncols != $3->length) {
	       errmsg ("Left matrice number of columns not equal to vector length ");
	     }
	     for (j = masgn_row1; j <= masgn_row2; j++) {
	       for (i = 0; i < $1->ncols; i++) {
		 $1->matrix[j][i] = $3->data[i];
	       }
             }
           }
         }
;


asgn:
        VAR_D '=' expr        { *($1) = $3; }
        | FITPARM '=' expr    { nonl_parms[$1].value = $3; }
        | FITPMAX '=' expr    { nonl_parms[$1].max = $3; }
        | FITPMIN '=' expr    { nonl_parms[$1].min = $3; }
        | array indx '=' expr {
        		          if ($2 >= $1->length) {
        		              yyerror("Access beyond  bounds");
        		              return 1;
        		          }
        		          $1->data[$2] = $4;
        		      }
	;

lside_array: array {
            target tgt;
            switch ($1->type) {
            case GRARR_SET:
                if (find_set_bydata($1->data, &tgt) == RETURN_SUCCESS) {
                    vasgn_gno   = tgt.gno;
                    vasgn_setno = tgt.setno;
                } else {
                    errmsg("Internal error");
		    return 1;
                }
                break;
            case GRARR_VEC:
                vasgn_gno   = -1;
                vasgn_setno = -1;
                break;
            default:
                /* It can NOT be a tmp array on the left side! */
                errmsg("Internal error");
	        return 1;
            }
            $$ = $1;
        }
        ;

vasgn:
	lside_array '=' vexpr
	{
	    int i;
	    if ($1->length != $3->length) {
                errmsg("Left and right vectors are of different lengths");
                return 1;
            }
	    for (i = 0; i < $1->length; i++) {
	        $1->data[i] = $3->data[i];
	    }
	}
	| lside_array '=' expr
	{
	    int i;
	    for (i = 0; i < $1->length; i++) {
	        $1->data[i] = $3;
	    }
	}
	| lside_array '=' mexpr
	{
	   int i ,j ,k;
	   if ($3->nrows * $3->ncols != $1->length) {
	     errmsg ("Right matrice size incompatible with left vector length ");
	   }
	   k = 0;
	   for (i = 0; i < $3->nrows; i++) {
	     for (j = 0; j < $3->ncols; j++) {
	       $1->data[k] = $3->matrix[i][j];
	     k++;
	     }
	   }
	}
        ;

flowcontrol:
         IF '(' onoff ')' '{'  {
  	 printf ("flowcontrol: IF detected \n");
	 ptree = ptree_if_begin (ptree);
         }
         | '}' {
  	 printf ("flowcontrol: } detected \n");
         }

defines:
	DEFINE NEW_TOKEN
        {
	    symtab_entry tmpkey;
            double *var;
            var		  = xmalloc(SIZEOF_DOUBLE);
            *var	  = 0.0;
	    tmpkey.s	  = $2;
	    tmpkey.type	  = KEY_VAR;
	    tmpkey.status = VAR_USER;
	    tmpkey.data	  = (void *) var;
	    if (addto_symtab (tmpkey) != RETURN_SUCCESS) {
	        yyerror("Adding new symbol failed");
	    }

            xfree($2);
        }
	| DEFINE NEW_TOKEN '[' ']'
        {
	    if (define_parser_arr ($2 ,VAR_USER) == NULL) {
	        yyerror("Adding new symbol failed");
	    }

            xfree($2);
        }
	| DEFINE NEW_TOKEN '[' nexpr ']'
        {
	    grarr *var;
            if ((var = define_parser_arr ($2 ,VAR_USER)) == NULL) {
	        yyerror("Adding new symbol failed");
	    } else {
                realloc_vrbl(var, $4);
            }

            xfree($2);
        }
	| DEFINE NEW_TOKEN '[' nexpr ']' '[' nexpr ']'
        {
	    grarr *var;
            if ((var = define_parser_arr2 ($2 ,VAR_USER ,$4 ,$7)) == NULL) {
	        yyerror("Adding new 2D symbol failed");
            }
            xfree($2);
        }
	| DEFINE VAR_D { yyerror("Keyword already exists"); }
	| DEFINE VEC_D { yyerror("Keyword already exists"); }
	| DEFINE MAT_D { yyerror("Keyword already exists"); }
	| CLEAR VAR_D  { undefine_parser_var((void *) $2); xfree($2); }
	| CLEAR VEC_D  {
            realloc_vrbl($2, 0);
            undefine_parser_var((void *) $2);
            xfree($2);
        }
	| CLEAR MAT_D   {
	   int i;
	   for (i = 0; i < $2->nrows; i++) xfree ($2->matrix[i]);
	   xfree ($2->matrix);
	   $2->length = 0;
	   $2->nrows = 0;
	   $2->ncols = 0;
            undefine_parser_var((void *) $2);
            xfree($2);
          }
	| CLEAR ALLVARS { undefine_all_users_var (); }
	| ALIAS sexpr sexpr {
	    int position;

	    lowtoupper($3);
	    if ((position = findf(key, $3)) >= 0) {
	        symtab_entry tmpkey;
		tmpkey.s = $2;
		tmpkey.type   = key[position].type;
		tmpkey.data   = key[position].data;
		if (addto_symtab(tmpkey) != RETURN_SUCCESS) {
		    yyerror("Keyword already exists");
		}
	    } else {
	        yyerror("Aliased keyword not found");
	    }
	    xfree($2);
	    xfree($3);
	}
	| ALIAS FORCE onoff {
	    alias_force = $3;
	}
	| USE sexpr TYPE proctype FROM sexpr {
	    if (load_module($6, $2, $2, $4) != 0) {
	        yyerror("DL module load failed");
	    }
	    xfree($2);
	    xfree($6);
	}
	| USE sexpr TYPE proctype FROM sexpr ALIAS sexpr {
	    if (load_module($6, $2, $8, $4) != 0) {
	        yyerror("DL module load failed");
	    }
	    xfree($2);
	    xfree($6);
	    xfree($8);
	}
        ;

regionset:
        REGNUM onoff 		 { rg[$1].active = $2; }
        | REGNUM TYPE regiontype { rg[$1].type   = $3;   }
        | REGNUM color_select 	 { rg[$1].color  = $2;  }
        | REGNUM lines_select 	 { rg[$1].lines  = $2;  }
        | REGNUM linew_select 	 { rg[$1].linew  = $2;  }
        | REGNUM LINE expr ',' expr ',' expr ',' expr
        {
	    rg[$1].x1 = $3;
	    rg[$1].y1 = $5;
	    rg[$1].x2 = $7;
	    rg[$1].y2 = $9;
	}
	| REGNUM XY expr ',' expr
	{
	    rg[$1].x = xrealloc(rg[$1].x, (rg[$1].n + 1) * SIZEOF_DOUBLE);
	    rg[$1].y = xrealloc(rg[$1].y, (rg[$1].n + 1) * SIZEOF_DOUBLE);
	    rg[$1].x[rg[$1].n] = $3;
	    rg[$1].y[rg[$1].n] = $5;
	    rg[$1].n++;
	}
	| LINK REGNUM TO selectgraph { rg[$2].linkto = $4; }
	;

parmset:
        VERSION nexpr {
	  file_project_version = $2;
            if (set_project_version($2) != RETURN_SUCCESS) {
                errmsg("Project version is newer than software!");
            }
            if (get_project_version() < 50001) {
                map_fonts(FONT_MAP_ACEGR);
            } else {
                map_fonts(FONT_MAP_DEFAULT);
            }
        }
  	| PAGE RESIZE nexpr ',' nexpr 	{ set_page_dimensions 	  ($3 ,$5 ,TRUE);  }
  	| PAGE SIZE nexpr ',' nexpr   	{ set_page_dimensions 	  ($3 ,$5 ,FALSE); }
  	| PAGE SIZE sexpr pageorient  	{ set_page_dimensions_std ($3 ,$4 ,FALSE); }
  	| PAGE RESIZE sexpr pageorient  { set_page_dimensions_std ($3 ,$4 ,TRUE); }
	| DEVICE sexpr PAGE SIZE nexpr ',' nexpr {
            int device_id;
            Device_entry dev;

            device_id = get_device_by_name($2);
            xfree($2);
            if (device_id < 0) {
                yyerror("Unknown device");
            } else {
                dev = get_device_props(device_id);
                dev.pg.width =  (long) ($5*dev.pg.dpi/72);
                dev.pg.height = (long) ($7*dev.pg.dpi/72);
                set_device_props(device_id, dev);
            }
        }
        | DEVICE sexpr DPI expr {
            int device_id;
            Device_entry dev;

            device_id = get_device_by_name($2);
            if (device_id < 0) {
                yyerror("Unknown device");
            } else {
                dev = get_device_props(device_id);
                dev.pg.dpi = $4;
                set_device_props(device_id, dev);
            }
            xfree($2);
        }
        | DEVICE sexpr FONTP ANTIALIASING onoff {
            int device_id;
            Device_entry dev;

            device_id = get_device_by_name($2);
            if (device_id < 0) {
                yyerror("Unknown device");
            } else {
                dev = get_device_props(device_id);
                dev.fontaa = $5;
                set_device_props(device_id, dev);
            }
            xfree($2);
        }
        | DEVICE sexpr FONTP onoff {
            int device_id;
            Device_entry dev;

            device_id = get_device_by_name($2);
            if (device_id < 0) {
                yyerror("Unknown device");
            } else {
                dev = get_device_props(device_id);
                dev.devfonts = $4;
                set_device_props(device_id, dev);
            }
            xfree($2);
        }
        | DEVICE sexpr OP sexpr {
            int device_id;

            device_id = get_device_by_name($2);
            if (device_id < 0) {
                yyerror("Unknown device");
            } else {
                if (parse_device_options(device_id, $4) !=
                                                        RETURN_SUCCESS) {
                    yyerror("Incorrect device option string");
                }
            }
            xfree($2);
            xfree($4);
        }
  	| HARDCOPY DEVICE sexpr   	 { set_printer_by_name    ($3); xfree($3); }
  	| REFERENCE DATE jrawdate 	 { set_ref_date 	  ($3); }
  	| DATE WRAP onoff         	 { allow_two_digits_years ($3); }
  	| DATE WRAP YEAR iexpr    	 { set_wrap_year 	  ($4); }
  	| BACKGROUND color_select 	 { setbgcolor             ($2); }
| PAGE BACKGROUND FILL onoff   	 { } //43g  setbgfill              ($4); }   Problems with @ page background fill off
	| PAGE SCROLL expr '%' 		 { scroll_proc((int) $3);	}
	| PAGE INOUT expr '%' 		 { scrollinout_proc((int) $3);	}
	| LINK PAGE onoff 		 { scrolling_islinked = $3;	}
	| STACK WORLD expr ',' expr ',' expr ',' expr	{ add_world(gn, $3, $5, $7, $9); }
	| TIMER nexpr 			 { timer_delay = $2; }

	| TARGET selectset {
	    target_set = *($2);
	    cur_obj_typ = Q_Set;
	    set_parser_setno(target_set.gno, target_set.setno);
	}
	| TARGET POLYLINE nexpr {
	  curtype = SET_XY; /* we use this although it is not a set */
	  cur_obj_typ = Q_Polyline;
        }
	| WITH selectgraph { set_parser_gno($2); }
	| WITH selectset   { set_parser_setno($2->gno, $2->setno); }

/* Hot links */
	| selectset LINK sourcetype sexpr {
	    set_hotlink($1->gno, $1->setno, 1, $4, $3);
	    xfree($4);
	}
	| selectset LINK onoff {
	    set_hotlink($1->gno, $1->setno, $3, NULL, 0);
	}

        | thislegend onoff                   { g[gn].l.active  = $2; }
        | thislegend LAYER nexpr             { set_legend_layer (gn ,$3); }
        | thislegend LOCTYPE worldview       { g[gn].l.loctype = $3; }
        | thislegend VGAP nexpr              { g[gn].l.vgap    = $3; }
        | thislegend HGAP nexpr              { g[gn].l.hgap    = $3; }
        | thislegend LENGTH nexpr            { g[gn].l.len     = $3; }
        | thislegend INVERT onoff            { g[gn].l.invert  = $3; }
        | thislegend BOX FILL color_select   { g[gn].l.boxfillpen.color   = $4; }
        | thislegend BOX FILL pattern_select { g[gn].l.boxfillpen.pattern = $4; }
        | thislegend BOX color_select        { g[gn].l.boxpen.color   = $3; }
        | thislegend BOX pattern_select      { g[gn].l.boxpen.pattern = $3; }
        | thislegend BOX lines_select        { g[gn].l.boxlines       = $3; }
        | thislegend BOX linew_select        { g[gn].l.boxlinew       = $3; }
        | thislegend expr ',' expr           { g[gn].l.legx = $2;
                                               g[gn].l.legy = $4;
	}
        | thislegend CHAR SIZE expr          { g[gn].l.charsize    = $4; }
        | thislegend font_select             { g[gn].l.font        = $2; }
        | thislegend color_select            { g[gn].l.color       = $2; }


/* timestamp */

| TIMESTAMP HIDDEN onoff          	{ objs[iTS].hidden   = $3; }
	| TIMESTAMP ANCHOR expr ',' expr  { objs[iTS].x1 = $3;
	    		        	    objs[iTS].y1 = $5;      }
        | TIMESTAMP onoff          	{ objs[iTS].hidden   	= ($2 == TRUE) ? 0 : 1; }
	| TIMESTAMP LAYER nexpr    	{ objs[iTS].layer     	= $3; }
	| TIMESTAMP font_select    	{ objs[iTS].font     	= $2; }
	| TIMESTAMP JUST     nexpr 	{ objs[iTS].just     	= $3; }
	| TIMESTAMP CHAR SIZE expr 	{ objs[iTS].charsize 	= $4; }
	| TIMESTAMP ROT nexpr      	{ objs[iTS].rot      	= $3; }
	| TIMESTAMP color_select   	{ objs[iTS].color    	= $2; }
 	| TIMESTAMP lines_select        { objs[iTS].lines    	= $2; }
 	| TIMESTAMP FILL pattern_select { objs[iTS].fillpattern = $3; }
 	| TIMESTAMP FILL color_select   { objs[iTS].fillcolor   = $3; }
	| TIMESTAMP expr ',' expr  	{ objs[iTS].x1 = $2;
	    		   	    	  objs[iTS].y1 = $4; }
	| TIMESTAMP DEF sexpr      	{ set_plotstr_string (&objs[iTS], $3); xfree($3); }

/* defaults */
        | DEFAULT lines_select 	   { grdefaults.lines    	= $2; }
        | DEFAULT linew_select 	   { grdefaults.linew    	= $2; }
        | DEFAULT color_select 	   { grdefaults.color    	= $2; }
        | DEFAULT pattern_select   { grdefaults.pattern  	= $2; }
        | DEFAULT CHAR SIZE expr   { grdefaults.charsize 	= $4; string_size = $4; }
        | DEFAULT font_select 	   { grdefaults.font     	= $2; string_font = $2; }
        | DEFAULT SYMBOL SIZE expr { grdefaults.symsize = $4; }
        | DEFAULT SFORMAT sexpr    { strcpy(sformat, $3); xfree($3);  }
        | MAP FONTP nexpr TO sexpr ',' sexpr {
            			   if ((map_font_by_name($5, $3) != RETURN_SUCCESS) && 
            			       (map_font_by_name($7, $3) != RETURN_SUCCESS)) {
            			       errmsg("Failed mapping a font");
            			   }
            			   xfree($5);
            			   xfree($7);
        }
	| MAP COLOR nexpr TO '(' nexpr ',' nexpr ',' nexpr ')' ',' sexpr {
	    CMap_entry cmap;
            cmap.rgb.red   = $6;
            cmap.rgb.green = $8;
            cmap.rgb.blue  = $10;
            cmap.ctype = COLOR_MAIN;
            cmap.cname = $13;
            if (store_color($3, cmap) == RETURN_FAILURE) {
                errmsg("Failed mapping a color");
            }
	    xfree($13);
        }

	| WORLD expr ',' expr ',' expr ',' expr {
			   if (bad_gno ()) return 1;
	    g[gn].w.xg1 = $2;
	    g[gn].w.yg1 = $4;
	    g[gn].w.xg2 = $6;
	    g[gn].w.yg2 = $8;
	}
	| ZNORM expr {
	    set_graph_znorm(gn, $2);
	}
	| VIEW expr ',' expr ',' expr ',' expr {
			   if (bad_gno ()) return 1;
	    g[gn].v.xv1 = $2;
	    g[gn].v.yv1 = $4;
	    g[gn].v.xv2 = $6;
	    g[gn].v.yv2 = $8;
        }

  	| TITLE sexpr         { if (bad_gno ()) return 1; set_plotstr_string (&g[gn].labs.title, $2); xfree($2); }
  	| TITLE LAYER nexpr   { if (bad_gno ()) return 1; g[gn].labs.title.layer     = $3; }
  	| TITLE font_select   { if (bad_gno ()) return 1; g[gn].labs.title.font      = $2; }
  	| TITLE SIZE expr     { if (bad_gno ()) return 1; g[gn].labs.title.charsize  = $3; }
  	| TITLE color_select  { if (bad_gno ()) return 1; g[gn].labs.title.color     = $2; }
  	| TITLE JUST nexpr 	{ if (bad_gno ()) return 1; g[gn].labs.title.just   = $3; }
  	| TITLE OFFSET expr ',' expr {
  	                        if (bad_gno ()) return 1; g[gn].labs.title.x1     = $3;
                                                    g[gn].labs.title.y1     = $5; }
  	| SUBTITLE sexpr        { if (bad_gno ()) return 1; set_plotstr_string (&g[gn].labs.stitle, $2); xfree($2); }
  	| SUBTITLE LAYER nexpr  { if (bad_gno ()) return 1; g[gn].labs.stitle.layer    = $3; }
  	| SUBTITLE font_select  { if (bad_gno ()) return 1; g[gn].labs.stitle.font     = $2; }
  	| SUBTITLE SIZE expr    { if (bad_gno ()) return 1; g[gn].labs.stitle.charsize = $3; }
  	| SUBTITLE color_select { if (bad_gno ()) return 1; g[gn].labs.stitle.color    = $2; }
  	| SUBTITLE JUST nexpr   { if (bad_gno ()) return 1; g[gn].labs.stitle.just   = $3; }
  	| SUBTITLE OFFSET expr ',' expr {
  	                          if (bad_gno ()) return 1; g[gn].labs.stitle.x1     = $3;
                            	  if (bad_gno ()) return 1; g[gn].labs.stitle.y1     = $5; }
  	| XAXES SCALE scaletype { if (bad_gno ()) return 1; g[gn].xscale  	       	= $3; }
  	| YAXES SCALE scaletype { if (bad_gno ()) return 1; g[gn].yscale  	       	= $3; }
  	| XAXES INVERT onoff    { if (bad_gno ()) return 1; g[gn].xinvert 	       	= $3; }
  	| YAXES INVERT onoff    { if (bad_gno ()) return 1; g[gn].yinvert 	       	= $3; }
  	| AUTOSCALE ONREAD NONE   { autoscale_onread = AUTOSCALE_NONE; }
  	| AUTOSCALE ONREAD XAXES  { autoscale_onread = AUTOSCALE_X;    }
  	| AUTOSCALE ONREAD YAXES  { autoscale_onread = AUTOSCALE_Y;    }
  	| AUTOSCALE ONREAD XYAXES { autoscale_onread = AUTOSCALE_XY;   }
	| DESCRIPTION sexpr {
            char *s;
            s = copy_string(NULL, get_project_description());
            s = concat_strings(s, $2);
	    xfree($2);
            s = concat_strings(s, "\n");
            set_project_description(s);
            xfree(s);
	}
  | CLEAR DESCRIPTION              { set_project_description(NULL); }

  | LEGEND onoff                   { if (bad_gno ())  return 1; g[gn].l.active      	   = $2; }
  | LEGEND LAYER nexpr             { if (bad_gno ())  return 1;        set_legend_layer (gn ,$3); }
  | LEGEND LOCTYPE worldview       { if (bad_gno ())  return 1; g[gn].l.loctype     	   = $3; }
  | LEGEND VGAP nexpr              { if (bad_gno ())  return 1; g[gn].l.vgap        	   = $3; }
  | LEGEND HGAP nexpr              { if (bad_gno ())  return 1; g[gn].l.hgap        	   = $3; }
  | LEGEND LENGTH nexpr            { if (bad_gno ())  return 1; g[gn].l.len         	   = $3; }
  | LEGEND INVERT onoff            { if (bad_gno ())  return 1; g[gn].l.invert      	   = $3; }
  | LEGEND BOX FILL color_select   { if (bad_gno ())  return 1; g[gn].l.boxfillpen.color   = $4; }
  | LEGEND BOX FILL pattern_select { if (bad_gno ())  return 1; g[gn].l.boxfillpen.pattern = $4; }
  | LEGEND BOX color_select        { if (bad_gno ())  return 1; g[gn].l.boxpen.color       = $3; }
  | LEGEND BOX pattern_select      { if (bad_gno ())  return 1; g[gn].l.boxpen.pattern     = $3; }
  | LEGEND BOX lines_select        { if (bad_gno ())  return 1; g[gn].l.boxlines           = $3; }
  | LEGEND BOX linew_select        { if (bad_gno ())  return 1; g[gn].l.boxlinew           = $3; }
  | LEGEND expr ',' expr           { if (bad_gno ())  return 1; g[gn].l.legx               = $2;
                                                                g[gn].l.legy               = $4; }
  | LEGEND CHAR SIZE expr          { if (bad_gno ())  return 1; g[gn].l.charsize           = $4; }
  | LEGEND font_select             { if (bad_gno ())  return 1; g[gn].l.font               = $2; }
  | LEGEND color_select            { if (bad_gno ())  return 1; g[gn].l.color              = $2; }

  | FRAMEP onoff                   { if (bad_gno ())  return 1; g[gn].f.pen.pattern        = $2; }
  | FRAMEP LAYER nexpr             { if (bad_gno ())  return 1; g[gn].f.layer        	   = $3; }
  | FRAMEP TYPE nexpr              { if (bad_gno ())  return 1; g[gn].f.type               = $3; }
  | FRAMEP lines_select            { if (bad_gno ())  return 1; g[gn].f.lines              = $2; }
  | FRAMEP linew_select            { if (bad_gno ())  return 1; g[gn].f.linew              = $2; }
  | FRAMEP color_select            { if (bad_gno ())  return 1; g[gn].f.pen.color          = $2; }
  | FRAMEP pattern_select          { if (bad_gno ())  return 1; g[gn].f.pen.pattern        = $2; }
  | FRAMEP BACKGROUND color_select { if (bad_gno ())  return 1; g[gn].f.fillpen.color      = $3; }
  | FRAMEP BACKGROUND pattern_select { if (bad_gno ())  return 1; g[gn].f.fillpen.pattern  = $3; }


/* ---------------- Graphs ---------------- */
  | selectgraph onoff                 { set_graph_hidden  ($1, !$2); }
  | selectgraph HIDDEN onoff          { set_graph_hidden  ($1, $3); }
  | selectgraph TYPE graphtype        { set_graph_type    ($1, $3); }
  | selectgraph STACKED onoff         { set_graph_stacked ($1, $3); }
  | selectgraph BAR HGAP expr         { set_graph_bargap  ($1, $4); }
  | selectgraph FIXEDPOINT onoff      { g[$1].locator.pointset = $3; }
  | selectgraph FIXEDPOINT TYPE nexpr { g[$1].locator.pt_type = $4; }
  | selectgraph FIXEDPOINT FORMAT formatchoice formatchoice {
                                        g[$1].locator.fx = $4;
                                        g[$1].locator.fy = $5; }
  | selectgraph FIXEDPOINT PREC expr ',' expr {
                                        g[$1].locator.px = $4;
                                        g[$1].locator.py = $6; }
  | selectgraph FIXEDPOINT XY expr ',' expr {
                                        g[$1].locator.dsx = $4;
                                        g[$1].locator.dsy = $6; }

  | TYPE xytype { curtype = $2;}

  /* ---------------- I/O filters ---------------- */
    | DEFINE filtertype sexpr filtermethod sexpr {
      		     		    if (add_io_filter ($2, $4, $5, $3) != 0) {
      		     		      yyerror("Failed adding i/o filter");
      		     		    }
      		     		    xfree ($3);
      		     		    xfree ($5);
      		     		    }
   | CLEAR filtertype                { clear_io_filters($2); }
   | SOURCE sourcetype               { cursource = $2; }
   | FORMAT formatchoice             { readxformat = $2; }
   | FIT nonlfitopts                 { }
   | FITPARM  CONSTRAINTS onoff      { nonl_parms[$2].constr = $3; }


;  /* ---------------- End of parmset: ---------------- */


objtyp:
       LINE       { curobjtyp = Q_Line; curpmask = MA_2_HANDLES; $$ = Q_Line;}
       | BOX      { curobjtyp = Q_Box; 	curpmask = 0;            $$ = Q_Box;}
       | ELLIPSE  { curobjtyp = Q_Arc; 	curpmask = MARC_Ellipse; $$ = Q_Arc;}
       | ARC      { curobjtyp = Q_Arc;                           $$ = Q_Arc;}
       | STRING   { curobjtyp = Q_String;                        $$ = Q_String;}
       | POLYLINE { curobjtyp = Q_Polyline;                      $$ = Q_Polyline;}
      ;

thiscompound:
      COMPOUND nexpr {
        curid = $2;
        if (obj_tid_is_valid (Q_Compound ,curid)) {
          jo = cur_obj_num = obj_geom_get_num (Q_Compound ,curid);
	  $$ = &objs[jo];
        } else {
	  char buf[32];
	  sprintf (buf ,"Compound %d not active" ,curid);
          yyerror(buf);
        }
      }
     ;

incompound:
        selobj
        | thiscompound
        ;

thislegend:
  LEGEND selectgraph {
    if (!is_valid_gno ($2)) {
      yyerror("No valid graph selected");
      return 1;
    }
    gn = $2;
  }


setpobj: WITH objtyp       { curid = -2; jo = cur_obj_num = obj_t_next ($2 ,&curid ,$2 ,-1); }
       | WITH objtyp nexpr { if (obj_tid_is_valid (curobjtyp ,$3))  {
			      jo = cur_obj_num = obj_geom_get_num ($2 ,$3);
			      curid 	  = $3;
             		     } else {
			       int id 	  = $3;
                               jo = obj_t_next ($2 ,&id ,$2 ,-1);
			       if (jo < 0) {
				 yyerror ("error in  WITH objtyp nexpr");
				 jo = cur_obj_num;
			       } else {
				 cur_obj_num = jo;
				 curid       = $3;
			       }
                             }
        }
;  /* ---------------- End of setpobj: ---------------- */

selobj: objtyp       { $$ = &objs[jo]; }          /* use the current obj */
| objtyp nexpr 	     { if (obj_tid_is_valid (curobjtyp ,$2))  {
	    		  jo = cur_obj_num = obj_geom_get_num ($1 ,$2);
		       	  $$ = &objs[jo];
		       } else {
		       	  yyerror ("Object not active");
      	  	       }
	}
;  /* ---------------- End of selobj: ---------------- */


/* objs with no id are attached to template by default and
 *  moved to a graph if needed after the file is red
 */
objects: objtyp DEF 	  { objs[jo].active = TRUE;  }
       | objtyp nexpr DEF { if (obj_tid_is_valid (curobjtyp ,$2))  {
	    		    jo = cur_obj_num = obj_geom_get_num ($1 ,$2);
		       	    } else {
		       	    	  yyerror ("Object not active");
      	  	       	    }
       }
       | objtyp DEF sexpr { objs[jo].active = TRUE;
			    objs[jo].s      = copy_string (objs[jo].s ,$3);
			    if (curobjtyp == Q_Polyline) curpmask = MA_N_HANDLES;
			    if (curobjtyp == Q_Line)	 curpmask = MA_2_HANDLES;
			    xfree ($3);
       }
       | objtyp nexpr DEF sexpr { if (obj_tid_is_valid (curobjtyp ,$2))  {
      				     objs[jo].s      = copy_string (objs[jo].s ,$4);
			             if (curobjtyp == Q_Polyline) curpmask = MA_N_HANDLES;
			             if (curobjtyp == Q_Line)	  curpmask = MA_2_HANDLES;
		       	   	   } else {
		       	   	     yyerror ("Object not active");
      	  	       	   	   }
			           xfree ($4);
      }
      | objtyp selectgraph 	   { obj_geom_attach_to_graph (curobjtyp ,curid ,$2 ,TRUE);  }

      | objtyp nexpr selectgraph   { if (obj_tid_is_valid (curobjtyp ,$2))  {
			      	       jo = cur_obj_num = obj_geom_get_num (curobjtyp ,$2);
				       obj_geom_attach_to_graph (curobjtyp ,$2 ,$3 ,TRUE);
	                             } else {
		       	    	       yyerror ("Object not active");
      	  	       	             }
      }

       | objtyp expr ',' expr ',' expr ',' expr {
 	     		    	        objs[jo].x1 = $2;
 	     		    	        objs[jo].y1 = $4;
 	     		    	        objs[jo].x2 = $6;
 	     		    	        objs[jo].y2 = $8;

       }
       | objtyp expr ',' expr  {
 	     		    	        objs[jo].x1 = $2;
 	     		    	        objs[jo].y1 = $4;
       }
       | selobj ':' expr ',' expr ',' expr ',' expr {
 	     		    	        objs[jo].x1 = $3;
 	     		    	        objs[jo].y1 = $5;
 	     		    	        objs[jo].x2 = $7;
 	     		    	        objs[jo].y2 = $9;
       }
       | selobj ANCHOR expr ',' expr {	objs[jo].x1 = $3;
 	     		    	        objs[jo].y1 = $5;
       }
       | selobj onoff                 { if (cur_obj_typ == Q_String) {
 					   objs[jo].active     = $2;
 					   objs[jo].hidden     = !$2;
					} else {
					   objs[jo].hidden     = !$2;
					}
       }
       | selobj LAYER nexpr                { objs[jo].layer        = $3; }
       | selobj HIDDEN  onoff       	   { objs[jo].hidden       = $3; }
       | selobj LOCTYPE worldview   	   { objs[jo].loctyp       = $3; }
       | selobj lines_select        	   { objs[jo].lines        = $2; }
       | selobj linew_select        	   { objs[jo].linew        = $2; }
       | selobj color_select        	   { objs[jo].color        = $2; }
       | selobj FILL color_select   	   { objs[jo].fillcolor    = $3; }
       | selobj FILL pattern_select 	   { objs[jo].fillpattern  = $3; }
       | selobj ARROW nexpr         	   { objs[jo].arrow_end    = $3; }
       | selobj ARROW LENGTH  expr  	   { objs[jo].arrow.length = $4; }
       | selobj ARROW TYPE   nexpr  	   { objs[jo].arrow.type   = $4; }
       | selobj ARROW LAYOUT expr ',' expr { objs[jo].arrow.dL_ff  = $4;
				    	     objs[jo].arrow.lL_ff  = $6; }
       | selobj ARROW SIZE expr     	   { objs[jo].arrow.length  = 2.0*$4; }
       | selobj TYPE nexpr          	   { if (curobjtyp == Q_Arc) {
 	  			   	       objs[jo].pmask = arc_pmask ($3);
 	  			   	     } else {
 	  			   	       yyerror("selobj TYPE is valid only for ARCs");
 	  			   	     }
 	}
        | selobj ANGLES nexpr ',' nexpr    { objs[jo].astart   = $3; objs[jo].aend = $5; }
        | selobj ROT    nexpr     	   { objs[jo].rot      = $3; }
        | selobj font_select       	   { objs[jo].font     = $2; }
        | selobj JUST   nexpr     	   { objs[jo].just     = $3; }
        | selobj CHAR SIZE  expr   	   { objs[jo].charsize = $4; }
        | selobj sexpr             	   { objs[jo].s        = copy_string (objs[jo].s , $2); }

        | selobj SPLINE '(' expr ',' expr ')' { if (curobjtyp == Q_Polyline)  {
			       			   objs[jo].pmask = objs[jo].pmask | MA_SMOOTH;
			       			   objs[jo].xspline_s = $4;
			       			   objs[jo].xspline_p = $6;
                                                } else {
		       	       	       	           yyerror ("Only polylines use SPLINE");
                                                }
        }
        | selobj SPLINE NONE                { objs[jo].pmask     = objs[jo].pmask & ~MA_SMOOTH; }
        | selobj CLOSE                      { objs[jo].pmask     = objs[jo].pmask | MA_IS_CLOSED; printf ("\n pmask = %d \n" ,objs[jo].pmask); }
        | selobj OPEN                       { objs[jo].pmask     = objs[jo].pmask & ~MA_IS_CLOSED; }
        | selobj LABEL TYPE   nexpr         { objs[jo].label_opt = $4; }
        | selobj LABEL FORMAT sexpr 	    { objs[jo].frmt      = copy_string (objs[jo].frmt ,$4); xfree ($4); }
        | selobj LABEL OFFSET expr ',' expr { objs[jo].perp	 = $4; objs[jo].para = $6; }


/* compounds */
	| NEW COMPOUND incompound 	 { int compound_id = -2;
	  		  	  	   int compound_num = obj_tid_next_compound (&compound_id ,objs[jo].father_typ ,objs[jo].father_id ,jo);
	  		  	  	   if (compound_num > 0) obj_make_current (compound_num);
	}
	| NEW COMPOUND sexpr incompound  { int compound_id = -2;
	    /*	  		  	  	   int compound_num    = obj_t_next_compound_orphaned (&compound_id ,cur_obj_num);*/
	  		  	  	   int compound_num = obj_tid_next_compound (&compound_id ,objs[jo].father_typ ,objs[jo].father_id ,jo);
	  		  	  	   if (compound_num > 0) {
					     obj_make_current (compound_num);
					     objs[compound_num].s = copy_string (objs[compound_num].s ,$3);
					   }
	}
	| NEW COMPOUND nexpr incompound  { int compound_id = $3;
	  		  	  	   int compound_num = obj_tid_next_compound (&compound_id ,objs[jo].father_typ ,objs[jo].father_id ,jo);
	  		  	  	   if (compound_num > 0) obj_make_current (compound_num);
	}
	| NEW COMPOUND nexpr sexpr incompound  { int compound_id = $3;
	  		  	  	   int compound_num = obj_tid_next_compound (&compound_id ,objs[jo].father_typ ,objs[jo].father_id ,jo);
	  		  	  	   if (compound_num > 0) {
					     obj_make_current (compound_num);
					     objs[compound_num].s = copy_string (objs[compound_num].s ,$4);
					   }
	}
	| COMPOUND nexpr  incompound     { obj_glue_add ($2 ,cur_obj_num); }
        | thiscompound HIDDEN  onoff     { objs[cur_obj_num].hidden = $3; }
        | thiscompound LOCTYPE worldview { objs[cur_obj_num].loctyp = $3; }
        | thiscompound selectgraph       { printf("thiscompound selectgraph A FAIRE\n"); }
      	| thiscompound  ':' expr ',' expr ',' expr ',' expr  {
	  			    	   objs[cur_obj_num].x1 = $3;
	  			    	   objs[cur_obj_num].y1 = $5;
	  			    	   objs[cur_obj_num].x2 = $7;
	  			    	   objs[cur_obj_num].y2 = $9;
	}
        | COMPOUND nexpr BREAK             { obj_break_compound (cur_obj_num); }


;  /*----------------  End of objects:  ---------------- */





actions:
       REDRAW { drawgraph (); }
       | UPDATEALL {
#ifdef WITH_GTK
	    if (inwin) gg_update_all();
#endif    /* WITH_GTK */
	}
	| CD sexpr   { set_workingdir ($2); xfree ($2); }
	| ECHO sexpr { echomsg ($2); 	    xfree ($2); }
	| ECHO expr  {
	              char buf[32];
	              set_locale_num(TRUE);
	              sprintf (buf, "%g", $2);
	              set_locale_num (FALSE);
	              echomsg (buf);
	}
	| CLOSE                { close_input = copy_string (close_input, ""); }
	| CLOSE sexpr          { close_input = copy_string (close_input, $2); }
	| CLOSE LOG FILEP      { if (fplog != NULL) fclose (fplog); }
	| EXIT                 { exit(0); }
	| EXIT '(' iexpr ')'   { exit($3); }
	| OPEN LOG FILEP sexpr { if ( (fplog = fopen ($4 ,"w")) != NULL) {
       				    printf ("LOG FILE %s  opened\n" ,$4);
       				  } else {
	 			    printf ("Fails to open LOG FILE  %s\n" ,$4);
       				  }
        }

	| PRINT                {
	                         if (!safe_mode) {
	                           do_hardcopy();
	                         } else {
	                           yyerror("File modifications are disabled in safe mode");
	                         }
	}
	| PRINT TO DEVICE { set_ptofile (FALSE); }
	| PRINT TO sexpr  { set_ptofile (TRUE);
	                   strcpy (print_file, $3);
	                   xfree ($3);
	}
	|  PAGE direction {
	                  switch ($2) {
	                  case UP:    graph_scroll (GSCROLL_UP);    break;
	                  case DOWN:  graph_scroll (GSCROLL_DOWN);  break;
	                  case RIGHT: graph_scroll (GSCROLL_RIGHT); break;
	                  case LEFT:  graph_scroll (GSCROLL_LEFT);  break;
	                  case IN:    graph_zoom   (GZOOM_SHRINK);  break;
	                  case OUT:   graph_zoom   (GZOOM_EXPAND);  break;
	                  }
	}
	| SLEEP expr { if ($2 > 0) msleep_wrap((unsigned int) (1000 * $2)); }

        | HELP {
#ifndef NONE_GUI
            if (inwin) {
#ifndef WITH_GTK
                HelpCB("doc/UsersGuide.html");
#else     /* WITH_GTK */
                gg_HelpCB("doc/UsersGuide.html");
#endif    /* WITH_GTK */
            }
#endif
	}
	| GETP sexpr {	    gotparams = TRUE;	    strcpy(paramfile, $2);	    xfree($2);	}
	| PUTP sexpr {
	    if (!safe_mode) {
                FILE *pp = grace_openw($2);
	        if (pp != NULL) {
	            putparms(gn, pp, 0);
	            grace_close(pp);
	        }
            } else {
                yyerror("File modifications are disabled in safe mode");
            }
            xfree($2);
        }



//13 provisoire pour XYCMAP
        | selectset HIDDEN onoff 	 { set_set_hidden ($1->gno, $1->setno, $3); }
        | selectset LENGTH nexpr 	 { setlength 	  ($1->gno, $1->setno, $3); }
        | VEC_D LENGTH nexpr 	 	 { realloc_vrbl   ($1, $3); }
        | selectset POINT expr ',' expr  { add_point 	  ($1->gno, $1->setno, $3, $5); }
        | selectset DROP nexpr ',' nexpr { int start = $3 - index_shift;
            				   int stop = $5 - index_shift;
            				   droppoints 	  ($1->gno, $1->setno, start, stop); }
        | SORT selectset sorton sortdir  { if (is_set_active ($2->gno, $2->setno)) {
                			   sortset 	  ($2->gno, $2->setno, $3, $4 == ASCENDING ? 0 : 1);
                                           }
        }
        | COPY selectset TO selectset 	 { do_copyset ($2->gno, $2->setno, $4->gno, $4->setno); }
        | APPEND selectset TO selectset {
            if ($2->gno != $4->gno) {
                errmsg ("Can't append sets from different graphs");
            } else {
                int sets[2];
                sets[0] = $4->setno;
                sets[1] = $2->setno;
                join_sets ($2->gno, sets, 2);
            }
        }
        | REVERSE selectset 	       { reverse_set  ($2->gno, $2->setno); }
        | SPLIT selectset nexpr        { do_splitsets ($2->gno, $2->setno, $3); }
        | MOVE selectset TO selectset  { do_moveset   ($2->gno, $2->setno, $4->gno, $4->setno); }
        | SWAP selectset AND selectset { do_swapset   ($2->gno, $2->setno, $4->gno, $4->setno); }
        | KILL selectset 	       { killset      ($2->gno, $2->setno); }
        | KILL selectset SAVEALL       { killsetdata  ($2->gno, $2->setno); }
        | KILL selectgraph 	       { kill_graph   ($2); }
        | KILL REGNUM 		       { kill_region  ($2); }
        | FLUSH 		       { wipeout      (); }
        | ARRANGE '(' nexpr ',' nexpr ',' expr ',' expr ',' expr ')' {
            arrange_graphs_simple($3, $5, 0, FALSE, $7, $9, $11);
        }
        | ARRANGE '(' nexpr ',' nexpr ',' expr ',' expr ',' expr ',' onoff ',' onoff ',' onoff ')' {
            int order = ($13 * GA_ORDER_HV_INV) |
                        ($15 * GA_ORDER_H_INV ) |
                        ($17 * GA_ORDER_V_INV );
            arrange_graphs_simple($3, $5, order, FALSE, $7, $9, $11);
        }
        | ARRANGE '(' nexpr ',' nexpr ',' expr ',' expr ',' expr ',' onoff ',' onoff ',' onoff ',' onoff ')' {
            int order = ($13 * GA_ORDER_HV_INV) |
                        ($15 * GA_ORDER_H_INV ) |
                        ($17 * GA_ORDER_V_INV );
            arrange_graphs_simple($3, $5, order, $19, $7, $9, $11);
        }
        | NONLFIT '(' selectset ',' nexpr ')' {
            gotnlfit  	 = TRUE;
            nlfit_gno 	 = $3->gno;
            nlfit_setno  = $3->setno;
            nlfit_nsteps = $5;
            nlfit_warray = NULL;
	    nlfit_trace  = FALSE;
        }
        | NONLFIT '(' selectset ',' vexpr ',' nexpr  ')' {
            if (getsetlength($3->gno, $3->setno) != $5->length) {
                errmsg("Data and weight arrays are of different lengths");
                return 1;
            } else {
                gotnlfit     = TRUE;
                nlfit_gno    = $3->gno;
                nlfit_setno  = $3->setno;
                nlfit_nsteps = $7;
                nlfit_warray = copy_data_column($5->data, $5->length);
		nlfit_trace  = FALSE;
            }
        }
        | NONLFIT '(' selectset ',' vexpr  ',' nexpr ',' sexpr ',' onoff ')' {
	  int i;
	  gotnlfit     = TRUE;
	  nlfit_gno    = $3->gno;
	  nlfit_setno  = $3->setno;
	  nlfit_nsteps = $7;
	  nlfit_warray = NULL;
	  nonl_opts.parnum  = $5->length;
	  for (i = 0; i < nonl_opts.parnum; i++) nonl_parms[i].value = $5->data[i];
	  nonl_opts.formula = copy_string (nonl_opts.formula ,$9);
	  nlfit_trace  = $11;
#ifdef WITH_GTK
	  gg_nonl_update ();
#endif
        }
        | REGRESS '(' selectset ',' nexpr ')' {
            do_regress($3->gno, $3->setno, $5, 0, -1, 0, -1);
        }
        | runtype '(' selectset ',' nexpr ')' {
            do_runavg($3->gno, $3->setno, $5, $1, -1, 0);
        }
        | runtype '(' selectset ',' nexpr ',' selectset ',' nexpr ',' nexpr')' {
          do_runavg12 ($3->gno ,$3->setno ,$7->gno ,$7->setno ,$5 ,$1 ,$9 ,$11);
        }
        | ffttype '(' selectset ',' nexpr ')' {
          do_fourier_command ($3->gno ,$3->setno ,$1 ,$5);
        }
        | ffttype '(' selectset ',' fourierdata ',' windowtype ','
                      fourierloadx ','  fourierloady ')' {
          do_fourier_command_1 ($3->gno, $3->setno
                                ,$1 ,$9 ,$5
                                ,$7
                                ,$11 ,-1 ,-1);
          }
        | ffttype '(' selectset ',' fourierdata ',' windowtype ','
                      fourierloadx ','  fourierloady ',' fourierinseg ','  fourieroutsg ')' {
          do_fourier_command_1 ($3->gno, $3->setno
                                ,$1 ,$9 ,$5
                                ,$7
                                ,$11 ,$13 ,$15);
        }
        | WAVELET '(' selectset  ',' waveletype  ',' nexpr ',' nexpr ',' nexpr ')' { 
	  do_wavelet_command ($3->gno, $3->setno ,$5 ,$7 ,$9 ,$11);
        }
        | INTERPOLATE '(' selectset ',' vexpr ',' interpmethod ',' onoff ')' {
            	       do_interp($3->gno, $3->setno, get_cg(), SET_SELECT_NEXT,
				 $5->data, $5->length, $7, $9
				 ,  /* Default for INTERP_XSPLINE xspline_s = */ -0.5);
        }
	| XYSPLINE  '(' selectset ',' expr ',' expr ')' {
	               do_xyspline ($3->gno ,$3->setno ,$5 ,$7);
	}
	| HISTOGRAM '(' selectset ',' vexpr ',' onoff ',' onoff ')' {
            		do_histo($3->gno, $3->setno, get_cg(), SET_SELECT_NEXT,
				 $5->data, $5->length - 1, $7, $9);
	}
	| DIFFERENCE '(' selectset ',' nexpr ')' {
	    		do_diff_command ($3->gno, $3->setno, $5);
	}
	| DIFFERENCE '(' selectset ',' nexpr ',' nexpr ',' nexpr ')' {
	    		do_diff_command_1 ($3->gno, $3->setno, $5 ,$7 ,$9);
	}
	| INTEGRATE '(' selectset ')' {
	    		do_int($3->gno, $3->setno, 0);
	}
 	| XCOR '(' selectset ',' selectset ',' nexpr ',' onoff ')' {
	    		do_xcor($3->gno, $3->setno, $5->gno, $5->setno, $7, $9);
	}
 	| LINCONV '(' selectset ',' selectset ')' {
	    		do_linearc($3->gno, $3->setno, $5->gno, $5->setno);
	}
 	| RESTRICT '(' selectset ',' vexpr ')' {
            int len = getsetlength($3->gno, $3->setno);
            if (len != $5->length) {
		errmsg("Filter expression is of a wrong length");
            } else {
                char *rarray;
                rarray = xmalloc(len*SIZEOF_CHAR);
                if (rarray) {
                    int i;
                    for (i = 0; i < len; i++) {
                        rarray[i] = CAST_DBL_TO_BOOL($5->data[i]);
                    }
                    filter_set($3->gno, $3->setno, rarray);
                    xfree(rarray);
                }
            }
	}
 	| RESTRICT '(' selectset ',' REGNUM ',' onoff ')' {
            int rtype;
            char *rarray;

            rtype = RESTRICT_REG0 + $5;

	    if (get_restriction_array($3->gno, $3->setno,
                rtype, $7, &rarray) != RETURN_SUCCESS) {
                errmsg("Error in region evaluation");
                return 1;
	    } else {
                filter_set($3->gno, $3->setno, rarray);
                xfree(rarray);
            }
	}
	| AUTOSCALE {
	    if (autoscale_graph (gn, AUTOSCALE_XY) != RETURN_SUCCESS) {
		errmsg ("Can't autoscale (no active sets?)");
	    }
	}
	| AUTOSCALE XAXES {
	    if (autoscale_graph (gn, AUTOSCALE_X) != RETURN_SUCCESS) {
		errmsg ("Can't autoscale (no active sets?)");
	    }
	}
	| AUTOSCALE YAXES {
	    if (autoscale_graph (gn, AUTOSCALE_Y) != RETURN_SUCCESS) {
		errmsg ("Can't autoscale (no active sets?)");
	    }
	}
	| AUTOSCALE selectset { autoscale_byset ($2->gno, $2->setno, AUTOSCALE_XY); }
        | AUTOTICKS 	      { autotick_axis (gn, ALL_AXES); }
	| FOCUS selectgraph {
	    int gno = $2;
            if (is_graph_hidden (gno) == FALSE) {
                select_graph (gno);
            } else {
		errmsg ("Graph is not active");
            }
	}
	| READ sexpr 	   	      { gotread = TRUE; strcpy (readfile, $2); xfree ($2); }
	| READ BATCH sexpr 	      { strcpy  (batchfile, $3); xfree ($3); }
	| READ BLOCK sexpr 	      { getdata (gn, $3, SOURCE_DISK, LOAD_BLOCK); xfree ($3); }
	| READ BLOCK sourcetype sexpr { getdata (gn, $4, $3, LOAD_BLOCK); xfree ($4); 	}
	| BLOCK xytype sexpr {
            int nc, *cols, scol;
            if (field_string_to_cols($3, &nc, &cols, &scol) != RETURN_SUCCESS) {
                errmsg("Erroneous field specifications");
	        xfree($3);
                return 1;
            } else {
	        xfree($3);
	        create_set_fromblock(gn, NEW_SET,
                    $2, nc, cols, scol, autoscale_onread);
                xfree(cols);
            }
	}
	| KILL BLOCK {
	    set_blockdata(NULL);
	}
        | SELXY sexpr { selxy_set ($2);	}
	| READ xytype sexpr {
	  if ($2 == SELXY) {
	    getdata (gn, $3, SOURCE_DISK, LOAD_SELXY);
	  } else {
	    gotread = TRUE;
	    curtype = $2;
	    strcpy(readfile, $3);
	  }
	    xfree($3);
	}
	| READ xytype sourcetype sexpr {
	    gotread = TRUE;
	    strcpy(readfile, $4);
	    curtype = $2;
	    cursource = $3;
	    xfree($4);
	}
	| READ NXY sexpr {
	    getdata(gn, $3, SOURCE_DISK, LOAD_NXY);
	    xfree($3);
	}
	| READ NXY sourcetype sexpr {
	    getdata(gn, $4, $3, LOAD_NXY);
	    xfree($4);
	}
	| WRITE selectset {
	    if (!safe_mode) {
                outputset($2->gno, $2->setno, "stdout", NULL);
            } else {
                yyerror("File modifications are disabled in safe mode");
            }
	}
	| WRITE selectset FORMAT sexpr {
	    if (!safe_mode) {
	        outputset($2->gno, $2->setno, "stdout", $4);
            } else {
                yyerror("File modifications are disabled in safe mode");
            }
	    xfree($4);
	}
	| WRITE selectset FILEP sexpr {
	    if (!safe_mode) {
	        outputset($2->gno, $2->setno, $4, NULL);
            } else {
                yyerror("File modifications are disabled in safe mode");
            }
	    xfree($4);
	}
	| WRITE selectset FILEP sexpr FORMAT sexpr {
	    if (!safe_mode) {
	        outputset($2->gno, $2->setno, $4, $6);
            } else {
                yyerror("File modifications are disabled in safe mode");
            }
	    xfree($4);
	    xfree($6);
	}
        | SAVEALL sexpr {
            if (!safe_mode) {
                save_project($2);
            } else {
                yyerror("File modifications are disabled in safe mode");
            }
            xfree($2);
        }
        | LOAD sexpr 	 { load_project      ($2); xfree ($2); }
        | NEW 		 { new_project 	     (NULL); }
        | NEW FROM sexpr { new_project 	     ($3); xfree ($3); }
	| PUSH 	       	 { push_world 	     (); }
	| POP 	       	 { pop_world 	     (); }
	| CYCLE        	 { cycle_world_stack (); }
	| STACK nexpr  	 { if ($2 > 0) show_world_stack ($2 - 1); }
	| CLEAR STACK  	 { clear_world_stack (); }
        | CLEAR objtyp 	 { objs_t_clear      (curobjtyp); }

        ;   /*  ------------ End of actions: ------------  */


options:
        PAGE LAYOUT pagelayout {
#ifndef NONE_GUI
            set_pagelayout($3);
#endif
        }
        | AUTO REDRAW onoff { auto_redraw = $3; }
        | FOCUS onoff 	    { draw_focus_flag = $2; }
        | FOCUS SET 	    { focus_policy = FOCUS_SET; }
        | FOCUS FOLLOWS     { focus_policy = FOCUS_FOLLOWS; }
        | FOCUS CLICK 	    { focus_policy = FOCUS_CLICK; }
        ;


set_setprop:
	setprop {}
	| setprop_obs {}
	;

setprop:
        selectset onoff 		       { set_set_hidden ($1->gno, $1->setno, !$2); }
        | selectset TYPE xytype 	       { curtype = $3;
					   	 set_dataset_type ($1->gno, $1->setno, $3); }
        | selectset TYPE XYCMAP nexpr ',' nexpr { curtype = SET_XYCMAP;
						  set_dataset_type ($1->gno, $1->setno, SET_XYCMAP);
	                                          set_xymap_nxy    ($1->gno, $1->setno, $4 ,$6);
	}
        | selectset LAYER nexpr 	       { int num = obj_tid_get_set_num ($1->gno, $1->setno);
	                                                       objs[num].layer              = $3; }
        | selectset SYMBOL nexpr	       { g[$1->gno].p[$1->setno].sym 	      	    = $3; }
        | selectset SYMBOL color_select        { g[$1->gno].p[$1->setno].sympen.color 	    = $3; }
        | selectset SYMBOL pattern_select      { g[$1->gno].p[$1->setno].sympen.pattern     = $3; }
        | selectset SYMBOL linew_select        { g[$1->gno].p[$1->setno].symlinew 	    = $3; }
        | selectset SYMBOL lines_select        { g[$1->gno].p[$1->setno].symlines 	    = $3; }
        | selectset SYMBOL FILL color_select   { g[$1->gno].p[$1->setno].symfillpen.color   = $4; }
        | selectset SYMBOL FILL pattern_select { g[$1->gno].p[$1->setno].symfillpen.pattern = $4; }
        | selectset SYMBOL SIZE expr 	       { g[$1->gno].p[$1->setno].symsize  	    = $4; }
        | selectset SYMBOL CHAR nexpr 	       { g[$1->gno].p[$1->setno].symchar  	    = $4; }
        | selectset SYMBOL CHAR font_select    { g[$1->gno].p[$1->setno].charfont 	    = $4; }
        | selectset SYMBOL SKIP nexpr 	       { g[$1->gno].p[$1->setno].symskip  	    = $4;
					     	 if (get_project_version () < 70000) {
					     	   g[$1->gno].p[$1->setno].avalue.skip = $4;
					     	   g[$1->gno].p[$1->setno].avalue.start= 0;
					     	   }
	}
        | selectset LINE TYPE nexpr            { g[$1->gno].p[$1->setno].linet    	    = $4; }
        | selectset LINE TYPE nexpr ',' nexpr ',' nexpr {
	    				       	 g[$1->gno].p[$1->setno].linet   = $4;
            				       	 g[$1->gno].p[$1->setno].method  = $6;
            				       	 g[$1->gno].p[$1->setno].meshlen = $8;
        }
        | selectset LINE TYPE nexpr ',' nexpr ',' nexpr ',' expr {
	    				       	 g[$1->gno].p[$1->setno].linet   = $4;
            				       	 g[$1->gno].p[$1->setno].method  = $6;
            				       	 g[$1->gno].p[$1->setno].meshlen = $8;
						 if ($6 == INTERP_XYSPLINE || $6 == INTERP_XSPLINE) {
						   QDobject *po = obj_tid_get_pointer (Q_Set ,$1->setno ,$1->gno);
						   if (po != NULL) po->xspline_s = $10;
                                                 }
        }
        | selectset LINE TYPE nexpr ',' nexpr ',' nexpr ',' expr  ',' expr {
	    				       	 g[$1->gno].p[$1->setno].linet   = $4;
            				       	 g[$1->gno].p[$1->setno].method  = $6;
            				       	 g[$1->gno].p[$1->setno].meshlen = $8;
						 if ($6 == INTERP_XYSPLINE || $6 == INTERP_XSPLINE) {
						   QDobject *po = obj_tid_get_pointer (Q_Set ,$1->setno ,$1->gno);
						   if (po != NULL) {
						     po->xspline_s = $10;
						     if ($6 == INTERP_XYSPLINE) po->xspline_p = $12;
                                                   }
                                                 }
        }
        | selectset LINE lines_select        { g[$1->gno].p[$1->setno].lines 	       = $3; }
        | selectset LINE linew_select        { g[$1->gno].p[$1->setno].linew 	       = $3; }
        | selectset LINE color_select        { g[$1->gno].p[$1->setno].linepen.color   = $3; }
        | selectset LINE pattern_select      { g[$1->gno].p[$1->setno].linepen.pattern = $3; }

        | selectset FILL TYPE nexpr          { g[$1->gno].p[$1->setno].filltype = $4; }
        | selectset FILL RULE nexpr          { g[$1->gno].p[$1->setno].fillrule = $4; }
        | selectset FILL color_select        {
            int prop = $3;

	    if (get_project_version() <= 40102 && get_project_version() >= 30000) {
                switch (filltype_obs) {
                case COLOR:
                    break;
                case PATTERN:
                    prop = 1;
                    break;
                default: /* NONE */
	            prop = 0;
                    break;
                }
	    }
	    g[$1->gno].p[$1->setno].setfillpen.color = prop;
	}
	| selectset FILL pattern_select
        {
	    int prop = $3;

	    if (get_project_version() <= 40102) {
                switch (filltype_obs) {
                case COLOR:
                    prop = 1;
                    break;
                case PATTERN:
                    break;
                default: /* NONE */
	            prop = 0;
                    break;
                }
	    }
	    g[$1->gno].p[$1->setno].setfillpen.pattern = prop;
	}


        | selectset BASELINE onoff { g[$1->gno].p[$1->setno].baseline = $3;     }
        | selectset BASELINE TYPE nexpr { g[$1->gno].p[$1->setno].baseline_type = $4;   }

        | selectset DROPLINE onoff 		    { g[$1->gno].p[$1->setno].dropline 	    	 = $3; }
        | selectset AVALUE onoff 		    { g[$1->gno].p[$1->setno].avalue.active 	 = $3; }
        | selectset AVALUE TYPE nexpr 		    { g[$1->gno].p[$1->setno].avalue.type 	 = $4; }
        | selectset AVALUE CHAR SIZE expr 	    { g[$1->gno].p[$1->setno].avalue.size 	 = $5; }
        | selectset AVALUE font_select 		    { g[$1->gno].p[$1->setno].avalue.font 	 = $3; }
        | selectset AVALUE color_select 	    { g[$1->gno].p[$1->setno].avalue.color 	 = $3; }
        | selectset AVALUE ROT nexpr 		    { g[$1->gno].p[$1->setno].avalue.angle 	 = $4; }
        | selectset AVALUE FORMAT formatchoice 	    { g[$1->gno].p[$1->setno].avalue.format 	 = $4; }
        | selectset AVALUE PREC nexpr 		    { g[$1->gno].p[$1->setno].avalue.prec 	 = $4; }
        | selectset AVALUE OFFSET expr ',' expr     { g[$1->gno].p[$1->setno].avalue.offset.x 	 = $4;
           				       	      g[$1->gno].p[$1->setno].avalue.offset.y 	 = $6; }
        | selectset AVALUE PREPEND sexpr 	    { strcpy(g[$1->gno].p[$1->setno].avalue.prestr, $4); xfree($4); }
        | selectset AVALUE APPEND sexpr 	    { strcpy(g[$1->gno].p[$1->setno].avalue.appstr, $4); xfree($4); }
        | selectset AVALUE SKIP nexpr  	    	    { g[$1->gno].p[$1->setno].avalue.skip      	 = $4;
	 				       	      g[$1->gno].p[$1->setno].avalue.start       = 0; }
        | selectset AVALUE SKIP nexpr ',' nexpr     { g[$1->gno].p[$1->setno].avalue.skip      	 = $4;
	 				       	      g[$1->gno].p[$1->setno].avalue.start       = $6; }
        | selectset ERRORBAR onoff 		    { g[$1->gno].p[$1->setno].errbar.active 	 = $3; }
        | selectset ERRORBAR opchoice_sel 	    { g[$1->gno].p[$1->setno].errbar.ptype 	 = $3; }
        | selectset ERRORBAR color_select 	    { g[$1->gno].p[$1->setno].errbar.pen.color	 = $3;
	                                             errbar_to_obj ($1->gno ,$1->setno ,&(g[$1->gno].p[$1->setno].errbar) ,file_project_version); }
        | selectset ERRORBAR pattern_select 	    { g[$1->gno].p[$1->setno].errbar.pen.pattern = $3; }
        | selectset ERRORBAR SIZE expr 		    { g[$1->gno].p[$1->setno].errbar.barsize 	 = $4; }
        | selectset ERRORBAR linew_select 	    { g[$1->gno].p[$1->setno].errbar.linew 	 = $3;
	                                             errbar_to_obj ($1->gno ,$1->setno ,&(g[$1->gno].p[$1->setno].errbar) ,file_project_version); }
        | selectset ERRORBAR lines_select 	    { g[$1->gno].p[$1->setno].errbar.lines 	 = $3;
	                                             errbar_to_obj ($1->gno ,$1->setno ,&(g[$1->gno].p[$1->setno].errbar) ,file_project_version); }
        | selectset ERRORBAR RISER linew_select     { g[$1->gno].p[$1->setno].errbar.riser_linew = $4; }
        | selectset ERRORBAR RISER lines_select     { g[$1->gno].p[$1->setno].errbar.riser_lines = $4; }
        | selectset ERRORBAR RISER CLIP onoff	    { g[$1->gno].p[$1->setno].errbar.arrow_clip  = $5; }
        | selectset ERRORBAR RISER CLIP LENGTH expr { g[$1->gno].p[$1->setno].errbar.cliplen 	 = $6; }

        | vectormap color_select                    { objs[$1].color = $2; }
        | vectormap linew_select                    { objs[$1].linew = $2; }
        | vectormap lines_select                    { objs[$1].lines = $2; }
        | vectormap ARROW TYPE nexpr                { objs[$1].arrow.type = $4; }
        | vectormap ARROW LENGTH expr               { objs[$1].arrow.length = $4; }
        | vectormap ARROW LAYOUT expr ',' expr      { objs[$1].arrow.dL_ff  = $4;
         			     		      objs[$1].arrow.lL_ff  = $6; }

        | selectset COLORBAR  ZSCALE expr ',' expr  {
	 				  	     g[$1->gno].p[$1->setno].contour.z_scaled = TRUE;
	 				  	     g[$1->gno].p[$1->setno].contour.z1 = $4;
	 				  	     g[$1->gno].p[$1->setno].contour.z2 = $6;
         				  	    }
        | selectset COLORBAR NTICKS nexpr           { g[$1->gno].p[$1->setno].contour.nticks = $4; }
        | colorbar  ':' expr ',' expr ',' expr ',' expr {
 	     		    	        	      objs[jo].x1 = $3;
 	     		    	        	      objs[jo].y1 = $5;
 	     		    	        	      objs[jo].x2 = $7;
 	     		    	        	      objs[jo].y2 = $9;
                                                    }
        | COLORBAR nexpr ':' expr ',' expr ',' expr ',' expr {
						    if (obj_tid_is_valid (Q_Colorbar ,$2))  {
	    					       jo = cur_obj_num = obj_geom_get_num (Q_Colorbar ,$2);
						       objs[jo].x1 = $4;
						       objs[jo].y1 = $6;
						       objs[jo].x2 = $8;
						       objs[jo].y2 = $10;
                                                       }
                                                    }
        | colorbar AUTOSCALE             	    { zc_autoscale (jo); }
        | colorbar LAYER nexpr                	    { objs[jo].layer        = $3; }
        | colorbar HIDDEN  onoff       	   	    { objs[jo].hidden       = $3; }
        | colorbar lines_select        	  	    { objs[jo].lines        = $2; }
        | colorbar linew_select        	  	    { objs[jo].linew        = $2; }
        | colorbar color_select        	            { objs[jo].color        = $2; }
        | colorbar JUST nexpr     	            { objs[jo].just         = $3; }
        | colorbar font_select       	   	    { objs[jo].font         = $2; }
        | colorbar CHAR SIZE  expr   	   	    { objs[jo].charsize     = $4; }

	| selectset CMAP nexpr                      { g[$1->gno].p[$1->setno].contour.nmap = $3; }
	| selectset CMAP sexpr                      { int nmap = zc_cmap_get_by_name ($3);
                                                      if (nmap >= 0) {
                                                        g[$1->gno].p[$1->setno].contour.nmap = nmap;
						      }
						      xfree ($3);
	                                            }

	| selectset CONTOUR HIDDEN onoff            { g[$1->gno].p[$1->setno].contour.hidden 	 = $4; }
	| selectset CONTOUR color_select            { g[$1->gno].p[$1->setno].contour.line_color = $3; }
	| selectset CONTOUR linew_select            { g[$1->gno].p[$1->setno].linew              = $3; }
	| selectset CONTOUR LABEL expr ',' expr ',' expr { zc_label_new ($1->gno ,$1->setno ,$4 ,$6 ,$8 ,0.0); }


        | selectset COMMENT sexpr 		    { strncpy(g[$1->gno].p[$1->setno].comments ,$3, MAX_STRING_LENGTH - 1); xfree($3); }
        | selectset LEGEND  sexpr 		    { strncpy(g[$1->gno].p[$1->setno].lstr     ,$3, MAX_STRING_LENGTH - 1); xfree($3); }
        | selectset LEGEND  onoff 		    { g[$1->gno].p[$1->setno].legend_on = $3; }
        ;  /* End of setprop: */

axisfeature:
	onoff 			      { if (bad_axis ()) return 1; g[gn].t[naxis]->active = $1;	}
        | LAYER nexpr                 { if (bad_axis ()) return 1; int num = obj_tid_get_num (Q_Axis ,naxis ,Q_Graph ,gn);
                                                                   objs[num].layer        = $2; }
	| TYPE ZERO onoff 	      { if (bad_axis ()) return 1; g[gn].t[naxis]->zero   = $3; }
	| TICKP tickattr              {}
	| TICKP tickattr_obs          {}
	| TICKLABEL ticklabelattr     {}
	| TICKLABEL ticklabelattr_obs {}
	| LABEL axislabeldesc         {}
	| LABEL axislabeldesc_obs     {}
	| BAR axisbardesc             {}
	| OFFSET expr ',' expr { if (bad_axis ()) return 1;
            			 g[gn].t[naxis]->offsx = $2;
	    			 g[gn].t[naxis]->offsy = $4;
        }
        ;

tickattr:
        onoff 			 { if (bad_axis ()) return 1; g[gn].t[naxis]->t_flag 	      = $1; }
        | MAJOR expr 		 { if (bad_axis ()) return 1; g[gn].t[naxis]->tmajor 	      = $2; }
        | MINOR TICKSP nexpr 	 { if (bad_axis ()) return 1; g[gn].t[naxis]->nminor 	      = $3; }
        | PLACE ROUNDED onoff 	 { if (bad_axis ()) return 1; g[gn].t[naxis]->t_round 	      = $3; }
        | OFFSETX expr 		 { if (bad_axis ()) return 1; g[gn].t[naxis]->offsx 	      = $2; }
        | OFFSETY expr 		 { if (bad_axis ()) return 1; g[gn].t[naxis]->offsy 	      = $2; }
        | DEFAULT nexpr 	 { if (bad_axis ()) return 1; g[gn].t[naxis]->t_autonum       = $2; }
        | inoutchoice 		 { if (bad_axis ()) return 1; g[gn].t[naxis]->t_inout 	      = $1; }
        | MAJOR SIZE expr 	 { if (bad_axis ()) return 1; g[gn].t[naxis]->props.size      = $3; }
        | MINOR SIZE expr 	 { if (bad_axis ()) return 1; g[gn].t[naxis]->mprops.size     = $3; }
        | color_select 		 { if (bad_axis ()) return 1; g[gn].t[naxis]->props.color     = g[gn].t[naxis]->mprops.color = $1; }
        | MAJOR color_select 	 { if (bad_axis ()) return 1; g[gn].t[naxis]->props.color     = $2; }
        | MINOR color_select 	 { if (bad_axis ()) return 1; g[gn].t[naxis]->mprops.color    = $2; }
        | linew_select 		 { if (bad_axis ()) return 1; g[gn].t[naxis]->props.linew     = g[gn].t[naxis]->mprops.linew = $1; }
        | MAJOR linew_select 	 { if (bad_axis ()) return 1; g[gn].t[naxis]->props.linew     = $2;        }
        | MINOR linew_select     { if (bad_axis ()) return 1; g[gn].t[naxis]->mprops.linew    = $2; }
        | MAJOR lines_select     { if (bad_axis ()) return 1; g[gn].t[naxis]->props.lines     = $2; }
        | MINOR lines_select     { if (bad_axis ()) return 1; g[gn].t[naxis]->mprops.lines    = $2; }
        | MAJOR GRID onoff   	 { if (bad_axis ()) return 1; g[gn].t[naxis]->props.gridflag  = $3; }
        | MINOR GRID onoff   	 { if (bad_axis ()) return 1; g[gn].t[naxis]->mprops.gridflag = $3; }
        | opchoice_sel	     	 { if (bad_axis ()) return 1; g[gn].t[naxis]->t_op   	      = $1; }
        | SPEC TYPE tickspectype { if (bad_axis ()) return 1; g[gn].t[naxis]->t_spec 	      = $3; }
        | SPEC nexpr 		 { if (bad_axis ()) return 1; g[gn].t[naxis]->nticks 	      = $2; }
        | MAJOR nexpr ',' expr 	 { if (bad_axis ()) return 1;
            						      g[gn].t[naxis]->tloc[$2].wtpos = $4;
            						      g[gn].t[naxis]->tloc[$2].type = TICK_TYPE_MAJOR;
        }
        | MINOR nexpr ',' expr 	 { if (bad_axis ()) return 1;
            						      g[gn].t[naxis]->tloc[$2].wtpos = $4;
            						      g[gn].t[naxis]->tloc[$2].type = TICK_TYPE_MINOR;
        }
	;

ticklabelattr:
	onoff 		      { if (bad_axis ()) return 1; g[gn].t[naxis]->tl_flag   	= $1; }
	| PREC nexpr	      { if (bad_axis ()) return 1; g[gn].t[naxis]->tl_prec   	= $2; }
	| FORMAT formatchoice { if (bad_axis ()) return 1; g[gn].t[naxis]->tl_format 	= $2; }
	| FORMAT expr 	      { if (bad_axis ()) return 1; g[gn].t[naxis]->tl_format 	= $2; }
	| APPEND sexpr 	      { if (bad_axis ()) return 1; strncpy (g[gn].t[naxis]->tl_appstr, $2 ,63); xfree($2); }
	| PREPEND sexpr       { if (bad_axis ()) return 1; strncpy (g[gn].t[naxis]->tl_prestr, $2 ,63); xfree($2); }
	| ANGLE nexpr 	      { if (bad_axis ()) return 1; g[gn].t[naxis]->tl_angle  	= $2; }
	| SKIP nexpr 	      { if (bad_axis ()) return 1; g[gn].t[naxis]->tl_skip   	= $2; }
	| STAGGER nexpr       { if (bad_axis ()) return 1; g[gn].t[naxis]->tl_staggered = $2; }
	| opchoice_sel        { if (bad_axis ()) return 1; g[gn].t[naxis]->tl_op     	= $1; }
	| FORMULA sexpr       { if (bad_axis ()) return 1; g[gn].t[naxis]->tl_formula =
            		        			     copy_string(g[gn].t[naxis]->tl_formula, $2);
            		        			   xfree($2);
	}
	| START expr 	       { if (bad_axis ()) return 1; g[gn].t[naxis]->tl_start     = $2; }
	| STOP expr 	       { if (bad_axis ()) return 1; g[gn].t[naxis]->tl_stop 	    = $2; }
	| START TYPE SPEC      { if (bad_axis ()) return 1; g[gn].t[naxis]->tl_starttype = TYPE_SPEC; }
	| START TYPE AUTO      { if (bad_axis ()) return 1; g[gn].t[naxis]->tl_starttype = TYPE_AUTO; }
	| STOP TYPE SPEC       { if (bad_axis ()) return 1; g[gn].t[naxis]->tl_stoptype  = TYPE_SPEC; }
	| STOP TYPE AUTO       { if (bad_axis ()) return 1; g[gn].t[naxis]->tl_stoptype  = TYPE_AUTO; }
	| CHAR SIZE expr       { if (bad_axis ()) return 1; g[gn].t[naxis]->tl_charsize  = $3; }
	| font_select 	       { if (bad_axis ()) return 1; g[gn].t[naxis]->tl_font 	    = $1; }
	| color_select 	       { if (bad_axis ()) return 1; g[gn].t[naxis]->tl_color     = $1; }
	| nexpr ',' sexpr      { if (bad_axis ()) { xfree($3); return 1; }
	    		         if ($1 >= MAX_TICKS) {
	    		              yyerror("Number of ticks exceeds maximum");
	    		              xfree($3);
	    		              return 1;
	    		         }
	    		         g[gn].t[naxis]->tloc[$1].label =
            		             copy_string(g[gn].t[naxis]->tloc[$1].label, $3);
	    		         xfree($3);
	}
	| OFFSET AUTO 	       { if (bad_axis ()) return 1; g[gn].t[naxis]->tl_gaptype = TYPE_AUTO; }
	| OFFSET SPEC 	       { if (bad_axis ()) return 1; g[gn].t[naxis]->tl_gaptype = TYPE_SPEC; }
	| OFFSET expr ',' expr { if (bad_axis ()) return 1; g[gn].t[naxis]->tl_gap.x = $2;
	    					 	    g[gn].t[naxis]->tl_gap.y = $4; }
	;

axislabeldesc:
	sexpr                 { if (bad_axis ()) return 1; set_plotstr_string(&g[gn].t[naxis]->label, $1); xfree($1); }
	| LAYOUT PERP         { if (bad_axis ()) return 1; g[gn].t[naxis]->label_layout   = LAYOUT_PERPENDICULAR; }
	| LAYOUT PARA         { if (bad_axis ()) return 1; g[gn].t[naxis]->label_layout   = LAYOUT_PARALLEL; }
	| PLACE AUTO          { if (bad_axis ()) return 1; g[gn].t[naxis]->label_place    = TYPE_AUTO; }
	| PLACE SPEC          { if (bad_axis ()) return 1; g[gn].t[naxis]->label_place    = TYPE_SPEC; }
	| PLACE expr ',' expr { if (bad_axis ()) return 1; g[gn].t[naxis]->label.x1       = $2;
	                                                   g[gn].t[naxis]->label.y1       = $4; }
	| JUST justchoice     { if (bad_axis ()) return 1; g[gn].t[naxis]->label.just     = $2; }
	| CHAR SIZE expr      { if (bad_axis ()) return 1; g[gn].t[naxis]->label.charsize = $3; }
	| font_select         { if (bad_axis ()) return 1; g[gn].t[naxis]->label.font     = $1; }
	| color_select        { if (bad_axis ()) return 1; g[gn].t[naxis]->label.color    = $1; }
	| opchoice_sel        { if (bad_axis ()) return 1; g[gn].t[naxis]->label_op       = $1; }
	;
axisbardesc: onoff        { if (bad_axis ()) return 1; g[gn].t[naxis]->t_drawbar      = $1; }
	   | color_select { if (bad_axis ()) return 1; g[gn].t[naxis]->t_drawbarcolor = $1; }
	   | lines_select { if (bad_axis ()) return 1; g[gn].t[naxis]->t_drawbarlines = $1; }
	   | linew_select { if (bad_axis ()) return 1; g[gn].t[naxis]->t_drawbarlinew = $1; }
           ;
nonlfitopts:
	   TITLE sexpr             { nonl_opts.title     = copy_string (nonl_opts.title   ,$2);  xfree($2);}
	   | FORMULA sexpr         { nonl_opts.formula   = copy_string (nonl_opts.formula ,$2);  xfree($2);}
	   | WITH nexpr PARAMETERS { nonl_opts.parnum    = $2; }
	   | PREC expr             { nonl_opts.tolerance = $2; }
	   ;
selectgraph: GRAPHNO    { $$ = $1; }
           | GRAPH indx { $$ = $2; }
           ;

selectset:
	selectgraph '.' SETNUM
	{
	    int gno = $1, setno = $3;
            if (allocate_set(gno, setno) == RETURN_SUCCESS) {
                $$ = &trgt_pool[tgtn];
                $$->gno   = gno;
                $$->setno = setno;
                tgtn++;
            } else {
                errmsg("Can't allocate referred set");
                return 1;
            }
	}
	| selectgraph '.' SET indx
	{
	    int gno = $1, setno = $4;
            if (allocate_set(gno, setno) == RETURN_SUCCESS) {
                $$ = &trgt_pool[tgtn];
                $$->gno   = gno;
                $$->setno = setno;
                tgtn++;
            } else {
                errmsg("Can't allocate referred set");
                return 1;
            }
	}
	| SETNUM
	{
	    int gno = gn, setno = $1;
            if (allocate_set(gno, setno) == RETURN_SUCCESS) {
                $$ = &trgt_pool[tgtn];
                $$->gno   = gno;
                $$->setno = setno;
                tgtn++;
            } else {
                errmsg("Can't allocate referred set");
                return 1;
            }
	}
	| SET indx
	{
	    int gno = gn, setno = $2;
            if (allocate_set(gno, setno) == RETURN_SUCCESS) {
                $$ = &trgt_pool[tgtn];
                $$->gno   = gno;
                $$->setno = setno;
                tgtn++;
            } else {
                errmsg("Can't allocate referred set");
                return 1;
            }
	}
	;

setaxis:
	axis axisfeature {}
	| selectgraph axis axisfeature {}
	;

axis:
	XAXIS 	   { naxis = X_AXIS; }
	| YAXIS    { naxis = Y_AXIS; }
	| ALTXAXIS { naxis = ZX_AXIS; }
	| ALTYAXIS { naxis = ZY_AXIS; }
	;

proctype:
        KEY_CONST         { $$ = CONSTANT;  }
        | KEY_UNIT        { $$ = UCONSTANT; }
        | KEY_FUNC_I      { $$ = FUNC_I;    }
	| KEY_FUNC_D      { $$ = FUNC_D;    }
	| KEY_FUNC_ND     { $$ = FUNC_ND;   }
	| KEY_FUNC_NN     { $$ = FUNC_NN;   }
	| KEY_FUNC_DD     { $$ = FUNC_DD;   }
	| KEY_FUNC_NND    { $$ = FUNC_NND;  }
	| KEY_FUNC_PPD    { $$ = FUNC_PPD;  }
	| KEY_FUNC_PPPD   { $$ = FUNC_PPPD; }
	| KEY_FUNC_PPPPD  { $$ = FUNC_PPPPD; }
	| KEY_FUNC_PPPPPD { $$ = FUNC_PPPPPD; }
	;

tickspectype:
	NONE 	 { $$ = TICKS_SPEC_NONE; }
	| TICKSP { $$ = TICKS_SPEC_MARKS; }
	| BOTH 	 { $$ = TICKS_SPEC_BOTH; }
	;

filtertype:
        IFILTER	  { $$ = FILTER_INPUT; }
	| OFILTER { $$ = FILTER_OUTPUT; }
	;

filtermethod:
        MAGIC 	  { $$ = FILTER_MAGIC; }
	| PATTERN { $$ = FILTER_PATTERN; }
	;

xytype:	XY 	     { $$ = SET_XY; }
	| SELXY	     { $$ = SELXY; }
	| BAR 	     { $$ = SET_BAR; }
	| BARDY      { $$ = SET_BARDY; }
	| BARDYDY    { $$ = SET_BARDYDY; }
	| XYZ 	     { $$ = SET_XYZ; }
	| XYDX 	     { $$ = SET_XYDX; }
	| XYDY 	     { $$ = SET_XYDY; }
	| XYDXDX     { $$ = SET_XYDXDX; }
	| XYDYDY     { $$ = SET_XYDYDY; }
	| XYDXDY     { $$ = SET_XYDXDY; }
	| XYDXDXDYDY { $$ = SET_XYDXDXDYDY; }
	| XYHILO     { $$ = SET_XYHILO; }
	| XYR 	     { $$ = SET_XYR; }
	| XYSIZE     { $$ = SET_XYSIZE; }
	| XYCOLOR    { $$ = SET_XYCOLOR; }
	| XYCOLPAT   { $$ = SET_XYCOLPAT; }
	| XYCMAP     { $$ = SET_XYCMAP; }
	| XYVMAP     { $$ = SET_XYVMAP; }
	| XYBOXPLOT  { $$ = SET_BOXPLOT; }
	| XYSTRING   { $$ = SET_XY; }
	;

graphtype:
	XY 	{ $$ = GRAPH_XY; }
	| CHART { $$ = GRAPH_CHART; }
	| POLAR { $$ = GRAPH_POLAR; }
	| SMITH { $$ = GRAPH_SMITH; }
	| FIXED { $$ = GRAPH_FIXED; }
	| PIE   { $$ = GRAPH_PIE;   }
	;

pagelayout:
        FREE 	{ $$ = PAGE_FREE; }
        | FIXED { $$ = PAGE_FIXED; }
        ;

pageorient:
        LANDSCAPE  { $$ = PAGE_ORIENT_LANDSCAPE; }
        | PORTRAIT { $$ = PAGE_ORIENT_PORTRAIT;  }
        ;

regiontype:
	ABOVE 	  { $$ = REGION_ABOVE; }
	|  BELOW  { $$ = REGION_BELOW; }
	|  LEFT   { $$ = REGION_TOLEFT; }
	|  RIGHT  { $$ = REGION_TORIGHT; }
	|  POLYI  { $$ = REGION_POLYI; }
	|  POLYO  { $$ = REGION_POLYO; }
	|  HORIZI { $$ = REGION_HORIZI; }
	|  VERTI  { $$ = REGION_VERTI; }
	|  HORIZO { $$ = REGION_HORIZO; }
	|  VERTO  { $$ = REGION_VERTO; }
	;

scaletype: NORMAL     { $$ = SCALE_NORMAL; }
	| LOGARITHMIC { $$ = SCALE_LOG; }
	| RECIPROCAL  { $$ = SCALE_REC; }
	| LOGIT       { $$ = SCALE_LOGIT; }
	| DEGREES     { $$ = SCALE_DEGREES; }
	;

onoff: ON     { $$ = TRUE; }
	| OFF { $$ = FALSE; }
	;

runtype: RUNAVG  { $$ = RUN_AVG; }
	| RUNSTD { $$ = RUN_STD; }
	| RUNMED { $$ = RUN_MED; }
	| RUNMAX { $$ = RUN_MAX; }
	| RUNMIN { $$ = RUN_MIN; }
	;

sourcetype:
        DISK { $$ = SOURCE_DISK; }
	| PIPE {
            if (!safe_mode) {
                $$ = SOURCE_PIPE;
            } else {
                yyerror("Pipe inputs are disabled in safe mode");
                $$ = SOURCE_DISK;
            }
        }
	;

justchoice: RIGHT { $$ = JUST_RIGHT; }
	| LEFT 	  { $$ = JUST_LEFT; }
	| CENTER  { $$ = JUST_CENTER; }
	;

inoutchoice: IN { $$ = TICKS_IN; }
	| OUT 	{ $$ = TICKS_OUT; }
	| BOTH  { $$ = TICKS_BOTH; }
	;

formatchoice: DECIMAL 	 { $$ = FORMAT_DECIMAL; }
	| EXPONENTIAL 	 { $$ = FORMAT_EXPONENTIAL; }
	| GENERAL 	 { $$ = FORMAT_GENERAL; }
	| SCIENTIFIC   	 { $$ = FORMAT_SCIENTIFIC; }
	| ENGINEERING 	 { $$ = FORMAT_ENGINEERING; }
	| COMPUTING 	 { $$ = FORMAT_COMPUTING; }
	| POWER 	 { $$ = FORMAT_POWER; }
	| DDMMYY 	 { $$ = FORMAT_DDMMYY; }
	| MMDDYY 	 { $$ = FORMAT_MMDDYY; }
	| YYMMDD 	 { $$ = FORMAT_YYMMDD; }
	| MMYY 		 { $$ = FORMAT_MMYY; }
	| MMDD 		 { $$ = FORMAT_MMDD; }
	| MONTHDAY 	 { $$ = FORMAT_MONTHDAY; }
	| DAYMONTH 	 { $$ = FORMAT_DAYMONTH; }
	| MONTHS 	 { $$ = FORMAT_MONTHS; }
	| MONTHSY 	 { $$ = FORMAT_MONTHSY; }
	| MONTHL 	 { $$ = FORMAT_MONTHL; }
	| DAYOFWEEKS 	 { $$ = FORMAT_DAYOFWEEKS; }
	| DAYOFWEEKL 	 { $$ = FORMAT_DAYOFWEEKL; }
	| DAYOFYEAR 	 { $$ = FORMAT_DAYOFYEAR; }
	| HMS 		 { $$ = FORMAT_HMS; }
	| MMDDHMS 	 { $$ = FORMAT_MMDDHMS; }
	| MMDDYYHMS 	 { $$ = FORMAT_MMDDYYHMS; }
	| YYMMDDHMS 	 { $$ = FORMAT_YYMMDDHMS; }
	| DEGREESLON 	 { $$ = FORMAT_DEGREESLON; }
	| DEGREESMMLON 	 { $$ = FORMAT_DEGREESMMLON; }
	| DEGREESMMSSLON { $$ = FORMAT_DEGREESMMSSLON; }
	| MMSSLON 	 { $$ = FORMAT_MMSSLON; }
	| DEGREESLAT 	 { $$ = FORMAT_DEGREESLAT; }
	| DEGREESMMLAT 	 { $$ = FORMAT_DEGREESMMLAT; }
	| DEGREESMMSSLAT { $$ = FORMAT_DEGREESMMSSLAT; }
	| MMSSLAT 	 { $$ = FORMAT_MMSSLAT; }
	;

signchoice: NORMAL { $$ = SIGN_NORMAL; }
	| ABSOLUTE { $$ = SIGN_ABSOLUTE; }
	| NEGATE   { $$ = SIGN_NEGATE; }
	;

direction: UP  	{ $$ = UP; }
	| DOWN 	{ $$ = DOWN; }
	| RIGHT { $$ = RIGHT; }
	| LEFT 	{ $$ = LEFT; }
	| IN 	{ $$ = IN; }
	| OUT 	{ $$ = OUT; }
	;

worldview: WORLD { $$ = COORD_WORLD; }
	|  VIEW  { $$ = COORD_VIEW; }
	;

datacolumn: X_TOK { $$ = DATA_X; }
	| Y_TOK { $$ = DATA_Y; }
	| X0 { $$ = DATA_X; }
	| Y0 { $$ = DATA_Y; }
	| Y1 { $$ = DATA_Y1; }
	| Y2 { $$ = DATA_Y2; }
	| Y3 { $$ = DATA_Y3; }
	| Y4 { $$ = DATA_Y4; }
	;

sortdir: ASCENDING   { $$ = ASCENDING; }
	| DESCENDING { $$ = DESCENDING; }
	;

sorton: X_TOK { $$ = DATA_X; }
	| Y_TOK { $$ = DATA_Y; }
	;

	/* For grace-5.1.22 compatibility, we keep DFT/FFT */
ffttype: DFT 	 { $$ = 0; }
	| FFT 	 { $$ = 0; }
	| INVDFT { $$ = 1; }
	| INVFFT { $$ = 1; }
	;

fourierdata:
	REAL {$$=0;}
	| COMPLEX {$$=1;}
	;

fourierloadx:
	INDEX       {$$ = FFT_XSCALE_INDEX;}
	| FREQUENCY {$$ = FFT_XSCALE_NU;}
	| PERIOD    {$$ = FFT_XSCALE_OMEGA;}
	;

fourierloady:
	MAGNITUDE      {$$ = FFT_OUTPUT_MAGNITUDE;}
	| PHASE        {$$ = FFT_OUTPUT_PHASE;}
	| COEFFICIENTS {$$ = FFT_OUTPUT_COEFF;}
        | COMPLEX      {$$ = FFT_OUTPUT_REIM;}
	;

fourierinseg:
	INDEX       {$$ = FFT_INPUT_SEG_INDEX;}
	| INSEGX    {$$ = FFT_INPUT_SEG_X ;}
	;

fourieroutsg:
	POSITIVE    {$$ = FFT_OUTSEGMT_POSITIVE;}
	| HALF      {$$ = FFT_OUTSEGMT_HALF_LENGTH;}
	| CENTER    {$$ = FFT_OUTSEGMT_CENTERED;}
	;

waveletype:
        DAUBECHIES 	      {$$ = WAVELET_DAUBECHIES;}
        | DAUBECHIES_CENTERED {$$ = WAVELET_DAUBECHIES_CENTERED;}
        | HAAR       	      {$$ = WAVELET_HARR;}
        | HAAR_CENTERED	      {$$ = WAVELET_HARR_CENTERED;}
        | BSPLINE	      {$$ = WAVELET_BSPLINE;}
        | BSPLINE_CENTERED    {$$ = WAVELET_BSPLINE_CENTERED;}
        ;

windowtype:
	NONE         {$$ = FFT_WINDOW_NONE;}
	| TRIANGULAR {$$ = FFT_WINDOW_TRIANGULAR;}
	| PARZEN     {$$ = FFT_WINDOW_PARZEN;}
	| WELCH      {$$ = FFT_WINDOW_WELCH;}
	| HANNING    {$$ = FFT_WINDOW_HANNING;}
	| HAMMING    {$$ = FFT_WINDOW_HAMMING;}
	| BLACKMAN   {$$ = FFT_WINDOW_BLACKMAN;}
	;
interpmethod:
        LINEAR     { $$ = INTERP_LINEAR; }
        | SPLINE   { $$ = INTERP_SPLINE; }
        | ASPLINE  { $$ = INTERP_ASPLINE; }
        | XSPLINE  { $$ = INTERP_XSPLINE; }
        | XYSPLINE { $$ = INTERP_XYSPLINE; }
	;

stattype: MINP { $$ = MINP; }
	| MAXP { $$ = MAXP; }
        | AVG  { $$ = AVG; }
	| SD   { $$ = SD; }
	| SUM  { $$ = SUM; }
	| IMIN { $$ = IMIN; }
	| IMAX { $$ = IMAX; }
	;

font_select:
        FONTP nexpr
        {
            $$ = get_mapped_font($2);
        }
        | FONTP sexpr
        {
            $$ = get_font_by_name($2);
            xfree($2);
        }
        ;

lines_select:
        LINESTYLE nexpr
        {
	    int lines = $2;
            if (lines >= 0 && lines < number_of_linestyles()) {
	        $$ = lines;
	    } else {
	        errmsg("invalid linestyle");
	        $$ = 1;
	    }
        }
        ;

pattern_select:
        PATTERN nexpr
        {
	    int patno = $2;
            if (patno >= 0 && patno < number_of_patterns()) {
	        $$ = patno;
	    } else {
	        errmsg("invalid pattern number");
	        $$ = 1;
	    }
        }
        ;

color_select:
        COLOR nexpr
        {
            int c = $2;
            if (c >= 0 && c < number_of_colors()) {
                $$ = c;
            } else {
                errmsg("Invalid color ID");
                $$ = 1;
            }
        }
        | COLOR sexpr
        {
            int c = get_color_by_name($2);
            if (c == BAD_COLOR) {
                errmsg("Invalid color name");
                c = 1;
            }
            xfree($2);
            $$ = c;
        }
        | COLOR '(' nexpr ',' nexpr ',' nexpr ')'
        {
            int c;
            CMap_entry cmap;
            cmap.rgb.red = $3;
            cmap.rgb.green = $5;
            cmap.rgb.blue = $7;
            cmap.ctype = COLOR_MAIN;
            cmap.cname = NULL;
            c = add_color(cmap);
            if (c == BAD_COLOR) {
                errmsg("Can't allocate requested color");
                c = 1;
            }
            $$ = c;
        }
        ;

linew_select:
        LINEWIDTH expr
        {
            double linew;
            linew = $2;
            if (linew < 0.0) {
                yyerror("Negative linewidth");
                linew = 0.0;
            } else if (linew > MAX_LINEWIDTH) {
                yyerror("Linewidth too large");
                linew = MAX_LINEWIDTH;
            }
            $$ = linew;
        }
        ;

opchoice_sel: PLACE opchoice
        {
            $$ = $2;
        }
        ;

opchoice: NORMAL   { $$ = PLACEMENT_NORMAL; }
	| OPPOSITE { $$ = PLACEMENT_OPPOSITE; }
	| BOTH 	   { $$ = PLACEMENT_BOTH; }
	;


parmset_obs:
        PAGE LAYOUT pageorient {
            		 	int wpp, hpp;
            		 	if ($3 == PAGE_ORIENT_LANDSCAPE) {
            		 	    wpp = 792;
            		 	    hpp = 612;
            		 	} else {
            		 	    wpp = 612;
            		 	    hpp = 792;
            		 	}
            		 	set_page_dimensions(wpp, hpp, FALSE);
        }
        | PAGE SIZE NUMBER NUMBER { set_page_dimensions((int) $3, (int) $4, FALSE); }
	| PAGE nexpr 		  { scroll_proc($2); }
	| PAGE INOUT nexpr 	  { scrollinout_proc($3); }

	| DEFAULT FONTP SOURCE expr { }

	| STACK WORLD expr ',' expr ',' expr ',' expr TICKP expr ',' expr ',' expr ',' expr {
	    				  add_world(gn, $3, $5, $7, $9); }
//24 	| BOX FILL colpat_obs {filltype_obs = $3;}
//24 	| ELLIPSE FILL colpat_obs {filltype_obs = $3;}
//24 	| STRING linew_select { }

	| TIMESTAMP linew_select { }

	| TITLE linew_select 	{ }
	| SUBTITLE linew_select { }
	| LEGEND BOX onoff { if ( bad_gno ()) return 1;
	    		     if ($3 == FALSE && get_project_version() <= 40102) {
            		         g[gn].l.boxpen.pattern = 0;
            		     }
	}
	| LEGEND X1 expr 	    { if ( bad_gno ()) return 1; g[gn].l.legx = $3;	}
	| LEGEND Y1 expr 	    { if ( bad_gno ()) return 1; g[gn].l.legy = $3;	}
 	| LEGEND STRING nexpr sexpr { if (is_valid_setno(gn, $3)) {
             			          strncpy(g[gn].p[$3].lstr, $4, MAX_STRING_LENGTH - 1);
 	    			      } else {
             			          yyerror("Unallocated set");
             			      }
             			      xfree($4);
 	}
	| LEGEND BOX FILL onoff { }
	| LEGEND BOX FILL WITH colpat_obs {filltype_obs = $5;}
	| LEGEND lines_select { }
	| LEGEND linew_select { }

        | selectgraph LABEL onoff      { }
        | selectgraph TYPE LOGX        { g[$1].type = GRAPH_XY;    g[$1].xscale  = SCALE_LOG; }
        | selectgraph TYPE LOGY        { g[$1].type = GRAPH_XY;    g[$1].yscale  = SCALE_LOG; }
        | selectgraph TYPE LOGXY       { g[$1].type = GRAPH_XY;    g[$1].xscale  = SCALE_LOG; g[$1].yscale  = SCALE_LOG; }
        | selectgraph TYPE BAR         { g[$1].type = GRAPH_CHART; g[$1].xyflip  = FALSE;     g[$1].stacked = FALSE; }
        | selectgraph TYPE HBAR        { g[$1].type = GRAPH_CHART; g[$1].xyflip  = TRUE; }
        | selectgraph TYPE STACKEDBAR  { g[$1].type = GRAPH_CHART; g[$1].stacked = TRUE; }
        | selectgraph TYPE STACKEDHBAR { g[$1].type = GRAPH_CHART; g[$1].stacked = TRUE;      g[$1].xyflip = TRUE; }

	| WORLD XMIN expr { if ( bad_gno ()) return 1; g[gn].w.xg1 = $3; }
	| WORLD XMAX expr { if ( bad_gno ()) return 1; g[gn].w.xg2 = $3; }
	| WORLD YMIN expr { if ( bad_gno ()) return 1; g[gn].w.yg1 = $3; }
	| WORLD YMAX expr { if ( bad_gno ()) return 1; g[gn].w.yg2 = $3; }
	| VIEW XMIN expr  { if ( bad_gno ()) return 1; g[gn].v.xv1 = $3; }
	| VIEW XMAX expr  { if ( bad_gno ()) return 1; g[gn].v.xv2 = $3; }
	| VIEW YMIN expr  { if ( bad_gno ()) return 1; g[gn].v.yv1 = $3; }
	| VIEW YMAX expr  { if ( bad_gno ()) return 1; g[gn].v.yv2 = $3; }

	| LEGEND LAYOUT expr {	}

	| FRAMEP FILL onoff {  if ( bad_gno ()) return 1;            g[gn].f.fillpen.pattern = $3;        }
	| selectgraph AUTOSCALE TYPE AUTO {        }
	| selectgraph AUTOSCALE TYPE SPEC {        }
        | HARDCOPY DEVICE expr        { }
        | PS LINEWIDTH BEGIN expr     { }
        | PS LINEWIDTH INCREMENT expr { }
        | PS linew_select { }
        ;


axislabeldesc_obs:
	linew_select { }
	| opchoice_sel_obs {
	    if (!is_valid_axis(gn, naxis)) {
                yyerror("No valid axis selected");
                return 1;
            }
	    g[gn].t[naxis]->label_op = $1;
	}
        ;

setprop_obs:
	selectset SYMBOL FILL nexpr {
	    switch ($4){
	    case 0:
	        g[$1->gno].p[$1->setno].symfillpen.pattern = 0;
	        break;
	    case 1:
	        g[$1->gno].p[$1->setno].symfillpen.pattern = 1;
	        break;
	    case 2:
	        g[$1->gno].p[$1->setno].symfillpen.pattern = 1;
	        g[$1->gno].p[$1->setno].symfillpen.color = getbgcolor();
	        break;
	    }
	}
	| selectset SKIP nexpr { g[$1->gno].p[$1->setno].symskip = $3; }
	| selectset FILL nexpr {
	    switch ($3) {
            case 0:
                g[$1->gno].p[$1->setno].filltype = SETFILL_NONE;
                break;
            case 1:
                g[$1->gno].p[$1->setno].filltype = SETFILL_POLYGON;
                break;
            case 2:
                g[$1->gno].p[$1->setno].filltype = SETFILL_BASELINE;
                g[$1->gno].p[$1->setno].baseline_type = BASELINE_TYPE_0;
                break;
            case 6:
                g[$1->gno].p[$1->setno].filltype = SETFILL_BASELINE;
                g[$1->gno].p[$1->setno].baseline_type = BASELINE_TYPE_GMIN;
                break;
            case 7:
                g[$1->gno].p[$1->setno].filltype = SETFILL_BASELINE;
                g[$1->gno].p[$1->setno].baseline_type = BASELINE_TYPE_GMAX;
                break;
            }
	}
	| selectset ERRORBAR TYPE opchoice_obs { g[$1->gno].p[$1->setno].errbar.ptype = $4; }
	| selectset SYMBOL CENTER onoff  { }
	| selectset lines_select 	 { g[$1->gno].p[$1->setno].lines = $2; }
	| selectset linew_select 	 { g[$1->gno].p[$1->setno].linew = $2; }
	| selectset color_select 	 { g[$1->gno].p[$1->setno].linepen.color = $2; }
	| selectset FILL WITH colpat_obs { filltype_obs = $4; }
	| selectset XYZ expr ',' expr    { }
	| selectset ERRORBAR LENGTH expr { g[$1->gno].p[$1->setno].errbar.barsize = $4;	}
	| selectset ERRORBAR RISER onoff { }
/*
 * 	| selectset SYMBOL COLOR '-' N_NUMBER {
 * 	    g[$1->gno].p[$1->setno].sympen.color = -1;
 * 	}
 */
        ;


tickattr_obs:
        MAJOR onoff   { if (bad_axis ()) return 1; g[gn].t[naxis]->active = $2; }/* <= xmgr-4.1 */
	| MINOR onoff { }
	| ALT onoff   { }
	| MINP NUMBER { }
	| MAXP NUMBER { }
	| LOG onoff   { }
	| TYPE AUTO   { if (bad_axis ()) return 1; g[gn].t[naxis]->t_spec = TICKS_SPEC_NONE; }
	| TYPE SPEC   { if (bad_axis ()) return 1;
	    		if (!is_valid_axis(gn, naxis)) {
	    		   if (g[gn].t[naxis]->t_spec != TICKS_SPEC_BOTH) {
            		       g[gn].t[naxis]->t_spec = TICKS_SPEC_MARKS;
            		   }
	    		}
	}
	| MINOR expr { if (bad_axis ()) return 1;
	    	       if ($2 != 0.0) {
            	           g[gn].t[naxis]->nminor =
            	                       (int) rint(g[gn].t[naxis]->tmajor / $2 - 1);
            	       } else {
            	           g[gn].t[naxis]->nminor = 0;
            	       }
	}
	| SIZE expr        { if (bad_axis ()) return 1; g[gn].t[naxis]->props.size = $2; }
	| nexpr ',' expr   { if (bad_axis ()) return 1;
	    		     g[gn].t[naxis]->tloc[$1].wtpos = $3;
	    		     g[gn].t[naxis]->tloc[$1].type = TICK_TYPE_MAJOR; }
	| opchoice_sel_obs { if (bad_axis ()) return 1;g[gn].t[naxis]->t_op = $1; }
        ;

ticklabelattr_obs:
	linew_select { }
	| TYPE AUTO { if (bad_axis ()) return 1;
	    	      if (g[gn].t[naxis]->t_spec == TICKS_SPEC_BOTH) {
            	          g[gn].t[naxis]->t_spec = TICKS_SPEC_MARKS;
            	      }
	}
	| TYPE SPEC { if (bad_axis ()) return 1; g[gn].t[naxis]->t_spec = TICKS_SPEC_BOTH; }
	| LAYOUT SPEC 	    { }
	| LAYOUT HORIZONTAL { if (bad_axis ()) return 1; g[gn].t[naxis]->tl_angle = 0; }
	| LAYOUT VERTICAL   { if (bad_axis ()) return 1; g[gn].t[naxis]->tl_angle = 90;	}
	| PLACE ON TICKSP   { }
	| PLACE BETWEEN TICKSP { }
	| opchoice_sel_obs  { if (bad_axis ()) return 1;g[gn].t[naxis]->tl_op = $1; }
	| SIGN signchoice   { if (bad_axis ()) return 1;
	    		      switch($2) {
            		      case SIGN_NEGATE:
            		          g[gn].t[naxis]->tl_formula =
            		              copy_string(g[gn].t[naxis]->tl_formula, "-$t");
            		          break;
            		      case SIGN_ABSOLUTE:
            		          g[gn].t[naxis]->tl_formula =
            		              copy_string(g[gn].t[naxis]->tl_formula, "abs($t)");
            		          break;
            		      default:
            		          g[gn].t[naxis]->tl_formula =
            		              copy_string(g[gn].t[naxis]->tl_formula, NULL);
            		          break;
            		      }
	}
        ;

colpat_obs: NONE
	| COLOR
	| PATTERN
	;

opchoice_sel_obs: OP opchoice_obs { $$ = $2; }
        ;

opchoice_obs: TOP { $$ = PLACEMENT_OPPOSITE; }
	| BOTTOM  { $$ = PLACEMENT_NORMAL; }
	| LEFT	  { $$ = PLACEMENT_NORMAL; }
	| RIGHT   { $$ = PLACEMENT_OPPOSITE; }
	| BOTH 	  { $$ = PLACEMENT_BOTH; }
	;

vectormap: selectset VMAP { $$ = obj_tid_get_set_num ($1->gno ,$1->setno); }
        ;

colorbar: selectset COLORBAR {
	  int numset = obj_tid_get_set_num ($1->gno ,$1->setno);
	  jo = objs[numset].child;
	  if (jo < 0) {
	    errmsg("No colorbar defined");
	  }
	}
;


%%

/* -----------------------  EPILOGUE  ----------------------- */

/* list of intrinsic functions and keywords */
symtab_entry ikey[] = {
	{"A0"			, FITPARM	  , VAR_INTRINSIC , NULL}				,
	{"A0MAX"		, FITPMAX	  , VAR_INTRINSIC , NULL}				,
	{"A0MIN"		, FITPMIN	  , VAR_INTRINSIC , NULL}				,
	{"A1"			, FITPARM	  , VAR_INTRINSIC , NULL}				,
	{"A1MAX"		, FITPMAX	  , VAR_INTRINSIC , NULL}				,
	{"A1MIN"		, FITPMIN	  , VAR_INTRINSIC , NULL}				,
	{"A2"			, FITPARM	  , VAR_INTRINSIC , NULL}				,
	{"A2MAX"		, FITPMAX	  , VAR_INTRINSIC , NULL}				,
	{"A2MIN"		, FITPMIN	  , VAR_INTRINSIC , NULL}				,
	{"A3"			, FITPARM	  , VAR_INTRINSIC , NULL}				,
	{"A3MAX"		, FITPMAX	  , VAR_INTRINSIC , NULL}				,
	{"A3MIN"		, FITPMIN	  , VAR_INTRINSIC , NULL}				,
	{"A4"			, FITPARM	  , VAR_INTRINSIC , NULL}				,
	{"A4MAX"		, FITPMAX	  , VAR_INTRINSIC , NULL}				,
	{"A4MIN"		, FITPMIN	  , VAR_INTRINSIC , NULL}				,
	{"A5"			, FITPARM	  , VAR_INTRINSIC , NULL}				,
	{"A5MAX"		, FITPMAX	  , VAR_INTRINSIC , NULL}				,
	{"A5MIN"		, FITPMIN	  , VAR_INTRINSIC , NULL}				,
	{"A6"			, FITPARM	  , VAR_INTRINSIC , NULL}				,
	{"A6MAX"		, FITPMAX	  , VAR_INTRINSIC , NULL}				,
	{"A6MIN"		, FITPMIN	  , VAR_INTRINSIC , NULL}				,
	{"A7"			, FITPARM	  , VAR_INTRINSIC , NULL}				,
	{"A7MAX"		, FITPMAX	  , VAR_INTRINSIC , NULL}				,
	{"A7MIN"		, FITPMIN	  , VAR_INTRINSIC , NULL}				,
	{"A8"			, FITPARM	  , VAR_INTRINSIC , NULL}				,
	{"A8MAX"		, FITPMAX	  , VAR_INTRINSIC , NULL}				,
	{"A8MIN"		, FITPMIN	  , VAR_INTRINSIC , NULL}				,
	{"A9"			, FITPARM	  , VAR_INTRINSIC , NULL}				,
	{"A9MAX"		, FITPMAX	  , VAR_INTRINSIC , NULL}				,
	{"A9MIN"		, FITPMIN	  , VAR_INTRINSIC , NULL}				,
	{"ABOVE"		, ABOVE		  , VAR_INTRINSIC , NULL}				,
	{"ABS"			, FUNC_D	  , VAR_INTRINSIC , (void *) fabs}		,
	{"ABSOLUTE"		, ABSOLUTE	  , VAR_INTRINSIC , NULL}				,
	{"ACOS"			, FUNC_D	  , VAR_INTRINSIC , (void *) acos}		,
	{"ACOSH"		, FUNC_D	  , VAR_INTRINSIC , (void *) acosh}		,
	{"AI"			, FUNC_D	  , VAR_INTRINSIC , (void *) ai_wrap}		,
	{"ALLVARS"		, ALLVARS	  , VAR_INTRINSIC , NULL}				,
	{"ALIAS"		, ALIAS		  , VAR_INTRINSIC , NULL}				,
	{"ALT"			, ALT		  , VAR_INTRINSIC , NULL}				,
	{"ALTXAXIS"		, ALTXAXIS	  , VAR_INTRINSIC , NULL}				,
	{"ALTYAXIS"		, ALTYAXIS	  , VAR_INTRINSIC , NULL}				,
	{"AND"			, AND		  , VAR_INTRINSIC , NULL}				,
	{"ANCHOR"		, ANCHOR	  , VAR_INTRINSIC , NULL}				,
	{"ANGLE"		, ANGLE		  , VAR_INTRINSIC , NULL}				,
	{"ANGLES"		, ANGLES	  , VAR_INTRINSIC , NULL}				,
	{"ANTIALIASING"		, ANTIALIASING	  , VAR_INTRINSIC , NULL}				,
	{"APPEND"		, APPEND	  , VAR_INTRINSIC , NULL}				,
	{"ARC"			, ARC		  , VAR_INTRINSIC , NULL}				,
	{"ARRANGE"		, ARRANGE	  , VAR_INTRINSIC , NULL}				,
	{"ARROW"		, ARROW		  , VAR_INTRINSIC , NULL}				,
	{"ASCENDING"		, ASCENDING	  , VAR_INTRINSIC , NULL}				,
	{"ASIN"			, FUNC_D	  , VAR_INTRINSIC , (void *) asin}		,
	{"ASINH"		, FUNC_D	  , VAR_INTRINSIC , (void *) asinh}		,
	{"ASPLINE"		, ASPLINE	  , VAR_INTRINSIC , NULL}				,
	{"ATAN"			, FUNC_D	  , VAR_INTRINSIC , (void *) atan}		,
	{"ATAN2"		, FUNC_DD	  , VAR_INTRINSIC , (void *) atan2}		,
	{"ATANH"		, FUNC_D	  , VAR_INTRINSIC , (void *) atanh}		,
	{"AUTO"			, AUTO		  , VAR_INTRINSIC , NULL}				,
	{"AUTOSCALE"		, AUTOSCALE	  , VAR_INTRINSIC , NULL}				,
	{"AUTOTICKS"		, AUTOTICKS	  , VAR_INTRINSIC , NULL}				,
	{"AVALUE"		, AVALUE	  , VAR_INTRINSIC , NULL}				,
	{"AVG"			, AVG		  , VAR_INTRINSIC , NULL}				,
	{"BACKGROUND"		, BACKGROUND	  , VAR_INTRINSIC , NULL}				,
	{"BAR"			, BAR		  , VAR_INTRINSIC , NULL}				,
	{"BARDY"		, BARDY		  , VAR_INTRINSIC , NULL}				,
	{"BARDYDY"		, BARDYDY	  , VAR_INTRINSIC , NULL}				,
	{"BASELINE"		, BASELINE	  , VAR_INTRINSIC , NULL}				,
	{"BATCH"		, BATCH		  , VAR_INTRINSIC , NULL}				,
        {"BEGIN"		, BEGIN		  , VAR_INTRINSIC , NULL}				,
	{"BELOW"		, BELOW		  , VAR_INTRINSIC , NULL}				,
	{"BETA"			, FUNC_DD	  , VAR_INTRINSIC , (void *) beta}		,
	{"BETWEEN"		, BETWEEN	  , VAR_INTRINSIC , NULL}				,
	{"BI"			, FUNC_D	  , VAR_INTRINSIC , (void *) bi_wrap}		,
	{"BLACKMAN"		, BLACKMAN	  , VAR_INTRINSIC , NULL}				,
	{"BLOCK"		, BLOCK		  , VAR_INTRINSIC , NULL}				,
	{"BOTH"			, BOTH		  , VAR_INTRINSIC , NULL}				,
	{"BOTTOM"		, BOTTOM	  , VAR_INTRINSIC , NULL}				,
	{"BOX"			, BOX		  , VAR_INTRINSIC , NULL}				,
	{"BREAK"		, BREAK		  , VAR_INTRINSIC , NULL}				,
	{"BSPLINE"		, BSPLINE	  , VAR_INTRINSIC , NULL}				,
	{"BSPLINE_CENTERED"	, BSPLINE_CENTERED, VAR_INTRINSIC , NULL}				,
	{"CD"			, CD		  , VAR_INTRINSIC , NULL}				,
	{"CEIL"			, FUNC_D	  , VAR_INTRINSIC , (void *) ceil}		,
	{"CENTER"		, CENTER	  , VAR_INTRINSIC , NULL}				,
	{"CHAR"			, CHAR		  , VAR_INTRINSIC , NULL}				,
	{"CHART"		, CHART		  , VAR_INTRINSIC , NULL}				,
	{"CHDTR"		, FUNC_DD	  , VAR_INTRINSIC , (void *) chdtr}		,
	{"CHDTRC"		, FUNC_DD	  , VAR_INTRINSIC , (void *) chdtrc}		,
	{"CHDTRI"		, FUNC_DD	  , VAR_INTRINSIC , (void *) chdtri}		,
	{"CHI"			, FUNC_D	  , VAR_INTRINSIC , (void *) chi_wrap}		,
	{"CI"			, FUNC_D	  , VAR_INTRINSIC , (void *) ci_wrap}		,
	{"CLEAR"		, CLEAR		  , VAR_INTRINSIC , NULL}				,
	{"CLICK"		, CLICK		  , VAR_INTRINSIC , NULL}				,
	{"CLIP"			, CLIP		  , VAR_INTRINSIC , NULL}				,
	{"CLOSE"		, CLOSE		  , VAR_INTRINSIC , NULL}				,
	{"CMAP"		        , CMAP		  , VAR_INTRINSIC , NULL}				,
	{"COEFFICIENTS"		, COEFFICIENTS	  , VAR_INTRINSIC , NULL}				,
	{"COLOR"		, COLOR		  , VAR_INTRINSIC , NULL}				,
	{"COLORBAR"		, COLORBAR	  , VAR_INTRINSIC , NULL}				,
	{"COMMENT"		, COMMENT	  , VAR_INTRINSIC , NULL}				,
	{"COMPLEX"		, COMPLEX	  , VAR_INTRINSIC , NULL}				,
	{"COMPOUND"		, COMPOUND	  , VAR_INTRINSIC , NULL}				,
	{"COMPUTING"		, COMPUTING	  , VAR_INTRINSIC , NULL}				,
	{"CONST"		, KEY_CONST	  , VAR_INTRINSIC , NULL}				,
	{"CONSTRAINTS"		, CONSTRAINTS	  , VAR_INTRINSIC , NULL}				,
	{"CONTOUR"		, CONTOUR	  , VAR_INTRINSIC , NULL}				,
	{"COPY"			, COPY		  , VAR_INTRINSIC , NULL}				,
	{"COS"			, FUNC_D	  , VAR_INTRINSIC , (void *) cos}			,
	{"COSH"			, FUNC_D	  , VAR_INTRINSIC , (void *) cosh}		,
	{"CYCLE"		, CYCLE		  , VAR_INTRINSIC , NULL}				,
	{"DAUBECHIES"		, DAUBECHIES	  , VAR_INTRINSIC , NULL}				,
	{"DAUBECHIES_CENTERED"	, DAUBECHIES_CENTERED , VAR_INTRINSIC , NULL}				,
	{"DATE"			, DATE		  , VAR_INTRINSIC , NULL}				,
	{"DAWSN"		, FUNC_D	  , VAR_INTRINSIC , (void *) dawsn}		,
	{"DAYMONTH"		, DAYMONTH	  , VAR_INTRINSIC , NULL}				,
	{"DAYOFWEEKL"		, DAYOFWEEKL	  , VAR_INTRINSIC , NULL}				,
	{"DAYOFWEEKS"		, DAYOFWEEKS	  , VAR_INTRINSIC , NULL}				,
	{"DAYOFYEAR"		, DAYOFYEAR	  , VAR_INTRINSIC , NULL}				,
	{"DDMMYY"		, DDMMYY	  , VAR_INTRINSIC , NULL}				,
	{"DECIMAL"		, DECIMAL	  , VAR_INTRINSIC , NULL}				,
	{"DEF"			, DEF		  , VAR_INTRINSIC , NULL}				,
	{"DEFAULT"		, DEFAULT	  , VAR_INTRINSIC , NULL}				,
	{"DEFINE"		, DEFINE	  , VAR_INTRINSIC , NULL}				,
	{"DEG"			, UCONSTANT	  , VAR_INTRINSIC , (void *) deg_uconst}		,
	{"DEGREES"		, DEGREES	  , VAR_INTRINSIC , NULL}				,
	{"DEGREESLAT"		, DEGREESLAT	  , VAR_INTRINSIC , NULL}				,
	{"DEGREESLON"		, DEGREESLON	  , VAR_INTRINSIC , NULL}				,
	{"DEGREESMMLAT"		, DEGREESMMLAT	  , VAR_INTRINSIC , NULL}				,
	{"DEGREESMMLON"		, DEGREESMMLON	  , VAR_INTRINSIC , NULL}				,
	{"DEGREESMMSSLAT"	, DEGREESMMSSLAT  , VAR_INTRINSIC , NULL}				,
	{"DEGREESMMSSLON"	, DEGREESMMSSLON  , VAR_INTRINSIC , NULL}				,
	{"DESCENDING"		, DESCENDING	  , VAR_INTRINSIC , NULL}				,
	{"DESCRIPTION"		, DESCRIPTION	  , VAR_INTRINSIC , NULL}				,
	{"DEVICE"		, DEVICE	  , VAR_INTRINSIC , NULL}				,
	{"DFT"			, DFT		  , VAR_INTRINSIC , NULL}				,
	{"DIFF"			, DIFFERENCE	  , VAR_INTRINSIC , NULL}				,
	{"DIFFERENCE"		, DIFFERENCE	  , VAR_INTRINSIC , NULL}				,
	{"DISK"			, DISK		  , VAR_INTRINSIC , NULL}				,
	{"DOWN"			, DOWN		  , VAR_INTRINSIC , NULL}				,
	{"DPI"			, DPI		  , VAR_INTRINSIC , NULL}				,
	{"DROP"			, DROP		  , VAR_INTRINSIC , NULL}				,
	{"DROPLINE"		, DROPLINE	  , VAR_INTRINSIC , NULL}				,
	{"ECHO"			, ECHO		  , VAR_INTRINSIC , NULL}				,
	{"ELLIE"		, FUNC_DD	  , VAR_INTRINSIC , (void *) ellie}		,
	{"ELLIK"		, FUNC_DD	  , VAR_INTRINSIC , (void *) ellik}		,
	{"ELLIPSE"		, ELLIPSE	  , VAR_INTRINSIC , NULL}				,
	{"ELLPE"		, FUNC_D	  , VAR_INTRINSIC , (void *) ellpe_wrap}		,
	{"ELLPK"		, FUNC_D	  , VAR_INTRINSIC , (void *) ellpk_wrap}		,
	{"ENGINEERING"		, ENGINEERING	  , VAR_INTRINSIC , NULL}				,
	{"EQ"			, EQ		  , VAR_INTRINSIC , NULL}				,
	{"ER"			, ERRORBAR	  , VAR_INTRINSIC , NULL}				,
	{"ERF"			, FUNC_D	  , VAR_INTRINSIC , (void *) erf}			,
	{"ERFC"			, FUNC_D	  , VAR_INTRINSIC , (void *) erfc}		,
	{"ERRORBAR"		, ERRORBAR	  , VAR_INTRINSIC , NULL}				,
	{"EXIT"			, EXIT		  , VAR_INTRINSIC , NULL}				,
	{"EXP"			, FUNC_D	  , VAR_INTRINSIC , (void *) exp}			,
	{"EXPN"			, FUNC_ND	  , VAR_INTRINSIC , (void *) expn}		,
	{"EXPONENTIAL"		, EXPONENTIAL	  , VAR_INTRINSIC , NULL}				,
	{"FAC"			, FUNC_I	  , VAR_INTRINSIC , (void *) fac}			,
	{"FALSE"		, OFF		  , VAR_INTRINSIC , NULL}				,
	{"FDTR"			, FUNC_NND	  , VAR_INTRINSIC , (void *) fdtr}		,
	{"FDTRC"		, FUNC_NND	  , VAR_INTRINSIC , (void *) fdtrc}		,
	{"FDTRI"		, FUNC_NND	  , VAR_INTRINSIC , (void *) fdtri}		,
	{"FFT"			, FFT		  , VAR_INTRINSIC , NULL}				,
	{"FILE"			, FILEP		  , VAR_INTRINSIC , NULL}				,
	{"FILL"			, FILL		  , VAR_INTRINSIC , NULL}				,
	{"FIT"			, FIT		  , VAR_INTRINSIC , NULL}				,
	{"FIXED"		, FIXED		  , VAR_INTRINSIC , NULL}				,
	{"FIXEDPOINT"		, FIXEDPOINT	  , VAR_INTRINSIC , NULL}				,
	{"FLOOR"		, FUNC_D	  , VAR_INTRINSIC , (void *) floor}		,
	{"FLUSH"		, FLUSH		  , VAR_INTRINSIC , NULL}				,
	{"FOCUS"		, FOCUS		  , VAR_INTRINSIC , NULL}				,
	{"FOLLOWS"		, FOLLOWS	  , VAR_INTRINSIC , NULL}				,
	{"FONT"			, FONTP		  , VAR_INTRINSIC , NULL}				,
	{"FORCE"		, FORCE		  , VAR_INTRINSIC , NULL}				,
	{"FORMAT"		, FORMAT	  , VAR_INTRINSIC , NULL}				,
	{"FORMULA"		, FORMULA	  , VAR_INTRINSIC , NULL}				,
	{"FRAME"		, FRAMEP	  , VAR_INTRINSIC , NULL}				,
	{"FREE"			, FREE		  , VAR_INTRINSIC , NULL}				,
	{"FREQUENCY"		, FREQUENCY	  , VAR_INTRINSIC , NULL}				,
	{"FRESNLC"		, FUNC_D	  , VAR_INTRINSIC , (void *) fresnlc_wrap}	,
	{"FRESNLS"		, FUNC_D	  , VAR_INTRINSIC , (void *) fresnls_wrap}	,
	{"FROM"			, FROM		  , VAR_INTRINSIC , NULL}				,
	{"F_OF_D"		, KEY_FUNC_D	  , VAR_INTRINSIC , NULL}				,
	{"F_OF_DD"		, KEY_FUNC_DD	  , VAR_INTRINSIC , NULL}				,
        {"F_OF_I"		, KEY_FUNC_I	  , VAR_INTRINSIC , NULL}				,
	{"F_OF_ND"		, KEY_FUNC_ND	  , VAR_INTRINSIC , NULL}				,
	{"F_OF_NN"		, KEY_FUNC_NN	  , VAR_INTRINSIC , NULL}				,
	{"F_OF_NND"		, KEY_FUNC_NND	  , VAR_INTRINSIC , NULL}				,
	{"F_OF_PPD"		, KEY_FUNC_PPD	  , VAR_INTRINSIC , NULL}				,
	{"F_OF_PPPD"		, KEY_FUNC_PPPD	  , VAR_INTRINSIC , NULL}				,
	{"F_OF_PPPPD"		, KEY_FUNC_PPPPD  , VAR_INTRINSIC , NULL}				,
	{"F_OF_PPPPPD"		, KEY_FUNC_PPPPPD , VAR_INTRINSIC , NULL}				,
	{"GAMMA"		, FUNC_D	  , VAR_INTRINSIC , (void *) true_gamma}		,
	{"GDTR"			, FUNC_PPD	  , VAR_INTRINSIC , (void *) gdtr}		,
	{"GDTRC"		, FUNC_PPD	  , VAR_INTRINSIC , (void *) gdtrc}		,
	{"GE"			, GE		  , VAR_INTRINSIC , NULL}				,
	{"GENERAL"		, GENERAL	  , VAR_INTRINSIC , NULL}				,
	{"GETP"			, GETP		  , VAR_INTRINSIC , NULL}				,
	{"GRAPH"		, GRAPH		  , VAR_INTRINSIC , NULL}				,
	{"GRID"			, GRID		  , VAR_INTRINSIC , NULL}				,
	{"GT"			, GT		  , VAR_INTRINSIC , NULL}				,
	{"HAAR"			, HAAR		  , VAR_INTRINSIC , NULL}				,
	{"HAAR_CENTERED"	, HAAR_CENTERED	  , VAR_INTRINSIC , NULL}				,
	{"HALF"			, HALF		  , VAR_INTRINSIC , NULL}				,
	{"HAMMING"		, HAMMING	  , VAR_INTRINSIC , NULL}				,
	{"HANNING"		, HANNING	  , VAR_INTRINSIC , NULL}				,
	{"HARDCOPY"		, HARDCOPY	  , VAR_INTRINSIC , NULL}				,
	{"HBAR"			, HBAR		  , VAR_INTRINSIC , NULL}				,
	{"HELP"			, HELP		  , VAR_INTRINSIC , NULL}				,
	{"HGAP"			, HGAP		  , VAR_INTRINSIC , NULL}				,
	{"HIDDEN"		, HIDDEN	  , VAR_INTRINSIC , NULL}				,
	{"HISTOGRAM"		, HISTOGRAM	  , VAR_INTRINSIC , NULL}				,
	{"HMS"			, HMS		  , VAR_INTRINSIC , NULL}				,
	{"HORIZI"		, HORIZI	  , VAR_INTRINSIC , NULL}				,
	{"HORIZO"		, HORIZO	  , VAR_INTRINSIC , NULL}				,
	{"HORIZONTAL"		, HORIZONTAL	  , VAR_INTRINSIC , NULL}				,
	{"HYP2F1"		, FUNC_PPPD	  , VAR_INTRINSIC , (void *) hyp2f1}		,
	{"HYPERG"		, FUNC_PPD	  , VAR_INTRINSIC , (void *) hyperg}		,
	{"HYPOT"		, FUNC_DD	  , VAR_INTRINSIC , (void *) hypot}		,
	{"I0E"			, FUNC_D	  , VAR_INTRINSIC , (void *) i0e}			,
	{"I1E"			, FUNC_D	  , VAR_INTRINSIC , (void *) i1e}			,
	{"ID"			, ID		  , VAR_INTRINSIC , NULL}				,
	{"IF"			, IF		  , VAR_INTRINSIC , NULL}				,
	{"IFILTER"		, IFILTER	  , VAR_INTRINSIC , NULL}				,
	{"IGAM"			, FUNC_DD	  , VAR_INTRINSIC , (void *) igam}		,
	{"IGAMC"		, FUNC_DD	  , VAR_INTRINSIC , (void *) igamc}		,
	{"IGAMI"		, FUNC_DD	  , VAR_INTRINSIC , (void *) igami}		,
	{"IMAX"			, IMAX		  , VAR_INTRINSIC , NULL}				,
	{"IMIN"			, IMIN		  , VAR_INTRINSIC , NULL}				,
	{"IN"			, IN		  , VAR_INTRINSIC , NULL}				,
	{"INCBET"		, FUNC_PPD	  , VAR_INTRINSIC , (void *) incbet}		,
	{"INCBI"		, FUNC_PPD	  , VAR_INTRINSIC , (void *) incbi}		,
	{"INCREMENT"		, INCREMENT	  , VAR_INTRINSIC , NULL}				,
	{"INDEX"		, INDEX		  , VAR_INTRINSIC , NULL}				,
	{"INOUT"		, INOUT		  , VAR_INTRINSIC , NULL}				,
	{"INSEGX"		, INSEGX	  , VAR_INTRINSIC , NULL}				,
	{"INT"			, INT		  , VAR_INTRINSIC , NULL}				,
	{"INTEGRATE"		, INTEGRATE	  , VAR_INTRINSIC , NULL}				,
	{"INTERPOLATE"		, INTERPOLATE	  , VAR_INTRINSIC , NULL}				,
	{"INVDFT"		, INVDFT	  , VAR_INTRINSIC , NULL}				,
	{"INVERT"		, INVERT	  , VAR_INTRINSIC , NULL}				,
	{"INVFFT"		, INVFFT	  , VAR_INTRINSIC , NULL}				,
	{"IRAND"		, FUNC_I	  , VAR_INTRINSIC , (void *) irand_wrap}		,
	{"IV"			, FUNC_DD	  , VAR_INTRINSIC , (void *) iv_wrap}		,
	{"JUST"			, JUST		  , VAR_INTRINSIC , NULL}				,
	{"JV"			, FUNC_DD	  , VAR_INTRINSIC , (void *) jv_wrap}		,
	{"K0E"			, FUNC_D	  , VAR_INTRINSIC , (void *) k0e}			,
	{"K1E"			, FUNC_D	  , VAR_INTRINSIC , (void *) k1e}			,
	{"KILL"			, KILL		  , VAR_INTRINSIC , NULL}				,
	{"KN"			, FUNC_ND	  , VAR_INTRINSIC , (void *) kn_wrap}		,
	{"LABEL"		, LABEL		  , VAR_INTRINSIC , NULL}				,
	{"LANDSCAPE"		, LANDSCAPE	  , VAR_INTRINSIC , NULL}				,
	{"LAYER"		 ,LAYER		  , VAR_INTRINSIC  ,NULL}				,
	{"LAYOUT"		, LAYOUT	  , VAR_INTRINSIC , NULL}				,
	{"LBETA"		, FUNC_DD	  , VAR_INTRINSIC , (void *) lbeta}		,
	{"LE"			, LE		  , VAR_INTRINSIC , NULL}				,
	{"LEFT"			, LEFT		  , VAR_INTRINSIC , NULL}				,
	{"LEGEND"		, LEGEND	  , VAR_INTRINSIC , NULL}				,
	{"LENGTH"		, LENGTH	  , VAR_INTRINSIC , NULL}				,
	{"LGAMMA"		, FUNC_D	  , VAR_INTRINSIC , (void *) lgamma}		,
	{"LINCONV"		, LINCONV	  , VAR_INTRINSIC , NULL}				,
	{"LINE"			, LINE		  , VAR_INTRINSIC , NULL}				,
	{"LINEAR"		, LINEAR	  , VAR_INTRINSIC , NULL}				,
	{"LINESTYLE"		, LINESTYLE	  , VAR_INTRINSIC , NULL}				,
	{"LINEWIDTH"		, LINEWIDTH	  , VAR_INTRINSIC , NULL}				,
	{"LINK"			, LINK		  , VAR_INTRINSIC , NULL}				,
	{"LN"			, FUNC_D	  , VAR_INTRINSIC , (void *) log}			,
	{"LOAD"			, LOAD		  , VAR_INTRINSIC , NULL}				,
	{"LOCTYPE"		, LOCTYPE	  , VAR_INTRINSIC , NULL}				,
	{"LOG"			, LOG		  , VAR_INTRINSIC , NULL}				,
	{"LOG10"		, FUNC_D	  , VAR_INTRINSIC , (void *) log10}		,
	{"LOG2"			, FUNC_D	  , VAR_INTRINSIC , (void *) log2}		,
	{"LOGARITHMIC"		 ,LOGARITHMIC	  , VAR_INTRINSIC  ,NULL}				,
	{"LOGX"			, LOGX		  , VAR_INTRINSIC , NULL}				,
	{"LOGXY"		, LOGXY		  , VAR_INTRINSIC , NULL}				,
	{"LOGY"			, LOGY		  , VAR_INTRINSIC , NULL}				,
	{"LOGIT"		, LOGIT		  , VAR_INTRINSIC , NULL}				,
	{"LT"			, LT		  , VAR_INTRINSIC , NULL}				,
	{"MAGIC"		, MAGIC		  , VAR_INTRINSIC , NULL}				,
	{"MAGNITUDE"		, MAGNITUDE	  , VAR_INTRINSIC , NULL}				,
	{"MAJOR"		, MAJOR		  , VAR_INTRINSIC , NULL}				,
	{"MAP"			, MAP		  , VAR_INTRINSIC , NULL}				,
	{"MAX"			, MAXP		  , VAR_INTRINSIC , NULL}				,
	{"MAXOF"		, FUNC_DD	  , VAR_INTRINSIC , (void *) max_wrap}		,
	{"MESH"			, MESH		  , VAR_INTRINSIC , NULL}				,
	{"MESH2D"		, MESH2D	  , VAR_INTRINSIC , NULL}				,
	{"MIN"			, MINP		  , VAR_INTRINSIC , NULL}				,
	{"MINOF"		, FUNC_DD	  , VAR_INTRINSIC , (void *) min_wrap}		,
	{"MINOR"		, MINOR		  , VAR_INTRINSIC , NULL}				,
	{"MMDD"			, MMDD		  , VAR_INTRINSIC , NULL}				,
	{"MMDDHMS"		, MMDDHMS	  , VAR_INTRINSIC , NULL}				,
	{"MMDDYY"		, MMDDYY	  , VAR_INTRINSIC , NULL}				,
	{"MMDDYYHMS"		, MMDDYYHMS	  , VAR_INTRINSIC , NULL}				,
	{"MMSSLAT"		, MMSSLAT	  , VAR_INTRINSIC , NULL}				,
	{"MMSSLON"		, MMSSLON	  , VAR_INTRINSIC , NULL}				,
	{"MMYY"			, MMYY		  , VAR_INTRINSIC , NULL}				,
	{"MOD"			, FUNC_DD	  , VAR_INTRINSIC , (void *) fmod}		,
	{"MONTHDAY"		, MONTHDAY	  , VAR_INTRINSIC , NULL}				,
	{"MONTHL"		, MONTHL	  , VAR_INTRINSIC , NULL}				,
	{"MONTHS"		, MONTHS	  , VAR_INTRINSIC , NULL}				,
	{"MONTHSY"		, MONTHSY	  , VAR_INTRINSIC , NULL}				,
	{"MOVE"			, MOVE		  , VAR_INTRINSIC , NULL}				,
	{"NDTR"			, FUNC_D	  , VAR_INTRINSIC , (void *) ndtr}		,
	{"NDTRI"		, FUNC_D	  , VAR_INTRINSIC , (void *) ndtri}		,
	{"NE"			, NE		  , VAR_INTRINSIC , NULL}				,
	{"NEGATE"		, NEGATE	  , VAR_INTRINSIC , NULL}				,
	{"NEW"			, NEW		  , VAR_INTRINSIC , NULL}				,
	{"NONE"			, NONE		  , VAR_INTRINSIC , NULL}				,
	{"NONLFIT"		, NONLFIT	  , VAR_INTRINSIC , NULL}				,
	{"NORM"			, FUNC_D	  , VAR_INTRINSIC , (void *) fx}			,
	{"NORMAL"		, NORMAL	  , VAR_INTRINSIC , NULL}				,
	{"NOT"			, NOT		  , VAR_INTRINSIC , NULL}				,
	{"NTICKS"		, NTICKS 	  , VAR_INTRINSIC , NULL}				,
	{"NXY"			, NXY		  , VAR_INTRINSIC , NULL}				,
	{"OBJ"			, OBJ		  , VAR_INTRINSIC , NULL}				,
	{"OFF"			, OFF		  , VAR_INTRINSIC , NULL}				,
	{"OFFSET"		, OFFSET	  , VAR_INTRINSIC , NULL}				,
	{"OFFSETX"		, OFFSETX	  , VAR_INTRINSIC , NULL}				,
	{"OFFSETY"		, OFFSETY	  , VAR_INTRINSIC , NULL}				,
	{"OFILTER"		, OFILTER	  , VAR_INTRINSIC , NULL}				,
	{"ON"			, ON		  , VAR_INTRINSIC , NULL}				,
	{"ONREAD"		, ONREAD	  , VAR_INTRINSIC , NULL}				,
	{"OP"			, OP		  , VAR_INTRINSIC , NULL}				,
	{"OPEN"			, OPEN		  , VAR_INTRINSIC , NULL}				,
	{"OPPOSITE"		, OPPOSITE	  , VAR_INTRINSIC , NULL}				,
	{"OR"			, OR		  , VAR_INTRINSIC , NULL}				,
	{"OUT"			, OUT		  , VAR_INTRINSIC , NULL}				,
	{"PAGE"			, PAGE		  , VAR_INTRINSIC , NULL}				,
	{"PARA"			, PARA		  , VAR_INTRINSIC , NULL}				,
	{"PARAMETERS"		, PARAMETERS	  , VAR_INTRINSIC , NULL}				,
	{"PARZEN"		, PARZEN	  , VAR_INTRINSIC , NULL}				,
	{"PATTERN"		, PATTERN	  , VAR_INTRINSIC , NULL}				,
	{"PDTR"			, FUNC_ND	  , VAR_INTRINSIC , (void *) pdtr}		,
	{"PDTRC"		, FUNC_ND	  , VAR_INTRINSIC , (void *) pdtrc}		,
	{"PDTRI"		, FUNC_ND	  , VAR_INTRINSIC , (void *) pdtri}		,
	{"PERIOD"		, PERIOD	  , VAR_INTRINSIC , NULL}				,
	{"PERP"			, PERP		  , VAR_INTRINSIC , NULL}				,
	{"PHASE"		, PHASE		  , VAR_INTRINSIC , NULL}				,
	{"PI"			, CONSTANT	  , VAR_INTRINSIC , (void *) pi_const}		,
	{"PIE"			, PIE		  , VAR_INTRINSIC , NULL}				,
	{"PIPE"			, PIPE		  , VAR_INTRINSIC , NULL}				,
	{"PLACE"		, PLACE		  , VAR_INTRINSIC , NULL}				,
	{"POINT"		, POINT		  , VAR_INTRINSIC , NULL}				,
	{"POLAR"		, POLAR		  , VAR_INTRINSIC , NULL}				,
	{"POLYI"		, POLYI		  , VAR_INTRINSIC , NULL}				,
	{"POLYLINE"		, POLYLINE	  , VAR_INTRINSIC , NULL}				,
	{"POLYO"		, POLYO		  , VAR_INTRINSIC , NULL}				,
	{"POP"			, POP		  , VAR_INTRINSIC , NULL}				,
	{"PORTRAIT"		, PORTRAIT	  , VAR_INTRINSIC , NULL}				,
	{"POSITIVE"		, POSITIVE	  , VAR_INTRINSIC , NULL}				,
	{"POWER"		, POWER		  , VAR_INTRINSIC , NULL}				,
	{"PREC"			, PREC		  , VAR_INTRINSIC , NULL}				,
	{"PREPEND"		, PREPEND	  , VAR_INTRINSIC , NULL}				,
	{"PRINT"		, PRINT		  , VAR_INTRINSIC , NULL}				,
	{"PS"			, PS		  , VAR_INTRINSIC , NULL}				,
	{"PSI"			, FUNC_D	  , VAR_INTRINSIC , (void *) psi}			,
	{"PUSH"			, PUSH		  , VAR_INTRINSIC , NULL}				,
	{"PUTP"			, PUTP		  , VAR_INTRINSIC , NULL}				,
	{"RAD"			, UCONSTANT	  , VAR_INTRINSIC , (void *) rad_uconst}		,
	{"RAND"			, RAND		  , VAR_INTRINSIC , NULL}				,
	{"READ"			, READ		  , VAR_INTRINSIC , NULL}				,
	{"REAL"			, REAL		  , VAR_INTRINSIC , NULL}				,
	{"RECIPROCAL"		, RECIPROCAL	  , VAR_INTRINSIC , NULL}				,
	{"REDRAW"		, REDRAW	  , VAR_INTRINSIC , NULL}				,
	{"REFERENCE"		, REFERENCE	  , VAR_INTRINSIC , NULL}				,
	{"REGRESS"		, REGRESS	  , VAR_INTRINSIC , NULL}				,
	{"RESIZE"		, RESIZE	  , VAR_INTRINSIC , NULL}				,
	{"RESTRICT"		, RESTRICT	  , VAR_INTRINSIC , NULL}				,
	{"REVERSE"		, REVERSE	  , VAR_INTRINSIC , NULL}				,
	{"RGAMMA"		, FUNC_D	  , VAR_INTRINSIC , (void *) rgamma}		,
	{"RIGHT"		, RIGHT		  , VAR_INTRINSIC , NULL}				,
	{"RINT"			, FUNC_D	  , VAR_INTRINSIC , (void *) rint}		,
	{"RISER"		, RISER		  , VAR_INTRINSIC , NULL}				,
	{"RNORM"		, FUNC_DD	  , VAR_INTRINSIC , (void *) rnorm}		,
	{"ROT"			, ROT		  , VAR_INTRINSIC , NULL}				,
	{"ROUNDED"		, ROUNDED	  , VAR_INTRINSIC , NULL}				,
	{"RSUM"			, RSUM		  , VAR_INTRINSIC , NULL}				,
	{"RULE"			, RULE		  , VAR_INTRINSIC , NULL}				,
	{"RUNAVG"		, RUNAVG	  , VAR_INTRINSIC , NULL}				,
	{"RUNMAX"		, RUNMAX	  , VAR_INTRINSIC , NULL}				,
	{"RUNMED"		, RUNMED	  , VAR_INTRINSIC , NULL}				,
	{"RUNMIN"		, RUNMIN	  , VAR_INTRINSIC , NULL}				,
	{"RUNSTD"		, RUNSTD	  , VAR_INTRINSIC , NULL}				,
	{"SAVEALL"		, SAVEALL	  , VAR_INTRINSIC , NULL}				,
	{"SCALE"		, SCALE		  , VAR_INTRINSIC , NULL}				,
	{"SCIENTIFIC"		, SCIENTIFIC	  , VAR_INTRINSIC , NULL}				,
	{"SCROLL"		, SCROLL	  , VAR_INTRINSIC , NULL}				,
	{"SD"			, SD		  , VAR_INTRINSIC , NULL}				,
	{"SET"			, SET		  , VAR_INTRINSIC , NULL}				,
	{"SELXY"		, SELXY		  , VAR_INTRINSIC , NULL}				,
	{"SFORMAT"		, SFORMAT	  , VAR_INTRINSIC , NULL}				,
	{"SGN"			, FUNC_D	  , VAR_INTRINSIC , (void *) sign_wrap}		,
	{"SHI"			, FUNC_D	  , VAR_INTRINSIC , (void *) shi_wrap}		,
	{"SI"			, FUNC_D	  , VAR_INTRINSIC , (void *) si_wrap}		,
	{"SIGN"			, SIGN		  , VAR_INTRINSIC , NULL}				,
	{"SIN"			, FUNC_D	  , VAR_INTRINSIC , (void *) sin}			,
	{"SINH"			, FUNC_D	  , VAR_INTRINSIC , (void *) sinh}		,
	{"SIZE"			, SIZE		  , VAR_INTRINSIC , NULL}				,
	{"SKIP"			, SKIP		  , VAR_INTRINSIC , NULL}				,
	{"SLEEP"		, SLEEP		  , VAR_INTRINSIC , NULL}				,
	{"SMITH"		, SMITH		  , VAR_INTRINSIC , NULL}				,
	{"SORT"			, SORT		  , VAR_INTRINSIC , NULL}				,
	{"SOURCE"		, SOURCE	  , VAR_INTRINSIC , NULL}				,
	{"SPEC"			, SPEC		  , VAR_INTRINSIC , NULL}				,
	{"SPENCE"		, FUNC_D	  , VAR_INTRINSIC , (void *) spence}		,
	{"SPLINE"		, SPLINE	  , VAR_INTRINSIC , NULL}				,
	{"SPLIT"		, SPLIT		  , VAR_INTRINSIC , NULL}				,
	{"SQR"			, FUNC_D	  , VAR_INTRINSIC , (void *) sqr_wrap}		,
	{"SQRT"			, FUNC_D	  , VAR_INTRINSIC , (void *) sqrt}		,
	{"STACK"		, STACK		  , VAR_INTRINSIC , NULL}				,
	{"STACKED"		, STACKED	  , VAR_INTRINSIC , NULL}				,
	{"STACKEDBAR"		, STACKEDBAR	  , VAR_INTRINSIC , NULL}				,
	{"STACKEDHBAR"		, STACKEDHBAR	  , VAR_INTRINSIC , NULL}				,
	{"STAGGER"		, STAGGER	  , VAR_INTRINSIC , NULL}				,
	{"START"		, START		  , VAR_INTRINSIC , NULL}				,
	{"STDTR"		, FUNC_ND	  , VAR_INTRINSIC , (void *) stdtr}		,
	{"STDTRI"		, FUNC_ND	  , VAR_INTRINSIC , (void *) stdtri}		,
	{"STOP"			, STOP		  , VAR_INTRINSIC , NULL}				,
	{"STRING"		, STRING	  , VAR_INTRINSIC , NULL}				,
	{"STRUVE"		, FUNC_DD	  , VAR_INTRINSIC , (void *) struve}		,
	{"SUBTITLE"		, SUBTITLE	  , VAR_INTRINSIC , NULL}				,
	{"SUM"			, SUM		  , VAR_INTRINSIC , NULL}				,
	{"SWAP"			, SWAP		  , VAR_INTRINSIC , NULL}				,
	{"SYMBOL"		, SYMBOL	  , VAR_INTRINSIC , NULL}				,
	{"TAN"			, FUNC_D	  , VAR_INTRINSIC , (void *) tan}			,
	{"TANH"			, FUNC_D	  , VAR_INTRINSIC , (void *) tanh}		,
	{"TARGET"		, TARGET	  , VAR_INTRINSIC , NULL}				,
	{"TICK"			, TICKP		  , VAR_INTRINSIC , NULL}				,
	{"TICKLABEL"		, TICKLABEL	  , VAR_INTRINSIC , NULL}				,
	{"TICKS"		, TICKSP	  , VAR_INTRINSIC , NULL}				,
	{"TIMER"		, TIMER		  , VAR_INTRINSIC , NULL}				,
	{"TIMESTAMP"		, TIMESTAMP	  , VAR_INTRINSIC , NULL}				,
	{"TITLE"		, TITLE		  , VAR_INTRINSIC , NULL}				,
	{"TO"			, TO		  , VAR_INTRINSIC , NULL}				,
	{"TOP"			, TOP		  , VAR_INTRINSIC , NULL}				,
	{"TRIANGULAR"		, TRIANGULAR	  , VAR_INTRINSIC , NULL}				,
	{"TRUE"			, ON		  , VAR_INTRINSIC , NULL}				,
	{"TYPE"			, TYPE		  , VAR_INTRINSIC , NULL}				,
	{"UNIT"			, KEY_UNIT	  , VAR_INTRINSIC , NULL}				,
	{"UP"			, UP		  , VAR_INTRINSIC , NULL}				,
	{"UPDATEALL"		, UPDATEALL	  , VAR_INTRINSIC , NULL}				,
	{"USE"			, USE		  , VAR_INTRINSIC , NULL}				,
	{"VERSION"		, VERSION	  , VAR_INTRINSIC , NULL}				,
	{"VERTI"		, VERTI		  , VAR_INTRINSIC , NULL}				,
	{"VERTICAL"		, VERTICAL	  , VAR_INTRINSIC , NULL}				,
	{"VERTO"		, VERTO		  , VAR_INTRINSIC , NULL}				,
	{"VGAP"			, VGAP		  , VAR_INTRINSIC , NULL}				,
	{"VIEW"			, VIEW		  , VAR_INTRINSIC , NULL}				,
	{"VMAP"			, VMAP		  , VAR_INTRINSIC , NULL}				,
	{"VOIGT"		, FUNC_PPD	  , VAR_INTRINSIC , (void *) voigt}		,
	{"VX1"			, VX1		  , VAR_INTRINSIC , NULL}				,
	{"VX2"			, VX2		  , VAR_INTRINSIC , NULL}				,
	{"VXMAX"		, VXMAX		  , VAR_INTRINSIC , NULL}				,
	{"VY1"			, VY1		  , VAR_INTRINSIC , NULL}				,
	{"VY2"			, VY2		  , VAR_INTRINSIC , NULL}				,
	{"VYMAX"		, VYMAX		  , VAR_INTRINSIC , NULL}				,
	{"WAVELET"		, WAVELET	  , VAR_INTRINSIC , NULL}				,
	{"WELCH"		, WELCH		  , VAR_INTRINSIC , NULL}				,
	{"WITH"			, WITH		  , VAR_INTRINSIC , NULL}				,
	{"WORLD"		, WORLD		  , VAR_INTRINSIC , NULL}				,
	{"WRAP"			, WRAP		  , VAR_INTRINSIC , NULL}				,
	{"WRITE"		, WRITE		  , VAR_INTRINSIC , NULL}				,
	{"WX1"			, WX1		  , VAR_INTRINSIC , NULL}				,
	{"WX2"			, WX2		  , VAR_INTRINSIC , NULL}				,
	{"WY1"			, WY1		  , VAR_INTRINSIC , NULL}				,
	{"WY2"			, WY2		  , VAR_INTRINSIC , NULL}				,
	{"X"			, X_TOK		  , VAR_INTRINSIC , NULL}				,
	{"X0"			, X0		  , VAR_INTRINSIC , NULL}				,
	{"X1"			, X1		  , VAR_INTRINSIC , NULL}				,
	{"XAXES"		, XAXES		  , VAR_INTRINSIC , NULL}				,
	{"XAXIS"		, XAXIS		  , VAR_INTRINSIC , NULL}				,
	{"XCOR"			, XCOR		  , VAR_INTRINSIC , NULL}				,
	{"XMAX"			, XMAX		  , VAR_INTRINSIC , NULL}				,
	{"XMIN"			, XMIN		  , VAR_INTRINSIC , NULL}				,
	{"XSPLINE"		, XSPLINE	  , VAR_INTRINSIC , NULL}				,
	{"XY"			, XY		  , VAR_INTRINSIC , NULL}				,
	{"XYAXES"		, XYAXES	  , VAR_INTRINSIC , NULL}				,
	{"XYBOXPLOT"		, XYBOXPLOT	  , VAR_INTRINSIC , NULL}				,
	{"XYCOLOR"		, XYCOLOR	  , VAR_INTRINSIC , NULL}				,
	{"XYCOLPAT"		, XYCOLPAT	  , VAR_INTRINSIC , NULL}				,
	{"XYDX"			, XYDX		  , VAR_INTRINSIC , NULL}				,
	{"XYDXDX"		, XYDXDX	  , VAR_INTRINSIC , NULL}				,
	{"XYDXDXDYDY"		, XYDXDXDYDY	  , VAR_INTRINSIC , NULL}				,
	{"XYDXDY"		, XYDXDY	  , VAR_INTRINSIC , NULL}				,
	{"XYDY"			, XYDY		  , VAR_INTRINSIC , NULL}				,
	{"XYDYDY"		, XYDYDY	  , VAR_INTRINSIC , NULL}				,
	{"XYHILO"		, XYHILO	  , VAR_INTRINSIC , NULL}				,
	{"XYR"			, XYR		  , VAR_INTRINSIC , NULL}				,
	{"XYSIZE"		, XYSIZE	  , VAR_INTRINSIC , NULL}				,
	{"XYSPLINE"		, XYSPLINE	  , VAR_INTRINSIC , NULL}				,
	{"XYSTRING"		, XYSTRING	  , VAR_INTRINSIC , NULL}				,
	{"XYCMAP"		, XYCMAP	  , VAR_INTRINSIC , NULL}				,
	{"XYVMAP"		, XYVMAP	  , VAR_INTRINSIC , NULL}				,
	{"XYZ"			, XYZ		  , VAR_INTRINSIC , NULL}				,
	{"Y"			, Y_TOK		  , VAR_INTRINSIC , NULL}				,
	{"Y0"			, Y0		  , VAR_INTRINSIC , NULL}				,
	{"Y1"			, Y1		  , VAR_INTRINSIC , NULL}				,
	{"Y2"			, Y2		  , VAR_INTRINSIC , NULL}				,
	{"Y3"			, Y3		  , VAR_INTRINSIC , NULL}				,
	{"Y4"			, Y4		  , VAR_INTRINSIC , NULL}				,
	{"YAXES"		, YAXES		  , VAR_INTRINSIC , NULL}				,
	{"YAXIS"		, YAXIS		  , VAR_INTRINSIC , NULL}				,
	{"YEAR"			, YEAR		  , VAR_INTRINSIC , NULL}				,
	{"YMAX"			, YMAX		  , VAR_INTRINSIC , NULL}				,
	{"YMIN"			, YMIN		  , VAR_INTRINSIC , NULL}				,
	{"YV"			, FUNC_DD	  , VAR_INTRINSIC , (void *) yv_wrap}		,
	{"YYMMDD"		, YYMMDD	  , VAR_INTRINSIC , NULL}				,
	{"YYMMDDHMS"		, YYMMDDHMS	  , VAR_INTRINSIC , NULL}				,
	{"ZERO"			, ZERO		  , VAR_INTRINSIC , NULL}				,
	{"ZEROXAXIS"		, ALTXAXIS	  , VAR_INTRINSIC , NULL}				,
	{"ZEROYAXIS"		, ALTYAXIS	  , VAR_INTRINSIC , NULL}				,
	{"ZETA"			, FUNC_DD	  , VAR_INTRINSIC , (void *) zeta}		,
	{"ZETAC"		, FUNC_D	  , VAR_INTRINSIC , (void *) zetac}		,
	{"ZNORM"		, ZNORM		  , VAR_INTRINSIC , NULL}                       ,
	{"ZSCALE"		, ZSCALE	  , VAR_INTRINSIC , NULL}
};

const  int maxintrinsics = sizeof(ikey) / sizeof(symtab_entry);
static int maxfunc = sizeof(ikey) / sizeof(symtab_entry);

int get_parser_gno(void)
{
    return gn;
}

int set_parser_gno(int gno)
{
    if (is_valid_gno(gno) == TRUE) {
      gn = gno;
      /* For compatibility with Grace */
      lock_dirtystate (TRUE);
      if (gno == 0) set_graph_hidden (0 ,FALSE);
      lock_dirtystate (FALSE);
        return RETURN_SUCCESS;
    } else {
        return RETURN_FAILURE;
    }
}

int get_parser_setno(void)
{
    return sn;
}

int set_parser_setno(int gno, int setno)
{
    if (is_valid_setno(gno, setno) == TRUE) {
        gn = gno;
        sn = setno;
        /* those will usually be overridden except when evaluating
           a _standalone_ vexpr */
        vasgn_gno = gno;
        vasgn_setno = setno;
        return RETURN_SUCCESS;
    } else {
        return RETURN_FAILURE;
    }
}

/* Resize user defined vectors */
void realloc_vrbl(grarr *vrbl, int len)
{
    double *a;
    int i, oldlen;

    if (vrbl->type != GRARR_VEC) {
        errmsg("Internal error");
        return;
    }
    oldlen = vrbl->length;
    if (oldlen == len) {
        return;
    } else {
        a = xrealloc(vrbl->data, len*SIZEOF_DOUBLE);
        if (a != NULL || len == 0) {
            vrbl->data = a;
            vrbl->length = len;
            for (i = oldlen; i < len; i++) {
                vrbl->data[i] = 0.0;
            }
        } else {
            errmsg("Malloc failed in realloc_vrbl()");
        }
    }
}


//flow  static int com_depth = 0;
//flow  static char **com_tree = NULL;  /* command tree */

static int parser (const char *s, int type)
{
    char *seekpos;
    int i;

    if (s == NULL || s[0] == '\0') {
        if (type == PARSER_TYPE_VOID) {
            /* don't consider an empty string as error for generic parser */
            return RETURN_SUCCESS;
        } else {
            return RETURN_FAILURE;
        }
    } else if (fplog != NULL) {
       fprintf (fplog ,"@ %s " ,s);
    }

//flow      if (type == PARSER_TYPE_PUSH) {  /* push lines on the tree */
//flow      }

    strncpy(f_string, s, MAX_PARS_STRING_LENGTH - 2);
    f_string[MAX_PARS_STRING_LENGTH - 2] = '\0';
    strcat(f_string, " ");

    seekpos = f_string;

    while ((seekpos - f_string < MAX_PARS_STRING_LENGTH - 1) && (*seekpos == ' ' || *seekpos == '\t')) {
        seekpos++;
    }
    if (*seekpos == '\n' || *seekpos == '#') {
        if (type == PARSER_TYPE_VOID) {
            /* don't consider an empty string as error for generic parser */
            return RETURN_SUCCESS;
        } else {
            return RETURN_FAILURE;
        }
    }

    lowtoupper(f_string);

    pos = 0;
    interr = 0;
    expr_parsed  = FALSE;
    vexpr_parsed = FALSE;

    yyparse();

    /* free temp. arrays; for a vector expression keep the last one
     * (which is none but v_result), given there have been no errors
     * and it's what we've been asked for
     */
    if (vexpr_parsed && !interr && type == PARSER_TYPE_VEXPR) {
        for (i = 0; i < fcnt - 1; i++) {
            free_grarr (&(freelist[i]));
        }
    } else {
        for (i = 0; i < fcnt; i++) {
            free_grarr (&(freelist[i]));
        }
    }
    fcnt = 0;

    tgtn = 0;

    if ((type == PARSER_TYPE_VEXPR && !vexpr_parsed) ||
        (type == PARSER_TYPE_EXPR  && !expr_parsed)) {
        return RETURN_FAILURE;
    } else {
        return (interr ? RETURN_FAILURE:RETURN_SUCCESS);
    }
}

int s_scanner(char *s, double *res)
{
    int retval = parser(s, PARSER_TYPE_EXPR);
    *res = s_result;
    return retval;
}

int v_scanner (const char *s, int *reslen, double **vres)
{
    int retval = parser(s, PARSER_TYPE_VEXPR);
    if (retval != RETURN_SUCCESS) {
        return RETURN_FAILURE;
    } else {
        *reslen = v_result->length;
        if (v_result->type == GRARR_TMP) {
            *vres = v_result->data;
            v_result->length = 0;
            v_result->data = NULL;
        } else {
            *vres = copy_data_column(v_result->data, v_result->length);
        }
        return RETURN_SUCCESS;
    }
}

int scanner(char *s)
{
    int retval = parser(s, PARSER_TYPE_VOID);
    if (retval != RETURN_SUCCESS) {
        return RETURN_FAILURE;
    }

    if (gotparams) {
	gotparams = FALSE;
        getparms(paramfile);
    }

    if (gotread) {
	gotread = FALSE;
        getdata(gn, readfile, cursource, LOAD_SINGLE);
    }

    if (gotnlfit) {
	gotnlfit = FALSE;
        do_nonlfit(nlfit_gno, nlfit_setno, nlfit_warray, NULL, nlfit_nsteps ,nlfit_gno ,nlfit_trace);
        XCFREE(nlfit_warray);
    }
    return retval;
}

/******** Management of grace arrays grarr *********/

static void free_grarr (grarr *gar)
{
    if (gar->type == GRARR_TMP) {
        XCFREE(gar->data);
	gar->length = 0;
	gar->nrows  = 0;
	gar->ncols  = 0;
    } else if (gar->type == GRARR_MATTMP) {
      int i;
      for (i = 0; i < gar->nrows; i++) XCFREE (gar->matrix[i]);
      XCFREE (gar->matrix);
      gar->length = 0;
      gar->nrows  = 0;
      gar->ncols  = 0;
    }
}

static void copy_grarr (grarr *dest, grarr *src)
{
  int i;
  dest->type = src->type;
  switch (src->type) {
  case GRARR_MAT:
    dest->matrix = xmalloc (src->nrows * sizeof(double *));
    for (i = 0; i < src->nrows; i++) {
      dest->matrix[i] = xmalloc (src->ncols * SIZEOF_DOUBLE);
      memcpy (dest->matrix[i], src->matrix[i], src->ncols*SIZEOF_DOUBLE);
    }
    dest->length = src->length;
    dest->ncols = src->ncols;
    dest->nrows = src->nrows;
    break;
  default:
    dest->data = xmalloc(src->length*SIZEOF_DOUBLE);
    if (dest->data == NULL) {
      errmsg("Malloc failed in copy_grarr ()");
    } else {
      memcpy(dest->data, src->data, src->length*SIZEOF_DOUBLE);
      dest->length = src->length;
    }
    break;
  }
}

grarr *get_parser_arr_by_name(char * const name)
{
     int position;
     char *s;

     s = copy_string(NULL, name);
     lowtoupper(s);

     position = findf(key, s);
     xfree(s);

     if (position >= 0) {
         if (key[position].type == KEY_VEC) {
            return (grarr *) key[position].data;
         } else if (key[position].type == KEY_MAT) {
            return (grarr *) key[position].data;
	 }
     }

     return NULL;
}

grarr *define_parser_arr(char * const name ,VarStatus status)
{
     if (get_parser_arr_by_name(name) == NULL) {
	symtab_entry tmpkey;
        grarr *var;

        var = xmalloc(sizeof(grarr));
        var->type = GRARR_VEC;
        var->length = 0;
        var->data = NULL;

	tmpkey.s      = name;
	tmpkey.type   = KEY_VEC;
	tmpkey.status = status;
	tmpkey.data   = (void *) var;
	if (addto_symtab(tmpkey) == RETURN_SUCCESS) {
	    return var;
	} else {
            return NULL;
        }
     } else {
        return NULL;
     }
}

/**
 * Define an array with two indices (a matrix), allocate memory
 *   and initialise to zero.
 */
grarr *define_parser_arr2 (char * const name ,VarStatus status ,int nrows ,int ncols)
{
  int i ,j;
     if (get_parser_arr_by_name(name) == NULL) {
	symtab_entry tmpkey;
        grarr *var;
        /* first create the entry */
        var = xmalloc(sizeof(grarr));
        var->type   = GRARR_MAT;
        var->length = 0;
        var->data   = NULL;
        var->matrix = NULL;
        /* and add to symbol table */
	tmpkey.s      = name;
	tmpkey.type   = KEY_MAT;
	tmpkey.status = status;
	tmpkey.data   = (void *) var;
	if (addto_symtab(tmpkey) == RETURN_SUCCESS) {
	  /* then allocate memory */
	  var->matrix = malloc (nrows * sizeof(double *));
	  for (i = 0; i < nrows; i++) {
	    var->matrix[i] = malloc (ncols * SIZEOF_DOUBLE);
	    for (j = 0; j < ncols; j++) var->matrix[i][j] = 0.0;
	  }
	  var->length = nrows * ncols;
	  var->nrows  = nrows;
	  var->ncols  = ncols;
	  return var;
	} else {
            return NULL;
        }
     } else {
        return NULL;
     }
}


/******** Management of symbols in key array *********/


int undefine_parser_var(void *ptr)
{
    int i;

    for (i = 0; i < maxfunc; i++) {
	if (key[i].data == ptr) {
            xfree(key[i].s);
            maxfunc--;
            if (i != maxfunc) {
                memmove(&(key[i]), &(key[i + 1]), (maxfunc - i)*sizeof(symtab_entry));
            }
            key = xrealloc(key, maxfunc*sizeof(symtab_entry));
            return RETURN_SUCCESS;
        }
    }
    return RETURN_FAILURE;
}

static void undefine_all_users_var (void)
{
  int i;
  grarr *var;
  /* Free user defined and temporary matrices and vectors */
  for (i = 0; i < matcnt; i++) free_grarr (&mattmp[i]);
  for (i = 0; i < fcnt;   i++) free_grarr (&freelist[i]);
  /* free symbol table entry and data */
  for (i = maxfunc; i > 0; i--) {
    if (key[i].status == VAR_USER) {
      var = (grarr*) (&key[i].data);
      xfree (key[i].s);
      if (key[i].type == GRARR_VEC) {
	realloc_vrbl (var, 0);
      } else if (key[i].type == GRARR_MAT) {
	free_grarr (var);
      }
      maxfunc--;
      if (i != maxfunc) {
	memmove(&(key[i]), &(key[i + 1]), (maxfunc - i)*sizeof(symtab_entry));
      }
      key = xrealloc (key, maxfunc*sizeof(symtab_entry));
    }
  }
}


static int find_set_bydata(double *data, target *tgt)
{
    int gno, setno, ncol;

    if (data == NULL) {
        return RETURN_FAILURE;
    } else {
        for (gno = 0; gno < number_of_graphs(); gno++) {
            for (setno = 0; setno < number_of_sets(gno); setno++) {
                for (ncol = 0; ncol < MAX_SET_COLS; ncol++) {
                    if (getcol(gno, setno, ncol) == data) {
                        tgt->gno   = gno;
                        tgt->setno = setno;
                        return RETURN_SUCCESS;
                    }
                }
            }
        }
    }
    return RETURN_FAILURE;
}

static int findf(symtab_entry *keytable, char *s)
{

    int low, high, mid;

    low = 0;
    high = maxfunc - 1;
    while (low <= high) {
	mid = (low + high) / 2;
	if (strcmp(s, keytable[mid].s) < 0) {
	    high = mid - 1;
	} else {
	    if (strcmp(s, keytable[mid].s) > 0) {
		low = mid + 1;
	    } else {
		return (mid);
	    }
	}
    }
    return (-1);
}

static int compare_keys (const void *a, const void *b)
{
    return (int) strcmp (((const symtab_entry*)a)->s,
                         ((const symtab_entry*)b)->s);
}

/* add new entry to the symbol table */
int addto_symtab(symtab_entry newkey)
{
    int position;
    char *s;

    s = copy_string(NULL, newkey.s);
    lowtoupper(s);
    if ((position = findf(key, s)) < 0) {
        if ((key = (symtab_entry *) xrealloc(key, (maxfunc + 1)*sizeof(symtab_entry))) != NULL) {
	    key[maxfunc].type	= newkey.type;
	    key[maxfunc].status	= newkey.status;
	    key[maxfunc].data	= newkey.data;
	    key[maxfunc].s	= s;
	    maxfunc++;
	    qsort(key, maxfunc, sizeof(symtab_entry), compare_keys);
	    return RETURN_SUCCESS;
	} else {
	    xfree(s);
	    return RETURN_FAILURE;
	}
    } else if (alias_force == TRUE) { /* already exists but alias_force enabled */
        key[position].type   = newkey.type;
	key[position].data   = newkey.data;
	key[position].status = newkey.status;
	return RETURN_SUCCESS;
    } else {
	xfree(s);
        return RETURN_FAILURE;
    }
}

/* initialize symbol table */
void init_symtab(void)
{
    int i;

    if ((key = (symtab_entry *) xmalloc(maxfunc*sizeof(symtab_entry))) != NULL) {
    	memcpy (key, ikey, maxfunc*sizeof(symtab_entry));
	for (i = 0; i < maxfunc; i++) {
	    key[i].s = xmalloc(strlen(ikey[i].s) + 1);
	    strcpy(key[i].s, ikey[i].s);
	}
	qsort(key, maxfunc, sizeof(symtab_entry), compare_keys);
	return;
    } else {
	key = ikey;
	return;
    }
}

static int getcharstr(void)
{
    if (pos >= strlen(f_string))
	 return EOF;
    return (f_string[pos++]);
}

static void ungetchstr(void)
{
    if (pos > 0)
	pos--;
}

static int yylex(void)
{
    int c, i;
    int found;
    char sbuf[MAX_PARS_STRING_LENGTH + 40];

    while ((c = getcharstr()) == ' ' || c == '\t');
    if (c == EOF) {
	return (0);
    }
    if (c == '"') {
	i = 0;
	while ((c = getcharstr()) != '"' && c != EOF) {
	    if (c == '\\') {
		int ctmp;
		ctmp = getcharstr();
		if (ctmp != '"') {
		    ungetchstr();
		}
		else {
		    c = ctmp;
		}
	    }
	    sbuf[i] = c;
	    i++;
	}
	if (c == EOF) {
	    yyerror("Nonterminating string");
	    return 0;
	}
	sbuf[i] = '\0';
	yylval.sval = copy_string(NULL, sbuf);
	return CHRSTR;
    }
    if (c == '.' || isdigit(c)) {
	double d;
	int i, gotdot = 0;

	i = 0;
	while (c == '.' || isdigit(c)) {
	    if (c == '.') {
		if (gotdot) {
		    yyerror("Reading number, too many dots");
	    	    return 0;
		} else {
		    gotdot = 1;
		}
	    }
	    sbuf[i++] = c;
	    c = getcharstr();
	}
	if (c == 'E' || c == 'e') {
	    sbuf[i++] = c;
	    c = getcharstr();
	    if (c == '+' || c == '-') {
		sbuf[i++] = c;
		c = getcharstr();
	    }
	    while (isdigit(c)) {
		sbuf[i++] = c;
		c = getcharstr();
	    }
	}
	if (gotdot && i == 1) {
	    ungetchstr();
	    return '.';
	}
	sbuf[i] = '\0';
	ungetchstr();
	sscanf(sbuf, "%lf", &d);
	yylval.dval = d;
	return NUMBER;
    }
/* graphs, sets, regions resp. */
    if (c == 'G' || c == 'S' || c == 'R') {
	int i = 0, ctmp = c, gna, sna, rn;
	c = getcharstr();
	while (isdigit(c) || c == '$' || c == '_') {
	    sbuf[i++] = c;
	    c = getcharstr();
	}
	if (i == 0) {
	    c = ctmp;
	    ungetchstr();
	} else {
	    ungetchstr();
	    if (ctmp == 'G') {
	        sbuf[i] = '\0';
		if (i == 1 && sbuf[0] == '_') {
                    gna = get_recent_gno();
                } else if (i == 1 && sbuf[0] == '$') {
                    gna = gn;
                } else {
                    gna = atoi(sbuf);
                }
		if (is_valid_gno(gna) || graph_allocate(gna) == RETURN_SUCCESS) {
		    yylval.ival = gna;
		    return GRAPHNO;
		}
	    } else if (ctmp == 'S') {
	        sbuf[i] = '\0';
		if (i == 1 && sbuf[0] == '_') {
                    sna = get_recent_setno();
                } else if (i == 1 && sbuf[0] == '$') {
                    sna = sn;
                } else {
		    sna = atoi(sbuf);
                }
		yylval.ival = sna;
		return SETNUM;
	    } else if (ctmp == 'R') {
	        sbuf[i] = '\0';
		rn = atoi(sbuf);
		if (rn >= 0 && rn < MAXREGION) {
		    yylval.ival = rn;
		    return REGNUM;
		} else {
                    errmsg("Invalid region number");
                }
	    }
	}
    }
    if (isalpha(c) || c == '$') {
	char *p = sbuf;

	do {
	    *p++ = c;
	} while ((c = getcharstr()) != EOF && (isalpha(c) || isdigit(c) ||
                  c == '_' || c == '$'));
	ungetchstr();
	*p = '\0';
#ifdef DEBUG
        if (get_debuglevel() == 2) {
	    printf("->%s<-\n", sbuf);
	}
#endif
	found = -1;
	if ((found = findf(key, sbuf)) >= 0) {
	    if (key[found].type == FITPARM) {
		int index = sbuf[1] - '0';
		yylval.ival = index;
		return FITPARM;
	    }
	    else if (key[found].type == FITPMAX) {
		int index = sbuf[1] - '0';
		yylval.ival = index;
		return FITPMAX;
	    }
	    else if (key[found].type == FITPMIN) {
		int index = sbuf[1] - '0';
		yylval.ival = index;
		return FITPMIN;
	    }

	    else if (key[found].type == KEY_VAR) {
		yylval.dptr = (double *) key[found].data;
		return VAR_D;
	    }
	    else if (key[found].type == KEY_VEC) {
		yylval.vrbl = (grarr *) key[found].data;
		return VEC_D;
	    }
	    else if (key[found].type == KEY_MAT) {
		yylval.matp = (grarr *) key[found].data;
		return MAT_D;
	    }

	    else if (key[found].type == FUNC_I) {
		yylval.ival = found;
		return FUNC_I;
	    }
	    else if (key[found].type == CONSTANT) {
		yylval.ival = found;
		return CONSTANT;
	    }
	    else if (key[found].type == UCONSTANT) {
		yylval.ival = found;
		return UCONSTANT;
	    }
	    else if (key[found].type == FUNC_D) {
		yylval.ival = found;
		return FUNC_D;
	    }
	    else if (key[found].type == FUNC_ND) {
		yylval.ival = found;
		return FUNC_ND;
	    }
	    else if (key[found].type == FUNC_DD) {
		yylval.ival = found;
		return FUNC_DD;
	    }
	    else if (key[found].type == FUNC_NND) {
		yylval.ival = found;
		return FUNC_NND;
	    }
	    else if (key[found].type == FUNC_PPD) {
		yylval.ival = found;
		return FUNC_PPD;
	    }
	    else if (key[found].type == FUNC_PPPD) {
		yylval.ival = found;
		return FUNC_PPPD;
	    }
	    else if (key[found].type == FUNC_PPPPD) {
		yylval.ival = found;
		return FUNC_PPPPD;
	    }
	    else if (key[found].type == FUNC_PPPPPD) {
		yylval.ival = found;
		return FUNC_PPPPPD;
	    }
	    else {
	        yylval.ival = key[found].type;
	        return key[found].type;
	    }
	} else {
	    yylval.sval = copy_string(NULL, sbuf);
	    return NEW_TOKEN;
	}
    }
    switch (c) {
    case '>':
	return follow('=', GE, GT);
    case '<':
	return follow('=', LE, LT);
    case '=':
	return follow('=', EQ, '=');
    case '!':
	return follow('=', NE, NOT);
    case '|':
	return follow('|', OR, '|');
    case '&':
	return follow('&', AND, '&');
    case '\n':
	return '\n';
    default:
	return c;
    }
}

static int follow(int expect, int ifyes, int ifno)
{
    int c = getcharstr();

    if (c == expect) {
	return ifyes;
    }
    ungetchstr();
    return ifno;
}

static void yyerror(char *s)
{
    char *buf;

    buf = copy_string(NULL, s);
    buf = concat_strings(buf, ": ");
    buf = concat_strings(buf, f_string);
    errmsg(buf);
    xfree(buf);
    interr = 1;
}

static int bad_gno (void)
{
  if (!is_valid_gno (gn)) {
    yyerror("No valid graph selected");
    return TRUE;
  }
  return FALSE;
}

static int bad_axis (void)
{
  if (!is_valid_axis (gn, naxis)) {
    yyerror("No valid axis selected");
    return TRUE;
  }
  return FALSE;
}

static int div_null (double div)
{
  if (div == 0.0) {
    yyerror("Divide by zero");
    return TRUE;
  }
  return FALSE;

}

static grarr *tmp_vec (grarr *src)
{
  grarr *retval;
  if (fcnt > MAX_GRARR_TMP-2) {
    yyerror ("Max number of temporary vectors exceeded");
    return NULL;
  }
  retval = &freelist[fcnt++];
  copy_grarr  (retval, src);
  retval->type = GRARR_TMP;
  return retval;
}

static grarr *tmp_mat (grarr *src)
{
  grarr *retval;
  if (matcnt > MAX_GRARR_TMP-2) {
    yyerror ("Max number of temporary matrices  exceeded");
    return NULL;
  }
  retval = &mattmp[fcnt++];
  copy_grarr (retval, src);
  retval->type = GRARR_TMP;
  return retval;
}
