/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.util;

import java.awt.Dimension;
import java.awt.Insets;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;

import java.util.ArrayList;

import junit.framework.TestCase;

/**
 * Tests JaxoGeometry.
 */
public class JaxoGeometryTest extends TestCase {

    /** Constructor: just calls super.
     * @param testName The name of this test.
     */
    public JaxoGeometryTest(String testName) {
        super(testName);
    }

    /**
     * Test of add method, of class net.sf.jaxodraw.util.JaxoGeometry.
     */
    public void testAdd() {
        final Rectangle r = new Rectangle(10, 10, 10, 10);

        JaxoGeometry.add(r, new Rectangle());
        assertEquals("Wrong add!", r, new Rectangle(10, 10, 10, 10));
        r.setBounds(10, 10, 10, 10);

        JaxoGeometry.add(r, new Rectangle(20, 10, 10, 10));
        assertEquals("Wrong add!", r, new Rectangle(10, 10, 20, 10));
        r.setBounds(10, 10, 10, 10);

        JaxoGeometry.add(new Rectangle(), r);
        assertEquals("Wrong add!", r, new Rectangle(10, 10, 10, 10));
        r.setBounds(10, 10, 10, 10);

        JaxoGeometry.add(r, 20, 10, 10, 10);
        assertEquals("Wrong add!", r, new Rectangle(10, 10, 20, 10));
        r.setBounds(10, 10, 10, 10);

        JaxoGeometry.add(r, 20, 10, 0, 0);
        assertEquals("Wrong add!", r, new Rectangle(10, 10, 10, 10));
    }

    /**
     * Test of clear method, of class net.sf.jaxodraw.util.JaxoGeometry.
     */
    public void testClear() {
        Rectangle2D r = new Rectangle(10, 10, 10, 10);
        JaxoGeometry.clear(r);
        assertTrue("Clear didn't clear!", r.isEmpty());
    }

    /**
     * Test of grow method, of class net.sf.jaxodraw.util.JaxoGeometry.
     */
    public void testGrow() {
        Rectangle r = new Rectangle(10, 10, 10, 10);
        Insets n = new Insets(2, 3, 4, 5);
        JaxoGeometry.grow(r, n);
        assertEquals("Wrong grow!", r, new Rectangle(7, 8, 18, 16));

        Dimension d = new Dimension(10, 10);
        JaxoGeometry.grow(d, n);
        assertEquals("Wrong grow!", d, new Dimension(18, 16));
    }

    /**
     * Test of shrink method, of class net.sf.jaxodraw.util.JaxoGeometry.
     */
    public void testShrink() {
        Rectangle r = new Rectangle(10, 10, 10, 10);
        Insets n = new Insets(2, 3, 4, 5);
        JaxoGeometry.shrink(r, n);
        assertEquals("Wrong shrink!", r, new Rectangle(13, 12, 2, 4));

        Dimension d = new Dimension(10, 10);
        JaxoGeometry.shrink(d, n);
        assertEquals("Wrong shrink!", d, new Dimension(2, 4));
    }

    /**
     * Test of getBounds method, of class net.sf.jaxodraw.util.JaxoGeometry.
     */
    public void testGetBounds() {
        ArrayList<Object> c = new ArrayList<Object>(5);
        assertNull(JaxoGeometry.getBounds(c));
        c.add(new Point(10, 20));
        assertNull(JaxoGeometry.getBounds(c));
        c.add(new Rectangle(10, 10, 10, 10));
        assertEquals("wrong bounds!", new Rectangle(10, 10, 10, 10),
            JaxoGeometry.getBounds(c));
        c.add(new Rectangle(20, 10, 10, 10));
        assertEquals("wrong bounds!", new Rectangle(10, 10, 20, 10),
            JaxoGeometry.getBounds(c));
    }

    /**
     * Test of scaledPoint method, of class net.sf.jaxodraw.util.JaxoGeometry.
     */
    public void testScaledPoint() {
        Point2D result = JaxoGeometry.scaledPoint(0.d, 0.d, 2.d, 1.d, 1.d);
        assertTrue(JaxoUtils.equal((float) result.getX(), 2.f));
        assertTrue(JaxoUtils.equal((float) result.getY(), 2.f));
    }

    /**
     * Test of curveParameter method, of class net.sf.jaxodraw.util.JaxoGeometry.
     */
    public void testCurveParameter() {
        float result = (float) JaxoGeometry.curveParameter(0.234f);
        assertTrue(JaxoUtils.equal(result, 0.234f));
        result = (float) JaxoGeometry.curveParameter(-0.0000001f);
        assertTrue(JaxoUtils.equal(result, 0.f));
        result = (float) JaxoGeometry.curveParameter(1.000001f);
        assertTrue(JaxoUtils.equal(result, 1.f));

        try {
            result = (float) JaxoGeometry.curveParameter(1.234f);
            fail("curveParameter should fail!");
        } catch (IllegalArgumentException e) {
            // expected
            return;
        }
    }

    /**
     * Test of translate method, of class net.sf.jaxodraw.util.JaxoGeometry.
     */
    public void testTranslate() {
        Point2D p = new Point(10, 10);
        JaxoGeometry.translate(p, 10.d, 10.d);
        assertTrue("Wrong translate!", p.getX() > 15.d);
        assertTrue("Wrong translate!", p.getY() > 15.d);
    }
}
