/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.object.vertex;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Point;
import java.awt.Rectangle;

import net.sf.jaxodraw.JaxoTestUtils;
import net.sf.jaxodraw.io.imports.JaxoAxodraw;
import net.sf.jaxodraw.object.JaxoObject;
import net.sf.jaxodraw.object.JaxoObjectFactory;
import net.sf.jaxodraw.object.loop.JaxoFLoop;
import net.sf.jaxodraw.util.JaxoColor;
import net.sf.jaxodraw.util.JaxoConstants;
import net.sf.jaxodraw.util.JaxoPrefs;


/** Tests JaxoVertexT6. */
public class JaxoVertexT6Test extends AbstractJaxoVertexTest {
    private JaxoVertexT6 vertex = new JaxoVertexT6();

    /** Constructor: just calls super().
     * @param testName The name of this test.
     */
    public JaxoVertexT6Test(String testName) {
        super(testName);
    }

    /** JUnit setup: initialize the object. */
    @Override
    protected void setUp() {
        super.setUp();
        init();
    }

    private void init() {
        vertex.setLocation(100, 100);
        vertex.setRelWAndH(100, 100);
        vertex.setStrokeWidth(1.f);
        vertex.setColor(JaxoColor.GREEN);
    }

    /** {@inheritDoc} */
    public JaxoObject getInstance() {
        return vertex;
    }

    /** Tests serialization. */
    public void testSerialization() {
        vertex.setColor(JaxoColor.RED);
        vertex.setStrokeWidth(2.f);
        JaxoTestUtils.checkSerializedEquality(vertex);
    }

    /** Tests that a copy of an object is still the same object. */
    public void testCopy() {
        vertex.setColor(JaxoColor.RED);
        vertex.setStrokeWidth(2.f);
        JaxoTestUtils.checkCopyEquality(vertex);

        JaxoVertexT6 vertex2 = new JaxoVertexT6();
        vertex2.setLocation(100, 100);
        vertex2.setRelWAndH(100, 100);
        vertex2.setStrokeWidth(1.f);
        vertex2.setColor(JaxoColor.GREEN);
        assertFalse(vertex2.isCopy(vertex));

        vertex2.setColor(JaxoColor.RED);
        vertex2.setStrokeWidth(2.f);
        assertTrue(vertex2.isCopy(vertex));

        JaxoFLoop loop = new JaxoFLoop();
        assertFalse(vertex.isCopy(loop));

    }

    /** Tests identity. */
    public void testIdentity() {
        JaxoVertexT6 vertex2 = vertex;
        assertEquals(vertex, vertex2);
    }

    /** Tests the latex command. */
    public void testLatexCommand() {
        Dimension dim = new Dimension(100, 100);

        vertex.setRelWAndH(50, 0);
        String expected =
            "\\CTri(29.289,0)(100,70.711)(170.711,0){Green}{Green}\\CTri(29.289,0)(100,-70.711)(170.711,0){Green}{Green}";
        String actual = vertex.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        expected = "%";
        actual = vertex.latexCommand(100.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

    }

    /** Tests axodraw. */
    public void testAxodraw() {
        JaxoAxodraw axo = new JaxoAxodraw();
        String command;

        command =
            "\\CTri(29.29,41)(100,111.71)(170.71,41){Black}{Black}\\CTri(29.29,41)(100,-29.71)(170.71,41){Black}{Black}";
        vertex = (JaxoVertexT6) axo.getJaxoObject(command);
        checkVertexParameters(JaxoColor.BLACK);

    }

    private void checkVertexParameters(Color color) {
        assertEquals("VertexT6 doesn't have the right color!", color,
            vertex.getColor());
        assertEquals("VertexT6 doesn't have the right relw!", 36,
            vertex.getRelw());
        assertEquals("VertexT6 doesn't have the right relh!", 36,
            vertex.getRelh());
        assertEquals("VertexT6 doesn't have the right x!", 100, vertex.getX());
        assertEquals("VertexT6 doesn't have the right y!", -41, vertex.getY());
    }

    /** Tests the getBounds() method. */
    public void testGetBounds() {
        Rectangle bBox = vertex.getBounds();
        Rectangle expected = new Rectangle(-101, -101, 402, 402);
        assertEquals("Bounding box not correct!", expected, bBox);

        assertNotNull(vertex.getBounds2D());
    }

    /** Tests createNewObject in JaxoObjectFactory. */
    public void testCreateNewObject() {
        Point p = new Point(50, 50);
        JaxoObject ob =
            JaxoObjectFactory.newObject(JaxoConstants.VERTEXT6, new Point[] {p, p});

        assertTrue("createNewObject did not return a VertexT6!",
            (ob instanceof JaxoVertexT6));

        vertex = (JaxoVertexT6) ob;
        assertEquals("VertexT6 doesn't have the right x!", 50, vertex.getX());
        assertEquals("VertexT6 doesn't have the right y!", 50, vertex.getY());
        assertEquals("VertexT6 doesn't have the right color!",
            JaxoColor.getColor(JaxoPrefs.getStringPref(JaxoPrefs.PREF_COLOR)),
            vertex.getColor());
    }

    /** {@inheritDoc} */
    public String className() {
        return "JaxoVertexT6";
    }

    /** {@inheritDoc} */
    public void testJaxo20Compat() {
        JaxoVertexT6 object =
                (JaxoVertexT6) instanceFrom20OpenGraph(JaxoVertexT6.class);
        assertNotNull(object);
        check20Object("opened wrong object!", object);

        object =
                (JaxoVertexT6) instanceFrom20ImportGraph(JaxoVertexT6.class);
        assertNotNull(object);
        check20Object("imported wrong object!", object);
    }

    private void check20Object(String message, JaxoVertexT6 object) {
        assertEquals(message, 483, object.getX());
        assertEquals(message, 456, object.getY());
        assertEquals(message, 21, (int) Math.round(object.getRadius()));

        assertEquals(message, JaxoColor.TAN, object.getColor());
    }
}
