/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.object.text;

import java.awt.Dimension;
import java.awt.Point;
import java.awt.Rectangle;

import net.sf.jaxodraw.JaxoTestUtils;
import net.sf.jaxodraw.io.imports.JaxoAxodraw;
import net.sf.jaxodraw.object.JaxoObject;
import net.sf.jaxodraw.object.JaxoObjectFactory;
import net.sf.jaxodraw.util.JaxoColor;
import net.sf.jaxodraw.util.JaxoConstants;
import net.sf.jaxodraw.util.JaxoPrefs;


/** Tests JaxoLatexText. */
public class JaxoLatexTextTest extends AbstractJaxoTextObjectTest {
    private JaxoLatexText text = new JaxoLatexText();

    /** Constructor: just calls super().
     * @param testName The name of this test.
     */
    public JaxoLatexTextTest(String testName) {
        super(testName);
    }

    /** JUnit setup: initialize the object. */
    @Override
    protected void setUp() {
        super.setUp();
        init();
    }

    private void init() {
        text.setLocation(100, 100);
        text.setColor(JaxoColor.BLACK);
        text.setRotationAngle(0);
        text.setTextString("Feynman");
        text.setAllign(JaxoLatexText.ALIGN_LB);
        text.setLatexTextSize(JaxoLatexText.XLARGE);
    }

    /** {@inheritDoc} */
    public JaxoObject getInstance() {
        return text;
    }

    /** Tests serialization. */
    public void testSerialization() {
        text.setRotationAngle(15);
        text.setTextString("blabla");
        text.setAllign(JaxoLatexText.ALIGN_CB);
        text.setLatexTextSize(JaxoLatexText.SMALL);
        JaxoTestUtils.checkSerializedEquality(text);
    }

    /** Tests that a copy of an object is still the same object. */
    public void testCopy() {
        JaxoLatexText text2 = new JaxoLatexText();
        text2.setState(text);

        text.setRotationAngle(15);
        text.setTextString("blabla");
        text.setAllign(JaxoLatexText.ALIGN_CB);
        text.setLatexTextSize(JaxoLatexText.SMALL);
        JaxoTestUtils.checkCopyEquality(text);

        assertFalse(text2.isCopy(text));

        text2.setRotationAngle(15);
        text2.setTextString("blabla");
        text2.setAllign(JaxoLatexText.ALIGN_CB);
        text2.setLatexTextSize(JaxoLatexText.SMALL);
        assertTrue(text2.isCopy(text));

        JaxoLatexText loop = new JaxoLatexText();
        assertFalse(text.isCopy(loop));

    }

    /** Tests identity. */
    public void testIdentity() {
        JaxoLatexText text2 = text;
        assertEquals(text, text2);
    }

    /** Tests the latex command. */
    public void testLatexCommand() {
        Dimension dim = new Dimension(30, 16);

        String expected = "\\Text(100,-84)[]{\\small{\\Black{$Feynman$}}}";
        text.setLatexTextSize(JaxoLatexText.SMALL);
        text.setAllign(JaxoLatexText.ALIGN_CC);
        String actual = text.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        text.setRotationAngle(15);
        text.setLatexTextSize(JaxoLatexText.HUGE);
        text.setAllign(JaxoLatexText.ALIGN_RT);
        expected =
            "\\Text(100,-84)[rt]{\\huge{\\Black{$\\rput[rt]{-15}{Feynman}$}}}";
        actual = text.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        text.setAllign(JaxoLatexText.ALIGN_CC);
        expected =
            "\\Text(100,-84)[]{\\huge{\\Black{$\\rput{-15}{Feynman}$}}}";
        actual = text.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        text.setColor(JaxoColor.GRAYSCALE15);
        expected = "\\Text(100,-84)[]{\\huge{\\Gray{$\\rput{-15}{Feynman}$}}}";
        actual = text.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);
    }

    /** Tests axodraw. */
    public void testAxodraw() {
        JaxoAxodraw axo = new JaxoAxodraw();
        String command;

        command = "\\Text(100,-84)[]{\\small{\\Black{$Feynman$}}}";
        text = (JaxoLatexText) axo.getJaxoObject(command);
        checkTextParameters(JaxoLatexText.ALIGN_CC, JaxoLatexText.SMALL, "Feynman");

        command =
            "\\Text(100,-76)[rt]{\\huge{\\Black{$\\rput[rt]{-15}{blabla}$}}}";
        text = (JaxoLatexText) axo.getJaxoObject(command);
        checkTextParameters(JaxoLatexText.ALIGN_RT, JaxoLatexText.HUGE, "blabla");

        command = "\\Text(100,-76)[]{\\huge{\\Black{$\\rput{-15}{blabla}$}}}";
        text = (JaxoLatexText) axo.getJaxoObject(command);
        checkTextParameters(JaxoLatexText.ALIGN_CC, JaxoLatexText.HUGE, "blabla");

    }

    private void checkTextParameters(int allign, int fs, String test) {
        assertEquals("Mismatch in LatexText allign!", allign, text.getAllign());
        assertEquals("Mismatch in LatexText font size!", fs, text.getLatexTextSize());
        assertEquals("Mismatch in LatexText text string!", test,
            text.getTextString());

    }

    /** Tests the getBounds() method. */
    public void testGetBounds() {
        Rectangle bBox = text.getBounds();
        Rectangle expected = new Rectangle(103, 79, 32, 16);
        assertEquals("Bounding box not correct!", expected, bBox);
        text.setRotationAngle(45);
        bBox = text.getBounds();
        expected = new Rectangle(105, 87, 35, 35);
        assertEquals("Rotated bounding box not correct!", expected, bBox);

        assertNotNull(text.getBounds2D());

        assertEquals(0, text.getWidth());
        assertEquals(0, text.getHeight());
    }


    /** Tests isRolledOver. */
    public void testIsRolledOver() {
        text.setAllign(JaxoLatexText.ALIGN_LT);
        assertTrue("Latex text is not rolled over! 0", text.isRolledOver(105, 102));

        text.setAllign(JaxoLatexText.ALIGN_LC);
        assertTrue("Latex text is not rolled over! 1", text.isRolledOver(105, 92));

        text.setAllign(JaxoLatexText.ALIGN_LB);
        assertTrue("Latex text is not rolled over! 2", text.isRolledOver(105, 82));

        text.setAllign(JaxoLatexText.ALIGN_CT);
        assertTrue("Latex text is not rolled over! 3", text.isRolledOver(86, 102));

        text.setAllign(JaxoLatexText.ALIGN_CC);
        assertTrue("Latex text is not rolled over! 4", text.isRolledOver(86, 92));

        text.setAllign(JaxoLatexText.ALIGN_CB);
        assertTrue("Latex text is not rolled over! 5", text.isRolledOver(86, 82));

        text.setAllign(JaxoLatexText.ALIGN_RT);
        assertTrue("Latex text is not rolled over! 6", text.isRolledOver(68, 102));

        text.setAllign(JaxoLatexText.ALIGN_RC);
        assertTrue("Latex text is not rolled over! 7", text.isRolledOver(68, 92));

        text.setAllign(JaxoLatexText.ALIGN_RB);
        assertTrue("Latex text is not rolled over! 8", text.isRolledOver(68, 82));

        assertFalse("Latex text is rolled over!", text.isRolledOver(50, 98));
    }

    /** Tests createNewObject in JaxoObjectFactory. */
    public void testCreateNewObject() {
        Point p = new Point(50, 50);
        JaxoObject ob =
            JaxoObjectFactory.newObject(JaxoConstants.LATEX, new Point[] {p});

        assertTrue("createNewObject did not return a LatexText!",
            (ob instanceof JaxoLatexText));

        text = (JaxoLatexText) ob;
        assertEquals("LatexText doesn't have the right x!", 50, text.getX());
        assertEquals("LatexText doesn't have the right y!", 50, text.getY());
        assertEquals("LatexText doesn't have the right color!",
            JaxoColor.getColor(JaxoPrefs.getStringPref(JaxoPrefs.PREF_COLOR)),
            text.getColor());
        assertEquals("LatexText doesn't have the right text!",
            "", text.getTextString());
    }

    /** {@inheritDoc} */
    public String className() {
        return "JaxoLatexText";
    }

    /** {@inheritDoc} */
    public void testJaxo20Compat() {
        JaxoLatexText object =
                (JaxoLatexText) instanceFrom20OpenGraph(JaxoLatexText.class);
        assertNotNull(object);
        check20Object("opened wrong object!", object);

        object =
                (JaxoLatexText) instanceFrom20ImportGraph(JaxoLatexText.class);
        assertNotNull(object);
        check20Object("imported wrong object!", object);
    }

    private void check20Object(String message, JaxoLatexText object) {
        assertEquals(message, 521, object.getX());
        assertEquals(message, 355, object.getY());

        assertEquals(message, "\\vec{r}_i", object.getTextString());
        assertEquals(message, 7, object.getRotationAngle());
        assertEquals(message, JaxoLatexText.ALIGN_RT, object.getAllign());
        assertEquals(message, JaxoLatexText.HUGE, object.getLatexTextSize());

        assertEquals(message, JaxoColor.ROYAL_BLUE, object.getColor());
    }

    /** {@inheritDoc} */
    public String[] serializableProperties() {
        return new String[] {
            "allign", "color", "latexTextSize", "pointCount", "points",
            "rotationAngle", "textString"
        }; // 7
    }
}
