/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.object.loop;

import java.awt.Dimension;
import java.awt.Point;
import java.awt.Rectangle;

import net.sf.jaxodraw.JaxoTestUtils;
import net.sf.jaxodraw.io.imports.JaxoAxodraw;
import net.sf.jaxodraw.object.JaxoObject;
import net.sf.jaxodraw.object.JaxoObjectFactory;
import net.sf.jaxodraw.object.arc.JaxoFArc;
import net.sf.jaxodraw.util.JaxoColor;
import net.sf.jaxodraw.util.JaxoConstants;
import net.sf.jaxodraw.util.JaxoPrefs;


/** Tests JaxoGlLoop. */
public class JaxoGlLoopTest extends AbstractJaxoLoopObjectTest {
    private JaxoGlLoop loop = new JaxoGlLoop();

    /** Constructor: just calls super().
     * @param testName The name of this test.
     */
    public JaxoGlLoopTest(String testName) {
        super(testName);
    }

    /** JUnit setup: initialize the object. */
    @Override
    protected void setUp() {
        super.setUp();
        init();
    }

    private void init() {
        loop.setLocation(67, 100);
        loop.setRelWAndH(-33, 39);
        loop.setColor(JaxoColor.RED);
        loop.setStrokeWidth(1.f);
        loop.setAmp(15);
        loop.setWiggles(17);
        loop.setDoubleLine(false);
        loop.setDLSeparation(2.f);
    }

    /** {@inheritDoc} */
    public JaxoObject getInstance() {
        return loop;
    }

    /** Tests serialization. */
    public void testSerialization() {
        loop.setStrokeWidth(1.6f);
        loop.setAmp(25);
        loop.setDoubleLine(true);
        JaxoTestUtils.checkSerializedEquality(loop);
    }

    /** Tests that a copy of an object is still the same object. */
    public void testCopy() {
        JaxoGlLoop loop2 = new JaxoGlLoop();
        loop2.setState(loop);

        loop.setStrokeWidth(1.6f);
        loop.setAmp(25);
        loop.setDoubleLine(true);
        JaxoTestUtils.checkCopyEquality(loop);

        assertFalse(loop2.isCopy(loop));

        loop2.setStrokeWidth(1.6f);
        loop2.setAmp(25);
        loop2.setDoubleLine(true);
        assertTrue(loop2.isCopy(loop));

        JaxoFArc arc = new JaxoFArc();
        assertFalse(loop.isCopy(arc));

    }

    /** Tests identity. */
    public void testIdentity() {
        JaxoGlLoop loop2 = loop;
        assertEquals(loop, loop2);
    }

    /** Tests the latex command. */
    public void testLatexCommand() {
        Dimension dim = new Dimension(119, 119);

        String expected = "\\GluonArc(67,19)(51.088,-130,230){7.5}{17}";
        String actual = loop.latexCommand(1.f, dim);

        assertEquals("Latex commands do not match!", expected, actual);

        // reset to initial state
        loop.setPaintArrow(false);
        loop.setFlip(false);

        loop.setDoubleLine(true);
        loop.setDLSeparation(4.f);
        loop.setRelWAndH(-33, -39);

        expected = "\\GluonArc[double,sep=4](67,19)(51.088,130,490){7.5}{17}";
        actual = loop.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        loop.setRelWAndH(0, 0);
        expected = "%";
        actual = loop.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

    }

    /** Tests axodraw. */
    public void testAxodraw() {
        JaxoAxodraw axo = new JaxoAxodraw();
        axo.checkRequired("\\begin{picture}(119,119) (40,-40)");
        JaxoObject ob = axo.getJaxoObject("\\SetWidth{1.0}");
        assertNull(ob);
        String command;

        command = "\\GlueArc(100,19)(51.09,-130,230){7.5}{17}";
        loop = (JaxoGlLoop) axo.getJaxoObject(command);
        checkLoopParameters(false);

        axo.checkRequired("\\begin{picture}(126,126) (36,-36)");

        command =
            "\\GlueArc(100,26)(54.09,-130,230){7.5}{17}\\GlueArc(100,26)(48.09,-130,230){7.5}{17}%%JaxoDrawID:DoubleLine(3)";
        loop = (JaxoGlLoop) axo.getJaxoObject(command);
        checkLoopParameters(true);

    }

    private void checkLoopParameters(boolean doubleLine) {
        assertEquals("GlLoop doesn't have the right color!", JaxoColor.BLACK,
            loop.getColor());
        assertEquals("GlLoop doesn't have the right amp!", 15, loop.getAmp());
        assertEquals("GlLoop doesn't have the right wiggles!", 17, loop.getWiggles());
        assertEquals("GlLoop doesn't have the right x!", 100, loop.getX());
        assertEquals("GlLoop doesn't have the right y!", 100, loop.getY());
        assertEquals("Mismatch in GlLoop doubleLine!", doubleLine,
            loop.isDoubleLine());

        float dlsep = 2.f;

        if (doubleLine) {
            dlsep = 3.f;
        }

        assertTrue("Mismatch in GlLoop doubleLine separation!",
            loop.dlSepIs(dlsep));
    }

    /** Tests the getBounds() method. */
    public void testGetBounds() {
        Rectangle bBox = loop.getBounds();
        Rectangle expected = new Rectangle(7, 40, 120, 120);
        assertEquals("Bounding box not correct!", expected, bBox);
        loop.setDoubleLine(true);
        bBox = loop.getBounds();
        expected.setRect(6, 39, 122, 122);
        assertEquals("DL Bounding box not correct!", expected, bBox);

        assertNotNull(loop.getBounds2D());
    }

    /** Tests createNewObject in JaxoObjectFactory. */
    public void testCreateNewObject() {
        Point p = new Point(50, 50);
        JaxoObject ob =
            JaxoObjectFactory.newObject(JaxoConstants.GLLOOP, new Point[] {p, p});

        assertTrue("createNewObject did not return a GlLoop!",
            (ob instanceof JaxoGlLoop));

        loop = (JaxoGlLoop) ob;
        assertEquals("GlLoop doesn't have the right x!", 50, loop.getX());
        assertEquals("GlLoop doesn't have the right y!", 50, loop.getY());
        assertEquals("GlLoop doesn't have the right color!",
            JaxoColor.getColor(JaxoPrefs.getStringPref(JaxoPrefs.PREF_COLOR)),
            loop.getColor());
        assertTrue("GlLoop doesn't have the right stroke!",
            loop.strokeIs(JaxoPrefs.getFloatPref(JaxoPrefs.PREF_LINEWIDTH)));
        assertEquals("GlLoop doesn't have the right double line!", false,
            loop.isDoubleLine());
        assertTrue("GlLoop doesn't have the right dlsep!",
            loop.dlSepIs(JaxoPrefs.getFloatPref(JaxoPrefs.PREF_DLSEP)));
        assertEquals("GlLoop doesn't have the right amplitude!",
            JaxoPrefs.getIntPref(JaxoPrefs.PREF_AMPLITUDE), loop.getAmp());
    }

    /** {@inheritDoc} */
    public String className() {
        return "JaxoGlLoop";
    }

    /** {@inheritDoc} */
    public void testJaxo20Compat() {
        JaxoGlLoop object = (JaxoGlLoop) instanceFrom20OpenGraph(JaxoGlLoop.class);
        assertNotNull(object);
        check20Object("opened wrong object!", object);

        object = (JaxoGlLoop) instanceFrom20ImportGraph(JaxoGlLoop.class);
        assertNotNull(object);
        check20Object("imported wrong object!", object);
    }

    private void check20Object(String message, JaxoGlLoop object) {
        assertEquals(message, 229, object.getX());
        assertEquals(message, 462, object.getY());
        assertEquals(message, 35, (int) Math.round(object.getRadius()));

        assertFalse(message, object.isDoubleLine());
        assertTrue(message, object.strokeIs(1.4f));

        assertEquals(message, JaxoColor.SPRING_GREEN, object.getColor());

        assertEquals(message, 10, object.getAmp());
        assertEquals(message, 14, object.getWiggles());
    }
}
