/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.object.loop;

import java.awt.Point;
import java.awt.Rectangle;
import net.sf.jaxodraw.object.AbstractJaxoParticleObjectTest;
import net.sf.jaxodraw.object.arrow.JaxoArrow;
import net.sf.jaxodraw.util.JaxoConstants;


/**
 * Base Test for JaxoLoopObjects.
 */
public abstract class AbstractJaxoLoopObjectTest
    extends AbstractJaxoParticleObjectTest {

    private JaxoLoopObject instance;

    /**
     * Constructor.
     *
     * @param testName The name of this test.
     */
    protected AbstractJaxoLoopObjectTest(String testName) {
        super(testName);
    }

    /** JUnit setup: get a new instance of the object. */
    @Override
    protected void setUp() {
        super.setUp();
        this.instance = (JaxoLoopObject) getInstance();
    }

      //
     // tests
    //

    /** Tests moveBy. */
    public void testMoveBy() {
        instance.setLocation(100, 100, 200, 200);
        instance.moveBy(23, -45);

        assertEquals("Wrong moveBy x!", 123, instance.getX());
        assertEquals("Wrong moveBy y!", 55, instance.getY());
        assertEquals("Wrong moveBy x2!", 223, instance.getX2());
        assertEquals("Wrong moveBy y2!", 155, instance.getY2());
    }

    /** Tests rescaleObject. */
    public void testRescaleObject() {
        instance.setLocation(100, 100, 200, 100);
        instance.rescaleObject(instance.getX(), instance.getY(), 2.f);
        assertEquals("Loop x didn't get rescaled right!", 200, instance.getRelw());
        assertEquals("Loop y didn't get rescaled right!", 0, instance.getRelh());

        instance.setLocation(50, 50, 100, 100);
        instance.setAmp(5);
        instance.setWiggles(25);
        instance.rescaleObject(instance.getX(), instance.getY(), 3.f);
        assertEquals("Loop amp wrongly rescaled!", 15, instance.getAmp());
        assertEquals("Loop wiggles wrongly rescaled!", 75, instance.getWiggles());
    }

    /** Tests getGrabbedHandle. */
    public void testGetGrabbedHandle() {
        instance.setLocation(100, 100, 200, 200);

        instance.setRelWAndH(50, 0);
        checkHandle(150, 100, JaxoLoopObject.SELECT_P2);
        checkHandle(500, 500, JaxoLoopObject.SELECT_NONE);

        instance.setRelWAndH(-50, 0);
        checkHandle(100, 100, JaxoLoopObject.SELECT_P1);
        checkHandle(50, 100, JaxoLoopObject.SELECT_P2);
        checkHandle(500, 500, JaxoLoopObject.SELECT_NONE);

    }

    /** Tests canBeSelected. */
    public void testCanBeSelected() {
        instance.setLocation(100, 100, 200, 200);

        int mode = JaxoConstants.MOVE;
        int handle = JaxoLoopObject.SELECT_P1;
        assertTrue("P1 in MOVE mode", instance.canBeSelected(handle, mode));
        handle = JaxoLoopObject.SELECT_P2;
        assertTrue("P2 in MOVE mode", instance.canBeSelected(handle, mode));

        mode = JaxoConstants.RESIZE;
        handle = JaxoLoopObject.SELECT_P1;
        assertFalse("P1 in RESIZE mode", instance.canBeSelected(handle, mode));
        handle = JaxoLoopObject.SELECT_P2;
        assertTrue("P2 in RESIZE mode", instance.canBeSelected(handle, mode));
    }

    /** Tests smallestDistanceTo. */
    public void testSmallestDistanceTo() {
        instance.setLocation(100, 100, 200, 200);

        float dist = instance.smallestDistanceTo(103, 104);
        assertEquals("Wrong distance", dist, 5.f);
        dist = instance.smallestDistanceTo(203, 204);
        assertEquals("Wrong distance", dist, 5.f);
        dist = instance.smallestDistanceTo(150, 200);
        assertEquals("Wrong distance", dist, 50.f);
    }

    /** Tests all getters/setters. */
    public void testGetSetXY() {
        assertEquals("Wrong PointCount!", 2, instance.getPointCount());
        try {
            instance.setPointCount(3);
            fail("setPointCount(3) should throw IllegalArgumentException");
        } catch (IllegalArgumentException e) {
            // expected, now check it doesn't throw with proper value
            instance.setPointCount(2);
        }
    }

    /** {@inheritDoc} */
    public void testContains() {
        instance.setLocation(100, 100, 200, 200);

        assertTrue(getInstance().contains(75, 75));
        assertTrue(getInstance().contains(5, 5));
        assertFalse(getInstance().contains(325, 325));

        assertTrue(getInstance().contains(75, 75, 5, 5));
        assertTrue(getInstance().contains(5, 5, 5, 5));
        assertFalse(getInstance().contains(325, 325, 5, 5));

        assertTrue(getInstance().contains(new Point(75, 75)));
        assertFalse(getInstance().contains(new Point(325, 325)));

        assertTrue(getInstance().contains(new Rectangle(75, 75, 5, 5)));
        assertFalse(getInstance().contains(new Rectangle(325, 325, 5, 5)));
    }

    /** {@inheritDoc} */
    public void testIntersects() {
        instance.setLocation(100, 100, 200, 200);

        assertTrue(getInstance().intersects(5, 5, 100, 100));
        assertFalse(getInstance().intersects(325, 325, 1, 1));

        assertTrue(getInstance().intersects(new Rectangle(130, 130)));
        assertFalse(getInstance().intersects(new Rectangle(330, 330, 5, 5)));
    }

    /** {@inheritDoc} */
    public void testArrowCoordinates() {
        instance.setLocation(100, 100, 200, 200);
        instance.setArrowPosition(0.5f);

        JaxoArrow.Coordinates arrow = instance.arrowCoordinates();
        assertEquals(200, (int) Math.round(arrow.getX()));
        assertEquals(200, (int) Math.round(arrow.getY()));
        assertEquals((float) (-Math.PI / 4.d), (float) arrow.getAngle());

        instance.setFlip(true);
        arrow = instance.arrowCoordinates();
        assertEquals(200, (int) Math.round(arrow.getX()));
        assertEquals(200, (int) Math.round(arrow.getY()));
        assertEquals((float) (3.d * Math.PI / 4.d), (float) arrow.getAngle());
    }

      //
     // abstract
    //

    /** Tests serialization. */
    public abstract void testSerialization();

}
