/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.object.line;

import java.awt.Dimension;
import java.awt.Point;
import java.awt.Rectangle;

import net.sf.jaxodraw.JaxoTestUtils;
import net.sf.jaxodraw.io.imports.JaxoAxodraw;
import net.sf.jaxodraw.object.JaxoObject;
import net.sf.jaxodraw.object.JaxoObjectFactory;
import net.sf.jaxodraw.object.loop.JaxoFLoop;
import net.sf.jaxodraw.util.JaxoColor;
import net.sf.jaxodraw.util.JaxoConstants;
import net.sf.jaxodraw.util.JaxoPrefs;


/** Tests JaxoPLine. */
public class JaxoPLineTest extends AbstractJaxoLineObjectTest {
    private JaxoPLine line = new JaxoPLine();

    /** Constructor: just calls super().
     * @param testName The name of this test.
     */
    public JaxoPLineTest(String testName) {
        super(testName);
    }

    /** JUnit setup: initialize the object. */
    @Override
    protected void setUp() {
        super.setUp();
        init();
    }

    private void init() {
        line.setLocation(100, 100);
        line.setRelWAndH(100, 100);
        line.setStrokeWidth(1.f);
        line.setColor(JaxoColor.GREEN);
        line.setAmp(15);
        line.setWiggles(12);
        line.setDoubleLine(false);
        line.setDLSeparation(2.f);
    }

    /** {@inheritDoc} */
    public JaxoObject getInstance() {
        return line;
    }

    /** Tests serialization. */
    public void testSerialization() {
        line.setAmp(20);
        line.setDoubleLine(true);
        line.setDLSeparation(4.f);
        JaxoTestUtils.checkSerializedEquality(line);
    }

    /** Tests that a copy of an object is still the same object. */
    public void testCopy() {
        JaxoPLine line2 = new JaxoPLine();
        line2.setState(line);

        line.setAmp(20);
        line.setDoubleLine(true);
        line.setDLSeparation(4.f);
        JaxoTestUtils.checkCopyEquality(line);

        assertFalse(line2.isCopy(line));

        line2.setAmp(20);
        line2.setDoubleLine(true);
        line2.setDLSeparation(4.f);
        assertTrue(line2.isCopy(line));

        JaxoFLoop loop = new JaxoFLoop();
        assertFalse(line.isCopy(loop));

    }

    /** Tests identity. */
    public void testIdentity() {
        JaxoPLine line2 = line;
        assertEquals(line, line2);
    }

    /** Tests the latex command. */
    public void testLatexCommand() {
        Dimension dim = new Dimension(100, 112);

        String expected = "\\Photon(100,12)(200,-88){7.5}{12}";
        String actual = line.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        dim.setSize(106, 106);
        line.setSymmetric(true);
        expected = "\\Photon(100,6)(200,-94){7.5}{12.5}";
        actual = line.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        dim.setSize(100, 112);
        line.setSymmetric(false);
        line.setRelWAndH(-100, -100);
        expected = "\\Photon(100,12)(0,112){7.5}{12}";
        actual = line.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        line.setDoubleLine(true);
        line.setDLSeparation(4.f);

        expected = "\\Photon[double,sep=4](100,12)(0,112){7.5}{12}";
        actual = line.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        line.setRelWAndH(0, 0);
        expected = "%";
        actual = line.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

    }

    /** Tests axodraw. */
    public void testAxodraw() {
        JaxoAxodraw axo = new JaxoAxodraw();
        JaxoObject ob = axo.getJaxoObject("\\SetWidth{1.0}");
        assertNull(ob);
        String command;

        command = "\\Photon(100,12)(200,-88){7.5}{6}";
        line = (JaxoPLine) axo.getJaxoObject(command);
        checkLineParameters(line, false);

        command =
            "\\Photon(102.12,-79.88)(202.12,-179.88){7.5}{6}\\Photon(97.88,-84.12)(197.88,-184.12){7.5}{6}%%JaxoDrawID:DoubleLine(3)";
        line = (JaxoPLine) axo.getJaxoObject(command);
        checkLineParameters(line, true);

    }

    private void checkLineParameters(JaxoPLine aline, boolean doubleLine) {
        assertEquals("PLine doesn't have the right color!", JaxoColor.BLACK,
            aline.getColor());
        assertTrue("PLine doesn't have the right stroke!", aline.strokeIs(1.f));
        assertEquals("PLine doesn't have the right amp!", 15, aline.getAmp());
        assertEquals("PLine doesn't have the right wiggles!", 6, aline.getWiggles());
        assertEquals("Mismatch in PLine doubleLine!", doubleLine,
            aline.isDoubleLine());

        float dlsep = 2.f;
        int x = 100;
        int y = -12;
        if (doubleLine) {
            dlsep = 3.f;
            x = 100;
            y = 82;
        }

        assertEquals("PLine doesn't have the right x!", x, aline.getX());
        assertEquals("PLine doesn't have the right y!", y, aline.getY());
        assertTrue("Mismatch in PLine doubleLine separation!",
            aline.dlSepIs(dlsep));

    }

    /** Tests the getBounds() method. */
    public void testGetBounds() {
        Rectangle bBox = line.getBounds();
        Rectangle expected = new Rectangle(99, 96, 102, 108);
        assertEquals("Bounding box not correct!", expected, bBox);
        line.setDoubleLine(true);
        bBox = line.getBounds();
        expected.setRect(99, 95, 103, 110);
        assertEquals("DL Bounding box not correct!", expected, bBox);

        assertNotNull(line.getBounds2D());
    }

    /** Tests createNewObject in JaxoObjectFactory. */
    public void testCreateNewObject() {
        Point p = new Point(50, 50);
        JaxoObject ob =
            JaxoObjectFactory.newObject(JaxoConstants.PLINE, new Point[] {p, p});

        assertTrue("createNewObject did not return a PLine!",
            (ob instanceof JaxoPLine));

        line = (JaxoPLine) ob;
        assertEquals("PLine doesn't have the right x!", 50, line.getX());
        assertEquals("PLine doesn't have the right y!", 50, line.getY());
        assertEquals("PLine doesn't have the right color!",
            JaxoColor.getColor(JaxoPrefs.getStringPref(JaxoPrefs.PREF_COLOR)),
            line.getColor());
        assertTrue("PLine doesn't have the right stroke!",
            line.strokeIs(JaxoPrefs.getFloatPref(JaxoPrefs.PREF_LINEWIDTH)));
        assertEquals("PLine doesn't have the right double line!", false,
            line.isDoubleLine());
        assertTrue("PLine doesn't have the right dlsep!",
            line.dlSepIs(JaxoPrefs.getFloatPref(JaxoPrefs.PREF_DLSEP)));
        assertEquals("PLine doesn't have the right amplitude!",
            JaxoPrefs.getIntPref(JaxoPrefs.PREF_AMPLITUDE), line.getAmp());
    }

    /** {@inheritDoc} */
    public String className() {
        return "JaxoPLine";
    }

    /** {@inheritDoc} */
    public void testJaxo20Compat() {
        JaxoPLine object = (JaxoPLine) instanceFrom20OpenGraph(JaxoPLine.class);
        assertNotNull(object);
        check20Object("opened wrong object!", object);

        object = (JaxoPLine) instanceFrom20ImportGraph(JaxoPLine.class);
        assertNotNull(object);
        check20Object("imported wrong object!", object);
    }

    private void check20Object(String message, JaxoPLine object) {
        assertEquals(message, 47, object.getX());
        assertEquals(message, 286, object.getY());
        assertEquals(message, 116, object.getX2());
        assertEquals(message, 336, object.getY2());

        assertFalse(message, object.isDoubleLine());
        assertTrue(message, object.strokeIs(1.2f));

        assertEquals(message, JaxoColor.RAWSIENNA, object.getColor());

        assertEquals(message, 10, object.getAmp());
        assertEquals(message, 5, object.getWiggles());
        assertTrue(message, object.isSymmetric());
    }

    /** {@inheritDoc} */
    @Override
    public String[] serializableProperties() {
        return new String[] {
            "DLSeparation", "amp", "arrow", "arrowPosition", "color",
            "dash", "doubleLine", "flip", "paintArrow", "pointCount",
            "points", "strokeWidth", "wiggles", "symmetric"
        }; // 14
    }
}
