/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.object.line;

import java.awt.Dimension;
import java.awt.Point;
import java.awt.Rectangle;

import net.sf.jaxodraw.JaxoTestUtils;
import net.sf.jaxodraw.io.imports.JaxoAxodraw;
import net.sf.jaxodraw.object.JaxoObject;
import net.sf.jaxodraw.object.JaxoObjectFactory;
import net.sf.jaxodraw.object.loop.JaxoFLoop;
import net.sf.jaxodraw.util.JaxoColor;
import net.sf.jaxodraw.util.JaxoConstants;
import net.sf.jaxodraw.util.JaxoPrefs;


/** Tests JaxoGlLine. */
public class JaxoGlLineTest extends AbstractJaxoLineObjectTest {
    private JaxoGlLine line = new JaxoGlLine();

    /** Constructor: just calls super().
     * @param testName The name of this test.
     */
    public JaxoGlLineTest(String testName) {
        super(testName);
    }

    /** JUnit setup: initialize the object. */
    @Override
    protected void setUp() {
        super.setUp();
        init();
    }

    private void init() {
        line.setLocation(100, 100);
        line.setRelWAndH(100, 100);
        line.setStrokeWidth(1.f);
        line.setColor(JaxoColor.GREEN);
        line.setAmp(15);
        line.setWiggles(17);
        line.setDoubleLine(false);
        line.setDLSeparation(2.f);
    }

    /** {@inheritDoc} */
    public JaxoObject getInstance() {
        return line;
    }

    /** Tests serialization. */
    public void testSerialization() {
        line.setAmp(20);
        line.setWiggles(20);
        line.setDoubleLine(true);
        line.setDLSeparation(4.f);
        JaxoTestUtils.checkSerializedEquality(line);
    }

    /** Tests that a copy of an object is still the same object. */
    public void testCopy() {
        JaxoGlLine line2 = new JaxoGlLine();
        line2.setState(line);

        line.setAmp(20);
        line.setWiggles(20);
        line.setDoubleLine(true);
        line.setDLSeparation(4.f);
        JaxoTestUtils.checkCopyEquality(line);

        assertFalse(line2.isCopy(line));

        line2.setAmp(20);
        line2.setWiggles(20);
        line2.setDoubleLine(true);
        line2.setDLSeparation(4.f);
        assertTrue(line2.isCopy(line));

        JaxoFLoop loop = new JaxoFLoop();
        assertFalse(line.isCopy(loop));

    }

    /** Tests identity. */
    public void testIdentity() {
        JaxoGlLine line2 = line;
        assertEquals(line, line2);
    }

    /** Tests the latex command. */
    public void testLatexCommand() {
        Dimension dim = new Dimension(100, 100);

        String expected = "\\Gluon(100,0)(200,-100){7.5}{17}";
        String actual = line.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        line.setDoubleLine(true);
        line.setDLSeparation(4.f);

        expected = "\\Gluon[double,sep=4](100,0)(200,-100){7.5}{17}";
        actual = line.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        line.setRelWAndH(0, 0);
        expected = "%";
        actual = line.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        // reset to initial state
        line.setDoubleLine(false);
        line.setRelWAndH(-100, -100);
        expected = "\\Gluon(100,0)(0,100){7.5}{17}";
        actual = line.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

    }

    /** Tests axodraw. */
    public void testAxodraw() {
        JaxoAxodraw axo = new JaxoAxodraw();
        JaxoObject ob = axo.getJaxoObject("\\SetWidth{1.0}");
        assertNull(ob);
        String command;

        command = "\\Gluon(100,2)(200,-98){7.5}{17}";
        line = (JaxoGlLine) axo.getJaxoObject(command);
        checkLineParameters(line, false);

        command =
            "\\Gluon(102.12,-89.88)(202.12,-189.88){7.5}{17}\\Gluon(97.88,-94.12)(197.88,-194.12){7.5}{17}%%JaxoDrawID:DoubleLine(3)";
        line = (JaxoGlLine) axo.getJaxoObject(command);
        checkLineParameters(line, true);

    }

    private void checkLineParameters(JaxoGlLine aline, boolean doubleLine) {
        assertEquals("GlLine doesn't have the right color!", JaxoColor.BLACK,
            aline.getColor());
        assertTrue("GlLine doesn't have the right stroke!", aline.strokeIs(1.f));
        assertEquals("GlLine doesn't have the right amp!", 15, aline.getAmp());
        assertEquals("GlLine doesn't have the right wiggles!", 17, aline.getWiggles());
        assertEquals("Mismatch in GlLine doubleLine!", doubleLine,
            aline.isDoubleLine());

        float dlsep = 2.f;
        int x = 100;
        int y = -2;
        if (doubleLine) {
            dlsep = 3.f;
            x = 100;
            y = 92;
        }

        assertEquals("GlLine doesn't have the right x!", x, aline.getX());
        assertEquals("GlLine doesn't have the right y!", y, aline.getY());
        assertTrue("Mismatch in GlLine doubleLine separation!",
            aline.dlSepIs(dlsep));

    }

    /** Tests the getBounds() method. */
    public void testGetBounds() {
        Rectangle bBox = line.getBounds();
        Rectangle expected = new Rectangle(98, 96, 106, 106);
        assertEquals("Bounding box not correct!", expected, bBox);
        line.setDoubleLine(true);
        bBox = line.getBounds();
        expected.setRect(97, 95, 108, 108);
        assertEquals("DL Bounding box not correct!", expected, bBox);

        assertNotNull(line.getBounds2D());
    }

    /** Tests createNewObject in JaxoObjectFactory. */
    public void testCreateNewObject() {
        Point p = new Point(50, 50);
        JaxoObject ob =
            JaxoObjectFactory.newObject(JaxoConstants.GLLINE, new Point[] {p, p});

        assertTrue("createNewObject did not return a GlLine!",
            (ob instanceof JaxoGlLine));

        line = (JaxoGlLine) ob;
        assertEquals("GlLine doesn't have the right x!", 50, line.getX());
        assertEquals("GlLine doesn't have the right y!", 50, line.getY());
        assertEquals("GlLine doesn't have the right color!",
            JaxoColor.getColor(JaxoPrefs.getStringPref(JaxoPrefs.PREF_COLOR)),
            line.getColor());
        assertTrue("GlLine doesn't have the right stroke!",
            line.strokeIs(JaxoPrefs.getFloatPref(JaxoPrefs.PREF_LINEWIDTH)));
        assertEquals("GlLine doesn't have the right double line!", false,
            line.isDoubleLine());
        assertTrue("GlLine doesn't have the right dlsep!",
            line.dlSepIs(JaxoPrefs.getFloatPref(JaxoPrefs.PREF_DLSEP)));
        assertEquals("GlLine doesn't have the right amplitude!",
            JaxoPrefs.getIntPref(JaxoPrefs.PREF_AMPLITUDE), line.getAmp());
    }

    /** {@inheritDoc} */
    public String className() {
        return "JaxoGlLine";
    }


    /** {@inheritDoc} */
    public void testJaxo20Compat() {
        JaxoGlLine object = (JaxoGlLine) instanceFrom20OpenGraph(JaxoGlLine.class);
        assertNotNull(object);
        check20Object("opened wrong object!", object);

        object = (JaxoGlLine) instanceFrom20ImportGraph(JaxoGlLine.class);
        assertNotNull(object);
        check20Object("opened wrong object!", object);
    }

    private void check20Object(String message, JaxoGlLine object) {
        assertEquals(message, 30, object.getX());
        assertEquals(message, 368, object.getY());
        assertEquals(message, 109, object.getX2());
        assertEquals(message, 428, object.getY2());

        assertTrue(message, object.isDoubleLine());
        assertTrue(message, object.dlSepIs(1.5f));
        assertTrue(message, object.strokeIs(1.4f));

        assertEquals(message, JaxoColor.BLUE, object.getColor());

        assertEquals(message, 13, object.getAmp());
        assertEquals(message, 5, object.getWiggles());
    }
}
