/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.object.line;

import java.awt.Dimension;
import java.awt.Point;
import java.awt.Rectangle;

import net.sf.jaxodraw.JaxoTestUtils;
import net.sf.jaxodraw.io.imports.JaxoAxodraw;
import net.sf.jaxodraw.object.JaxoObject;
import net.sf.jaxodraw.object.JaxoObjectFactory;
import net.sf.jaxodraw.object.loop.JaxoFLoop;
import net.sf.jaxodraw.util.JaxoColor;
import net.sf.jaxodraw.util.JaxoConstants;
import net.sf.jaxodraw.util.JaxoPrefs;


/** Tests JaxoFLine. */
public class JaxoFLineTest extends AbstractJaxoLineObjectTest {
    private JaxoFLine line = new JaxoFLine();

    /** Constructor: just calls super().
     * @param testName The name of this test.
     */
    public JaxoFLineTest(String testName) {
        super(testName);
    }

    /** JUnit setup: initialize the object. */
    @Override
    protected void setUp() {
        super.setUp();
        init();
    }

    private void init() {
        line.setLocation(100, 100);
        line.setRelWAndH(100, 100);
        line.setStrokeWidth(1.f);
        line.setColor(JaxoColor.GREEN);
        line.setPaintArrow(false);
        line.setArrowPosition(0.5f);
        line.setFlip(false);
        line.setDoubleLine(false);
        line.setDLSeparation(2.f);
    }

    /** {@inheritDoc} */
    public JaxoObject getInstance() {
        return line;
    }

    /** Tests serialization. */
    public void testSerialization() {
        line.setPaintArrow(true);
        line.setArrowPosition(1.f);
        line.setFlip(true);
        line.setDoubleLine(true);
        line.setDLSeparation(4.f);
        JaxoTestUtils.checkSerializedEquality(line);
    }

    /** Tests that a copy of an object is still the same object. */
    public void testCopy() {
        JaxoFLine line2 = new JaxoFLine();
        line2.setState(line);

        line.setPaintArrow(true);
        line.setArrowPosition(1.f);
        line.setFlip(true);
        line.setDoubleLine(true);
        line.setDLSeparation(4.f);
        JaxoTestUtils.checkCopyEquality(line);

        assertFalse(line2.isCopy(line));

        line2.setPaintArrow(true);
        line2.setArrowPosition(1.f);
        line2.setFlip(true);
        line2.setDoubleLine(true);
        line2.setDLSeparation(4.f);
        assertTrue(line2.isCopy(line));

        JaxoFLoop loop = new JaxoFLoop();
        assertFalse(line.isCopy(loop));

    }

    /** Tests identity. */
    public void testIdentity() {
        JaxoFLine line2 = line;
        assertEquals(line, line2);
    }

    /** Tests the latex command. */
    public void testLatexCommand() {
        Dimension dim = new Dimension(100, 100);

        String expected = "\\Line(100,0)(200,-100)";
        String actual = line.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        line.setPaintArrow(true);
        expected = "\\Line[arrow,arrowpos=0.5,arrowlength=5,arrowwidth=2,arrowinset=0.2](100,0)(200,-100)";
        actual = line.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        line.setArrowPosition(1.f);
        expected = "\\Line[arrow,arrowpos=1,arrowlength=5,arrowwidth=2,arrowinset=0.2](100,0)(200,-100)";
        actual = line.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        line.setArrowPosition(0.5f);
        line.setFlip(true);
        expected = "\\Line[arrow,arrowpos=0.5,arrowlength=5,arrowwidth=2,arrowinset=0.2,flip](100,0)(200,-100)";
        actual = line.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        // reset to initial state
        line.setPaintArrow(false);
        line.setFlip(false);

        line.setDoubleLine(true);
        line.setDLSeparation(4.f);

        expected = "\\Line[double,sep=4](100,0)(200,-100)";
        actual = line.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        line.setFlip(true);
        expected = "\\Line[double,sep=4](100,0)(200,-100)";
        actual = line.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        line.setRelWAndH(0, 0);
        expected = "%";
        actual = line.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        // reset to initial state
        line.setPaintArrow(false);
        line.setFlip(false);
        line.setDoubleLine(false);
        line.setRelWAndH(-100, -100);
        expected = "\\Line(100,0)(0,100)";
        actual = line.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

    }

    /** Tests axodraw. */
    public void testAxodraw() {
        JaxoAxodraw axo = new JaxoAxodraw();
        JaxoObject ob = axo.getJaxoObject("\\SetWidth{1.0}");
        assertNull(ob);
        String command;

        command = "\\Line(100,200)(200,300)";
        line = (JaxoFLine) axo.getJaxoObject(command);
        checkLineParameters(line, false);

        command = "\\ArrowLine(100,200)(200,300)";
        line = (JaxoFLine) axo.getJaxoObject(command);
        checkLineParameters(line, false);

        command = "\\LongArrow(100,200)(200,300)";
        line = (JaxoFLine) axo.getJaxoObject(command);
        checkLineParameters(line, false);

        // why do x and y values get shifted by one unit when in doubleline?
        command =
            "\\Line(98,202)(200,302)\\Line(100,196)(200,196)%%JaxoDrawID:DoubleLine(3)";
        line = (JaxoFLine) axo.getJaxoObject(command);
        checkLineParameters(line, true);

        command =
            "\\ArrowLine(98,202)(200,302)\\Line(100,196)(200,196)%%JaxoDrawID:DoubleLine(3)";
        line = (JaxoFLine) axo.getJaxoObject(command);
        checkLineParameters(line, true);

        command =
            "\\LongArrow(98,202)(200,302)\\Line(100,196)(200,196)%%JaxoDrawID:DoubleLine(3)";
        line = (JaxoFLine) axo.getJaxoObject(command);
        checkLineParameters(line, true);

    }

    private void checkLineParameters(JaxoFLine aline, boolean doubleLine) {
        assertEquals("FLine doesn't have the right color!", JaxoColor.BLACK,
            aline.getColor());
        assertTrue("FLine doesn't have the right stroke!",
            Math.abs(aline.getStrokeWidth() - 1.f) < 0.0001f);
        assertEquals("FLine doesn't have the right x!", 100, aline.getX());
        assertEquals("FLine doesn't have the right y!", -200, aline.getY());
        assertEquals("Mismatch in FLine doubleLine!", doubleLine,
            aline.isDoubleLine());

        float dlsep = 2.f;
        if (doubleLine) {
            dlsep = 3.f;
        }

        assertTrue("Mismatch in FLine doubleLine separation!",
            Math.abs(aline.getDLSeparation() - dlsep) < 0.0001f);

    }

    /** Tests the getBounds() method. */
    public void testGetBounds() {
        Rectangle bBox = line.getBounds();
        Rectangle expected = new Rectangle(99, 99, 102, 102);
        assertEquals("Bounding box not correct!", expected, bBox);
        line.setPaintArrow(true);
        line.setLocation(10, 10, 20, 10);
        bBox = line.getBounds();
        expected.setRect(9, 7, 12, 6);
        assertEquals("Line Bounding box not correct!", expected, bBox);
        line.setDoubleLine(true);
        bBox = line.getBounds();
        expected.setRect(9, 7, 12, 6);
        assertEquals("DL Bounding box not correct!", expected, bBox);

        assertNotNull(line.getBounds2D());
    }

    /** Tests createNewObject in JaxoObjectFactory. */
    public void testCreateNewObject() {
        Point p = new Point(50, 50);
        JaxoObject ob =
            JaxoObjectFactory.newObject(JaxoConstants.FLINE, new Point[] {p, p});

        assertTrue("createNewObject did not return a FLine!",
            (ob instanceof JaxoFLine));

        line = (JaxoFLine) ob;
        assertEquals("FLine doesn't have the right x!", 50, line.getX());
        assertEquals("FLine doesn't have the right y!", 50, line.getY());
        assertEquals("FLine doesn't have the right color!",
            JaxoColor.getColor(JaxoPrefs.getStringPref(JaxoPrefs.PREF_COLOR)),
            line.getColor());
        assertTrue("FLine doesn't have the right stroke!",
            line.strokeIs(JaxoPrefs.getFloatPref(JaxoPrefs.PREF_LINEWIDTH)));
        assertEquals("FLine doesn't have the right arrow!",
            JaxoPrefs.getBooleanPref(JaxoPrefs.PREF_ARROW), line.isPaintArrow());
        assertEquals("FLine doesn't have the right flip!", false,
            line.isFlip());
        assertEquals("FLine doesn't have the right double line!", false,
            line.isDoubleLine());
        assertTrue("FLine doesn't have the right dlsep!",
            line.dlSepIs(JaxoPrefs.getFloatPref(JaxoPrefs.PREF_DLSEP)));
    }

    /** {@inheritDoc} */
    public String className() {
        return "JaxoFLine";
    }

    /** {@inheritDoc} */
    public void testJaxo20Compat() {
        JaxoFLine object = (JaxoFLine) instanceFrom20OpenGraph(JaxoFLine.class);
        assertNotNull(object);
        check20Object("opened wrong object!", object);

        object = (JaxoFLine) instanceFrom20ImportGraph(JaxoFLine.class);
        assertNotNull(object);
        check20Object("imported wrong object!", object);
    }

    private void check20Object(String message, JaxoFLine object) {
        assertEquals(message, 69, object.getX());
        assertEquals(message, 27, object.getY());
        assertEquals(message, 136, object.getX2());
        assertEquals(message, 77, object.getY2());

        assertFalse(message, object.isDoubleLine());
        assertTrue(message, object.strokeIs(1.2f));

        assertEquals(message, JaxoColor.VIOLET_RED, object.getColor());

        assertTrue(message, object.isPaintArrow());
        assertTrue(message, object.isFlip());
        assertTrue(message, object.arrowPositionIs(1.f));
        assertArrow(message, object.getArrow(), 5.f, 2.f, 0.2f);
    }
}
