/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.object.line;

import java.awt.Point;
import java.awt.Rectangle;
import net.sf.jaxodraw.object.AbstractJaxoParticleObjectTest;
import net.sf.jaxodraw.object.arrow.JaxoArrow;
import net.sf.jaxodraw.util.JaxoConstants;


/**
 * Base Test for JaxoLineObjects.
 */
public abstract class AbstractJaxoLineObjectTest
    extends AbstractJaxoParticleObjectTest {

    private JaxoLineObject instance;

    /**
     * Constructor.
     *
     * @param testName The name of this test.
     */
    protected AbstractJaxoLineObjectTest(String testName) {
        super(testName);
    }

    /** JUnit setup: get a new instance of the object. */
    @Override
    protected void setUp() {
        super.setUp();
        this.instance = (JaxoLineObject) getInstance();
    }

      //
     // tests
    //

    /** Tests moveBy. */
    public void testMoveBy() {
        instance.setLocation(100, 100, 200, 200);
        instance.moveBy(23, -45);

        assertEquals("Wrong moveBy x!", 123, instance.getX());
        assertEquals("Wrong moveBy y!", 55, instance.getY());
        assertEquals("Wrong moveBy x2!", 223, instance.getX2());
        assertEquals("Wrong moveBy y2!", 155, instance.getY2());
    }

    /** Tests rescaleObject. */
    public void testRescaleObject() {
        instance.setLocation(100, 100, 200, 200);

        instance.rescaleObject(instance.getX(), instance.getY(), 2.f);
        assertEquals("Line x wrongly rescaled!", 200, instance.getRelw());
        assertEquals("Line y wrongly rescaled!", 200, instance.getRelh());

        instance.setLocation(50, 50, 100, 100);
        instance.setAmp(5);
        instance.setWiggles(25);
        instance.rescaleObject(instance.getX(), instance.getY(), 3.f);
        assertEquals("Line amp wrongly rescaled!", 15, instance.getAmp());
        assertEquals("Line wiggles wrongly rescaled!", 75, instance.getWiggles());
    }

    /** Tests getGrabbedHandle. */
    public void testGetGrabbedHandle() {
        instance.setLocation(100, 100, 200, 200);

        instance.setRelWAndH(100, 100);
        checkHandle(100, 100, JaxoLineObject.SELECT_P1);
        checkHandle(200, 200, JaxoLineObject.SELECT_P2);
        checkHandle(150, 150, JaxoLineObject.SELECT_P0);
        checkHandle(500, 500, JaxoLineObject.SELECT_NONE);

        instance.setRelWAndH(-100, -100);
        checkHandle(100, 100, JaxoLineObject.SELECT_P1);
        checkHandle(0, 0, JaxoLineObject.SELECT_P2);
        checkHandle(50, 50, JaxoLineObject.SELECT_P0);
        checkHandle(500, 500, JaxoLineObject.SELECT_NONE);
    }

    /** Tests canBeSelected. */
    public void testCanBeSelected() {
        instance.setLocation(100, 100, 200, 200);

        int mode = JaxoConstants.MOVE;
        int handle = JaxoLineObject.SELECT_P1;
        assertTrue("P1 in MOVE mode", instance.canBeSelected(handle, mode));
        handle = JaxoLineObject.SELECT_P2;
        assertTrue("P2 in MOVE mode", instance.canBeSelected(handle, mode));
        handle = JaxoLineObject.SELECT_P0;
        assertTrue("P0 in MOVE mode", instance.canBeSelected(handle, mode));

        mode = JaxoConstants.RESIZE;
        handle = JaxoLineObject.SELECT_P1;
        assertTrue("P1 in RESIZE mode", instance.canBeSelected(handle, mode));
        handle = JaxoLineObject.SELECT_P2;
        assertTrue("P2 in RESIZE mode", instance.canBeSelected(handle, mode));
        handle = JaxoLineObject.SELECT_P0;
        assertFalse("P0 in RESIZE mode", instance.canBeSelected(handle, mode));
    }

    /** Tests smallestDistanceTo. */
    public void testSmallestDistanceTo() {
        instance.setLocation(100, 100, 200, 200);

        float dist = instance.smallestDistanceTo(103, 104);
        assertEquals("Wrong distance", dist, 5.f);
        dist = instance.smallestDistanceTo(203, 204);
        assertEquals("Wrong distance", dist, 5.f);
        dist = instance.smallestDistanceTo(153, 154);
        assertEquals("Wrong distance", dist, 5.f);
    }

    /** Tests all getters/setters. */
    public void testGetSetXY() {
        assertEquals("Wrong PointCount!", 2, instance.getPointCount());
        try {
            instance.setPointCount(3);
            fail("setPointCount(3) should throw IllegalArgumentException");
        } catch (IllegalArgumentException e) {
            // expected, now check it doesn't throw with proper value
            instance.setPointCount(2);
        }
    }

    /** {@inheritDoc} */
    public void testContains() {
        instance.setLocation(100, 100, 200, 200);

        assertTrue(getInstance().contains(175, 175));
        assertFalse(getInstance().contains(5, 5));

        assertTrue(getInstance().contains(175, 175, 5, 5));
        assertFalse(getInstance().contains(5, 5, 5, 5));

        assertTrue(getInstance().contains(new Point(175, 175)));
        assertFalse(getInstance().contains(new Point(5, 5)));

        assertTrue(getInstance().contains(new Rectangle(175, 175, 5, 5)));
        assertFalse(getInstance().contains(new Rectangle(5, 5, 5, 5)));
    }

    /** {@inheritDoc} */
    public void testIntersects() {
        instance.setLocation(100, 100, 200, 200);

        assertTrue(getInstance().intersects(5, 5, 100, 100));
        assertFalse(getInstance().intersects(5, 5, 10, 10));

        assertTrue(getInstance().intersects(new Rectangle(130, 130)));
        assertFalse(getInstance().intersects(new Rectangle(30, 30)));
    }

    /** {@inheritDoc} */
    public void testArrowCoordinates() {
        instance.setLocation(100, 100, 200, 200);
        instance.setArrowPosition(0.5f);

        JaxoArrow.Coordinates arrow = instance.arrowCoordinates();
        assertEquals(150, (int) Math.round(arrow.getX()));
        assertEquals(150, (int) Math.round(arrow.getY()));
        assertEquals((float) (Math.PI / 4.d), (float) arrow.getAngle());

        instance.setFlip(true);
        arrow = instance.arrowCoordinates();
        assertEquals(150, (int) Math.round(arrow.getX()));
        assertEquals(150, (int) Math.round(arrow.getY()));
        assertEquals((float) (5.d * Math.PI / 4.d), (float) arrow.getAngle());
    }

      //
     // abstract
    //

    /** Tests serialization. */
    public abstract void testSerialization();

}
