/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.object.bezier;

import java.awt.Dimension;
import java.awt.Point;
import java.awt.Rectangle;

import net.sf.jaxodraw.JaxoTestUtils;
import net.sf.jaxodraw.io.imports.JaxoAxodraw4J;
import net.sf.jaxodraw.object.JaxoObject;
import net.sf.jaxodraw.object.JaxoObjectFactory;
import net.sf.jaxodraw.object.arc.JaxoGArc;
import net.sf.jaxodraw.object.loop.JaxoFLoop;
import net.sf.jaxodraw.util.JaxoColor;
import net.sf.jaxodraw.util.JaxoConstants;
import net.sf.jaxodraw.util.JaxoPrefs;


/** Tests JaxoPBezier. */
public class JaxoPBezierTest extends AbstractJaxoBezierObjectTest {
    private JaxoPBezier bezier = new JaxoPBezier();

    /** Constructor: just calls super().
     * @param testName The name of this test.
     */
    public JaxoPBezierTest(String testName) {
        super(testName);
    }

    /** JUnit setup: initialize the object. */
    @Override
    protected void setUp() {
        super.setUp();
        init();
    }

    private void init() {
        bezier.setLocation(50, 50, 100, 50, 100, 100, 50, 100);
        bezier.setColor(JaxoColor.GREEN);
        bezier.setStrokeWidth(1.f);
        bezier.setAmp(15);
        bezier.setWiggles(15);
        bezier.setDoubleLine(false);
        bezier.setDLSeparation(2.f);
    }

    /** {@inheritDoc} */
    public JaxoObject getInstance() {
        return bezier;
    }

    /** Tests serialization. */
    public void testSerialization() {
        bezier.setStrokeWidth(1.5f);
        bezier.setAmp(35);
        bezier.setWiggles(25);
        JaxoTestUtils.checkSerializedEquality(bezier);
    }

    /** Tests that a copy of an object is still the same object. */
    public void testCopy() {
        bezier.setStrokeWidth(1.5f);
        bezier.setAmp(35);
        bezier.setWiggles(25);
        JaxoTestUtils.checkCopyEquality(bezier);

        JaxoPBezier bezier2 = new JaxoPBezier();
        bezier2.setLocation(50, 50, 100, 50, 100, 100, 50, 100);
        bezier2.setColor(JaxoColor.GREEN);
        bezier2.setStrokeWidth(1.f);
        bezier2.setAmp(35);
        bezier2.setWiggles(25);
        bezier2.setDoubleLine(false);
        bezier2.setDLSeparation(2.f);
        assertFalse(bezier2.isCopy(bezier));

        bezier2.setStrokeWidth(1.5f);
        assertTrue(bezier2.isCopy(bezier));

        JaxoFLoop loop = new JaxoFLoop();
        assertFalse(bezier.isCopy(loop));

        JaxoPBezier pbezier = new JaxoPBezier();
        JaxoGArc garc = new JaxoGArc();
        assertFalse("PBezier is copy of GArc!", pbezier.isCopy(garc));

    }

    /** Tests the latex command. */
    public void testLatexCommand() {
        Dimension dim = new Dimension(100, 100);

        String expected = "%JaxoID:PBez(50,50)(100,50)(100,100)(50,100){15}{15}[true]";
        String actual = bezier.latexCommand(1.f, dim);
        assertTrue("PBezier Latex commands do not match!", actual.endsWith(expected));
    }

    /** Tests axodraw. */
    public void testAxodraw() {
        JaxoAxodraw4J axo = new JaxoAxodraw4J();
        JaxoObject ob = axo.getJaxoObject("\\SetWidth{1.0}");
        assertNull(ob);
        String command;

        command = "\\Bezier blahblah%JaxoID:PBez(50,50)(100,50)(100,100)(50,100){15}{15}[true]";
        bezier = (JaxoPBezier) axo.getJaxoObject(command);
        assertNotNull("PBezier not recognized!", bezier);

        assertEquals("PBezier doesn't have the right color!",
            JaxoColor.BLACK, bezier.getColor());
        assertTrue("PBezier doesn't have the right stroke!",
            Math.abs(bezier.getStrokeWidth() - 1.f) < 0.0001f);
        assertFalse("PBezier has an arrow!", bezier.isPaintArrow());

        assertEquals("PBezier doesn't have the right x1!", 50, bezier.getX());
        assertEquals("PBezier doesn't have the right y1!", 50, bezier.getY());
        assertEquals("PBezier doesn't have the right x2!", 100, bezier.getX2());
        assertEquals("PBezier doesn't have the right y2!", 50, bezier.getY2());
        assertEquals("PBezier doesn't have the right x3!", 100, bezier.getX3());
        assertEquals("PBezier doesn't have the right y3!", 100, bezier.getY3());
        assertEquals("PBezier doesn't have the right x4!", 50, bezier.getX4());
        assertEquals("PBezier doesn't have the right y4!", 100, bezier.getY4());
    }

    /** Tests the getBounds() method. */
    public void testGetBounds() {
        Rectangle bBox = bezier.getBounds();
        Rectangle expected = new Rectangle(49, 42, 47, 66);
        assertEquals("Bounding box not correct!", expected, bBox);

        assertNotNull(bezier.getBounds2D());
    }

    /** Tests createNewObject in JaxoObjectFactory. */
    public void testCreateNewObject() {
        Point p = new Point(50, 50);
        Point q = new Point(100, 50);
        Point r = new Point(100, 100);
        Point s = new Point(50, 100);
        JaxoObject ob =
            JaxoObjectFactory.newObject(JaxoConstants.PBEZ, new Point[] {p, q, r, s});

        assertTrue("createNewObject did not return a PBezier!",
            (ob instanceof JaxoPBezier));

        bezier = (JaxoPBezier) ob;
        assertEquals("PBezier doesn't have the right x1!", 50, bezier.getX());
        assertEquals("PBezier doesn't have the right y1!", 50, bezier.getY());
        assertEquals("PBezier doesn't have the right x2!", 100, bezier.getX2());
        assertEquals("PBezier doesn't have the right y2!", 50, bezier.getY2());
        assertEquals("PBezier doesn't have the right x3!", 100, bezier.getX3());
        assertEquals("PBezier doesn't have the right y3!", 100, bezier.getY3());
        assertEquals("PBezier doesn't have the right x4!", 50, bezier.getX4());
        assertEquals("PBezier doesn't have the right y4!", 100, bezier.getY4());
        assertEquals("PBezier doesn't have the right color!",
            JaxoColor.getColor(JaxoPrefs.getStringPref(JaxoPrefs.PREF_COLOR)),
            bezier.getColor());
        assertTrue("PBezier doesn't have the right stroke!",
            bezier.strokeIs(JaxoPrefs.getFloatPref(JaxoPrefs.PREF_LINEWIDTH)));
        assertEquals("PBezier doesn't have the right double line!", false,
            bezier.isDoubleLine());
        assertTrue("PBezier doesn't have the right dlsep!",
            bezier.dlSepIs(JaxoPrefs.getFloatPref(JaxoPrefs.PREF_DLSEP)));
    }
    /** {@inheritDoc} */
    public String className() {
        return "JaxoPBezier";
    }

    /** {@inheritDoc} */
    public void testJaxo20Compat() {
        JaxoPBezier object =
                (JaxoPBezier) instanceFrom20OpenGraph(JaxoPBezier.class);
        assertNotNull(object);
        check20Object("opened wrong object!", object);

        object =
                (JaxoPBezier) instanceFrom20ImportGraph(JaxoPBezier.class);
        assertNotNull(object);
        check20Object("imported wrong object!", object);
    }

    private void check20Object(String message, JaxoPBezier object) {

        assertEquals(message, 554, object.getX());
        assertEquals(message, 274, object.getY());
        assertEquals(message, 598, object.getX2());
        assertEquals(message, 390, object.getY2());
        assertEquals(message, 590, object.getX3());
        assertEquals(message, 245, object.getY3());
        assertEquals(message, 724, object.getX4());
        assertEquals(message, 336, object.getY4());

        assertTrue(message, object.strokeIs(1.4f));

        assertEquals(message, JaxoColor.SEPIA, object.getColor());

        assertEquals(message, 10, object.getAmp());
        assertEquals(message, 12, object.getWiggles());
        assertFalse(message, object.isNoFreqStretching());
    }
}
