/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */

package net.sf.jaxodraw.object.arrow;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Rectangle;

import java.beans.PropertyChangeEvent;

import java.beans.PropertyChangeSupport;
import net.sf.jaxodraw.AbstractJaxoBaseTest;
import net.sf.jaxodraw.gui.panel.edit.JaxoEditPanel;

/**
 * Test AbstractBaseArrow.
 */
public abstract class AbstractBaseArrowTest extends AbstractJaxoBaseTest {

    /**
     * Constructor: just calls super().
     *
     * @param testName The name of this test.
     */
    public AbstractBaseArrowTest(String testName) {
        super(testName);
    }

    /**
     * Test of get/setArrowWidth method, of class AbstractBaseArrow.
     */
    public void testArrowWidth() {
        AbstractBaseArrow arrow = getInstance();

        try {
            arrow.setArrowWidth(-2.f);
            fail();
        } catch (IllegalArgumentException e) {
            assertNotNull(e);
        }

        arrow.setArrowWidth(-0.00001f);
        assertEquals(0.f, arrow.getArrowWidth());

        arrow.setArrowWidth(1.234f);
        assertEquals(1.234f, arrow.getArrowWidth());
    }

    /**
     * Test of get/setArrowLength method, of class AbstractBaseArrow.
     */
    public void testArrowLength() {
        AbstractBaseArrow arrow = getInstance();

        try {
            arrow.setArrowLength(-2.f);
            fail();
        } catch (IllegalArgumentException e) {
            assertNotNull(e);
        }

        arrow.setArrowLength(-0.00001f);
        assertEquals(0.f, arrow.getArrowLength());

        arrow.setArrowLength(1.234f);
        assertEquals(1.234f, arrow.getArrowLength());
    }

    /**
     * Test of get/setArrowInset method, of class AbstractBaseArrow.
     */
    public void testArrowInset() {
        AbstractBaseArrow arrow = getInstance();

        try {
            arrow.setArrowInset(-2.f);
            fail();
        } catch (IllegalArgumentException e) {
            assertNotNull(e);
        }

        arrow.setArrowInset(-0.00001f);
        assertEquals(0.f, arrow.getArrowInset());

        arrow.setArrowInset(1.00001f);
        assertEquals(1.f, arrow.getArrowInset());

        try {
            arrow.setArrowInset(1.234f);
            fail();
        } catch (IllegalArgumentException e) {
            assertNotNull(e);
        }

        arrow.setArrowInset(0.234f);
        assertEquals(0.234f, arrow.getArrowInset());
    }

    /**
     * Test of propertyChange method, of class AbstractBaseArrow.
     */
    public void testPropertyChange() {
        AbstractBaseArrow arrow = getInstance();
        arrow.setColor(Color.BLUE);
        AbstractBaseArrow copy = (AbstractBaseArrow) arrow.copy();

        PropertyChangeSupport pcs = new PropertyChangeSupport(this);
        pcs.addPropertyChangeListener(arrow);
        assertEquals(Color.BLUE, arrow.getColor());
        assertTrue(copy.isCopy(arrow));
        pcs.firePropertyChange("color", Color.BLUE, Color.RED);
        assertEquals(Color.RED, arrow.getColor());
        assertFalse(copy.isCopy(arrow));

        copy = (AbstractBaseArrow) arrow.copy();
        assertTrue(copy.isCopy(arrow));
        pcs.firePropertyChange("blablabla", Color.RED, Color.GREEN);
        assertTrue(copy.isCopy(arrow));
    }

    protected class EmptyDefaultArrow extends AbstractBaseArrow {
        private static final long serialVersionUID = 314159L;

        public void paint(Graphics2D g2, Coordinates coords) {
            // nop
        }

        public Rectangle getBounds(Coordinates coords) {
            return new Rectangle();
        }

        public String latexCommand(float arPos, float scale) {
            return "";
        }

        public JaxoEditPanel getEditPanel() {
            return null;
        }

        public void propertyChange(PropertyChangeEvent evt) {
            // nop
        }
    }

    protected class NullArrow implements JaxoArrow {
        private static final long serialVersionUID = 314159L;

        public void paint(Graphics2D g2, Coordinates coords) {
            // nop
        }

        public JaxoArrow copy() {
            return null;
        }

        public boolean isCopy(JaxoArrow testArrow) {
            return false;
        }

        public Rectangle getBounds(Coordinates coords) {
            return null;
        }

        public JaxoEditPanel getEditPanel() {
            return null;
        }

        public String latexCommand(float arPos, float scale) {
            return null;
        }

        public void propertyChange(PropertyChangeEvent evt) {
            // nop
        }
    }

      //
     // abstract
    //

    /**
     * Test of latexCommand method, of class JaxoDefaultArrow.
     */
    public abstract void testLatexCommand();

    /**
     * Test of getBounds method, of class JaxoDefaultArrow.
     */
    public abstract void testGetBounds();

    /**
     * Test of getEditPanel method, of class JaxoDefaultArrow.
     */
    public abstract void testGetEditPanel();

    /**
     * Return a test instance of an arrow.
     *
     * @return an arrow to test.
     */
    public abstract AbstractBaseArrow getInstance();

}
