/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.object.arc;

import java.awt.Dimension;
import java.awt.Point;
import java.awt.Rectangle;

import net.sf.jaxodraw.JaxoTestUtils;
import net.sf.jaxodraw.io.imports.JaxoAxodraw;
import net.sf.jaxodraw.object.JaxoObject;
import net.sf.jaxodraw.object.JaxoObjectFactory;
import net.sf.jaxodraw.object.loop.JaxoFLoop;
import net.sf.jaxodraw.util.JaxoColor;
import net.sf.jaxodraw.util.JaxoConstants;
import net.sf.jaxodraw.util.JaxoPrefs;


/** Tests JaxoFArc. */
public class JaxoFArcTest extends AbstractJaxoArcObjectTest {
    private JaxoFArc arc = new JaxoFArc();

    /** Constructor: just calls super().
     * @param testName The name of this test.
     */
    public JaxoFArcTest(String testName) {
        super(testName);
    }

    /** JUnit setup: initialize the object. */
    @Override
    protected void setUp() {
        super.setUp();
        init();
    }

    private void init() {
        arc.setLocation(50, 50, 100, 50, 100, 100);
        arc.setColor(JaxoColor.GREEN);
        arc.setStrokeWidth(1.f);
        arc.setPaintArrow(false);
        arc.setArrowPosition(0.5f);
        arc.setFlip(false);
        arc.setDoubleLine(false);
        arc.setDLSeparation(2.f);
    }

    /** {@inheritDoc} */
    public JaxoObject getInstance() {
        return arc;
    }

    /** Tests serialization. */
    public void testSerialization() {
        arc.setStrokeWidth(1.5f);
        arc.setPaintArrow(true);
        arc.setFlip(true);
        JaxoTestUtils.checkSerializedEquality(arc);
    }

    /** Tests that a copy of an object is still the same object. */
    public void testCopy() {
        JaxoFArc arc2 = new JaxoFArc();
        arc2.setState(arc);

        arc.setStrokeWidth(1.5f);
        arc.setPaintArrow(true);
        arc.setFlip(true);
        JaxoTestUtils.checkCopyEquality(arc);

        assertFalse(arc2.isCopy(arc));

        arc2.setStrokeWidth(1.5f);
        arc2.setPaintArrow(true);
        arc2.setFlip(true);
        assertTrue(arc2.isCopy(arc));

        JaxoFLoop loop = new JaxoFLoop();
        assertFalse(arc.isCopy(loop));

        JaxoFArc farc = new JaxoFArc();
        JaxoGArc garc = new JaxoGArc();
        assertFalse("FArc is copy of GArc!", farc.isCopy(garc));

    }

    /** Tests identity. */
    public void testIdentity() {
        JaxoFArc arc2 = arc;
        assertEquals(arc, arc2);
    }

    /** Tests the latex command. */
    public void testLatexCommand() {
        Dimension dim = new Dimension(63, 63);

        String expected = "\\Arc[clock](75,-12)(35.355,135,-45)";
        String actual = arc.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        arc.setPaintArrow(true);
        expected = "\\Arc[arrow,arrowpos=0.5,arrowlength=5,arrowwidth=2,arrowinset=0.2,clock](75,-12)(35.355,135,-45)";
        actual = arc.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        arc.setFlip(true);
        expected = "\\Arc[arrow,arrowpos=0.5,arrowlength=5,arrowwidth=2,arrowinset=0.2,flip,clock](75,-12)(35.355,135,-45)";
        actual = arc.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        dim.setSize(63, 65);
        arc.setArrowPosition(1.f);
        expected = "\\Arc[arrow,arrowpos=1,arrowlength=5,arrowwidth=2,arrowinset=0.2,flip,clock](75,-10)(35.355,135,-45)";
        actual = arc.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        dim.setSize(65, 63);
        arc.setFlip(false);
        expected = "\\Arc[arrow,arrowpos=1,arrowlength=5,arrowwidth=2,arrowinset=0.2,clock](75,-12)(35.355,135,-45)";
        actual = arc.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        arc.setPaintArrow(false);
        arc.setArrowPosition(0.5f);
        arc.setDoubleLine(true);
        arc.setDLSeparation(4.f);
        expected =
            "\\Arc[double,sep=4,clock](75,-12)(35.355,135,-45)";
        actual = arc.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);
    }

    /** Tests axodraw. */
    public void testAxodraw() {
        JaxoAxodraw axo = new JaxoAxodraw();
        JaxoObject ob = axo.getJaxoObject("\\SetWidth{1.0}");
        assertNull(ob);
        String command;

        command = "\\CArc(100.0,200.0)(66.44,52.31,194.31)";
        arc = (JaxoFArc) axo.getJaxoObject(command);
        checkArcParameters(false);

        command = "\\ArrowArc(100.0,200.0)(66.44,52.31,194.31)";
        arc = (JaxoFArc) axo.getJaxoObject(command);
        checkArcParameters(false);

        command = "\\ArrowArcn(100.0,200.0)(66.44,52.31,194.31)";
        arc = (JaxoFArc) axo.getJaxoObject(command);
        checkArcParameters(false);

        command = "\\LongArrowArc(100.0,200.0)(66.44,52.31,194.31)";
        arc = (JaxoFArc) axo.getJaxoObject(command);
        checkArcParameters(false);

        command = "\\LongArrowArcn(100.0,200.0)(66.44,52.31,194.31)";
        arc = (JaxoFArc) axo.getJaxoObject(command);
        checkArcParameters(false);

        command =
            "\\CArc(100.0,200.0)(66.44,52.31,194.31)\\CArc(100.0,200.0)(66.44,52.31,194.31)%%JaxoDrawID:DoubleLine(3)";
        arc = (JaxoFArc) axo.getJaxoObject(command);
        checkArcParameters(true);

        command =
            "\\ArrowArc(100.0,200.0)(66.44,52.31,194.31)\\ArrowArc(100.0,200.0)(66.44,52.31,194.31)%%JaxoDrawID:DoubleLine(3)";
        arc = (JaxoFArc) axo.getJaxoObject(command);
        checkArcParameters(true);

        command =
            "\\ArrowArcn(100.0,200.0)(66.44,52.31,194.31)\\ArrowArcn(100.0,200.0)(66.44,52.31,194.31)%%JaxoDrawID:DoubleLine(3)";
        arc = (JaxoFArc) axo.getJaxoObject(command);
        checkArcParameters(true);

        command =
            "\\LongArrowArc(100.0,200.0)(66.44,52.31,194.31)\\LongArrowArc(100.0,200.0)(66.44,52.31,194.31)%%JaxoDrawID:DoubleLine(3)";
        arc = (JaxoFArc) axo.getJaxoObject(command);
        checkArcParameters(true);

        command =
            "\\LongArrowArcn(100.0,200.0)(66.44,52.31,194.31)\\LongArrowArcn(100.0,200.0)(66.44,52.31,194.31)%%JaxoDrawID:DoubleLine(3)";
        arc = (JaxoFArc) axo.getJaxoObject(command);
        checkArcParameters(true);

    }

    private void checkArcParameters(boolean doubleLine) {
        assertEquals("FArc doesn't have the right color!", JaxoColor.BLACK,
            arc.getColor());
        assertTrue("FArc doesn't have the right stroke!",
            Math.abs(arc.getStrokeWidth() - 1.f) < 0.0001f);
        assertEquals("Mismatch in FArc doubleLine!", doubleLine,
            arc.isDoubleLine());

        float dlsep = 2.f;
        if (doubleLine) {
            dlsep = 3.f;
        }

        assertTrue("Mismatch in FArc doubleLine separation!",
            Math.abs(arc.getDLSeparation() - dlsep) < 0.0001f);

    }

    /** Tests the getBounds() method. */
    public void testGetBounds() {
        Rectangle bBox = arc.getBounds();
        Rectangle expected = new Rectangle(49, 35, 66, 66);
        assertEquals("Bounding box not correct!", expected, bBox);
        arc.setLocation(50, 50, 100, 100, 150, 50);
        arc.setPaintArrow(true);
        bBox = arc.getBounds();
        expected.setRect(49, 49, 102, 54);
        assertEquals("Arc Bounding box not correct!", expected, bBox);
        arc.setDoubleLine(true);
        bBox = arc.getBounds();
        expected.setRect(48, 49, 104, 54);
        assertEquals("DL Bounding box not correct!", expected, bBox);

        assertNotNull(arc.getBounds2D());
    }

    /** Tests createNewObject in JaxoObjectFactory. */
    public void testCreateNewObject() {
        Point p = new Point(50, 50);
        Point q = new Point(100, 50);
        Point r = new Point(100, 100);
        JaxoObject ob =
            JaxoObjectFactory.newObject(JaxoConstants.FARC, new Point[] {p, q, r});

        assertTrue("createNewObject did not return a FArc!",
            (ob instanceof JaxoFArc));

        arc = (JaxoFArc) ob;
        assertEquals("FArc doesn't have the right x1!", 50, arc.getX());
        assertEquals("FArc doesn't have the right y1!", 50, arc.getY());
        assertEquals("FArc doesn't have the right x2!", 100, arc.getX2());
        assertEquals("FArc doesn't have the right y2!", 50, arc.getY2());
        assertEquals("FArc doesn't have the right x3!", 100, arc.getX3());
        assertEquals("FArc doesn't have the right y3!", 100, arc.getY3());
        assertEquals("FArc doesn't have the right color!",
            JaxoColor.getColor(JaxoPrefs.getStringPref(JaxoPrefs.PREF_COLOR)),
            arc.getColor());
        assertTrue("FArc doesn't have the right stroke!",
            arc.strokeIs(JaxoPrefs.getFloatPref(JaxoPrefs.PREF_LINEWIDTH)));
        assertEquals("FArc doesn't have the right arrow!",
            JaxoPrefs.getBooleanPref(JaxoPrefs.PREF_ARROW), arc.isPaintArrow());
        assertEquals("FArc doesn't have the right flip!", false, arc.isFlip());
        assertEquals("FArc doesn't have the right double line!", false,
            arc.isDoubleLine());
        assertTrue("FArc doesn't have the right dlsep!",
            arc.dlSepIs(JaxoPrefs.getFloatPref(JaxoPrefs.PREF_DLSEP)));

    }

    /** {@inheritDoc} */
    public String className() {
        return "JaxoFArc";
    }

    /** {@inheritDoc} */
    public void testJaxo20Compat() {
        JaxoFArc object =
                (JaxoFArc) instanceFrom20OpenGraph(JaxoFArc.class);
        assertNotNull(object);
        check20Object("opened wrong object!", object);

        object = (JaxoFArc) instanceFrom20ImportGraph(JaxoFArc.class);
        assertNotNull(object);
        check20Object("imported wrong object!", object);

    }

    private void check20Object(String message, JaxoFArc object) {

        assertEquals(message, 382, object.getX());
        assertEquals(message, 69, object.getY());
        assertEquals(message, 351, object.getX2());
        assertEquals(message, 36, object.getY2());
        assertEquals(message, 318, object.getX3());
        assertEquals(message, 67, object.getY3());

        assertTrue(message, object.strokeIs(1.6f));

        assertTrue(message, object.isDoubleLine());
        assertTrue(message, object.dlSepIs(3.f));

        assertEquals(message, JaxoColor.ROYAL_PURPLE, object.getColor());

        assertTrue(message, object.isPaintArrow());
        assertTrue(message, object.isFlip());
        assertTrue(message, object.arrowPositionIs(1.f));
        assertArrow(message, object.getArrow(), 15.f, 6.f, 0.2f);
    }
}
