/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.object;

import java.awt.Point;

import java.util.NoSuchElementException;

import net.sf.jaxodraw.AbstractJaxoBaseTest;
import net.sf.jaxodraw.util.JaxoConstants;


/**
 * Test for JaxoObjectFactory.
 */
public class JaxoObjectFactoryTest extends AbstractJaxoBaseTest {

    /**
     * Constructor.
     *
     * @param testName The name of this test.
     */
    public JaxoObjectFactoryTest(String testName) {
        super(testName);
    }


    /**
     * Tests object creation. Note: creation of particular instances
     * should be tested by individual object tests.
     */
    public void testNewObject() {
        JaxoObject ob = null;

        try {
            ob = JaxoObjectFactory.newObject(-12345);
            fail("Unknown object created!");
        } catch (NoSuchElementException e) {
            // expected
            ob = JaxoObjectFactory.newObject(JaxoConstants.BOX);
        }

        assertNotNull("Null object created!", ob);
        assertNull("Object has color!", ob.getColor());

        ob = null;

        try {
            ob = JaxoObjectFactory.newObject("");
            fail("Unknown object created!");
        } catch (NoSuchElementException e) {
            // expected
            ob = JaxoObjectFactory.newObject("net.sf.jaxodraw.object.fill.JaxoBox");
        }

        assertNotNull("Null object created!", ob);
        assertNull("Object has color!", ob.getColor());

        Point p = new Point(10, 10);
        Point[] q = new Point[] {p, p};

        ob = JaxoObjectFactory.newObject(JaxoConstants.BOX, q);
        assertNotNull("Null object created!", ob);
        assertNotNull("Object has no color!", ob.getColor());
        assertEquals("Object has wrong location!", 10, ob.getX());
        assertEquals("Object has wrong location!", 10, ob.getY());
    }

    /** Tests addObjectType. */
    public void testAddObjectType() {
        try {
            JaxoObjectFactory.addObjectType(-12345, null);
            fail("Null id should throw!");
        } catch (IllegalArgumentException e) {
            // expected
            JaxoObjectFactory.addObjectType(-12345, "");
        }

        try {
            JaxoObjectFactory.addObjectType(-12346,
                "net.sf.jaxodraw.object.fill.JaxoBox");
            fail("Duplicate id should throw!");
        } catch (IllegalArgumentException e) {
            // expected
            JaxoObjectFactory.addObjectType(-12346,
                "net.sf.jaxodraw.object.fill.JaxoBox.mine");
        }

        try {
            JaxoObjectFactory.addObjectType(JaxoConstants.BOX, "dummyObject");
            fail("Duplicate type should throw!");
        } catch (IllegalArgumentException e) {
            // expected
            JaxoObjectFactory.addObjectType(-12347, "dummyObject");
        }
    }
}
