/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.io.imports;

import junit.framework.TestCase;

import java.util.List;

import net.sf.jaxodraw.object.JaxoObject;
import net.sf.jaxodraw.object.line.JaxoPLine;
import net.sf.jaxodraw.object.text.JaxoLatexText;
import net.sf.jaxodraw.util.JaxoColor;


/** Tests JaxoAxodraw. */
public class JaxoAxodrawTest extends TestCase {
    private final String testString1;
    private final String testString2;
    private final String testString3;

    /** Constructor. */
    public JaxoAxodrawTest() {
        testString1 = "nonsense";
        testString2 = "\\Text(239,-142)[lb]{\\Large{\\Black{$test$}}}";
        testString3 = "\\Text(224,-64)[lb]{\\Large{\\Black{$\\mbox{text with embedded math $a$ in the middle}$}}}";
    }

    /** Tests getJaxoObject(). */
    public void testNull() {
        JaxoAxodraw axodraw = new JaxoAxodraw();
        assertNull("testString1 not Null!", axodraw.getJaxoObject(testString1));
        assertNotNull("testString2 gives Null!", axodraw.getJaxoObject(testString2));
        assertNotNull("testString3 gives Null!", axodraw.getJaxoObject(testString3));
    }

    /** Tests %%JaxoScale and %%JaxoComment. */
    public void testExtraCommands() {
        JaxoAxodraw axodraw = new JaxoAxodraw();
        axodraw.checkRequired("%%JaxoComment: Feynman was a bum.");
        assertEquals("JaxoComment not parsed correctly!",
            axodraw.getComment(), "Feynman was a bum.");
        axodraw.checkRequired("%%JaxoScale{1.0}");
        assertTrue("JaxoScale not parsed correctly!",
            (axodraw.getScaleFactor() - 1.f) < 0.0001f);
    }

    /** Tests \\SetColor and \\SetWidth. */
    public void testOtherCommands() {
        JaxoAxodraw axodraw = new JaxoAxodraw();
        JaxoObject ob = axodraw.getJaxoObject("\\SetColor{Green}");
        assertNull("ob should be null!", ob);
        ob = axodraw.getJaxoObject("\\SetWidth{0.8}");
        assertNull("ob should be null!", ob);
        ob = axodraw.getJaxoObject("\\Photon(144,-104)(415,-63){7.5}{13}");
        assertTrue("ob is not a JaxoPLine!", ob instanceof JaxoPLine);
        JaxoPLine pline = (JaxoPLine) ob;
        assertEquals("Object doesn't have the right color!", pline.getColor(),
            JaxoColor.GREEN);
        assertTrue("Object doesn't have the right stroke!",
            (pline.getStrokeWidth() - 1.6f) < 0.0001f);
    }

    /** Tests \\SetColor and \\SetWidth. */
    public void testRequiredCommands() {
        JaxoAxodraw axodraw = new JaxoAxodraw();
        boolean[] val = axodraw.getValid();
        assertFalse("val[] values not false!", val[0] || val[1] || val[2]);
        axodraw.checkRequired("\\documentclass");
        axodraw.checkRequired("\\begin{document}");
        axodraw.checkRequired("\\begin{picture}(100,100)(20,10)");
        axodraw.checkRequired(testString2);
        val = axodraw.getValid();
        assertTrue("val[] values not true!", val[0] && val[1] && val[2]);

    }

    /** Tests getPackages(). */
    public void testPackages() {
        JaxoAxodraw axodraw = new JaxoAxodraw();
        List<String> list = axodraw.getPackages();
        assertTrue("axodraw or color not contained in list!",
            list.contains("axodraw") && list.contains("color"));
        assertEquals("List does not only contain axodraw and color!",
            list.size(), 2);

        axodraw.checkRequired("\\usepackage{axodraw}");
        axodraw.checkRequired("\\usepackage{color}");
        list = axodraw.getPackages();
        assertEquals("Either axodraw or color got added a second time!",
            list.size(), 2);

        axodraw.checkRequired("\\usepackage{pstricks}");
        list = axodraw.getPackages();
        assertTrue("pstricks not contained in list!", list.contains("pstricks"));
        assertEquals("List does not only contain axodraw, color and pstricks!",
            list.size(), 3);

        axodraw.removePackage("axodraw");
        assertFalse("axodraw still contained in list!",
            list.contains("axodraw"));

        axodraw.addPackage("axodraw");
        assertTrue("axodraw not contained in list!",
            list.contains("axodraw"));

        int size = list.size();
        axodraw.addPackage("axodraw");
        assertEquals("axodraw got added a second time!",
            list.size(), size);
    }

    /** Tests instanceof JaxoLatexText(). */
    public void testLatexCommands() {
        JaxoAxodraw axodraw = new JaxoAxodraw();
        JaxoObject ob = axodraw.getJaxoObject(testString2);
        assertTrue("ob is not a JaxoLatexText!", ob instanceof JaxoLatexText);
        assertEquals("test", ((JaxoLatexText) ob).getTextString());

        ob = axodraw.getJaxoObject(testString3);
        assertTrue("ob is not a JaxoLatexText!", ob instanceof JaxoLatexText);
        assertEquals("\\mbox{text with embedded math $a$ in the middle}", ((JaxoLatexText) ob).getTextString());
    }
}
