/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.io;

import java.io.File;

import net.sf.jaxodraw.AbstractJaxoBaseTest;
import net.sf.jaxodraw.graph.JaxoGraph;
import net.sf.jaxodraw.io.imports.JaxoImportAxodraw;


/** Tests JaxoIO. */
public class JaxoIOTest extends AbstractJaxoBaseTest {
    private final JaxoIO io = new JaxoIO(null);
    private static final String BASE = "io/";

    /** Constructor.
     * @param testName The name of this test.
     */
    public JaxoIOTest(String testName) {
        super(testName);
    }

    /** Tests importLatex(String). */
    public void testImportLatex() {
        String all = getResourceFile(BASE, "all-2.0.tex").getAbsolutePath();
        JaxoGraph graph = (new JaxoImportAxodraw()).importGraph(all);

        assertNotNull("latex import gave Null!", graph);
        assertEquals("wrong number of elements!", 30, graph.getObjectList().size());
        assertFalse("graph should NOT be saved after import!", graph.isSaved());
        assertEquals("wrong saveFileName!", all, graph.getSaveFileName());
        assertFalse("graph contains marked Objects!",  graph.containsMarkedObjects());
        assertFalse("graph contains groups!",  graph.containsGroups());
        assertFalse("graph contains a PS text!",  graph.containsPSText());
        assertTrue("graph should contain a latex text!",  graph.containsLatexText());
    }

    /** Tests open(String). */
    public void testOpen() {
        File all = getResourceFile(BASE, "all-2.0.xml");
        JaxoGraph graph = io.open(all, false);

        assertNotNull("open gave Null!", graph);
        assertEquals("wrong number of elements!", 31, graph.getObjectList().size());
        assertTrue("graph should be saved after open!", graph.isSaved());
        assertEquals("wrong saveFileName!", all.getAbsolutePath(), graph.getSaveFileName());
        assertFalse("graph contains marked Objects!",  graph.containsMarkedObjects());
        assertFalse("graph contains groups!",  graph.containsGroups());
        assertTrue("graph should contain a PS text!",  graph.containsPSText());
        assertTrue("graph should contain a latex text!",  graph.containsLatexText());
    }

    /** Tests save(Graph,String,String). */
    public void testSave() {
        File saved = getOutputFile(BASE, "saved.xml");
        String saveFileName = saved.getAbsolutePath();

        JaxoGraph graph = io.open(getResourceFile(BASE, "all-2.0.xml"), false);
        graph.setSaveFileName(saveFileName);

        io.save(graph, "title", saveFileName);

        assertTrue("No file created!", saved.exists());
        assertTrue("Empty file created!", saved.length() > 0);

        // re-import and check it's the same
        JaxoGraph savedGraph = io.open(saved, false);
        assertTrue("Wrong saved graph!",
            savedGraph.getObjectList().isCopy(graph.getObjectList()));
    }

}
