/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.gui.panel.button;

import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import java.beans.PropertyChangeEvent;

import junit.framework.TestCase;

import net.sf.jaxodraw.JaxoTestUtils;
import net.sf.jaxodraw.util.JaxoConstants;
import net.sf.jaxodraw.util.JaxoLanguage;

/** Tests for JaxoActionButtons. */
public class JaxoActionButtonsTest extends TestCase {

    private static final int[] ACTION_MODES =
    {
        JaxoConstants.UNDO, JaxoConstants.REDO, JaxoConstants.CLEAR,
        JaxoConstants.REFRESH
    };
    private static final int NOF_BUTTONS = ACTION_MODES.length;
    private static final String[] TOOLTIP_KEYS =
    {
        "Undo_the_last_operation", "Redo_the_last_operation",
        "Clear_the_screen", "Refresh_the_Screen"
    };

    private final JaxoActionButtons actionButtons = new JaxoActionButtons();
    private String actionCommand;

    /** Constructor: just calls super().
     * @param testName The name of this test.
     */
    public JaxoActionButtonsTest(String testName) {
        super(testName);
    }

    /** Tests constructor. */
    public void testConstructor() {
        for (int i = 0; i < NOF_BUTTONS; i++) {
            String name = JaxoConstants.getModeAsString(ACTION_MODES[i]);
            Component component = JaxoTestUtils.getChildNamed(actionButtons, name);
            assertNotNull(component);
            JaxoPanelButton button = (JaxoPanelButton) component;
            assertEquals(JaxoConstants.getModeAsString(ACTION_MODES[i]),
                button.getName());
            assertEquals(JaxoLanguage.translate(TOOLTIP_KEYS[i]),
                button.getToolTipText());
            if (i == 0 || i == 1) {
                // undo and redo are intially disabled
                assertFalse(button.isEnabled());
            }
        }
    }

    /** Tests updateLanguage. */
    public void testUpdateLanguage() {
        actionButtons.updateLanguage();
        for (int i = 0; i < NOF_BUTTONS; i++) {
            String name = JaxoConstants.getModeAsString(ACTION_MODES[i]);
            JaxoPanelButton button =
                (JaxoPanelButton) JaxoTestUtils.getChildNamed(actionButtons, name);
            assertEquals(JaxoLanguage.translate(TOOLTIP_KEYS[i]),
                button.getToolTipText());
        }
    }

    /** Tests addActionListener. */
    public void testAddActionListener() {
        actionButtons.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent evt) {
                actionCommand = evt.getActionCommand();
            }
        });

        // test that undo/redo are disabled
        for (int i = 0; i < 2; i++) {
            String name = JaxoConstants.getModeAsString(ACTION_MODES[i]);
            JaxoPanelButton button =
                (JaxoPanelButton) JaxoTestUtils.getChildNamed(actionButtons, name);
            actionCommand = null;
            button.doClick();
            assertNull("Undo/Redo should be disabled!", actionCommand);
        }

        // now enable undo/redo buttons
        PropertyChangeEvent event =
            new PropertyChangeEvent(this, "Jaxo.canUndo", Boolean.FALSE, Boolean.TRUE);
        actionButtons.propertyChange(event);
        event =
            new PropertyChangeEvent(this, "Jaxo.canRedo", Boolean.FALSE, Boolean.TRUE);
        actionButtons.propertyChange(event);
        // this should do nothing
        event =
            new PropertyChangeEvent(this, "Jaxo.canDoAnything", Boolean.FALSE, Boolean.TRUE);
        actionButtons.propertyChange(event);

        for (int i = 0; i < NOF_BUTTONS; i++) {
            String name = JaxoConstants.getModeAsString(ACTION_MODES[i]);
            JaxoPanelButton button =
                (JaxoPanelButton) JaxoTestUtils.getChildNamed(actionButtons, name);
            actionCommand = null;
            button.doClick();
            assertEquals("wrong action command!", name, actionCommand);
        }
    }

}
