/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.graph;

import junit.framework.TestCase;

import java.awt.Rectangle;

import java.util.ArrayList;
import java.util.List;

import net.sf.jaxodraw.object.JaxoList;
import net.sf.jaxodraw.object.JaxoObject;
import net.sf.jaxodraw.object.JaxoObjectList;
import net.sf.jaxodraw.object.line.JaxoFLine;
import net.sf.jaxodraw.object.loop.JaxoFLoop;
import net.sf.jaxodraw.object.loop.JaxoGLoop;
import net.sf.jaxodraw.util.JaxoColor;


/** Tests JaxoSaveGraph. */
public class JaxoSaveGraphTest extends TestCase {
    /** Constructor: just calls super().
     * @param testName The name of this test.
     */
    public JaxoSaveGraphTest(String testName) {
        super(testName);
    }

    /**
     * Test of getObjectList method, of class net.sf.jaxodraw.graph.JaxoSaveGraph.
     */
    public void testGetObjectList() {
        JaxoList<JaxoObject> expResult = new JaxoObjectList<JaxoObject>(JaxoSaveGraph.LIST_INIT_SIZE);
        JaxoSaveGraph instance = new JaxoSaveGraph(expResult);
        JaxoList<JaxoObject> result = instance.getObjectList();
        assertEquals(expResult, result);
    }

    /**
     * Test of setObjectList method, of class net.sf.jaxodraw.graph.JaxoSaveGraph.
     */
    public void testSetObjectList() {
        JaxoList<JaxoObject> list = new JaxoObjectList<JaxoObject>(JaxoSaveGraph.LIST_INIT_SIZE);
        JaxoSaveGraph instance = new JaxoSaveGraph();
        instance.setObjectList(list);
        JaxoList<JaxoObject> result = instance.getObjectList();
        assertEquals(list, result);
    }

    /**
     * Test of getPackageList method, of class net.sf.jaxodraw.graph.JaxoSaveGraph.
     */
    public void testGetPackageList() {
        JaxoList<JaxoObject> list = new JaxoObjectList<JaxoObject>(JaxoSaveGraph.LIST_INIT_SIZE);
        List<String> expResult = new ArrayList<String>(JaxoSaveGraph.LIST_INIT_SIZE);
        JaxoSaveGraph instance1 = new JaxoSaveGraph(list, "");
        List<String> result = instance1.getPackageList();
        expResult.add("axodraw4j");
        expResult.add("pstricks");
        expResult.add("color");
        assertEquals(expResult, result);

        JaxoSaveGraph instance2 = new JaxoSaveGraph(list, "", expResult);
        result = instance2.getPackageList();
        assertEquals(expResult, result);
    }

    /**
     * Test of setPackageList method, of class net.sf.jaxodraw.graph.JaxoSaveGraph.
     */
    public void testSetPackageList() {
        List<String> expResult = new ArrayList<String>(JaxoSaveGraph.LIST_INIT_SIZE);
        JaxoSaveGraph instance = new JaxoSaveGraph();
        instance.setPackageList(expResult);
        expResult.add("axodraw4j");
        expResult.add("pstricks");
        expResult.add("color");
        assertEquals(expResult, instance.getPackageList());
        JaxoSaveGraph instance2 = new JaxoSaveGraph();
        instance2.setPackageList(expResult);
        assertEquals(expResult, instance2.getPackageList());
    }

    /**
     * Test of setDescription method, of class net.sf.jaxodraw.graph.JaxoSaveGraph.
     */
    public void testSetDescription() {
        String describe = "description";
        JaxoSaveGraph instance = new JaxoSaveGraph();
        instance.setDescription(describe);
        String description = instance.getDescription();
        assertEquals(describe, description);
    }

    /**
     * Test of getDescription method, of class net.sf.jaxodraw.graph.JaxoSaveGraph.
     */
    public void testGetDescription() {
        String describe = "description";
        JaxoList<JaxoObject> expResult = new JaxoObjectList<JaxoObject>(JaxoSaveGraph.LIST_INIT_SIZE);
        List<String> pack = new ArrayList<String>(JaxoSaveGraph.LIST_INIT_SIZE);
        JaxoSaveGraph instance =
            new JaxoSaveGraph(expResult, describe, pack);
        String description = instance.getDescription();
        assertEquals(describe, description);
    }

    /**
     * Test of listElementAt method, of class net.sf.jaxodraw.graph.JaxoSaveGraph.
     */
    public void testListElementAt() {
        int i = 0;
        JaxoSaveGraph instance = new JaxoSaveGraph();

        JaxoObject expResult = new JaxoFLine();
        instance.addObject(expResult);
        JaxoObject result = instance.listElementAt(i);
        assertEquals(expResult, result);

        i = 1;
        assertNull(instance.listElementAt(i));
    }

    /**
     * Test of listSize method, of class net.sf.jaxodraw.graph.JaxoSaveGraph.
     */
    public void testListSize() {
        JaxoSaveGraph instance = new JaxoSaveGraph();

        int expResult = 0;
        int result = instance.listSize();
        assertEquals(expResult, result);

        instance.addObject(new JaxoFLine());
        expResult = 1;
        result = instance.listSize();
        assertEquals(expResult, result);
    }

    /**
     * Test of foreground method, of class net.sf.jaxodraw.graph.JaxoSaveGraph.
     */
    public void testForeground() {
        JaxoObject line = new JaxoFLine();
        JaxoObject loop = new JaxoFLoop();
        JaxoSaveGraph instance = new JaxoSaveGraph();
        instance.addObject(line);
        instance.addObject(loop);

        boolean result = instance.foreground(loop);
        assertTrue(result);
        result = instance.foreground(line);
        assertFalse(result);
    }

    /**
     * Test of background method, of class net.sf.jaxodraw.graph.JaxoSaveGraph.
     */
    public void testBackground() {
        JaxoObject line = new JaxoFLine();
        JaxoObject loop = new JaxoFLoop();
        JaxoSaveGraph instance = new JaxoSaveGraph();
        instance.addObject(line);
        instance.addObject(loop);

        boolean result = instance.background(line);
        assertTrue(result);
        result = instance.background(loop);
        assertFalse(result);
    }

    /**
     * Test of delete method, of class net.sf.jaxodraw.graph.JaxoSaveGraph.
     */
    public void testDelete() {
        JaxoObject line = new JaxoFLine();
        JaxoSaveGraph instance = new JaxoSaveGraph();
        instance.addObject(line);
        assertTrue(instance.getObjectList().contains(line));
        instance.delete(line);
        assertFalse(instance.getObjectList().contains(line));
    }

    /**
     * Test of addObject method, of class net.sf.jaxodraw.graph.JaxoSaveGraph.
     */
    public void testAddObject() {
        JaxoSaveGraph instance = new JaxoSaveGraph();

        assertFalse(instance.addObject(null));
        assertEquals(0, instance.listSize());

        assertTrue(instance.addObject(new JaxoFLine()));
        assertEquals(1, instance.listSize());

        assertFalse(instance.addObject(0, null));
        assertEquals(1, instance.listSize());

        assertTrue(instance.addObject(0, new JaxoFLine()));
        assertEquals(2, instance.listSize());
    }

    /**
     * Test of addPackage method, of class net.sf.jaxodraw.graph.JaxoSaveGraph.
     */
    public void testAddPackage() {
        String pack = "pack";
        JaxoSaveGraph instance = new JaxoSaveGraph();

        instance.addPackage(pack);
        assertTrue(instance.getPackageList().contains(pack));
    }

    /**
     * Test of clear method, of class net.sf.jaxodraw.graph.JaxoSaveGraph.
     */
    public void testClear() {
        JaxoSaveGraph instance = new JaxoSaveGraph();

        boolean result = instance.clear();
        assertTrue(result);
        instance.addObject(new JaxoFLine());
        result = instance.clear();
        assertFalse(result);
        assertEquals(0, instance.listSize());
    }

    /**
     * Test of clearAll method, of class net.sf.jaxodraw.graph.JaxoSaveGraph.
     */
    public void testClearAll() {
        JaxoSaveGraph instance = new JaxoSaveGraph();

        boolean result = instance.clearAll();
        assertTrue(result);
        instance.setDescription("describe");
        result = instance.clearAll();
        assertFalse(result);
        assertEquals(0, instance.listSize());
    }

    /**
     * Test of getBounds method, of class net.sf.jaxodraw.graph.JaxoSaveGraph.
     */
    public void testGetBounds() {
        JaxoSaveGraph instance = new JaxoSaveGraph();

        Rectangle result = instance.getBounds();
        assertNull(result);

        JaxoFLine line = new JaxoFLine();
        boolean modified = instance.addObject(line);
        assertTrue("Adding an object should modify the graph", modified);
        result = instance.getBounds();
        assertNotNull(result);

        line.setLocation(100, 100);
        line.setRelWAndH(100, 100);
        line.setStrokeWidth(1.f);
        line.setPaintArrow(false);
        result = instance.getBounds();
        Rectangle expected = new Rectangle(99, 99, 102, 102);
        assertEquals("Bounding box not correct!", expected, result);

        assertNull(instance.getBounds((Rectangle) null));
        assertNull(instance.getBounds(new Rectangle(0, 0, 10, 10)));
        assertNotNull(instance.getBounds(new Rectangle(100, 100, 10, 10)));

        JaxoFLine line2 = new JaxoFLine();
        line2.setLocation(110, 110, 220, 220);
        line2.setStrokeWidth(1.f);
        line2.setPaintArrow(false);
        instance.addObject(line2);
        result = instance.getBounds();
        expected = new Rectangle(99, 99, 122, 122);
        assertEquals("Bounding box not correct!", expected, result);
    }

    /**
     * Test of copyOf method, of class net.sf.jaxodraw.graph.JaxoSaveGraph.
     */
    public void testCopyOf() {
        JaxoSaveGraph instance = new JaxoSaveGraph();
        JaxoFLine line = new JaxoFLine();
        line.setLocation(100, 100);
        line.setRelWAndH(100, 100);
        line.setColor(JaxoColor.GREEN);
        line.setStrokeWidth(1.f);
        line.setPaintArrow(false);
        instance.addObject(line);
        instance.addPackage("balooney");
        instance.setDescription("describe");

        JaxoSaveGraph result = instance.copyOf();

        assertEquals(instance.getObjectList().size(),
            result.getObjectList().size());
        assertTrue(instance.listElementAt(0).isCopy(result.listElementAt(0)));
        assertEquals(instance.getPackageList(), result.getPackageList());
        assertEquals(instance.getDescription(), result.getDescription());

        result.deleteAll(result.getObjectList());
        assertEquals(0, result.getObjectList().size());

        instance.addObject(new JaxoGLoop());
        assertTrue(instance.listElementAt(0) instanceof JaxoFLine);
        instance.move(0, 1);
        assertTrue(instance.listElementAt(0) instanceof JaxoGLoop);
        instance.move(1, 1);
        assertTrue(instance.listElementAt(0) instanceof JaxoGLoop);
    }

    /**
     * Test of isCopy method, of class net.sf.jaxodraw.graph.JaxoSaveGraph.
     */
    public void testIsCopy() {
        JaxoSaveGraph instance = new JaxoSaveGraph();
        JaxoFLine line = new JaxoFLine();
        line.setLocation(100, 100);
        line.setRelWAndH(100, 100);
        line.setColor(JaxoColor.GREEN);
        line.setStrokeWidth(1.f);
        line.setPaintArrow(false);
        instance.addObject(line);
        instance.addPackage("balooney");
        instance.setDescription("describe");
        instance.setJaxoDrawVersion("1.3.2");

        JaxoSaveGraph result = instance.copyOf();
        assertTrue(result.isCopy(instance));

        result.setJaxoDrawVersion(null);
        assertFalse(result.isCopy(instance));
        instance.setJaxoDrawVersion(null);
        assertTrue(result.isCopy(instance));

        result.setDescription(null);
        assertFalse(result.isCopy(instance));
        instance.setDescription(null);
        assertTrue(result.isCopy(instance));

        result.addPackage("boah");
        assertFalse(result.isCopy(instance));
        instance.addPackage("boah");
        assertTrue(result.isCopy(instance));

        result.addObject(new JaxoFLoop());
        assertFalse(result.isCopy(instance));
        instance.addObject(new JaxoFLoop());
        assertTrue(result.isCopy(instance));
    }

    /**
     * Test of getBoundsExcept method, of class net.sf.jaxodraw.graph.JaxoSaveGraph.
     */
    public void testGetBoundsExcept() {
        JaxoSaveGraph instance = new JaxoSaveGraph();
        JaxoFLine line = new JaxoFLine();
        line.setLocation(100, 100);
        line.setRelWAndH(100, 100);
        line.setStrokeWidth(1.f);
        line.setPaintArrow(false);

        JaxoFLine line2 = new JaxoFLine();
        line2.setLocation(110, 110, 220, 220);
        line2.setStrokeWidth(1.f);
        line2.setPaintArrow(false);

        instance.addObject(line);
        instance.addObject(line2);

        JaxoList<JaxoObject> except = new JaxoObjectList<JaxoObject>(1);
        except.add(line2);

        Rectangle expected = new Rectangle(99, 99, 102, 102);
        Rectangle result = instance.getBoundsExcept(except);
        assertEquals("A Bounding box not correct!", expected, result);

        except.remove(0);
        expected = new Rectangle(99, 99, 122, 122);
        result = instance.getBoundsExcept(except);
        assertEquals("B Bounding box not correct!", expected, result);
    }

}
