/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.util;

import java.io.IOException;

import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.AudioInputStream;
import javax.sound.sampled.AudioSystem;
import javax.sound.sampled.DataLine;
import javax.sound.sampled.FloatControl;
import javax.sound.sampled.LineUnavailableException;
import javax.sound.sampled.SourceDataLine;
import javax.sound.sampled.UnsupportedAudioFileException;


/** A class to handle sound events.
 * @since 2.0
 */
public class JaxoSoundsHandler implements Runnable {
    private static final int EXTERNAL_BUFFER_SIZE = 524288; // 128Kb
    private final String filename;
    private final int curPosition;

    /**
     * Constructor.
     * @param wavfile The sound file.
     */
    public JaxoSoundsHandler(final String wavfile) {
        filename = wavfile;
        curPosition = 0;
    }

    /**
     * Constructor.
     * @param wavfile The sound file.
     * @param curP The position to start playing.
     */
    public JaxoSoundsHandler(final String wavfile, final int curP) {
        filename = wavfile;
        curPosition = curP;
    }

    /** {@inheritDoc} */
    public void run() {
        AudioInputStream audioInputStream = null;
        try {
            audioInputStream =
                AudioSystem.getAudioInputStream(Thread.currentThread().getContextClassLoader()
                                                                       .getResource(filename));
        } catch (UnsupportedAudioFileException e1) {
            JaxoLog.debug(e1);
            return;
        } catch (IOException e1) {
            JaxoLog.debug(e1);
            return;
        }

        final AudioFormat format = audioInputStream.getFormat();
        SourceDataLine auline = null;
        final DataLine.Info info = new DataLine.Info(SourceDataLine.class, format);

        try {
            auline = (SourceDataLine) AudioSystem.getLine(info);
            auline.open(format);
        } catch (LineUnavailableException e) {
            JaxoLog.debug(e);
            return;
        } catch (Exception e) {
            JaxoLog.debug(e);
            return;
        }

        if (auline.isControlSupported(FloatControl.Type.PAN)) {
            final FloatControl pan =
                (FloatControl) auline.getControl(FloatControl.Type.PAN);
            if (curPosition == 1) {
                pan.setValue(1.0f);
            } else if (curPosition == -1) {
                pan.setValue(-1.0f);
            }
        }

        auline.start();
        int nBytesRead = 0;
        final byte[] abData = new byte[EXTERNAL_BUFFER_SIZE];

        try {
            while (nBytesRead != -1) {
                nBytesRead = audioInputStream.read(abData, 0, abData.length);
                if (nBytesRead >= 0) {
                    auline.write(abData, 0, nBytesRead);
                }
            }
        } catch (IOException e) {
            JaxoLog.debug(e);
            return;
        } finally {
            auline.drain();
            auline.close();
        }
    }
}
