/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.util;


/** Defines constants used throughout JaxoDraw.
 * @since 2.0
 */
public final class JaxoConstants {
    /** A mode indicating an invalid JaxoDraw mode. */
    public static final int NO_MODE = -1;

    // start valid modes

    /** All modes have to be above that. */
    private static final int MIN_MODE = 0;

    // Possible modes (to be filled in with new modes as soon as we coded them)

    /** Default drawing mode: do nothing. */
    public static final int STANDBY = 0;

    // particle modes

    /** All particle modes have to be above that. */
    private static final int PARTICLE_MIN = 3;

    /** Fermion line. */
    public static final int FLINE = 3;

    /** Fermion loop. */
    public static final int FLOOP = 4;

    /** Fermion arc. */
    public static final int FARC = 5;

    /** Scalar line. */
    public static final int SLINE = 6;

    /** Scalar loop. */
    public static final int SLOOP = 7;

    /** Scalar arc. */
    public static final int SARC = 8;

    /** Ghost line. */
    public static final int GLINE = 9;

    /** Ghost loop. */
    public static final int GLOOP = 10;

    /** Ghost arc. */
    public static final int GARC = 11;

    /** Photon line. */
    public static final int PLINE = 12;

    /** Photon loop. */
    public static final int PLOOP = 13;

    /** Photon arc. */
    public static final int PARC = 14;

    /** Gluon line. */
    public static final int GLLINE = 15;

    /** Gluon loop. */
    public static final int GLLOOP = 16;

    /** Gluon arc. */
    public static final int GLARC = 17;

    /** Fermionic Bezier. */
    public static final int FBEZ = 18;

    /** Photonic Bezier. */
    public static final int PBEZ = 19;

    /** Gluonic Bezier. */
    public static final int GLBEZ = 20;

    /** Scalar Bezier. */
    public static final int SBEZ = 21;

    /** Ghost Bezier. */
    public static final int GBEZ = 22;

    /** All particle modes have to be below that. */
    private static final int PARTICLE_MAX = 22;

    // misc modes

    /** All misc modes have to be above that. */
    private static final int MISC_MIN = 230;

    /** Box (filled rectangle). */
    public static final int BOX = 230;

    /** Blob (filled ellipse). */
    public static final int BLOB = 231;

    /** A general vertex. */
    public static final int VERTEX = 232;

    /** ZigZag line. */
    public static final int ZIGZAG = 233;

    /** Normal text. */
    public static final int TEXT = 234;

    /** Latex text. */
    public static final int LATEX = 235;

    /** All misc modes have to be below that. */
    private static final int MISC_MAX = 235;

    // vertex modes

    /** All vertex modes have to be above that. */
    private static final int VERTEX_MIN = 336;

    /** A vertex type 1 object (a dot). */
    public static final int VERTEXT1 = 336;

    /** A vertex type 2 object (a cross with a circle). */
    public static final int VERTEXT2 = 337;

    /** A vertex type 3 object (a filled square). */
    public static final int VERTEXT3 = 338;

    /** A vertex type 4 object (a cross). */
    public static final int VERTEXT4 = 339;

    /** A vertex type 5 object (a triangle). */
    public static final int VERTEXT5 = 340;

    /** A vertex type 6 object (a diamond). */
    public static final int VERTEXT6 = 341;

    /** All vertex modes have to be below that. */
    private static final int VERTEX_MAX = 341;

    // edit modes

    /** All edit modes have to be above that. */
    private static final int EDIT_MIN = 450;

    /** Move an object. */
    public static final int MOVE = 450;

    /** Resize an object. */
    public static final int RESIZE = 451;

    /** Copy an object. */
    public static final int DUPLICATE = 452;

    /** Color an object. */
    public static final int COLOR = 453;

    /** Edit an object. */
    public static final int EDIT = 454;

    /** Delete an object. */
    public static final int DELETE = 455;

    /** Put an object in the background. */
    public static final int BACKGROUND = 456;

    /** Put an object in the foreground. */
    public static final int FOREGROUND = 457;

    /** Ungroup objects. */
    public static final int UNGROUP = 459;

    /** Select objects. */
    public static final int SELECT = 460;

    /** All edit modes have to be below that. */
    private static final int EDIT_MAX = 461;

    // file modes

    /** All file modes have to be above that. */
    private static final int FILE_MIN = 570;

    /** Start a new graph. */
    public static final int NEWG = 570;

    /** Open a JaxoDraw file. */
    public static final int OPEN = 571;

    /** Save a JaxoDraw graph. */
    public static final int SAVE = 572;

    /** Save a JaxoDraw graph in a given file. */
    public static final int SAVE_AS = 573;

    /** Export the graph. */
    public static final int EXPORT = 575;

    /** Print the graph. */
    public static final int PRINT = 576;

    /** Close the current buffer. */
    public static final int CLOSE = 578;

    /** Start a new graph tab. */
    public static final int NEWT = 580;

    /** Import a Latex file. */
    public static final int IMPORT = 581;

    /** Save the current selection of objects in a given file. */
    public static final int SAVE_SELECTION_AS = 582;

    /** Export the current selection of objects. */
    public static final int EXPORT_SELECTION = 583;

    /** All file modes have to be below that. */
    private static final int FILE_MAX = 583;

    //  Recent file mode

    /** Recent file mode. This is NOT a file mode! */
    public static final int RECENT_FILE = 685;

    // Zoom factor modes

    /** Zoom modes have to be above that. */
    private static final int ZOOM_FACTOR_MIN = 791;

    /** A zoom factor mode. */
    public static final int ZOOM_FACTOR = 791;

    /** A zoom factor mode x2. */
    public static final int ZOOM_FACTOR_X2 = 792;

    /** A zoom factor mode x4. */
    public static final int ZOOM_FACTOR_X4 = 793;

    /** A zoom factor mode x8. */
    public static final int ZOOM_FACTOR_X8 = 794;

    /** Zoom modes have to be below that. */
    private static final int ZOOM_FACTOR_MAX = 794;

    // action modes

    /** All action modes have to be above that. */
    private static final int ACTION_MIN = 1101;

    /** Undo the last operations. */
    public static final int UNDO = 1101;

    /** Clear the current graph. */
    public static final int CLEAR = 1102;

    /** Redo the last operations. */
    public static final int REDO = 1103;

    /** Refresh the screen. */
    public static final int REFRESH = 1104;

    /** Paste the most recently selected objects from the clipboard. */
    public static final int PASTE = 1105;

    /** Enter a description of the graph. */
    public static final int DESCRIBE = 1106;

    /** Add a  custom Latex package to the Latex output of JaxoDraw. */
    public static final int PACKAGE = 1107;

    /** Preview the graph. */
    public static final int PREVIEW = 1108;

    /** Watch file mode. */
    public static final int WATCHFILE = 1109;

    /** Rename a tab. */
    public static final int RENAME_TAB = 1110;

    /** Cut the most recently selected objects stored in the clipboard. */
    public static final int CUT = 1111;

    /** Copy into the clipboard the most recently selected objects. */
    public static final int SCOPY = 1112;

    /** Group the most recently selected objects. */
    public static final int SGROUP = 1113;

    /** Ungroup the most recently selected objects. */
    public static final int SUNGROUP = 1114;

    /** Put in the foreground the most recently selected objects. */
    public static final int SFORE = 1115;

    /** Put in the background the most recently selected objects. */
    public static final int SBACK = 1116;

    /** Preview the current selection of objects (compiles in LaTeX + dvips). */
    public static final int LATEX_PREVIEW_SELECTION = 1117;

    /** Move complete graph on canvas. */
    public static final int MOVE_GRAPH = 1118;

    /** All action modes have to be below that. */
    private static final int ACTION_MAX = 1118;

    // help events

    /** All help modes have to be above that. */
    private static final int HELP_MIN = 2151;

    /** About. */
    public static final int ABOUT = 2151;

    /** User guide. */
    public static final int USR_GUIDE = 2152;

    /** System info. */
    public static final int SYS_INFO = 2153;

    /** System info. */
    public static final int MAC_README = 2154;

    /** All help modes have to be below that. */
    private static final int HELP_MAX = 2154;

    // grid modes

    /** All grid modes have to be above that. */
    private static final int GRID_MIN = 3161;

    /** Grid. */
    public static final int GRID = 3161;

    /** Dynamic zoom. */
    public static final int ZOOM = 3162;

    /** Quit. */
    public static final int QUIT = 3163;

    /** Grid size. */
    public static final int GRID_SIZE = 3164;

    /** Rectangular grid. */
    public static final int RECTANGULAR_GRID = 3165;

    /** Hexagonal grid. */
    public static final int HEXAGONAL_GRID = 3166;

    /** Dot style. */
    public static final int GRID_STYLE_DOT = 3167;

    /** Cross style. */
    public static final int GRID_STYLE_CROSS = 3168;

    /** Line style. */
    public static final int GRID_STYLE_LINE = 3169;

    /** Honeycomb style. */
    public static final int GRID_STYLE_LINE_HONEYCOMB = 3170;

    /** All grid modes have to be below that. */
    private static final int GRID_MAX = 3170;

    // languages

    /** All language modes have to be above that. */
    private static final int LANG_MIN = 4171;

    /** English language mode. */
    public static final int ENGLISH = 4171;

    /** German language mode. */
    public static final int GERMAN = 4172;

    /** French language mode. */
    public static final int FRENCH = 4173;

    /** Italian language mode. */
    public static final int ITALIAN = 4174;

    /** Spanish language mode. */
    public static final int SPANISH = 4175;

    /** All language modes have to be below that. */
    private static final int LANG_MAX = 4175;

    // option modes

    /** All option modes have to be above that. */
    private static final int OPTION_MIN = 5182;

    /** Show toolbar. */
    public static final int TOOLBAR = 5182;

    /** Show statusbar. */
    public static final int STATUSBAR = 5183;

    /** Antialias on/off. */
    public static final int ANTIALIAS = 5184;

    /** Arrow on/off. */
    public static final int ARROW = 5185;

    /** LooknFeel dialog. */
    public static final int LOOKNFEEL = 5187;

    /** Preferences dialog. */
    public static final int PREFERENCES = 5188;

    /** Grid on/off. */
    public static final int GRID_ONOFF = 5189;

    /** Show gridbar. */
    public static final int GRIDBAR = 5190;

    /** Snap on/off. */
    public static final int SNAP_ONOFF = 5191;

    /** Plugin manager dialog. */
    public static final int PLUGINS = 5192;

    /** All option modes have to be below that. */
    private static final int OPTION_MAX = 5192;

    // possible default modes

    /** All default modes have to be above that. */
    private static final int DEFAULT_MIN = 6201;

    /** Default none. */
    public static final int DEF_NONE = 6201;

    /** Default fline. */
    public static final int DEF_FLINE = 6202;

    /** Default . */
    public static final int DEF_FLOOP = 6203;

    /** Default . */
    public static final int DEF_FARC = 6204;

    /** Default . */
    public static final int DEF_SLINE = 6205;

    /** Default . */
    public static final int DEF_SLOOP = 6206;

    /** Default . */
    public static final int DEF_SARC = 6207;

    /** Default . */
    public static final int DEF_GLINE = 6208;

    /** Default . */
    public static final int DEF_GLOOP = 6209;

    /** Default . */
    public static final int DEF_GARC = 6210;

    /** Default . */
    public static final int DEF_PLINE = 6211;

    /** Default . */
    public static final int DEF_PLOOP = 6212;

    /** Default . */
    public static final int DEF_PARC = 6213;

    /** Default . */
    public static final int DEF_GLLINE = 6214;

    /** Default . */
    public static final int DEF_GLLOOP = 6215;

    /** Default . */
    public static final int DEF_GLARC = 6216;

    /** Default . */
    public static final int DEF_BOX = 6217;

    /** Default . */
    public static final int DEF_BLOB = 6218;

    /** Default . */
    public static final int DEF_VERTEX = 6219;

    /** Default . */
    public static final int DEF_ZIGZAG = 6220;

    /** Default . */
    public static final int DEF_TEXT = 6221;

    /** Default . */
    public static final int DEF_LATEX = 6222;

    /** Default . */
    public static final int DEF_MOVE = 6223;

    /** Default . */
    public static final int DEF_RESIZE = 6224;

    /** Default . */
    public static final int DEF_DUPLICATE = 6225;

    /** Default . */
    public static final int DEF_COLOR = 6226;

    /** Default . */
    public static final int DEF_EDIT = 6227;

    /** Default . */
    public static final int DEF_SELECT = 6228;

    /** All default modes have to be below that. */
    private static final int DEFAULT_MAX = 6229;

    // end valid modes

    /** All modes have to be below that. */
    private static final int MAX_MODE = 6229;

    // dummy modes

    /** A dummy variable. */
    public static final int DUMMIE = 10000;

    /** Another dummy. */
    public static final int DUMMIE1 = 10001;
    private static final String[] DEF_ACTION_KEYS =
    {
        "None", "Fermion_Line", "Fermion_Loop", "Fermion_Arc", "Scalar_Line",
        "Scalar_Loop", "Scalar_Arc", "Ghost_Line", "Ghost_Loop", "Ghost_Arc",
        "Photon_Line", "Photon_Loop", "Photon_Arc", "Gluon_Line", "Gluon_Loop",
        "Gluon_Arc", "Box", "Blob", "Vertex", "Zigzag", "Text", "LaTex", "Move",
        "Resize", "Duplicate", "Color", "Edit", "Select"
    };
    private static final int[] DEF_MODES =
    {
        DEF_NONE, DEF_FLINE, DEF_FLOOP, DEF_FARC, DEF_SLINE, DEF_SLOOP,
        DEF_SARC, DEF_GLINE, DEF_GLOOP, DEF_GARC, DEF_PLINE, DEF_PLOOP,
        DEF_PARC, DEF_GLLINE, DEF_GLLOOP, DEF_GLARC, DEF_BOX, DEF_BLOB,
        DEF_VERTEX, DEF_ZIGZAG, DEF_TEXT, DEF_LATEX, DEF_MOVE, DEF_RESIZE,
        DEF_DUPLICATE, DEF_COLOR, DEF_EDIT, DEF_SELECT
    };

    /** Empty private constructor to prevent the class from being
     * explicitly instantiated.
     */
    private JaxoConstants() {
        // empty on purpose
    }

    /** Returns the default action keys.
     * @return The default action keys.
     */
    public static String[] defaultActionKeys() {
        return DEF_ACTION_KEYS.clone();
    }

    /** Returns the default modes.
     * @return The default modes.
     */
    public static int[] defaultModes() {
        return DEF_MODES.clone();
    }

    /** Determines whether the given mode is an edit mode.
     * @param mode The mode to check.
     * @return True if the given mode is an edit mode.
     */
    public static boolean isEditMode(final int mode) {
        boolean iem = true;

        if ((mode < EDIT_MIN) || (mode > EDIT_MAX)) {
            iem = false;
        }

        return iem;
    }

    /** Determines whether the given mode is a particle mode.
     * @param mode The mode to check.
     * @return True if the given mode is a particle mode.
     */
    public static boolean isParticleMode(final int mode) {
        boolean ipm = true;

        if ((mode < PARTICLE_MIN) || (mode > PARTICLE_MAX)) {
            ipm = false;
        }

        return ipm;
    }

    /** Determines whether the given mode is a misc mode.
     * @param mode The mode to check.
     * @return True if the given mode is a misc mode.
     */
    public static boolean isMiscMode(final int mode) {
        boolean imm = true;

        if ((mode < MISC_MIN) || (mode > MISC_MAX)) {
            imm = false;
        }

        return imm;
    }

    /** Determines whether the given mode is a file mode.
     * @param mode The mode to check.
     * @return True if the given mode is a file mode.
     */
    public static boolean isFileMode(final int mode) {
        boolean ifm = true;

        if ((mode < FILE_MIN) || (mode > FILE_MAX)) {
            ifm = false;
        }

        return ifm;
    }

    /** Determines whether the given mode is an action mode.
     * @param mode The mode to check.
     * @return True if the given mode is an action mode.
     */
    public static boolean isActionMode(final int mode) {
        boolean is = true;

        if ((mode < ACTION_MIN) || (mode > ACTION_MAX)) {
            is = false;
        }

        return is;
    }

    /** Determines whether the given mode is a grid mode.
     * @param mode The mode to check.
     * @return True if the given mode is a grid mode.
     */
    public static boolean isGridMode(final int mode) {
        boolean is = true;

        if ((mode < GRID_MIN) || (mode > GRID_MAX)) {
            is = false;
        }

        return is;
    }

    /** Determines whether the given mode is a vertex mode.
     * @param mode The mode to check.
     * @return True if the given mode is a vertex mode.
     */
    public static boolean isVertexMode(final int mode) {
        boolean is = true;

        if ((mode < VERTEX_MIN) || (mode > VERTEX_MAX)) {
            is = false;
        }

        return is;
    }

    /** Determines whether the given mode is a zoom factor mode.
     * @param mode The mode to check.
     * @return True if the given mode is a zoom factor mode.
     */
    public static boolean isZoomFactorMode(final int mode) {
        boolean is = true;

        if ((mode < ZOOM_FACTOR_MIN) || (mode > ZOOM_FACTOR_MAX)) {
            is = false;
        }

        return is;
    }

    /** Determines whether the given mode is a help mode.
     * @param mode The mode to check.
     * @return True if the given mode is a help mode.
     */
    public static boolean isHelpMode(final int mode) {
        boolean is = true;

        if ((mode < HELP_MIN) || (mode > HELP_MAX)) {
            is = false;
        }

        return is;
    }

    /** Determines whether the given mode is a language mode.
     * @param mode The mode to check.
     * @return True if the given mode is a language mode.
     */
    public static boolean isLanguageMode(final int mode) {
        boolean is = true;

        if ((mode < LANG_MIN) || (mode > LANG_MAX)) {
            is = false;
        }

        return is;
    }

    /** Determines whether the given mode is an option mode.
     * @param mode The mode to check.
     * @return True if the given mode is an option mode.
     */
    public static boolean isOptionMode(final int mode) {
        boolean is = true;

        if ((mode < OPTION_MIN) || (mode > OPTION_MAX)) {
            is = false;
        }

        return is;
    }

    /** Determines whether the given mode is a default mode.
     * @param mode The mode to check.
     * @return True if the given mode is a default mode.
     */
    public static boolean isDefaultMode(final int mode) {
        boolean is = true;

        if ((mode < DEFAULT_MIN) || (mode > DEFAULT_MAX)) {
            is = false;
        }

        return is;
    }

    /**
     * Determines whether the given mode is a mode that will create a new JaxoObject.
     *
     * @param mode The mode to check.
     * @return True if the given mode is a particle, misc or vertex mode.
     */
    public static boolean isNewObjectMode(final int mode) {
        return (isParticleMode(mode) || isMiscMode(mode) || isVertexMode(mode));
    }

    /** Determines whether the given mode is a line mode.
     * @param mode The mode to check.
     * @return True if the given mode is a line mode.
     */
    public static boolean isLineMode(final int mode) {
        boolean is = false;

        if ((mode == FLINE) || (mode == GLINE) || (mode == GLLINE)
                || (mode == PLINE) || (mode == SLINE)) {
            is = true;
        }

        return is;
    }

    /** Determines whether the given mode is an arc mode.
     * @param mode The mode to check.
     * @return True if the given mode is an arc mode.
     */
    public static boolean isArcMode(final int mode) {
        boolean is = false;

        if ((mode == FARC) || (mode == GARC) || (mode == GLARC)
                || (mode == PARC) || (mode == SARC)) {
            is = true;
        }

        return is;
    }

    /** Determines whether the given mode is a Bezier mode.
     * @param mode The mode to check.
     * @return True if the given mode is an arc mode.
     */
    public static boolean isBezierMode(final int mode) {
        boolean is = false;

        if ((mode == FBEZ) || (mode == PBEZ) || (mode == GLBEZ)
                || (mode == SBEZ) || (mode == GBEZ)) {
            is = true;
        }

        return is;
    }

    /** Determines whether the given mode is a loop mode.
     * @param mode The mode to check.
     * @return True if the given mode is a loop mode.
     */
    public static boolean isLoopMode(final int mode) {
        boolean is = false;

        if ((mode == FLOOP) || (mode == GLOOP) || (mode == GLLOOP)
                || (mode == PLOOP) || (mode == SLOOP)) {
            is = true;
        }

        return is;
    }

    /** Determines whether the given mode is a text mode
     * (either LATEX or TEXT).
     * @param mode The mode to check.
     * @return True if the given mode is a text mode.
     */
    public static boolean isTextMode(final int mode) {
        return ((mode == LATEX) || (mode == TEXT));
    }

    /** Determines whether the given mode is a select mode
     * (mode allowing an object to be selected).
     * @param mode The mode to check.
     * @return True if the given mode is a select mode.
     */
    public static boolean isSelectMode(final int mode) {
        return (mode == EDIT) || (mode == COLOR) || (mode == BACKGROUND)
        || (mode == FOREGROUND) || (mode == DELETE) || (mode == UNGROUP)
                || (mode == SELECT);
    }

    /** Returns a string representation of the given mode.
     * @param mode The mode to convert.
     * @return The mode as a string, or "NO_MODE" if it is not a valid mode.
     */
    public static String getModeAsString(final int mode) {
        String result = "NO_MODE";
        if ((mode >= MIN_MODE) && (mode <= MAX_MODE)) {
            result = Integer.toString(mode);
        }
        return result;
    }

    /** Converts the given string to an integer and checks if it corresponds
     * to a JaxoDraw mode.
     * @param modeString A string holding an integer.
     * @return The JaxoDraw mode as defined by this class if modeString
     * corresponds to one of them, or JaxoConstants.NO_MODE otherwise.
     */
    public static int getModeAsInt(final String modeString) {
        int intvalue = NO_MODE;
        try {
            intvalue = Integer.parseInt(modeString);
        } catch (NumberFormatException exc) {
            JaxoLog.debug(exc);
        }

        int i = NO_MODE;

        while ((i != intvalue) && (i <= MAX_MODE)) {
            i++;
        }

        if (i >= MAX_MODE) {
            i = NO_MODE;
        }

        return i;
    }

    /** Mode constants corresponding to a DEF_XYZ constant.
     * @param defaultAction One of the default modes defined by this class.
     * @return One of the modes corresponding to defaultAction,
     * or STANDBY if defaultAction is not a valid mode.
     */
    public static int defaultActionToMode(final int defaultAction) {
        switch (defaultAction) {
            case DEF_NONE:
                return STANDBY;
            case DEF_FLINE:
                return FLINE;
            case DEF_FLOOP:
                return FLOOP;
            case DEF_FARC:
                return FARC;
            case DEF_SLINE:
                return SLINE;
            case DEF_SLOOP:
                return SLOOP;
            case DEF_SARC:
                return SARC;
            case DEF_GLINE:
                return GLINE;
            case DEF_GLOOP:
                return GLOOP;
            case DEF_GARC:
                return GARC;
            case DEF_PLINE:
                return PLINE;
            case DEF_PLOOP:
                return PLOOP;
            case DEF_PARC:
                return PARC;
            case DEF_GLLINE:
                return GLLINE;
            case DEF_GLLOOP:
                return GLLOOP;
            case DEF_GLARC:
                return GLARC;
            case DEF_BOX:
                return BOX;
            case DEF_BLOB:
                return BLOB;
            case DEF_VERTEX:
                return VERTEX;
            case DEF_ZIGZAG:
                return ZIGZAG;
            case DEF_TEXT:
                return TEXT;
            case DEF_LATEX:
                return LATEX;
            case DEF_MOVE:
                return MOVE;
            case DEF_RESIZE:
                return RESIZE;
            case DEF_DUPLICATE:
                return DUPLICATE;
            case DEF_COLOR:
                return COLOR;
            case DEF_EDIT:
                return EDIT;
            case DEF_SELECT:
                return SELECT;
            default:
                JaxoLog.warn("Unknown default mode: " + defaultAction);
                return STANDBY;
        }
    }
}
