/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.object.vertex;

import java.awt.Dimension;
import java.awt.Rectangle;
import java.awt.geom.AffineTransform;
import java.awt.geom.GeneralPath;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;

import java.io.IOException;
import java.io.ObjectInputStream;

import net.sf.jaxodraw.object.JaxoObject;
import net.sf.jaxodraw.object.JaxoObjectEditPanel;
import net.sf.jaxodraw.util.JaxoColor;
import net.sf.jaxodraw.util.graphics.JaxoGraphics2D;


/** A vertex (a filled diamond).
 * @since 2.0
 */
public class JaxoVertexT6 extends JaxoVertex {
    private static final long serialVersionUID = 314159L;
    private transient Rectangle2D square = new Rectangle2D.Float();

    private void readObject(final ObjectInputStream in)
        throws IOException, ClassNotFoundException {
        in.defaultReadObject();
        square = new Rectangle2D.Float();
    }

    /** {@inheritDoc} */
    @Override
    public final boolean isCopy(final JaxoObject comp) {
        boolean isCopy = false;
        if (comp instanceof JaxoVertexT6) {
            isCopy = super.isCopy(comp);
        }

        return isCopy;
    }

    /** {@inheritDoc} */
    public final void paint(final JaxoGraphics2D g2) {
        final GeneralPath gp = getObjectPath();
        g2.setColor(getColor());
        g2.fill(gp);
        g2.draw(gp);
    }

    /**
     * Returns the bounding box of this object.
     *
     * @return the bounding box of this object.
     */
    public Rectangle getBounds() {
        return getStroke().createStrokedShape(getObjectPath()).getBounds();
    }

    /** {@inheritDoc} */
    public final String latexCommand(final float scale, final Dimension canvasDim) {
        final float radius = getLaTexRadius(scale) * ((float) Math.sqrt(2));

        if ((int) radius == 0) {
            return "%";
        }

        final Point2D center = getLaTexCenter(scale, canvasDim.height);

        final Point2D p1 = new Point2D.Float((float) (center.getX() - radius),
                (float) center.getY());
        final Point2D p2 = new Point2D.Float((float) center.getX(),
                (float) (center.getY() + radius));
        final Point2D p3 = new Point2D.Float((float) (center.getX() + radius),
                (float) center.getY());
        final Point2D p4 = new Point2D.Float((float) center.getX(),
                (float) (center.getY() - radius));

        final String tlc = JaxoColor.getColorName(getColor());

        return "\\CTri" + "(" + D_FORMAT.format(p1.getX()) + ","
            + D_FORMAT.format(p1.getY()) + ")" + "("
            + D_FORMAT.format(p2.getX()) + ","
            + D_FORMAT.format(p2.getY()) + ")" + "("
            + D_FORMAT.format(p3.getX()) + ","
            + D_FORMAT.format(p3.getY()) + ")" + "{" + tlc + "}" + "{"
            + tlc + "}" + "\\CTri" + "(" + D_FORMAT.format(p1.getX())
            + "," + D_FORMAT.format(p1.getY()) + ")" + "("
            + D_FORMAT.format(p4.getX()) + ","
            + D_FORMAT.format(p4.getY()) + ")" + "("
            + D_FORMAT.format(p3.getX()) + ","
            + D_FORMAT.format(p3.getY()) + ")" + "{" + tlc + "}" + "{"
            + tlc + "}";
    }

    /** {@inheritDoc} */
    public void prepareEditPanel(final JaxoObjectEditPanel editPanel) {
        prepareGenericPanel(editPanel, "vertexT6.png");
    }

    private GeneralPath getObjectPath() {
        final GeneralPath gp = getGeneralPath();
        gp.reset();

        final double length =
            Math.sqrt((getWidth() * getWidth()) + (getHeight() * getHeight()));

        square.setFrame(-length + getX(), -length + getY(), 2 * length,
            2 * length);
        gp.append(square, false);

        final AffineTransform at = new AffineTransform();
        at.rotate(Math.PI / 4.d, getX(), getY());
        gp.transform(at);

        return gp;
    }
}
