/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.object.vertex;

import java.awt.Dimension;
import java.awt.Rectangle;
import java.awt.geom.AffineTransform;
import java.awt.geom.GeneralPath;
import java.awt.geom.Point2D;

import net.sf.jaxodraw.object.JaxoHandle;
import net.sf.jaxodraw.object.JaxoObject;
import net.sf.jaxodraw.object.JaxoObjectEditPanel;
import net.sf.jaxodraw.util.JaxoConstants;
import net.sf.jaxodraw.util.graphics.JaxoGraphics2D;


/** A vertex (a cross).
 * @since 2.0
 */
public class JaxoVertexT4 extends JaxoVertex {
    private static final long serialVersionUID = 314159L;

    /** {@inheritDoc} */
    @Override
    public final boolean isCopy(final JaxoObject comp) {
        boolean isCopy = false;
        if (comp instanceof JaxoVertexT4) {
            isCopy = super.isCopy(comp);
        }

        return isCopy;
    }

    /** {@inheritDoc} */
    @Override
    public boolean canBeSelected(final int handle, final int mode) {
        boolean active = ((handle == SELECT_P1) || (handle == SELECT_P2));
        if ((mode == JaxoConstants.RESIZE)) {
            active = (handle == SELECT_P2);
        }
        return active;
    }

    /** {@inheritDoc} */
    @Override
    public void paintHandles(final JaxoGraphics2D g2, final JaxoHandle h, final int editMode) {
        if (editMode == JaxoConstants.UNGROUP) {
            return;
        }

        h.paint(g2, getX(), getY(), isMarked(), !canBeSelected(SELECT_P1, editMode));
        h.paint(g2, getX2(), getY2(), isMarked(), !canBeSelected(SELECT_P2, editMode));
    }

    /** {@inheritDoc} */
    public final void paint(final JaxoGraphics2D g2) {
        g2.setColor(getColor());
        g2.setStroke(getStroke());
        g2.draw(getObjectPath());
    }

    /**
     * Returns the bounding box of this object.
     *
     * @return the bounding box of this object.
     */
    public Rectangle getBounds() {
        return getStroke().createStrokedShape(getObjectPath()).getBounds();
    }

    /** {@inheritDoc} */
    public final String latexCommand(final float scale, final Dimension canvasDim) {
        final float radius = getLaTexRadius(scale);

        if ((int) radius == 0) {
            return "%";
        }

        final Point2D center = getLaTexCenter(scale, canvasDim.height);

        final Point2D invec1 =
            new Point2D.Float((float) (center.getX() - (radius * 0.707f)),
                (float) (center.getY() - (radius * 0.707f)));
        final Point2D finvec1 =
            new Point2D.Float((float) (center.getX() + (radius * 0.707f)),
                (float) (center.getY() + (radius * 0.707f)));
        final Point2D invec2 =
            new Point2D.Float((float) (center.getX() - (radius * 0.707f)),
                (float) (center.getY() + (radius * 0.707f)));
        final Point2D finvec2 =
            new Point2D.Float((float) (center.getX() + (radius * 0.707f)),
                (float) (center.getY() - (radius * 0.707f)));

        final double theta = (Math.PI / 4) - Math.atan2(getRelh(), getRelw());

        final AffineTransform at = new AffineTransform();
        at.rotate(theta, center.getX(), center.getY());

        final Point2D invect1 = at.transform(invec1, null);
        final Point2D finvect1 = at.transform(finvec1, null);
        final Point2D invect2 = at.transform(invec2, null);
        final Point2D finvect2 = at.transform(finvec2, null);

        final String command1 =
            "\\Line" + "(" + D_FORMAT.format(invect1.getX()) + ","
            + D_FORMAT.format(invect1.getY()) + ")" + "("
            + D_FORMAT.format(finvect1.getX()) + ","
            + D_FORMAT.format(finvect1.getY()) + ")";

        final String command2 =
            "\\Line" + "(" + D_FORMAT.format(invect2.getX()) + ","
            + D_FORMAT.format(invect2.getY()) + ")" + "("
            + D_FORMAT.format(finvect2.getX()) + ","
            + D_FORMAT.format(finvect2.getY()) + ")";

        return command1 + command2;
    }

    /** {@inheritDoc} */
    public void prepareEditPanel(final JaxoObjectEditPanel editPanel) {
        final int r = (int) Math.round(getRadius());

        editPanel.addXYRPanel(getX(), getY(), r, 0, 0);
        editPanel.addStrokePanel(getStrokeWidth(), 0, 1);
        editPanel.addColorPanel(getColor(), JaxoObjectEditPanel.TYPE_LINE_COLOR, 1, 0);
        editPanel.addRotationPanel(getRotationAngle(), 1, 1);

        editPanel.setTitleAndIcon("Vertex_parameters", "vertexT4.png");
    }

    private GeneralPath getObjectPath() {
        final GeneralPath gp = getGeneralPath();
        gp.reset();

        final int sepx = getRelh();
        final int sepy = getRelw();

        gp.moveTo(getX() + sepx, getY() - sepy);
        gp.lineTo(getX() - sepx, getY() + sepy);
        gp.moveTo(getX() - sepy, getY() - sepx);
        gp.lineTo(getX() + sepy, getY() + sepx);

        return gp;
    }
}
