/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.object.vertex;

import java.awt.geom.Point2D;

import net.sf.jaxodraw.object.JaxoExtendedObject;
import net.sf.jaxodraw.object.JaxoObject;
import net.sf.jaxodraw.object.JaxoObjectEditPanel;
import net.sf.jaxodraw.object.JaxoRotationObject;
import net.sf.jaxodraw.util.JaxoPrefs;
import net.sf.jaxodraw.util.JaxoGeometry;
import net.sf.jaxodraw.util.graphics.JaxoGraphics2D;


/** A vertex object.
 * @since 2.0
 */
public abstract class JaxoVertex extends JaxoExtendedObject
    implements JaxoRotationObject {

    private static final long serialVersionUID = 2L;

    private static final int LP4 = 12;

    /**
     * Returns the rotation angle of this vertex.
     *
     * @return The rotation angle of this vertex (in degrees).
     */
    public final int getRotationAngle() {
        final double theta = Math.atan2((float) getRelh(), (float) getRelw());
        return ((int) Math.round(Math.toDegrees(theta)));
    }

    /**
     * Sets the rotation angle of this vertex, this only acts on P2.
     *
     * @param angle The rotation angle of this vertex (in degrees).
     */
    public final void setRotationAngle(final int angle) {
        final double radAngle = Math.toRadians(angle);
        final double length =
            Math.sqrt((getWidth() * getWidth()) + (getHeight() * getHeight()));
        final int newRelW = (int) Math.round(length * Math.cos(radAngle));
        final int newRelH = (int) Math.round(length * Math.sin(radAngle));
        setRelWAndH(newRelW, newRelH);
    }

    /** Returns an exact copy of this JaxoVertex.
     * @param temp An instance of the vertex to copy to.
     * @return A copy of this JaxoVertex.
     */
    public final JaxoObject copy(final JaxoVertex temp) {
        temp.copyFrom(this);

        return temp;
    }

    /** {@inheritDoc} */
    @Override
    public boolean isCopy(final JaxoObject comp) {
        boolean isCopy = false;

        if (comp instanceof JaxoVertex) {
            isCopy = super.isCopy(comp);
        }

        return isCopy;
    }

    /** {@inheritDoc} */
    @Override
    public final void paintVisualAid(final JaxoGraphics2D g2) {
        final int x = getX();
        final int y = getY();

        g2.drawLine(x - LP4, y, x + LP4, y);
        g2.drawLine(x, y - LP4, x, y + LP4);
    }

    /** {@inheritDoc} */
    public final void rescaleObject(final int orx, final int ory, final float scale) {
        final int newRelWidth = Math.round(this.getRelSize().width * scale);
        final int newRelHeight = Math.round(this.getRelSize().height * scale);

        final Point2D newP =
            JaxoGeometry.scaledPoint(orx, ory, scale, getX(), getY());

        setLocation((int) Math.round(newP.getX()),
            (int) Math.round(newP.getY()));
        this.setX2(getX() + newRelWidth);
        this.setY2(getY() + newRelHeight);

        //this.setRelWAndH(newRelWidth, newRelHeight);
    }

    /** Returns a center point which is used by the latexCommand method.
     * @param scaleFactor A scale factor.
     * @param canvasHeight The height of the current canvas.
     * @return A Point2D object holding the center point.
     */
    protected Point2D getLaTexCenter(final float scaleFactor, final int canvasHeight) {
        return getLaTexLocation(scaleFactor, canvasHeight);
    }

    /** Returns a radius which is used by the latexCommand method.
     * @param scaleFactor A scale factor.
     * @return The latex radius.
     */
    protected float getLaTexRadius(final float scaleFactor) {
        return (float) Math.sqrt(((double) getWidth() * (double) getWidth())
                + ((double) getHeight() * (double) getHeight())) / scaleFactor;
    }

    /** {@inheritDoc} */
    @Override
    public void setPreferences() {
        super.setPreferences();
        final int size = JaxoPrefs.getIntPref(JaxoPrefs.PREF_VERTEXSIZE);
        if (size > 0) {
            setRelWAndH(size, size);
        }
    }

    /**
     * Prepares a simple edit panel to edit coordinates and color.
     *
     * @param editPanel the panel to prepare.
     * @param iconName the name of the icon for the panel.
     */
    protected void prepareGenericPanel(final JaxoObjectEditPanel editPanel, final String iconName) {
        final int r = (int) Math.round(getRadius());

        editPanel.addXYRPanel(getX(), getY(), r, 0, 0);
        editPanel.addColorPanel(getColor(), JaxoObjectEditPanel.TYPE_LINE_COLOR, 1, 0);

        editPanel.setTitleAndIcon("Vertex_parameters", iconName);
    }
}
