/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.object.line;

import java.awt.BasicStroke;
import java.awt.geom.GeneralPath;

import java.io.IOException;
import java.io.ObjectInputStream;

import net.sf.jaxodraw.util.JaxoUtils;


/**
 * A dashed line.
 *
 * @since 2.0
 */
public abstract class JaxoDashLine extends JaxoLineObject {
    private static final long serialVersionUID = 314159L;

    private void readObject(final ObjectInputStream in)
        throws IOException, ClassNotFoundException {
        in.defaultReadObject();
        resetStroke();
    }

    /** {@inheritDoc} */
    @Override
    protected void resetStroke() {
        if (JaxoUtils.zero(getStrokeWidth()) || JaxoUtils.zero(getDash())) {
            // during initialization, either of them is still null
            // so wait until both are set
            return;
        }

        setStroke(new BasicStroke(getStrokeWidth(), BasicStroke.CAP_ROUND,
                BasicStroke.JOIN_MITER, 10.0f, strokeDashes(), 0.0f));
    }

    /** {@inheritDoc} */
    protected GeneralPath getObjectPath() {
        final GeneralPath gp = getGeneralPath();
        gp.reset();
        if (isDoubleLine()) {
            final float length = (float) getRadius();
            final float sepx = getRelh() / length * getDLSeparation() / 2.f;
            final float sepy = getRelw() / length * getDLSeparation() / 2.f;
            gp.moveTo(getX() - sepx, getY() + sepy);
            gp.lineTo(getX2() - sepx, getY2() + sepy);
            gp.moveTo(getX() + sepx, getY() - sepy);
            gp.lineTo(getX2() + sepx, getY2() - sepy);
        } else {
            gp.moveTo(getX(), getY());
            gp.lineTo(getX2(), getY2());
        }
        return gp;
    }

    /**
     * Returns an array that is used as the dash parameter in
     * {@link java.awt.BasicStroke} to paint this object.
     * @return a dash array.
     */
    protected abstract float[] strokeDashes();

    // Get the axodraw4j options set for this line

    /** {@inheritDoc} */
    protected String getAxo4JOptions(final float scale) {

        final float dashSize = getDash() / scale;
        final StringBuffer optioncmd = new StringBuffer("dash,dashsize=").append(
                    D_FORMAT.format(dashSize));

        if (isPaintArrow()) {
            final float arpos = getArrowPosition();
            final StringBuffer arrowcmd = new StringBuffer(this.getArrow().latexCommand(arpos, scale));

            if (isFlip()) {
                arrowcmd.append(",flip");
            }

            optioncmd.append(',').append(arrowcmd);

            if (isDoubleLine()) {
                optioncmd.append(",double,sep=").append(
                    D_FORMAT.format(this.getDLSeparation()));
            }

        } else if (isDoubleLine()) {
            optioncmd.append(",double,sep=").append(
                D_FORMAT.format(this.getDLSeparation()));
        }

        optioncmd.insert(0, '[').append(']');

        return optioncmd.toString();
    }
}
